/*
 *
 * Copyright (C) 2002 George Staikos <staikos@kde.org>
 *               2004 Dirk Ziegelmeier <dziegel@gmx.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


#include <qimage.h>
#include <qwidget.h>
#include <qtimer.h>
#include <klocale.h>
#include "kdetv_xv.h"
#include <kdebug.h>
#include <math.h>
#include <assert.h>
#include <qapplication.h>

#include <kscreensaver_vroot.h>


KdetvXv::KdetvXv(Kdetv *ktv, QWidget *parent, const char* name)
    : KdetvSourcePlugin(ktv, "xv",parent,name)
{
    xvDevice = NULL;
    xvHandle = NULL;
    kdDebug() << "Kdetv XVideo plugin loaded successfully." << endl;

    resizeTimer = new QTimer();
    connect(resizeTimer, SIGNAL( timeout() ),
            this, SLOT( startVideo() ));

    connect( parent, SIGNAL( resized(int, int) ),
             this, SLOT( viewResized() ) );
}


KdetvXv::~KdetvXv()
{
    delete resizeTimer;
    stopVideo();
    delete xvHandle;
    kdDebug() << "Kdetv XVideo plugin unloaded." << endl;
}


bool KdetvXv::isTuner()
{
	if (xvDevice) {
		return xvDevice->getAttribute("XV_FREQ", NULL);
	}
    return false;
}


#define my_round(x) ((x)-int(x) > 0.50 ? int(x)+1 : int(x))

int KdetvXv::brightness()
{
	if (xvDevice) {
		int val, minv, maxv;
		xvDevice->getAttributeRange("XV_BRIGHTNESS", &minv, &maxv);
		return 65535
            * (xvDevice->getAttribute("XV_BRIGHTNESS", &val) ? val-minv : 0)
            / (maxv - minv);
	}
    
    return -1;
}


QColor KdetvXv::colourKey()
{
	if (xvDevice) {
		int val = 0;
		xvDevice->getAttribute("XV_COLORKEY", &val);
		XColor a;
		memset(&a, 0, sizeof(a));
		a.pixel = val;
		XQueryColor(qt_xdisplay(), 
#if QT_VERSION >= 0x030100
                    QPaintDevice::x11AppColormap( qt_xscreen() ),
#else
                    QPaintDevice::x11AppColormap(),
#endif
                    &a);
		return QColor(QColor(a.red, a.green, a.blue).rgb(), val);
	}
    
    return QColor();
}


int KdetvXv::colour()
{
	if (xvDevice) {
		int val, minv, maxv;
		xvDevice->getAttributeRange("XV_SATURATION", &minv, &maxv);
		return 65535
            * (xvDevice->getAttribute("XV_SATURATION", &val) ? val-minv : 0)
            / (maxv - minv);
	}
    
    return -1;
}


int KdetvXv::hue()
{
	if (xvDevice) {
		int val, minv, maxv;
		xvDevice->getAttributeRange("XV_HUE", &minv, &maxv);
		return 65535
            * (xvDevice->getAttribute("XV_HUE", &val) ? val-minv : 0)
            / (maxv - minv);
	}
    
    return -1;
}


int KdetvXv::contrast()
{
	if (xvDevice) {
		int val, minv, maxv;
		xvDevice->getAttributeRange("XV_CONTRAST", &minv, &maxv);
		return 65535
            * (xvDevice->getAttribute("XV_CONTRAST", &val) ? val-minv : 0)
            / (maxv - minv);
	}
    
    return -1;
}


int KdetvXv::whiteness()
{
    return 0; // unsupported
}


int KdetvXv::signal()
{
	if (xvDevice) {
		int v;
		if (xvDevice->getAttribute("XV_SIGNAL", &v))
			return v;
	}
    
    return -1;
}


int KdetvXv::frequency()
{
	if (xvDevice) {
		int v;
		if (xvDevice->getAttribute("XV_FREQ", &v))
			return v*125/2;
	}
    
    return -1;
}


/* ARGH, this doesn't work on my card.  It always returns 0! */
bool KdetvXv::muted()
{
	if (xvDevice) {
		int v;
		if (xvDevice->getAttribute("XV_MUTE", &v)) {
			kdDebug() << "XV_MUTE == " << v << endl;
			return (v==1);
		}
	}
    
    return false;
}


const QStringList& KdetvXv::broadcastedAudioModes()
{
    static QStringList empty;
    return empty;
}


const QString& KdetvXv::defaultAudioMode()
{
    return QString::null;
}


int KdetvXv::setAudioMode( const QString& )
{
    return -1;
}


int KdetvXv::doSetEncoding( const QString &encoding, const QString &source )
{
	if (_device.isEmpty() || !xvDevice)
		return -1;
    
	// This works together with parseXvEncoding. If XV encoding does not contain
	// "-", the whole string is stored in encoding attribute and source remains empty.
    QString enc;
	if (source.isEmpty()) {
        enc = encoding;
	} else {
        enc = encoding + "-" + source;
	}
    
	kdDebug() << "KdetvXv: doSetEncoding: " << enc << endl;
    
	if (xvDevice->encodings().contains(enc)) {
		xvDevice->setEncoding(enc);
		return 0;
	}
    
	return -2;
}


void KdetvXv::parseXvEncoding( const QString &xvEncoding, QString &source, QString &encoding )
{
    if (xvEncoding.contains("-")) {
        source   = xvEncoding.section("-", 1, 1);
		encoding = xvEncoding.section("-", 0, 0);
	} else {
		source   = QString::null;
        encoding = xvEncoding;
	}
}


int KdetvXv::setDevice( const QString& dev )
{
	KXvDeviceList& xvdl(xvHandle->devices());
	KXvDevice *xvdev = NULL;
	QString encoding;
    
	kdDebug() << "KdetvXv: setDevice: " << dev << endl;
	for (xvdev = xvdl.first(); xvdev; xvdev = xvdl.next()) {
		QString label = i18n("%1 - XVideo port %2").arg(xvdev->name()).arg(xvdev->port());
		if (dev == label) {
			stopVideo();
			xvDevice = xvdev;
			_device  = dev;
			// init _src and _encoding members and set default source/encoding
			_encoding = _encodings[dev].first();
			setSource(_sources[dev].first());
			return 0;
		}
	}

    return -1;
}


int KdetvXv::setSource( const QString &src )
{
	kdDebug() << "KdetvXv: setSource: " << src << endl;
    
    int rc = doSetEncoding(_encoding, src);
    
	if (rc == 0) {
        _source = src;
	}
	return rc;
}


int KdetvXv::setEncoding( const QString &encoding )
{
	kdDebug() << "KdetvXv: setEncoding: " << encoding << endl;
    
    int rc = doSetEncoding(encoding, _source);
    
	if (rc == 0) {
        _encoding = encoding;
	}
	return rc;
}


void KdetvXv::setFrequency( int freq )
{
	if (xvDevice && isTuner()) {
		xvDevice->setAttribute("XV_FREQ", freq*2/125);
		// I hate this hack.  How do we do this right?
	}
}


void KdetvXv::setMuted( bool muted )
{
	if (xvDevice) {
		xvDevice->setAttribute("XV_MUTE", muted ? 1 : 0);
	}
}


#define SCALE_FACTOR float factor = float(val)*float(maxv-minv)/float(65535) + float(minv)

void KdetvXv::setBrightness(int val)
{
	if (xvDevice) {
		int minv, maxv;
		xvDevice->getAttributeRange("XV_BRIGHTNESS", &minv, &maxv);
		SCALE_FACTOR;
		xvDevice->setAttribute("XV_BRIGHTNESS", my_round(factor));
	}
}


void KdetvXv::setColour(int val)
{
	if (xvDevice) {
		int minv, maxv;
		xvDevice->getAttributeRange("XV_SATURATION", &minv, &maxv);
		SCALE_FACTOR;
		xvDevice->setAttribute("XV_SATURATION", my_round(factor));
	}
}


void KdetvXv::setHue(int val)
{
	if (xvDevice) {
		int minv, maxv;
		xvDevice->getAttributeRange("XV_HUE", &minv, &maxv);
		SCALE_FACTOR;
		xvDevice->setAttribute("XV_HUE", my_round(factor));
	}
}


void KdetvXv::setContrast(int val)
{
	if (xvDevice) {
		int minv, maxv;
		xvDevice->getAttributeRange("XV_CONTRAST", &minv, &maxv);
		SCALE_FACTOR;
		xvDevice->setAttribute("XV_CONTRAST", my_round(factor));
	}
}

#undef SCALE_FACTOR

void KdetvXv::setWhiteness(int)
{
    //unsupported
}


void KdetvXv::viewResized()
{
    resizeTimer->start(100, true);
}
    

int KdetvXv::probeDevices()
{
	KXvDeviceList& xvdl(xvHandle->devices());
	KXvDevice *xvdev = NULL;
	QStringList enc;
    
    
	kdDebug() << "KdetvXv: probeDevices" << endl;
    
	_devices.clear();
	_sources.clear();
	_encodings.clear();
	_tuners.clear();

	for (xvdev = xvdl.first(); xvdev; xvdev = xvdl.next()) {
		if (xvdev->isVideoSource() && xvdev->supportsWidget(_widget)) {
			QString label = i18n("%1 - XVideo port %2").arg(xvdev->name()).arg(xvdev->port());
			kdDebug() << "KdetvXv: found device: " << label << endl;
			_devices << label;
            
			enc = xvdev->encodings();
			for (QStringList::Iterator it = enc.begin(); it != enc.end(); it++) {
                QString source;
				QString encoding;
				parseXvEncoding(*it, source, encoding);
				if (_sources[label].contains(source) == 0)
                    _sources[label].append(source);
				if (_encodings[label].contains(encoding) == 0)
                    _encodings[label].append(encoding);
			}
            
			_tuners[label] = xvdev->getAttribute("XV_FREQ", NULL);
		}
	}
    return 0;
}


int KdetvXv::startVideo()
{
	assert(_widget);

	if (!xvDevice || _isVideoDesktop)
		return -1;

	bool rc = xvDevice->startVideo(_widget, _widget->width(), _widget->height());
	setMuted(true); // work around xv driver bug
	setMuted(false);

    if (!rc) {
		kdDebug() << "Error starting video in Xv plugin!" << endl;
        emit errorMessage("Unable to start video playback.\n\
                           Video playback may not be possible for the current device with the XVIDEO plugin.");
        stopVideo();
        return -2;
    } else {
        return 0;
    }
}


int KdetvXv::stopVideo()
{
	setMuted(true);
	if (!xvDevice || _isVideoDesktop)
		return -1;
	return xvDevice->stopVideo() ? 0 : -1;
}


bool KdetvXv::canVideoDesktop() const
{
    return true;
}


int KdetvXv::setVideoDesktop(bool on)
{
	if (!xvDevice)
		return -1;
    
	if (on) {
		Window w = DefaultRootWindow(qt_xdisplay());
		const QRect& s = QApplication::desktop()->screenGeometry();
		stopVideo();
		bool rc = xvDevice->startVideo(w, s.width(), s.height());
		setMuted(false);
		_isVideoDesktop = true;
		if (rc != true) 
            return -1;
	} else {
		if (!_isVideoDesktop)
			return -1;
		_isVideoDesktop = false;
		xvDevice->stopVideo();
		setMuted(true);
		// Window w = DefaultRootWindow(qt_xdisplay());
		// FIXME: force a repaint of w.
		return startVideo();
	}
    return 0;
}


void KdetvXv::setFullscreen(bool)
{
}


bool KdetvXv::videoPlaying() const
{
	if (xvDevice)
		return xvDevice->videoPlaying();
	return false;
}


bool KdetvXv::canGrabStill() const
{
    return true;
}


bool KdetvXv::grabStill( QImage *pix )
{
    if (pix && xvDevice)
        return xvDevice->grabStill( pix, pix->width(), pix->height() );
    return false;
}


extern "C" {
    KdetvXv* create_xv(Kdetv *ktv, QWidget *w)
    {
        Drawable d;
        KXv *xv;
        KdetvXv *qv;
        
        if (!w || !KXv::haveXv())
            return NULL;
        
        d = w->winId();
        
        xv = KXv::connect(d);
        if (!xv)
            return NULL;
        
        qv = new KdetvXv(ktv, w);
        qv->xvHandle = xv;
        qv->setWidget(w);
        
        return qv;
    }
}

#include "kdetv_xv.moc"

