// -*- c++ -*-

/*
 *
 * Copyright (C) 2002 Richard Moore <rich@kde.org>
 * Copyright (C) 2002 George Staikos <staikos@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <klocale.h>
#include <qdom.h>
#include <qfile.h>
#include <qtextstream.h>

#include <kdebug.h>
#include <ksavefile.h>

#include "channel.h"
#include "channelstore.h"

#include "channelioxml.h"

//
// XML Format Handler
//

static const int CHANNELIO_XML_FORMAT_VERSION = 1;

ChannelIOFormatXML::ChannelIOFormatXML(Kdetv *ktv, QObject *parent, const char* name)
    : KdetvChannelPlugin(ktv, "XML Channels", parent, name)
{
    _fmtName  = "xml";
    _menuName = i18n("kdetv XML");
    _flags    = FormatRead|FormatWrite;
}

//
// XML Loading
//

bool ChannelIOFormatXML::load( ChannelStore *store, QIODevice *file, const QString& /*fmt*/ )
{
    kdDebug() << "IOFormatXML::load(...)" << endl;

    if ( !doc.setContent( file ) )
        return false;

    return readDocument( store );
}

bool ChannelIOFormatXML::readDocument( ChannelStore *store )
{
    kdDebug() << "IOFormatXML::readDocument(...)" << endl;

    this->store = store;
    QDomElement root = doc.documentElement();
    if ( root.tagName() != "kwintv" )
        return false;

    kdDebug() << "       Found a kdetv channel file" << endl;

    QDomNode n = root.firstChild();
    if ( n.isNull() || (!n.isElement()) )
        return false;

    root = n.toElement();
    if ( root.tagName() != "tvregion" )
        return false;

    kdDebug() << "       Found a region in the channel file" << endl;

    // Read in the region
    for (QDomNode n = root.firstChild(); !n.isNull() && n.isElement();
         n = n.nextSibling()) {

        root = n.toElement();
        kdDebug() << "     Found " << root.tagName() << endl;

        if ( root.tagName() == "info" ) {
            kdDebug() << "       Found the metainfo" << endl;
            for (QDomNode m = root.firstChild(); !m.isNull() && m.isElement();
                 m = m.nextSibling()) {
                QDomElement e = m.toElement();
                if (e.tagName() == "name") {
                    _metaInfo.setName(e.text());
                } else if (e.tagName() == "format") {
                    _metaInfo.setNorm(e.text());
                }
            }
        } else if ( root.tagName() == "channels" ) {
            kdDebug() << "       Found a block of channels" << endl;

            QDomNode z = n.firstChild();
            while( (!z.isNull()) && z.isElement() ) {
                QDomElement e = z.toElement();
                Channel *ch = readChannel( e );
                if ( !ch ) {
                    kdDebug() << "        Error reading channel" << endl;
                    break;
                }
                store->addChannel( ch );

                z = z.nextSibling();
            }
        }
    }
    return true;
}

KdetvChannelPlugin::ChannelFileMetaInfo ChannelIOFormatXML::getMetaInfo( QIODevice *file, const QString&)
{
    QDomDocument mydoc;
    KdetvChannelPlugin::ChannelFileMetaInfo x;
    kdDebug() << "IOFormatXML::getMetaInfo(...)" << endl;

    if ( !mydoc.setContent( file ) )
        return x;

    QDomElement root = mydoc.documentElement();
    if ( root.tagName() != "kwintv" )
        return x;

    kdDebug() << "       Found a kdetv channel file" << endl;

    QDomNode n = root.firstChild();
    if ( n.isNull() || (!n.isElement()) )
        return x;

    root = n.toElement();
    if ( root.tagName() != "tvregion" )
        return x;

    kdDebug() << "       Found a region in the channel file" << endl;

    // Read in the region
    for (QDomNode n = root.firstChild(); !n.isNull() && n.isElement();
         n = n.nextSibling()) {

        root = n.toElement();
        kdDebug() << "     Found " << root.tagName() << endl;

        if ( root.tagName() == "info" ) {
            kdDebug() << "       Found the metainfo" << endl;
            for (QDomNode m = root.firstChild(); !m.isNull() && m.isElement();
                 m = m.nextSibling()) {
                QDomElement e = m.toElement();
                if (e.tagName() == "name") {
                    x.setName(e.text());
                } else if (e.tagName() == "format") {
                    x.setNorm(e.text());
                }
            }
        }
    }

    return x;
}


Channel *ChannelIOFormatXML::readChannel( const QDomElement &elem )
{
    if ( elem.tagName() != "channel" ) {
        kdDebug() << "Error: tried to read " << elem.tagName()
                  << " where we expected a channel." << endl;
        return 0;
    }

    Channel *ch = new Channel( store );
    ch->setEnabled(readAttrBool(elem, "enabled"));
    for (QDomNode n = elem.firstChild(); !n.isNull() && n.isElement();
         n = n.nextSibling()) {
        QDomElement z = n.toElement();
        if (z.tagName() == "name")
            ch->setName(readText( z, "name" ));
        else if (z.tagName() == "source") 
            ch->setSource(readText( z, "source" ));
        else if (z.tagName() == "encoding") 
            ch->setEncoding(readText( z, "encoding" ));
        else if (z.tagName() == "number") 
            ch->setNumber(readTextInt( z, "number" ));
        else if (z.tagName() == "frequency") 
            ch->setFreq(readTextULong( z, "frequency" ));
    }

    //kdDebug() << "Found Channel: '" << ch->name() << "', " << ch->number() << ", " << ch->freq() << ", " << ch->enabled() << endl;
    return ch;
}

int ChannelIOFormatXML::readTextInt( const QDomElement &elem, const QString &tag )
{
    QString val = readText( elem, tag );
    if ( val.isNull() )
        return 0;

    return val.toInt();
}

unsigned long ChannelIOFormatXML::readTextULong( const QDomElement &elem, const QString &tag )
{
    QString val = readText( elem, tag );
    if ( val.isNull() )
        return 0;

    return val.toULong();
}

QString ChannelIOFormatXML::readText( const QDomElement &elem, const QString &tag )
{
    if ( elem.tagName() != tag )
        return QString::null;

    return elem.text();
}


bool ChannelIOFormatXML::readAttrBool(const QDomElement &elem, const QString &name)
{
    QString val = elem.attribute(name, "true");

    if (val == "true")
        return(true);
    else if (val == "false")
        return(false);

    return false;
} // readBool

//
// XML Saving
//

bool ChannelIOFormatXML::save( ChannelStore *store, QIODevice *file, const QString& /*fmt*/ )
{
    kdDebug() << "IOFormatXML::save(...)" << endl;

    QTextStream ts( file );
    writeDocument( store );

    ts << doc.toString().utf8();

    return true;
}

void ChannelIOFormatXML::writeDocument( ChannelStore *store )
{
    this->store = store;
    
    doc = QDomDocument(); // Reset the document
    QDomProcessingInstruction instr = doc.createProcessingInstruction("xml","version=\"1.0\" encoding=\"UTF-8\"");
    doc.appendChild(instr);

    QDomElement root = writeElement( doc, "kwintv" );
    root.setAttribute( "version", QString::number( CHANNELIO_XML_FORMAT_VERSION ) );

    QDomElement elem = writeElement( root, "tvregion" );

    QDomElement ie = writeElement(elem, "info");
    writeText(ie, "name", _metaInfo.name());
    writeText(ie, "norm", _metaInfo.norm());

    QDomElement elem2 = writeElement( elem, "channels" );
    for (uint i = 0; i < store->count(); i++)
        writeChannel( elem2, store->channelAt(i) );
}

QDomElement ChannelIOFormatXML::writeChannel( QDomElement &parent, Channel *ch )
{
    QDomElement channel = writeElement( parent, "channel" );
    writeAttrBool( channel, "enabled", ch->enabled() );
    writeText( channel, "name", ch->name() );
    writeText( channel, "encoding", ch->encoding() );
    writeText( channel, "source", ch->source() );
    writeTextInt( channel, "number", ch->number() );
    writeTextULong( channel, "frequency", ch->freq() );

    return channel;
}

QDomElement ChannelIOFormatXML::writeTextInt( QDomElement &parent, const QString &tag, int text )
{
    return writeText( parent, tag, QString("%1").arg(text) );
}

QDomElement ChannelIOFormatXML::writeTextULong( QDomElement &parent, const QString &tag, unsigned long text )
{
    return writeText( parent, tag, QString("%1").arg(text) );
}

QDomElement ChannelIOFormatXML::writeText( QDomElement &parent, const QString &tag, const QString &text )
{
    QDomElement elem = writeElement( parent, tag );
    QDomText t = doc.createTextNode( text ); 
    elem.appendChild( t );

    return elem;
}


void ChannelIOFormatXML::writeAttrBool(QDomElement &parent, const QString &name, const bool val)
{
    if (val == true)
        parent.setAttribute(name, "true");
    else if (val == false)
        parent.setAttribute(name, "false");
} // writeBool


QDomElement ChannelIOFormatXML::writeElement( QDomNode &parent, const QString &tag )
{
    QDomElement elem = doc.createElement( tag );
    parent.appendChild( elem );
    return elem;
}

extern "C" {
	ChannelIOFormatXML* create_xmlchannels(Kdetv *ktv)
    {
		return new ChannelIOFormatXML(ktv, 0, "XML Channel Plugin");
	}
}

