"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.GDBBackend = void 0;
/*********************************************************************
 * Copyright (c) 2018 QNX Software Systems and others
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *********************************************************************/
const child_process_1 = require("child_process");
const events = require("events");
const logger_1 = require("@vscode/debugadapter/lib/logger");
const mi = require("./mi");
const MIParser_1 = require("./MIParser");
const varManager_1 = require("./varManager");
const util_1 = require("./util");
class GDBBackend extends events.EventEmitter {
    constructor() {
        super(...arguments);
        this.parser = new MIParser_1.MIParser(this);
        this.varMgr = new varManager_1.VarManager(this);
        this.token = 0;
        this.gdbAsync = false;
        this.gdbNonStop = false;
        this.hardwareBreakpoint = false;
    }
    get varManager() {
        return this.varMgr;
    }
    spawn(requestArgs) {
        return __awaiter(this, void 0, void 0, function* () {
            const gdbPath = requestArgs.gdb || 'gdb';
            this.gdbVersion = yield (0, util_1.getGdbVersion)(gdbPath, (0, util_1.getGdbCwd)(requestArgs), requestArgs.environment);
            let args = ['--interpreter=mi2'];
            if (requestArgs.gdbArguments) {
                args = args.concat(requestArgs.gdbArguments);
            }
            const gdbEnvironment = requestArgs.environment
                ? (0, util_1.createEnvValues)(process.env, requestArgs.environment)
                : process.env;
            this.proc = (0, child_process_1.spawn)(gdbPath, args, {
                cwd: (0, util_1.getGdbCwd)(requestArgs),
                env: gdbEnvironment,
            });
            if (this.proc.stdin == null || this.proc.stdout == null) {
                throw new Error('Spawned GDB does not have stdout or stdin');
            }
            this.out = this.proc.stdin;
            this.hardwareBreakpoint = requestArgs.hardwareBreakpoint ? true : false;
            yield this.parser.parse(this.proc.stdout);
            if (this.proc.stderr) {
                this.proc.stderr.on('data', (chunk) => {
                    const newChunk = chunk.toString();
                    this.emit('consoleStreamOutput', newChunk, 'stderr');
                });
            }
            yield this.setNonStopMode(requestArgs.gdbNonStop);
            yield this.setAsyncMode(requestArgs.gdbAsync);
        });
    }
    spawnInClientTerminal(requestArgs, cb) {
        return __awaiter(this, void 0, void 0, function* () {
            const gdbPath = requestArgs.gdb || 'gdb';
            this.gdbVersion = yield (0, util_1.getGdbVersion)(gdbPath, (0, util_1.getGdbCwd)(requestArgs), requestArgs.environment);
            // Use dynamic import to remove need for natively building this adapter
            // Useful when 'spawnInClientTerminal' isn't needed, but adapter is distributed on multiple OS's
            const { Pty } = yield Promise.resolve().then(() => require('./native/pty'));
            const pty = new Pty();
            let args = [gdbPath, '-ex', `new-ui mi2 ${pty.slave_name}`];
            if (requestArgs.gdbArguments) {
                args = args.concat(requestArgs.gdbArguments);
            }
            yield cb(args);
            this.out = pty.writer;
            yield this.parser.parse(pty.reader);
            yield this.setNonStopMode(requestArgs.gdbNonStop);
            yield this.setAsyncMode(requestArgs.gdbAsync);
        });
    }
    setAsyncMode(isSet) {
        return __awaiter(this, void 0, void 0, function* () {
            const command = this.gdbVersionAtLeast('7.8')
                ? 'mi-async'
                : 'target-async';
            if (isSet === undefined) {
                isSet = true;
            }
            if (this.gdbNonStop) {
                isSet = true;
            }
            const onoff = isSet ? 'on' : 'off';
            try {
                yield this.sendCommand(`-gdb-set ${command} ${onoff}`);
                this.gdbAsync = isSet;
            }
            catch (_a) {
                // no async support - normally this only happens on Windows
                // when doing host debugging. We explicitly set this
                // to off here so that we get the error propogate if the -gdb-set
                // failed and to make it easier to read the log
                yield this.sendCommand(`-gdb-set ${command} off`);
                this.gdbAsync = false;
            }
        });
    }
    getAsyncMode() {
        return this.gdbAsync;
    }
    setNonStopMode(isSet) {
        return __awaiter(this, void 0, void 0, function* () {
            if (isSet === undefined) {
                isSet = false;
            }
            if (isSet) {
                yield this.sendCommand('-gdb-set pagination off');
            }
            const onoff = isSet ? 'on' : 'off';
            try {
                yield this.sendCommand(`-gdb-set non-stop ${onoff}`);
                this.gdbNonStop = isSet;
            }
            catch (_a) {
                // no non-stop support - normally this only happens on Windows.
                // We explicitly set this to off here so that we get the error
                // propogate if the -gdb-set failed and to make it easier to
                // read the log
                yield this.sendCommand(`-gdb-set non-stop off`);
                this.gdbNonStop = false;
            }
        });
    }
    isNonStopMode() {
        return this.gdbNonStop;
    }
    // getBreakpointOptions called before inserting the breakpoint and this
    // method could overridden in derived classes to dynamically control the
    // breakpoint insert options. If an error thrown from this method, then
    // the breakpoint will not be inserted.
    getBreakpointOptions(_, initialOptions) {
        return __awaiter(this, void 0, void 0, function* () {
            return initialOptions;
        });
    }
    isUseHWBreakpoint() {
        return this.hardwareBreakpoint;
    }
    pause(threadId) {
        if (this.gdbAsync) {
            mi.sendExecInterrupt(this, threadId);
        }
        else {
            if (!this.proc) {
                throw new Error('GDB is not running, nothing to interrupt');
            }
            logger_1.logger.verbose(`GDB signal: SIGINT to pid ${this.proc.pid}`);
            this.proc.kill('SIGINT');
        }
    }
    supportsNewUi(gdbPath, gdbCwd, environment) {
        return __awaiter(this, void 0, void 0, function* () {
            this.gdbVersion = yield (0, util_1.getGdbVersion)(gdbPath || 'gdb', gdbCwd, environment);
            return this.gdbVersionAtLeast('7.12');
        });
    }
    gdbVersionAtLeast(targetVersion) {
        if (!this.gdbVersion) {
            throw new Error('gdbVersion needs to be set first');
        }
        return (0, util_1.compareVersions)(this.gdbVersion, targetVersion) >= 0;
    }
    sendCommands(commands) {
        return __awaiter(this, void 0, void 0, function* () {
            if (commands) {
                for (const command of commands) {
                    yield this.sendCommand(command);
                }
            }
        });
    }
    sendCommand(command) {
        const token = this.nextToken();
        logger_1.logger.verbose(`GDB command: ${token} ${command}`);
        return new Promise((resolve, reject) => {
            if (this.out) {
                /* Set error to capture the stack where the request originated,
                   not the stack of reading the stream and parsing the message.
                */
                const failure = new Error();
                this.parser.queueCommand(token, (resultClass, resultData) => {
                    switch (resultClass) {
                        case 'done':
                        case 'running':
                        case 'connected':
                        case 'exit':
                            resolve(resultData);
                            break;
                        case 'error':
                            failure.message = resultData.msg;
                            reject(failure);
                            break;
                        default:
                            failure.message = `Unknown response ${resultClass}: ${JSON.stringify(resultData)}`;
                            reject(failure);
                    }
                });
                this.out.write(`${token}${command}\n`);
            }
            else {
                reject(new Error('gdb is not running.'));
            }
        });
    }
    sendEnablePrettyPrint() {
        return this.sendCommand('-enable-pretty-printing');
    }
    // Rewrite the argument escaping whitespace, quotes and backslash
    standardEscape(arg, needQuotes = true) {
        let result = '';
        for (const char of arg) {
            if (char === '\\' || char === '"') {
                result += '\\';
            }
            if (char == ' ') {
                needQuotes = true;
            }
            result += char;
        }
        if (needQuotes) {
            result = `"${result}"`;
        }
        return result;
    }
    sendFileExecAndSymbols(program) {
        return this.sendCommand(`-file-exec-and-symbols ${this.standardEscape(program)}`);
    }
    sendFileSymbolFile(symbols) {
        return this.sendCommand(`-file-symbol-file ${this.standardEscape(symbols)}`);
    }
    sendAddSymbolFile(symbols, offset) {
        return this.sendCommand(`add-symbol-file ${this.standardEscape(symbols)} ${offset}`);
    }
    sendLoad(imageFileName, imageOffset) {
        return this.sendCommand(`load ${this.standardEscape(imageFileName)} ${imageOffset || ''}`);
    }
    sendGDBSet(params) {
        return this.sendCommand(`-gdb-set ${params}`);
    }
    sendGDBShow(params) {
        return this.sendCommand(`-gdb-show ${params}`);
    }
    sendGDBExit() {
        return this.sendCommand('-gdb-exit');
    }
    nextToken() {
        return this.token++;
    }
}
exports.GDBBackend = GDBBackend;
//# sourceMappingURL=GDBBackend.js.map