/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __extends = (this && this.__extends) || (function () {
    var extendStatics = function (d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (Object.prototype.hasOwnProperty.call(b, p)) d[p] = b[p]; };
        return extendStatics(d, b);
    };
    return function (d, b) {
        if (typeof b !== "function" && b !== null)
            throw new TypeError("Class extends value " + String(b) + " is not a constructor or null");
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
var __values = (this && this.__values) || function(o) {
    var s = typeof Symbol === "function" && Symbol.iterator, m = s && o[s], i = 0;
    if (m) return m.call(o);
    if (o && typeof o.length === "number") return {
        next: function () {
            if (o && i >= o.length) o = void 0;
            return { value: o && o[i++], done: !o };
        }
    };
    throw new TypeError(s ? "Object is not iterable." : "Symbol.iterator is not defined.");
};
var __read = (this && this.__read) || function (o, n) {
    var m = typeof Symbol === "function" && o[Symbol.iterator];
    if (!m) return o;
    var i = m.call(o), r, ar = [], e;
    try {
        while ((n === void 0 || n-- > 0) && !(r = i.next()).done) ar.push(r.value);
    }
    catch (error) { e = { error: error }; }
    finally {
        try {
            if (r && !r.done && (m = i["return"])) m.call(i);
        }
        finally { if (e) throw e.error; }
    }
    return ar;
};
var __spreadArray = (this && this.__spreadArray) || function (to, from) {
    for (var i = 0, il = from.length, j = to.length; i < il; i++, j++)
        to[j] = from[i];
    return to;
};
import { onUnexpectedError } from '../../../base/common/errors';
import { Emitter } from '../../../base/common/event';
import { Disposable } from '../../../base/common/lifecycle';
import * as mime from '../../../base/common/mime';
import * as strings from '../../../base/common/strings';
import { LanguageIdentifier } from '../modes';
import { ModesRegistry } from '../modes/modesRegistry';
import { NULL_LANGUAGE_IDENTIFIER, NULL_MODE_ID } from '../modes/nullMode';
import { Extensions } from '../../../platform/configuration/common/configurationRegistry';
import { Registry } from '../../../platform/registry/common/platform';
var hasOwnProperty = Object.prototype.hasOwnProperty;
var LanguagesRegistry = /** @class */ (function (_super) {
    __extends(LanguagesRegistry, _super);
    function LanguagesRegistry(useModesRegistry, warnOnOverwrite) {
        if (useModesRegistry === void 0) { useModesRegistry = true; }
        if (warnOnOverwrite === void 0) { warnOnOverwrite = false; }
        var _this = _super.call(this) || this;
        _this._onDidChange = _this._register(new Emitter());
        _this.onDidChange = _this._onDidChange.event;
        _this._warnOnOverwrite = warnOnOverwrite;
        _this._nextLanguageId2 = 1;
        _this._languageIdToLanguage = [];
        _this._languageToLanguageId = Object.create(null);
        _this._languages = {};
        _this._mimeTypesMap = {};
        _this._nameMap = {};
        _this._lowercaseNameMap = {};
        if (useModesRegistry) {
            _this._initializeFromRegistry();
            _this._register(ModesRegistry.onDidChangeLanguages(function (m) { return _this._initializeFromRegistry(); }));
        }
        return _this;
    }
    LanguagesRegistry.prototype._initializeFromRegistry = function () {
        this._languages = {};
        this._mimeTypesMap = {};
        this._nameMap = {};
        this._lowercaseNameMap = {};
        var desc = ModesRegistry.getLanguages();
        this._registerLanguages(desc);
    };
    LanguagesRegistry.prototype._registerLanguages = function (desc) {
        var e_1, _a;
        var _this = this;
        try {
            for (var desc_1 = __values(desc), desc_1_1 = desc_1.next(); !desc_1_1.done; desc_1_1 = desc_1.next()) {
                var d = desc_1_1.value;
                this._registerLanguage(d);
            }
        }
        catch (e_1_1) { e_1 = { error: e_1_1 }; }
        finally {
            try {
                if (desc_1_1 && !desc_1_1.done && (_a = desc_1.return)) _a.call(desc_1);
            }
            finally { if (e_1) throw e_1.error; }
        }
        // Rebuild fast path maps
        this._mimeTypesMap = {};
        this._nameMap = {};
        this._lowercaseNameMap = {};
        Object.keys(this._languages).forEach(function (langId) {
            var language = _this._languages[langId];
            if (language.name) {
                _this._nameMap[language.name] = language.identifier;
            }
            language.aliases.forEach(function (alias) {
                _this._lowercaseNameMap[alias.toLowerCase()] = language.identifier;
            });
            language.mimetypes.forEach(function (mimetype) {
                _this._mimeTypesMap[mimetype] = language.identifier;
            });
        });
        Registry.as(Extensions.Configuration).registerOverrideIdentifiers(ModesRegistry.getLanguages().map(function (language) { return language.id; }));
        this._onDidChange.fire();
    };
    LanguagesRegistry.prototype._getLanguageId = function (language) {
        if (this._languageToLanguageId[language]) {
            return this._languageToLanguageId[language];
        }
        var languageId = this._nextLanguageId2++;
        this._languageIdToLanguage[languageId] = language;
        this._languageToLanguageId[language] = languageId;
        return languageId;
    };
    LanguagesRegistry.prototype._registerLanguage = function (lang) {
        var langId = lang.id;
        var resolvedLanguage;
        if (hasOwnProperty.call(this._languages, langId)) {
            resolvedLanguage = this._languages[langId];
        }
        else {
            var languageId = this._getLanguageId(langId);
            resolvedLanguage = {
                identifier: new LanguageIdentifier(langId, languageId),
                name: null,
                mimetypes: [],
                aliases: [],
                extensions: [],
                filenames: [],
                configurationFiles: []
            };
            this._languages[langId] = resolvedLanguage;
        }
        this._mergeLanguage(resolvedLanguage, lang);
    };
    LanguagesRegistry.prototype._mergeLanguage = function (resolvedLanguage, lang) {
        var _a, e_2, _b, e_3, _c, e_4, _d, e_5, _e;
        var langId = lang.id;
        var primaryMime = null;
        if (Array.isArray(lang.mimetypes) && lang.mimetypes.length > 0) {
            (_a = resolvedLanguage.mimetypes).push.apply(_a, __spreadArray([], __read(lang.mimetypes)));
            primaryMime = lang.mimetypes[0];
        }
        if (!primaryMime) {
            primaryMime = "text/x-" + langId;
            resolvedLanguage.mimetypes.push(primaryMime);
        }
        if (Array.isArray(lang.extensions)) {
            if (lang.configuration) {
                // insert first as this appears to be the 'primary' language definition
                resolvedLanguage.extensions = lang.extensions.concat(resolvedLanguage.extensions);
            }
            else {
                resolvedLanguage.extensions = resolvedLanguage.extensions.concat(lang.extensions);
            }
            try {
                for (var _f = __values(lang.extensions), _g = _f.next(); !_g.done; _g = _f.next()) {
                    var extension = _g.value;
                    mime.registerTextMime({ id: langId, mime: primaryMime, extension: extension }, this._warnOnOverwrite);
                }
            }
            catch (e_2_1) { e_2 = { error: e_2_1 }; }
            finally {
                try {
                    if (_g && !_g.done && (_b = _f.return)) _b.call(_f);
                }
                finally { if (e_2) throw e_2.error; }
            }
        }
        if (Array.isArray(lang.filenames)) {
            try {
                for (var _h = __values(lang.filenames), _j = _h.next(); !_j.done; _j = _h.next()) {
                    var filename = _j.value;
                    mime.registerTextMime({ id: langId, mime: primaryMime, filename: filename }, this._warnOnOverwrite);
                    resolvedLanguage.filenames.push(filename);
                }
            }
            catch (e_3_1) { e_3 = { error: e_3_1 }; }
            finally {
                try {
                    if (_j && !_j.done && (_c = _h.return)) _c.call(_h);
                }
                finally { if (e_3) throw e_3.error; }
            }
        }
        if (Array.isArray(lang.filenamePatterns)) {
            try {
                for (var _k = __values(lang.filenamePatterns), _l = _k.next(); !_l.done; _l = _k.next()) {
                    var filenamePattern = _l.value;
                    mime.registerTextMime({ id: langId, mime: primaryMime, filepattern: filenamePattern }, this._warnOnOverwrite);
                }
            }
            catch (e_4_1) { e_4 = { error: e_4_1 }; }
            finally {
                try {
                    if (_l && !_l.done && (_d = _k.return)) _d.call(_k);
                }
                finally { if (e_4) throw e_4.error; }
            }
        }
        if (typeof lang.firstLine === 'string' && lang.firstLine.length > 0) {
            var firstLineRegexStr = lang.firstLine;
            if (firstLineRegexStr.charAt(0) !== '^') {
                firstLineRegexStr = '^' + firstLineRegexStr;
            }
            try {
                var firstLineRegex = new RegExp(firstLineRegexStr);
                if (!strings.regExpLeadsToEndlessLoop(firstLineRegex)) {
                    mime.registerTextMime({ id: langId, mime: primaryMime, firstline: firstLineRegex }, this._warnOnOverwrite);
                }
            }
            catch (err) {
                // Most likely, the regex was bad
                onUnexpectedError(err);
            }
        }
        resolvedLanguage.aliases.push(langId);
        var langAliases = null;
        if (typeof lang.aliases !== 'undefined' && Array.isArray(lang.aliases)) {
            if (lang.aliases.length === 0) {
                // signal that this language should not get a name
                langAliases = [null];
            }
            else {
                langAliases = lang.aliases;
            }
        }
        if (langAliases !== null) {
            try {
                for (var langAliases_1 = __values(langAliases), langAliases_1_1 = langAliases_1.next(); !langAliases_1_1.done; langAliases_1_1 = langAliases_1.next()) {
                    var langAlias = langAliases_1_1.value;
                    if (!langAlias || langAlias.length === 0) {
                        continue;
                    }
                    resolvedLanguage.aliases.push(langAlias);
                }
            }
            catch (e_5_1) { e_5 = { error: e_5_1 }; }
            finally {
                try {
                    if (langAliases_1_1 && !langAliases_1_1.done && (_e = langAliases_1.return)) _e.call(langAliases_1);
                }
                finally { if (e_5) throw e_5.error; }
            }
        }
        var containsAliases = (langAliases !== null && langAliases.length > 0);
        if (containsAliases && langAliases[0] === null) {
            // signal that this language should not get a name
        }
        else {
            var bestName = (containsAliases ? langAliases[0] : null) || langId;
            if (containsAliases || !resolvedLanguage.name) {
                resolvedLanguage.name = bestName;
            }
        }
        if (lang.configuration) {
            resolvedLanguage.configurationFiles.push(lang.configuration);
        }
    };
    LanguagesRegistry.prototype.isRegisteredMode = function (mimetypeOrModeId) {
        // Is this a known mime type ?
        if (hasOwnProperty.call(this._mimeTypesMap, mimetypeOrModeId)) {
            return true;
        }
        // Is this a known mode id ?
        return hasOwnProperty.call(this._languages, mimetypeOrModeId);
    };
    LanguagesRegistry.prototype.getRegisteredModes = function () {
        return Object.keys(this._languages);
    };
    LanguagesRegistry.prototype.getRegisteredLanguageNames = function () {
        return Object.keys(this._nameMap);
    };
    LanguagesRegistry.prototype.getLanguageName = function (modeId) {
        if (!hasOwnProperty.call(this._languages, modeId)) {
            return null;
        }
        return this._languages[modeId].name;
    };
    LanguagesRegistry.prototype.getModeIdForLanguageNameLowercase = function (languageNameLower) {
        if (!hasOwnProperty.call(this._lowercaseNameMap, languageNameLower)) {
            return null;
        }
        return this._lowercaseNameMap[languageNameLower].language;
    };
    LanguagesRegistry.prototype.getConfigurationFiles = function (modeId) {
        if (!hasOwnProperty.call(this._languages, modeId)) {
            return [];
        }
        return this._languages[modeId].configurationFiles || [];
    };
    LanguagesRegistry.prototype.getMimeForMode = function (modeId) {
        if (!hasOwnProperty.call(this._languages, modeId)) {
            return null;
        }
        var language = this._languages[modeId];
        return (language.mimetypes[0] || null);
    };
    LanguagesRegistry.prototype.extractModeIds = function (commaSeparatedMimetypesOrCommaSeparatedIds) {
        var _this = this;
        if (!commaSeparatedMimetypesOrCommaSeparatedIds) {
            return [];
        }
        return (commaSeparatedMimetypesOrCommaSeparatedIds.
            split(',').
            map(function (mimeTypeOrId) { return mimeTypeOrId.trim(); }).
            map(function (mimeTypeOrId) {
            if (hasOwnProperty.call(_this._mimeTypesMap, mimeTypeOrId)) {
                return _this._mimeTypesMap[mimeTypeOrId].language;
            }
            return mimeTypeOrId;
        }).
            filter(function (modeId) {
            return hasOwnProperty.call(_this._languages, modeId);
        }));
    };
    LanguagesRegistry.prototype.getLanguageIdentifier = function (_modeId) {
        if (_modeId === NULL_MODE_ID || _modeId === 0 /* Null */) {
            return NULL_LANGUAGE_IDENTIFIER;
        }
        var modeId;
        if (typeof _modeId === 'string') {
            modeId = _modeId;
        }
        else {
            modeId = this._languageIdToLanguage[_modeId];
            if (!modeId) {
                return null;
            }
        }
        if (!hasOwnProperty.call(this._languages, modeId)) {
            return null;
        }
        return this._languages[modeId].identifier;
    };
    LanguagesRegistry.prototype.getModeIdsFromLanguageName = function (languageName) {
        if (!languageName) {
            return [];
        }
        if (hasOwnProperty.call(this._nameMap, languageName)) {
            return [this._nameMap[languageName].language];
        }
        return [];
    };
    LanguagesRegistry.prototype.getModeIdsFromFilepathOrFirstLine = function (resource, firstLine) {
        if (!resource && !firstLine) {
            return [];
        }
        var mimeTypes = mime.guessMimeTypes(resource, firstLine);
        return this.extractModeIds(mimeTypes.join(','));
    };
    LanguagesRegistry.prototype.getExtensions = function (languageName) {
        if (!hasOwnProperty.call(this._nameMap, languageName)) {
            return [];
        }
        var languageId = this._nameMap[languageName];
        return this._languages[languageId.language].extensions;
    };
    LanguagesRegistry.prototype.getFilenames = function (languageName) {
        if (!hasOwnProperty.call(this._nameMap, languageName)) {
            return [];
        }
        var languageId = this._nameMap[languageName];
        return this._languages[languageId.language].filenames;
    };
    return LanguagesRegistry;
}(Disposable));
export { LanguagesRegistry };
