/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __read = (this && this.__read) || function (o, n) {
    var m = typeof Symbol === "function" && o[Symbol.iterator];
    if (!m) return o;
    var i = m.call(o), r, ar = [], e;
    try {
        while ((n === void 0 || n-- > 0) && !(r = i.next()).done) ar.push(r.value);
    }
    catch (error) { e = { error: error }; }
    finally {
        try {
            if (r && !r.done && (m = i["return"])) m.call(i);
        }
        finally { if (e) throw e.error; }
    }
    return ar;
};
var __values = (this && this.__values) || function(o) {
    var s = typeof Symbol === "function" && Symbol.iterator, m = s && o[s], i = 0;
    if (m) return m.call(o);
    if (o && typeof o.length === "number") return {
        next: function () {
            if (o && i >= o.length) o = void 0;
            return { value: o && o[i++], done: !o };
        }
    };
    throw new TypeError(s ? "Object is not iterable." : "Symbol.iterator is not defined.");
};
import { Emitter } from '../../../base/common/event';
import { toDisposable } from '../../../base/common/lifecycle';
import * as strings from '../../../base/common/strings';
import { DEFAULT_WORD_REGEXP, ensureValidWordDefinition } from '../model/wordHelper';
import { IndentAction, AutoClosingPairs } from './languageConfiguration';
import { createScopedLineTokens } from './supports';
import { CharacterPairSupport } from './supports/characterPair';
import { BracketElectricCharacterSupport } from './supports/electricCharacter';
import { IndentRulesSupport } from './supports/indentRules';
import { OnEnterSupport } from './supports/onEnter';
import { RichEditBrackets } from './supports/richEditBrackets';
var RichEditSupport = /** @class */ (function () {
    function RichEditSupport(languageIdentifier, rawConf) {
        this._languageIdentifier = languageIdentifier;
        this._brackets = null;
        this._electricCharacter = null;
        this._conf = rawConf;
        this._onEnterSupport = (this._conf.brackets || this._conf.indentationRules || this._conf.onEnterRules ? new OnEnterSupport(this._conf) : null);
        this.comments = RichEditSupport._handleComments(this._conf);
        this.characterPair = new CharacterPairSupport(this._conf);
        this.wordDefinition = this._conf.wordPattern || DEFAULT_WORD_REGEXP;
        this.indentationRules = this._conf.indentationRules;
        if (this._conf.indentationRules) {
            this.indentRulesSupport = new IndentRulesSupport(this._conf.indentationRules);
        }
        else {
            this.indentRulesSupport = null;
        }
        this.foldingRules = this._conf.folding || {};
    }
    Object.defineProperty(RichEditSupport.prototype, "brackets", {
        get: function () {
            if (!this._brackets && this._conf.brackets) {
                this._brackets = new RichEditBrackets(this._languageIdentifier, this._conf.brackets);
            }
            return this._brackets;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(RichEditSupport.prototype, "electricCharacter", {
        get: function () {
            if (!this._electricCharacter) {
                this._electricCharacter = new BracketElectricCharacterSupport(this.brackets);
            }
            return this._electricCharacter;
        },
        enumerable: false,
        configurable: true
    });
    RichEditSupport.prototype.onEnter = function (autoIndent, previousLineText, beforeEnterText, afterEnterText) {
        if (!this._onEnterSupport) {
            return null;
        }
        return this._onEnterSupport.onEnter(autoIndent, previousLineText, beforeEnterText, afterEnterText);
    };
    RichEditSupport._handleComments = function (conf) {
        var commentRule = conf.comments;
        if (!commentRule) {
            return null;
        }
        // comment configuration
        var comments = {};
        if (commentRule.lineComment) {
            comments.lineCommentToken = commentRule.lineComment;
        }
        if (commentRule.blockComment) {
            var _a = __read(commentRule.blockComment, 2), blockStart = _a[0], blockEnd = _a[1];
            comments.blockCommentStartToken = blockStart;
            comments.blockCommentEndToken = blockEnd;
        }
        return comments;
    };
    return RichEditSupport;
}());
export { RichEditSupport };
var LanguageConfigurationChangeEvent = /** @class */ (function () {
    function LanguageConfigurationChangeEvent(languageIdentifier) {
        this.languageIdentifier = languageIdentifier;
    }
    return LanguageConfigurationChangeEvent;
}());
export { LanguageConfigurationChangeEvent };
var LanguageConfigurationEntry = /** @class */ (function () {
    function LanguageConfigurationEntry(configuration, priority, order) {
        this.configuration = configuration;
        this.priority = priority;
        this.order = order;
    }
    LanguageConfigurationEntry.cmp = function (a, b) {
        if (a.priority === b.priority) {
            // higher order last
            return a.order - b.order;
        }
        // higher priority last
        return a.priority - b.priority;
    };
    return LanguageConfigurationEntry;
}());
var LanguageConfigurationEntries = /** @class */ (function () {
    function LanguageConfigurationEntries(languageIdentifier) {
        this.languageIdentifier = languageIdentifier;
        this._resolved = null;
        this._entries = [];
        this._order = 0;
        this._resolved = null;
    }
    LanguageConfigurationEntries.prototype.register = function (configuration, priority) {
        var _this = this;
        var entry = new LanguageConfigurationEntry(configuration, priority, ++this._order);
        this._entries.push(entry);
        this._resolved = null;
        return toDisposable(function () {
            for (var i = 0; i < _this._entries.length; i++) {
                if (_this._entries[i] === entry) {
                    _this._entries.splice(i, 1);
                    _this._resolved = null;
                    break;
                }
            }
        });
    };
    LanguageConfigurationEntries.prototype.getRichEditSupport = function () {
        if (!this._resolved) {
            var config = this._resolve();
            if (config) {
                this._resolved = new RichEditSupport(this.languageIdentifier, config);
            }
        }
        return this._resolved;
    };
    LanguageConfigurationEntries.prototype._resolve = function () {
        var e_1, _a;
        if (this._entries.length === 0) {
            return null;
        }
        this._entries.sort(LanguageConfigurationEntry.cmp);
        var result = {};
        try {
            for (var _b = __values(this._entries), _c = _b.next(); !_c.done; _c = _b.next()) {
                var entry = _c.value;
                var conf = entry.configuration;
                result.comments = conf.comments || result.comments;
                result.brackets = conf.brackets || result.brackets;
                result.wordPattern = conf.wordPattern || result.wordPattern;
                result.indentationRules = conf.indentationRules || result.indentationRules;
                result.onEnterRules = conf.onEnterRules || result.onEnterRules;
                result.autoClosingPairs = conf.autoClosingPairs || result.autoClosingPairs;
                result.surroundingPairs = conf.surroundingPairs || result.surroundingPairs;
                result.autoCloseBefore = conf.autoCloseBefore || result.autoCloseBefore;
                result.folding = conf.folding || result.folding;
                result.__electricCharacterSupport = conf.__electricCharacterSupport || result.__electricCharacterSupport;
            }
        }
        catch (e_1_1) { e_1 = { error: e_1_1 }; }
        finally {
            try {
                if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
            }
            finally { if (e_1) throw e_1.error; }
        }
        return result;
    };
    return LanguageConfigurationEntries;
}());
var LanguageConfigurationRegistryImpl = /** @class */ (function () {
    function LanguageConfigurationRegistryImpl() {
        this._entries2 = new Map();
        this._onDidChange = new Emitter();
        this.onDidChange = this._onDidChange.event;
    }
    /**
     * @param priority Use a higher number for higher priority
     */
    LanguageConfigurationRegistryImpl.prototype.register = function (languageIdentifier, configuration, priority) {
        var _this = this;
        if (priority === void 0) { priority = 0; }
        var entries = this._entries2.get(languageIdentifier.id);
        if (!entries) {
            entries = new LanguageConfigurationEntries(languageIdentifier);
            this._entries2.set(languageIdentifier.id, entries);
        }
        var disposable = entries.register(configuration, priority);
        this._onDidChange.fire(new LanguageConfigurationChangeEvent(languageIdentifier));
        return toDisposable(function () {
            disposable.dispose();
            _this._onDidChange.fire(new LanguageConfigurationChangeEvent(languageIdentifier));
        });
    };
    LanguageConfigurationRegistryImpl.prototype._getRichEditSupport = function (languageId) {
        var entries = this._entries2.get(languageId);
        return entries ? entries.getRichEditSupport() : null;
    };
    LanguageConfigurationRegistryImpl.prototype.getIndentationRules = function (languageId) {
        var value = this._getRichEditSupport(languageId);
        return value ? value.indentationRules || null : null;
    };
    // begin electricCharacter
    LanguageConfigurationRegistryImpl.prototype._getElectricCharacterSupport = function (languageId) {
        var value = this._getRichEditSupport(languageId);
        if (!value) {
            return null;
        }
        return value.electricCharacter || null;
    };
    LanguageConfigurationRegistryImpl.prototype.getElectricCharacters = function (languageId) {
        var electricCharacterSupport = this._getElectricCharacterSupport(languageId);
        if (!electricCharacterSupport) {
            return [];
        }
        return electricCharacterSupport.getElectricCharacters();
    };
    /**
     * Should return opening bracket type to match indentation with
     */
    LanguageConfigurationRegistryImpl.prototype.onElectricCharacter = function (character, context, column) {
        var scopedLineTokens = createScopedLineTokens(context, column - 1);
        var electricCharacterSupport = this._getElectricCharacterSupport(scopedLineTokens.languageId);
        if (!electricCharacterSupport) {
            return null;
        }
        return electricCharacterSupport.onElectricCharacter(character, scopedLineTokens, column - scopedLineTokens.firstCharOffset);
    };
    // end electricCharacter
    LanguageConfigurationRegistryImpl.prototype.getComments = function (languageId) {
        var value = this._getRichEditSupport(languageId);
        if (!value) {
            return null;
        }
        return value.comments || null;
    };
    // begin characterPair
    LanguageConfigurationRegistryImpl.prototype._getCharacterPairSupport = function (languageId) {
        var value = this._getRichEditSupport(languageId);
        if (!value) {
            return null;
        }
        return value.characterPair || null;
    };
    LanguageConfigurationRegistryImpl.prototype.getAutoClosingPairs = function (languageId) {
        var characterPairSupport = this._getCharacterPairSupport(languageId);
        return new AutoClosingPairs(characterPairSupport ? characterPairSupport.getAutoClosingPairs() : []);
    };
    LanguageConfigurationRegistryImpl.prototype.getAutoCloseBeforeSet = function (languageId) {
        var characterPairSupport = this._getCharacterPairSupport(languageId);
        if (!characterPairSupport) {
            return CharacterPairSupport.DEFAULT_AUTOCLOSE_BEFORE_LANGUAGE_DEFINED;
        }
        return characterPairSupport.getAutoCloseBeforeSet();
    };
    LanguageConfigurationRegistryImpl.prototype.getSurroundingPairs = function (languageId) {
        var characterPairSupport = this._getCharacterPairSupport(languageId);
        if (!characterPairSupport) {
            return [];
        }
        return characterPairSupport.getSurroundingPairs();
    };
    LanguageConfigurationRegistryImpl.prototype.shouldAutoClosePair = function (autoClosingPair, context, column) {
        var scopedLineTokens = createScopedLineTokens(context, column - 1);
        return CharacterPairSupport.shouldAutoClosePair(autoClosingPair, scopedLineTokens, column - scopedLineTokens.firstCharOffset);
    };
    // end characterPair
    LanguageConfigurationRegistryImpl.prototype.getWordDefinition = function (languageId) {
        var value = this._getRichEditSupport(languageId);
        if (!value) {
            return ensureValidWordDefinition(null);
        }
        return ensureValidWordDefinition(value.wordDefinition || null);
    };
    LanguageConfigurationRegistryImpl.prototype.getWordDefinitions = function () {
        var e_2, _a;
        var result = [];
        try {
            for (var _b = __values(this._entries2), _c = _b.next(); !_c.done; _c = _b.next()) {
                var _d = __read(_c.value, 2), language = _d[0], entries = _d[1];
                var value = entries.getRichEditSupport();
                if (value) {
                    result.push([language, value.wordDefinition]);
                }
            }
        }
        catch (e_2_1) { e_2 = { error: e_2_1 }; }
        finally {
            try {
                if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
            }
            finally { if (e_2) throw e_2.error; }
        }
        return result;
    };
    LanguageConfigurationRegistryImpl.prototype.getFoldingRules = function (languageId) {
        var value = this._getRichEditSupport(languageId);
        if (!value) {
            return {};
        }
        return value.foldingRules;
    };
    // begin Indent Rules
    LanguageConfigurationRegistryImpl.prototype.getIndentRulesSupport = function (languageId) {
        var value = this._getRichEditSupport(languageId);
        if (!value) {
            return null;
        }
        return value.indentRulesSupport || null;
    };
    /**
     * Get nearest preceiding line which doesn't match unIndentPattern or contains all whitespace.
     * Result:
     * -1: run into the boundary of embedded languages
     * 0: every line above are invalid
     * else: nearest preceding line of the same language
     */
    LanguageConfigurationRegistryImpl.prototype.getPrecedingValidLine = function (model, lineNumber, indentRulesSupport) {
        var languageID = model.getLanguageIdAtPosition(lineNumber, 0);
        if (lineNumber > 1) {
            var lastLineNumber = void 0;
            var resultLineNumber = -1;
            for (lastLineNumber = lineNumber - 1; lastLineNumber >= 1; lastLineNumber--) {
                if (model.getLanguageIdAtPosition(lastLineNumber, 0) !== languageID) {
                    return resultLineNumber;
                }
                var text = model.getLineContent(lastLineNumber);
                if (indentRulesSupport.shouldIgnore(text) || /^\s+$/.test(text) || text === '') {
                    resultLineNumber = lastLineNumber;
                    continue;
                }
                return lastLineNumber;
            }
        }
        return -1;
    };
    /**
     * Get inherited indentation from above lines.
     * 1. Find the nearest preceding line which doesn't match unIndentedLinePattern.
     * 2. If this line matches indentNextLinePattern or increaseIndentPattern, it means that the indent level of `lineNumber` should be 1 greater than this line.
     * 3. If this line doesn't match any indent rules
     *   a. check whether the line above it matches indentNextLinePattern
     *   b. If not, the indent level of this line is the result
     *   c. If so, it means the indent of this line is *temporary*, go upward utill we find a line whose indent is not temporary (the same workflow a -> b -> c).
     * 4. Otherwise, we fail to get an inherited indent from aboves. Return null and we should not touch the indent of `lineNumber`
     *
     * This function only return the inherited indent based on above lines, it doesn't check whether current line should decrease or not.
     */
    LanguageConfigurationRegistryImpl.prototype.getInheritIndentForLine = function (autoIndent, model, lineNumber, honorIntentialIndent) {
        if (honorIntentialIndent === void 0) { honorIntentialIndent = true; }
        if (autoIndent < 4 /* Full */) {
            return null;
        }
        var indentRulesSupport = this.getIndentRulesSupport(model.getLanguageIdentifier().id);
        if (!indentRulesSupport) {
            return null;
        }
        if (lineNumber <= 1) {
            return {
                indentation: '',
                action: null
            };
        }
        var precedingUnIgnoredLine = this.getPrecedingValidLine(model, lineNumber, indentRulesSupport);
        if (precedingUnIgnoredLine < 0) {
            return null;
        }
        else if (precedingUnIgnoredLine < 1) {
            return {
                indentation: '',
                action: null
            };
        }
        var precedingUnIgnoredLineContent = model.getLineContent(precedingUnIgnoredLine);
        if (indentRulesSupport.shouldIncrease(precedingUnIgnoredLineContent) || indentRulesSupport.shouldIndentNextLine(precedingUnIgnoredLineContent)) {
            return {
                indentation: strings.getLeadingWhitespace(precedingUnIgnoredLineContent),
                action: IndentAction.Indent,
                line: precedingUnIgnoredLine
            };
        }
        else if (indentRulesSupport.shouldDecrease(precedingUnIgnoredLineContent)) {
            return {
                indentation: strings.getLeadingWhitespace(precedingUnIgnoredLineContent),
                action: null,
                line: precedingUnIgnoredLine
            };
        }
        else {
            // precedingUnIgnoredLine can not be ignored.
            // it doesn't increase indent of following lines
            // it doesn't increase just next line
            // so current line is not affect by precedingUnIgnoredLine
            // and then we should get a correct inheritted indentation from above lines
            if (precedingUnIgnoredLine === 1) {
                return {
                    indentation: strings.getLeadingWhitespace(model.getLineContent(precedingUnIgnoredLine)),
                    action: null,
                    line: precedingUnIgnoredLine
                };
            }
            var previousLine = precedingUnIgnoredLine - 1;
            var previousLineIndentMetadata = indentRulesSupport.getIndentMetadata(model.getLineContent(previousLine));
            if (!(previousLineIndentMetadata & (1 /* INCREASE_MASK */ | 2 /* DECREASE_MASK */)) &&
                (previousLineIndentMetadata & 4 /* INDENT_NEXTLINE_MASK */)) {
                var stopLine = 0;
                for (var i = previousLine - 1; i > 0; i--) {
                    if (indentRulesSupport.shouldIndentNextLine(model.getLineContent(i))) {
                        continue;
                    }
                    stopLine = i;
                    break;
                }
                return {
                    indentation: strings.getLeadingWhitespace(model.getLineContent(stopLine + 1)),
                    action: null,
                    line: stopLine + 1
                };
            }
            if (honorIntentialIndent) {
                return {
                    indentation: strings.getLeadingWhitespace(model.getLineContent(precedingUnIgnoredLine)),
                    action: null,
                    line: precedingUnIgnoredLine
                };
            }
            else {
                // search from precedingUnIgnoredLine until we find one whose indent is not temporary
                for (var i = precedingUnIgnoredLine; i > 0; i--) {
                    var lineContent = model.getLineContent(i);
                    if (indentRulesSupport.shouldIncrease(lineContent)) {
                        return {
                            indentation: strings.getLeadingWhitespace(lineContent),
                            action: IndentAction.Indent,
                            line: i
                        };
                    }
                    else if (indentRulesSupport.shouldIndentNextLine(lineContent)) {
                        var stopLine = 0;
                        for (var j = i - 1; j > 0; j--) {
                            if (indentRulesSupport.shouldIndentNextLine(model.getLineContent(i))) {
                                continue;
                            }
                            stopLine = j;
                            break;
                        }
                        return {
                            indentation: strings.getLeadingWhitespace(model.getLineContent(stopLine + 1)),
                            action: null,
                            line: stopLine + 1
                        };
                    }
                    else if (indentRulesSupport.shouldDecrease(lineContent)) {
                        return {
                            indentation: strings.getLeadingWhitespace(lineContent),
                            action: null,
                            line: i
                        };
                    }
                }
                return {
                    indentation: strings.getLeadingWhitespace(model.getLineContent(1)),
                    action: null,
                    line: 1
                };
            }
        }
    };
    LanguageConfigurationRegistryImpl.prototype.getGoodIndentForLine = function (autoIndent, virtualModel, languageId, lineNumber, indentConverter) {
        if (autoIndent < 4 /* Full */) {
            return null;
        }
        var richEditSupport = this._getRichEditSupport(languageId);
        if (!richEditSupport) {
            return null;
        }
        var indentRulesSupport = this.getIndentRulesSupport(languageId);
        if (!indentRulesSupport) {
            return null;
        }
        var indent = this.getInheritIndentForLine(autoIndent, virtualModel, lineNumber);
        var lineContent = virtualModel.getLineContent(lineNumber);
        if (indent) {
            var inheritLine = indent.line;
            if (inheritLine !== undefined) {
                var enterResult = richEditSupport.onEnter(autoIndent, '', virtualModel.getLineContent(inheritLine), '');
                if (enterResult) {
                    var indentation = strings.getLeadingWhitespace(virtualModel.getLineContent(inheritLine));
                    if (enterResult.removeText) {
                        indentation = indentation.substring(0, indentation.length - enterResult.removeText);
                    }
                    if ((enterResult.indentAction === IndentAction.Indent) ||
                        (enterResult.indentAction === IndentAction.IndentOutdent)) {
                        indentation = indentConverter.shiftIndent(indentation);
                    }
                    else if (enterResult.indentAction === IndentAction.Outdent) {
                        indentation = indentConverter.unshiftIndent(indentation);
                    }
                    if (indentRulesSupport.shouldDecrease(lineContent)) {
                        indentation = indentConverter.unshiftIndent(indentation);
                    }
                    if (enterResult.appendText) {
                        indentation += enterResult.appendText;
                    }
                    return strings.getLeadingWhitespace(indentation);
                }
            }
            if (indentRulesSupport.shouldDecrease(lineContent)) {
                if (indent.action === IndentAction.Indent) {
                    return indent.indentation;
                }
                else {
                    return indentConverter.unshiftIndent(indent.indentation);
                }
            }
            else {
                if (indent.action === IndentAction.Indent) {
                    return indentConverter.shiftIndent(indent.indentation);
                }
                else {
                    return indent.indentation;
                }
            }
        }
        return null;
    };
    LanguageConfigurationRegistryImpl.prototype.getIndentForEnter = function (autoIndent, model, range, indentConverter) {
        if (autoIndent < 4 /* Full */) {
            return null;
        }
        model.forceTokenization(range.startLineNumber);
        var lineTokens = model.getLineTokens(range.startLineNumber);
        var scopedLineTokens = createScopedLineTokens(lineTokens, range.startColumn - 1);
        var scopedLineText = scopedLineTokens.getLineContent();
        var embeddedLanguage = false;
        var beforeEnterText;
        if (scopedLineTokens.firstCharOffset > 0 && lineTokens.getLanguageId(0) !== scopedLineTokens.languageId) {
            // we are in the embeded language content
            embeddedLanguage = true; // if embeddedLanguage is true, then we don't touch the indentation of current line
            beforeEnterText = scopedLineText.substr(0, range.startColumn - 1 - scopedLineTokens.firstCharOffset);
        }
        else {
            beforeEnterText = lineTokens.getLineContent().substring(0, range.startColumn - 1);
        }
        var afterEnterText;
        if (range.isEmpty()) {
            afterEnterText = scopedLineText.substr(range.startColumn - 1 - scopedLineTokens.firstCharOffset);
        }
        else {
            var endScopedLineTokens = this.getScopedLineTokens(model, range.endLineNumber, range.endColumn);
            afterEnterText = endScopedLineTokens.getLineContent().substr(range.endColumn - 1 - scopedLineTokens.firstCharOffset);
        }
        var indentRulesSupport = this.getIndentRulesSupport(scopedLineTokens.languageId);
        if (!indentRulesSupport) {
            return null;
        }
        var beforeEnterResult = beforeEnterText;
        var beforeEnterIndent = strings.getLeadingWhitespace(beforeEnterText);
        var virtualModel = {
            getLineTokens: function (lineNumber) {
                return model.getLineTokens(lineNumber);
            },
            getLanguageIdentifier: function () {
                return model.getLanguageIdentifier();
            },
            getLanguageIdAtPosition: function (lineNumber, column) {
                return model.getLanguageIdAtPosition(lineNumber, column);
            },
            getLineContent: function (lineNumber) {
                if (lineNumber === range.startLineNumber) {
                    return beforeEnterResult;
                }
                else {
                    return model.getLineContent(lineNumber);
                }
            }
        };
        var currentLineIndent = strings.getLeadingWhitespace(lineTokens.getLineContent());
        var afterEnterAction = this.getInheritIndentForLine(autoIndent, virtualModel, range.startLineNumber + 1);
        if (!afterEnterAction) {
            var beforeEnter = embeddedLanguage ? currentLineIndent : beforeEnterIndent;
            return {
                beforeEnter: beforeEnter,
                afterEnter: beforeEnter
            };
        }
        var afterEnterIndent = embeddedLanguage ? currentLineIndent : afterEnterAction.indentation;
        if (afterEnterAction.action === IndentAction.Indent) {
            afterEnterIndent = indentConverter.shiftIndent(afterEnterIndent);
        }
        if (indentRulesSupport.shouldDecrease(afterEnterText)) {
            afterEnterIndent = indentConverter.unshiftIndent(afterEnterIndent);
        }
        return {
            beforeEnter: embeddedLanguage ? currentLineIndent : beforeEnterIndent,
            afterEnter: afterEnterIndent
        };
    };
    /**
     * We should always allow intentional indentation. It means, if users change the indentation of `lineNumber` and the content of
     * this line doesn't match decreaseIndentPattern, we should not adjust the indentation.
     */
    LanguageConfigurationRegistryImpl.prototype.getIndentActionForType = function (autoIndent, model, range, ch, indentConverter) {
        if (autoIndent < 4 /* Full */) {
            return null;
        }
        var scopedLineTokens = this.getScopedLineTokens(model, range.startLineNumber, range.startColumn);
        if (scopedLineTokens.firstCharOffset) {
            // this line has mixed languages and indentation rules will not work
            return null;
        }
        var indentRulesSupport = this.getIndentRulesSupport(scopedLineTokens.languageId);
        if (!indentRulesSupport) {
            return null;
        }
        var scopedLineText = scopedLineTokens.getLineContent();
        var beforeTypeText = scopedLineText.substr(0, range.startColumn - 1 - scopedLineTokens.firstCharOffset);
        // selection support
        var afterTypeText;
        if (range.isEmpty()) {
            afterTypeText = scopedLineText.substr(range.startColumn - 1 - scopedLineTokens.firstCharOffset);
        }
        else {
            var endScopedLineTokens = this.getScopedLineTokens(model, range.endLineNumber, range.endColumn);
            afterTypeText = endScopedLineTokens.getLineContent().substr(range.endColumn - 1 - scopedLineTokens.firstCharOffset);
        }
        // If previous content already matches decreaseIndentPattern, it means indentation of this line should already be adjusted
        // Users might change the indentation by purpose and we should honor that instead of readjusting.
        if (!indentRulesSupport.shouldDecrease(beforeTypeText + afterTypeText) && indentRulesSupport.shouldDecrease(beforeTypeText + ch + afterTypeText)) {
            // after typing `ch`, the content matches decreaseIndentPattern, we should adjust the indent to a good manner.
            // 1. Get inherited indent action
            var r = this.getInheritIndentForLine(autoIndent, model, range.startLineNumber, false);
            if (!r) {
                return null;
            }
            var indentation = r.indentation;
            if (r.action !== IndentAction.Indent) {
                indentation = indentConverter.unshiftIndent(indentation);
            }
            return indentation;
        }
        return null;
    };
    LanguageConfigurationRegistryImpl.prototype.getIndentMetadata = function (model, lineNumber) {
        var indentRulesSupport = this.getIndentRulesSupport(model.getLanguageIdentifier().id);
        if (!indentRulesSupport) {
            return null;
        }
        if (lineNumber < 1 || lineNumber > model.getLineCount()) {
            return null;
        }
        return indentRulesSupport.getIndentMetadata(model.getLineContent(lineNumber));
    };
    // end Indent Rules
    // begin onEnter
    LanguageConfigurationRegistryImpl.prototype.getEnterAction = function (autoIndent, model, range) {
        var scopedLineTokens = this.getScopedLineTokens(model, range.startLineNumber, range.startColumn);
        var richEditSupport = this._getRichEditSupport(scopedLineTokens.languageId);
        if (!richEditSupport) {
            return null;
        }
        var scopedLineText = scopedLineTokens.getLineContent();
        var beforeEnterText = scopedLineText.substr(0, range.startColumn - 1 - scopedLineTokens.firstCharOffset);
        // selection support
        var afterEnterText;
        if (range.isEmpty()) {
            afterEnterText = scopedLineText.substr(range.startColumn - 1 - scopedLineTokens.firstCharOffset);
        }
        else {
            var endScopedLineTokens = this.getScopedLineTokens(model, range.endLineNumber, range.endColumn);
            afterEnterText = endScopedLineTokens.getLineContent().substr(range.endColumn - 1 - scopedLineTokens.firstCharOffset);
        }
        var previousLineText = '';
        if (range.startLineNumber > 1 && scopedLineTokens.firstCharOffset === 0) {
            // This is not the first line and the entire line belongs to this mode
            var oneLineAboveScopedLineTokens = this.getScopedLineTokens(model, range.startLineNumber - 1);
            if (oneLineAboveScopedLineTokens.languageId === scopedLineTokens.languageId) {
                // The line above ends with text belonging to the same mode
                previousLineText = oneLineAboveScopedLineTokens.getLineContent();
            }
        }
        var enterResult = richEditSupport.onEnter(autoIndent, previousLineText, beforeEnterText, afterEnterText);
        if (!enterResult) {
            return null;
        }
        var indentAction = enterResult.indentAction;
        var appendText = enterResult.appendText;
        var removeText = enterResult.removeText || 0;
        // Here we add `\t` to appendText first because enterAction is leveraging appendText and removeText to change indentation.
        if (!appendText) {
            if ((indentAction === IndentAction.Indent) ||
                (indentAction === IndentAction.IndentOutdent)) {
                appendText = '\t';
            }
            else {
                appendText = '';
            }
        }
        else if (indentAction === IndentAction.Indent) {
            appendText = '\t' + appendText;
        }
        var indentation = this.getIndentationAtPosition(model, range.startLineNumber, range.startColumn);
        if (removeText) {
            indentation = indentation.substring(0, indentation.length - removeText);
        }
        return {
            indentAction: indentAction,
            appendText: appendText,
            removeText: removeText,
            indentation: indentation
        };
    };
    LanguageConfigurationRegistryImpl.prototype.getIndentationAtPosition = function (model, lineNumber, column) {
        var lineText = model.getLineContent(lineNumber);
        var indentation = strings.getLeadingWhitespace(lineText);
        if (indentation.length > column - 1) {
            indentation = indentation.substring(0, column - 1);
        }
        return indentation;
    };
    LanguageConfigurationRegistryImpl.prototype.getScopedLineTokens = function (model, lineNumber, columnNumber) {
        model.forceTokenization(lineNumber);
        var lineTokens = model.getLineTokens(lineNumber);
        var column = (typeof columnNumber === 'undefined' ? model.getLineMaxColumn(lineNumber) - 1 : columnNumber - 1);
        return createScopedLineTokens(lineTokens, column);
    };
    // end onEnter
    LanguageConfigurationRegistryImpl.prototype.getBracketsSupport = function (languageId) {
        var value = this._getRichEditSupport(languageId);
        if (!value) {
            return null;
        }
        return value.brackets || null;
    };
    return LanguageConfigurationRegistryImpl;
}());
export { LanguageConfigurationRegistryImpl };
export var LanguageConfigurationRegistry = new LanguageConfigurationRegistryImpl();
