/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
import * as strings from '../../../base/common/strings';
import * as platform from '../../../base/common/platform';
import * as buffer from '../../../base/common/buffer';
var _platformTextDecoder;
export function getPlatformTextDecoder() {
    if (!_platformTextDecoder) {
        _platformTextDecoder = new TextDecoder(platform.isLittleEndian() ? 'UTF-16LE' : 'UTF-16BE');
    }
    return _platformTextDecoder;
}
export var hasTextDecoder = (typeof TextDecoder !== 'undefined');
export var createStringBuilder;
export var decodeUTF16LE;
if (hasTextDecoder) {
    createStringBuilder = function (capacity) { return new StringBuilder(capacity); };
    decodeUTF16LE = standardDecodeUTF16LE;
}
else {
    createStringBuilder = function (capacity) { return new CompatStringBuilder(); };
    decodeUTF16LE = compatDecodeUTF16LE;
}
function standardDecodeUTF16LE(source, offset, len) {
    var view = new Uint16Array(source.buffer, offset, len);
    return getPlatformTextDecoder().decode(view);
}
function compatDecodeUTF16LE(source, offset, len) {
    var result = [];
    var resultLen = 0;
    for (var i = 0; i < len; i++) {
        var charCode = buffer.readUInt16LE(source, offset);
        offset += 2;
        result[resultLen++] = String.fromCharCode(charCode);
    }
    return result.join('');
}
var StringBuilder = /** @class */ (function () {
    function StringBuilder(capacity) {
        this._capacity = capacity | 0;
        this._buffer = new Uint16Array(this._capacity);
        this._completedStrings = null;
        this._bufferLength = 0;
    }
    StringBuilder.prototype.reset = function () {
        this._completedStrings = null;
        this._bufferLength = 0;
    };
    StringBuilder.prototype.build = function () {
        if (this._completedStrings !== null) {
            this._flushBuffer();
            return this._completedStrings.join('');
        }
        return this._buildBuffer();
    };
    StringBuilder.prototype._buildBuffer = function () {
        if (this._bufferLength === 0) {
            return '';
        }
        var view = new Uint16Array(this._buffer.buffer, 0, this._bufferLength);
        return getPlatformTextDecoder().decode(view);
    };
    StringBuilder.prototype._flushBuffer = function () {
        var bufferString = this._buildBuffer();
        this._bufferLength = 0;
        if (this._completedStrings === null) {
            this._completedStrings = [bufferString];
        }
        else {
            this._completedStrings[this._completedStrings.length] = bufferString;
        }
    };
    StringBuilder.prototype.write1 = function (charCode) {
        var remainingSpace = this._capacity - this._bufferLength;
        if (remainingSpace <= 1) {
            if (remainingSpace === 0 || strings.isHighSurrogate(charCode)) {
                this._flushBuffer();
            }
        }
        this._buffer[this._bufferLength++] = charCode;
    };
    StringBuilder.prototype.appendASCII = function (charCode) {
        if (this._bufferLength === this._capacity) {
            // buffer is full
            this._flushBuffer();
        }
        this._buffer[this._bufferLength++] = charCode;
    };
    StringBuilder.prototype.appendASCIIString = function (str) {
        var strLen = str.length;
        if (this._bufferLength + strLen >= this._capacity) {
            // This string does not fit in the remaining buffer space
            this._flushBuffer();
            this._completedStrings[this._completedStrings.length] = str;
            return;
        }
        for (var i = 0; i < strLen; i++) {
            this._buffer[this._bufferLength++] = str.charCodeAt(i);
        }
    };
    return StringBuilder;
}());
var CompatStringBuilder = /** @class */ (function () {
    function CompatStringBuilder() {
        this._pieces = [];
        this._piecesLen = 0;
    }
    CompatStringBuilder.prototype.reset = function () {
        this._pieces = [];
        this._piecesLen = 0;
    };
    CompatStringBuilder.prototype.build = function () {
        return this._pieces.join('');
    };
    CompatStringBuilder.prototype.write1 = function (charCode) {
        this._pieces[this._piecesLen++] = String.fromCharCode(charCode);
    };
    CompatStringBuilder.prototype.appendASCII = function (charCode) {
        this._pieces[this._piecesLen++] = String.fromCharCode(charCode);
    };
    CompatStringBuilder.prototype.appendASCIIString = function (str) {
        this._pieces[this._piecesLen++] = str;
    };
    return CompatStringBuilder;
}());
