/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __extends = (this && this.__extends) || (function () {
    var extendStatics = function (d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (Object.prototype.hasOwnProperty.call(b, p)) d[p] = b[p]; };
        return extendStatics(d, b);
    };
    return function (d, b) {
        if (typeof b !== "function" && b !== null)
            throw new TypeError("Class extends value " + String(b) + " is not a constructor or null");
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
var __values = (this && this.__values) || function(o) {
    var s = typeof Symbol === "function" && Symbol.iterator, m = s && o[s], i = 0;
    if (m) return m.call(o);
    if (o && typeof o.length === "number") return {
        next: function () {
            if (o && i >= o.length) o = void 0;
            return { value: o && o[i++], done: !o };
        }
    };
    throw new TypeError(s ? "Object is not iterable." : "Symbol.iterator is not defined.");
};
var __read = (this && this.__read) || function (o, n) {
    var m = typeof Symbol === "function" && o[Symbol.iterator];
    if (!m) return o;
    var i = m.call(o), r, ar = [], e;
    try {
        while ((n === void 0 || n-- > 0) && !(r = i.next()).done) ar.push(r.value);
    }
    catch (error) { e = { error: error }; }
    finally {
        try {
            if (r && !r.done && (m = i["return"])) m.call(i);
        }
        finally { if (e) throw e.error; }
    }
    return ar;
};
import { onUnexpectedError } from '../../../base/common/errors';
import * as strings from '../../../base/common/strings';
import { CursorCollection } from './cursorCollection';
import { CursorColumns, CursorContext, CursorState, EditOperationResult } from './cursorCommon';
import { DeleteOperations } from './cursorDeleteOperations';
import { TypeOperations, TypeWithAutoClosingCommand } from './cursorTypeOperations';
import { Range } from '../core/range';
import { Selection } from '../core/selection';
import { ViewCursorStateChangedEvent, ViewRevealRangeRequestEvent } from '../view/viewEvents';
import { dispose, Disposable } from '../../../base/common/lifecycle';
import { CursorStateChangedEvent } from '../viewModel/viewModelEventDispatcher';
/**
 * A snapshot of the cursor and the model state
 */
var CursorModelState = /** @class */ (function () {
    function CursorModelState(model, cursor) {
        this.modelVersionId = model.getVersionId();
        this.cursorState = cursor.getCursorStates();
    }
    CursorModelState.prototype.equals = function (other) {
        if (!other) {
            return false;
        }
        if (this.modelVersionId !== other.modelVersionId) {
            return false;
        }
        if (this.cursorState.length !== other.cursorState.length) {
            return false;
        }
        for (var i = 0, len = this.cursorState.length; i < len; i++) {
            if (!this.cursorState[i].equals(other.cursorState[i])) {
                return false;
            }
        }
        return true;
    };
    return CursorModelState;
}());
export { CursorModelState };
var AutoClosedAction = /** @class */ (function () {
    function AutoClosedAction(model, autoClosedCharactersDecorations, autoClosedEnclosingDecorations) {
        this._model = model;
        this._autoClosedCharactersDecorations = autoClosedCharactersDecorations;
        this._autoClosedEnclosingDecorations = autoClosedEnclosingDecorations;
    }
    AutoClosedAction.getAllAutoClosedCharacters = function (autoClosedActions) {
        var e_1, _a;
        var autoClosedCharacters = [];
        try {
            for (var autoClosedActions_1 = __values(autoClosedActions), autoClosedActions_1_1 = autoClosedActions_1.next(); !autoClosedActions_1_1.done; autoClosedActions_1_1 = autoClosedActions_1.next()) {
                var autoClosedAction = autoClosedActions_1_1.value;
                autoClosedCharacters = autoClosedCharacters.concat(autoClosedAction.getAutoClosedCharactersRanges());
            }
        }
        catch (e_1_1) { e_1 = { error: e_1_1 }; }
        finally {
            try {
                if (autoClosedActions_1_1 && !autoClosedActions_1_1.done && (_a = autoClosedActions_1.return)) _a.call(autoClosedActions_1);
            }
            finally { if (e_1) throw e_1.error; }
        }
        return autoClosedCharacters;
    };
    AutoClosedAction.prototype.dispose = function () {
        this._autoClosedCharactersDecorations = this._model.deltaDecorations(this._autoClosedCharactersDecorations, []);
        this._autoClosedEnclosingDecorations = this._model.deltaDecorations(this._autoClosedEnclosingDecorations, []);
    };
    AutoClosedAction.prototype.getAutoClosedCharactersRanges = function () {
        var result = [];
        for (var i = 0; i < this._autoClosedCharactersDecorations.length; i++) {
            var decorationRange = this._model.getDecorationRange(this._autoClosedCharactersDecorations[i]);
            if (decorationRange) {
                result.push(decorationRange);
            }
        }
        return result;
    };
    AutoClosedAction.prototype.isValid = function (selections) {
        var enclosingRanges = [];
        for (var i = 0; i < this._autoClosedEnclosingDecorations.length; i++) {
            var decorationRange = this._model.getDecorationRange(this._autoClosedEnclosingDecorations[i]);
            if (decorationRange) {
                enclosingRanges.push(decorationRange);
                if (decorationRange.startLineNumber !== decorationRange.endLineNumber) {
                    // Stop tracking if the range becomes multiline...
                    return false;
                }
            }
        }
        enclosingRanges.sort(Range.compareRangesUsingStarts);
        selections.sort(Range.compareRangesUsingStarts);
        for (var i = 0; i < selections.length; i++) {
            if (i >= enclosingRanges.length) {
                return false;
            }
            if (!enclosingRanges[i].strictContainsRange(selections[i])) {
                return false;
            }
        }
        return true;
    };
    return AutoClosedAction;
}());
var Cursor = /** @class */ (function (_super) {
    __extends(Cursor, _super);
    function Cursor(model, viewModel, coordinatesConverter, cursorConfig) {
        var _this = _super.call(this) || this;
        _this._model = model;
        _this._knownModelVersionId = _this._model.getVersionId();
        _this._viewModel = viewModel;
        _this._coordinatesConverter = coordinatesConverter;
        _this.context = new CursorContext(_this._model, _this._coordinatesConverter, cursorConfig);
        _this._cursors = new CursorCollection(_this.context);
        _this._hasFocus = false;
        _this._isHandling = false;
        _this._isDoingComposition = false;
        _this._selectionsWhenCompositionStarted = null;
        _this._columnSelectData = null;
        _this._autoClosedActions = [];
        _this._prevEditOperationType = 0 /* Other */;
        return _this;
    }
    Cursor.prototype.dispose = function () {
        this._cursors.dispose();
        this._autoClosedActions = dispose(this._autoClosedActions);
        _super.prototype.dispose.call(this);
    };
    Cursor.prototype.updateConfiguration = function (cursorConfig) {
        this.context = new CursorContext(this._model, this._coordinatesConverter, cursorConfig);
        this._cursors.updateContext(this.context);
    };
    Cursor.prototype.onLineMappingChanged = function (eventsCollector) {
        if (this._knownModelVersionId !== this._model.getVersionId()) {
            // There are model change events that I didn't yet receive.
            //
            // This can happen when editing the model, and the view model receives the change events first,
            // and the view model emits line mapping changed events, all before the cursor gets a chance to
            // recover from markers.
            //
            // The model change listener above will be called soon and we'll ensure a valid cursor state there.
            return;
        }
        // Ensure valid state
        this.setStates(eventsCollector, 'viewModel', 0 /* NotSet */, this.getCursorStates());
    };
    Cursor.prototype.setHasFocus = function (hasFocus) {
        this._hasFocus = hasFocus;
    };
    Cursor.prototype._validateAutoClosedActions = function () {
        if (this._autoClosedActions.length > 0) {
            var selections = this._cursors.getSelections();
            for (var i = 0; i < this._autoClosedActions.length; i++) {
                var autoClosedAction = this._autoClosedActions[i];
                if (!autoClosedAction.isValid(selections)) {
                    autoClosedAction.dispose();
                    this._autoClosedActions.splice(i, 1);
                    i--;
                }
            }
        }
    };
    // ------ some getters/setters
    Cursor.prototype.getPrimaryCursorState = function () {
        return this._cursors.getPrimaryCursor();
    };
    Cursor.prototype.getLastAddedCursorIndex = function () {
        return this._cursors.getLastAddedCursorIndex();
    };
    Cursor.prototype.getCursorStates = function () {
        return this._cursors.getAll();
    };
    Cursor.prototype.setStates = function (eventsCollector, source, reason, states) {
        var reachedMaxCursorCount = false;
        if (states !== null && states.length > Cursor.MAX_CURSOR_COUNT) {
            states = states.slice(0, Cursor.MAX_CURSOR_COUNT);
            reachedMaxCursorCount = true;
        }
        var oldState = new CursorModelState(this._model, this);
        this._cursors.setStates(states);
        this._cursors.normalize();
        this._columnSelectData = null;
        this._validateAutoClosedActions();
        return this._emitStateChangedIfNecessary(eventsCollector, source, reason, oldState, reachedMaxCursorCount);
    };
    Cursor.prototype.setCursorColumnSelectData = function (columnSelectData) {
        this._columnSelectData = columnSelectData;
    };
    Cursor.prototype.revealPrimary = function (eventsCollector, source, revealHorizontal, scrollType) {
        var viewPositions = this._cursors.getViewPositions();
        if (viewPositions.length > 1) {
            this._emitCursorRevealRange(eventsCollector, source, null, this._cursors.getViewSelections(), 0 /* Simple */, revealHorizontal, scrollType);
            return;
        }
        else {
            var viewPosition = viewPositions[0];
            var viewRange = new Range(viewPosition.lineNumber, viewPosition.column, viewPosition.lineNumber, viewPosition.column);
            this._emitCursorRevealRange(eventsCollector, source, viewRange, null, 0 /* Simple */, revealHorizontal, scrollType);
        }
    };
    Cursor.prototype._revealPrimaryCursor = function (eventsCollector, source, verticalType, revealHorizontal, scrollType) {
        var viewPositions = this._cursors.getViewPositions();
        if (viewPositions.length > 1) {
            this._emitCursorRevealRange(eventsCollector, source, null, this._cursors.getViewSelections(), verticalType, revealHorizontal, scrollType);
        }
        else {
            var viewPosition = viewPositions[0];
            var viewRange = new Range(viewPosition.lineNumber, viewPosition.column, viewPosition.lineNumber, viewPosition.column);
            this._emitCursorRevealRange(eventsCollector, source, viewRange, null, verticalType, revealHorizontal, scrollType);
        }
    };
    Cursor.prototype._emitCursorRevealRange = function (eventsCollector, source, viewRange, viewSelections, verticalType, revealHorizontal, scrollType) {
        eventsCollector.emitViewEvent(new ViewRevealRangeRequestEvent(source, viewRange, viewSelections, verticalType, revealHorizontal, scrollType));
    };
    Cursor.prototype.saveState = function () {
        var result = [];
        var selections = this._cursors.getSelections();
        for (var i = 0, len = selections.length; i < len; i++) {
            var selection = selections[i];
            result.push({
                inSelectionMode: !selection.isEmpty(),
                selectionStart: {
                    lineNumber: selection.selectionStartLineNumber,
                    column: selection.selectionStartColumn,
                },
                position: {
                    lineNumber: selection.positionLineNumber,
                    column: selection.positionColumn,
                }
            });
        }
        return result;
    };
    Cursor.prototype.restoreState = function (eventsCollector, states) {
        var desiredSelections = [];
        for (var i = 0, len = states.length; i < len; i++) {
            var state = states[i];
            var positionLineNumber = 1;
            var positionColumn = 1;
            // Avoid missing properties on the literal
            if (state.position && state.position.lineNumber) {
                positionLineNumber = state.position.lineNumber;
            }
            if (state.position && state.position.column) {
                positionColumn = state.position.column;
            }
            var selectionStartLineNumber = positionLineNumber;
            var selectionStartColumn = positionColumn;
            // Avoid missing properties on the literal
            if (state.selectionStart && state.selectionStart.lineNumber) {
                selectionStartLineNumber = state.selectionStart.lineNumber;
            }
            if (state.selectionStart && state.selectionStart.column) {
                selectionStartColumn = state.selectionStart.column;
            }
            desiredSelections.push({
                selectionStartLineNumber: selectionStartLineNumber,
                selectionStartColumn: selectionStartColumn,
                positionLineNumber: positionLineNumber,
                positionColumn: positionColumn
            });
        }
        this.setStates(eventsCollector, 'restoreState', 0 /* NotSet */, CursorState.fromModelSelections(desiredSelections));
        this.revealPrimary(eventsCollector, 'restoreState', true, 1 /* Immediate */);
    };
    Cursor.prototype.onModelContentChanged = function (eventsCollector, e) {
        this._knownModelVersionId = e.versionId;
        if (this._isHandling) {
            return;
        }
        var hadFlushEvent = e.containsEvent(1 /* Flush */);
        this._prevEditOperationType = 0 /* Other */;
        if (hadFlushEvent) {
            // a model.setValue() was called
            this._cursors.dispose();
            this._cursors = new CursorCollection(this.context);
            this._validateAutoClosedActions();
            this._emitStateChangedIfNecessary(eventsCollector, 'model', 1 /* ContentFlush */, null, false);
        }
        else {
            if (this._hasFocus && e.resultingSelection && e.resultingSelection.length > 0) {
                var cursorState = CursorState.fromModelSelections(e.resultingSelection);
                if (this.setStates(eventsCollector, 'modelChange', e.isUndoing ? 5 /* Undo */ : e.isRedoing ? 6 /* Redo */ : 2 /* RecoverFromMarkers */, cursorState)) {
                    this._revealPrimaryCursor(eventsCollector, 'modelChange', 0 /* Simple */, true, 0 /* Smooth */);
                }
            }
            else {
                var selectionsFromMarkers = this._cursors.readSelectionFromMarkers();
                this.setStates(eventsCollector, 'modelChange', 2 /* RecoverFromMarkers */, CursorState.fromModelSelections(selectionsFromMarkers));
            }
        }
    };
    Cursor.prototype.getSelection = function () {
        return this._cursors.getPrimaryCursor().modelState.selection;
    };
    Cursor.prototype.getTopMostViewPosition = function () {
        return this._cursors.getTopMostViewPosition();
    };
    Cursor.prototype.getBottomMostViewPosition = function () {
        return this._cursors.getBottomMostViewPosition();
    };
    Cursor.prototype.getCursorColumnSelectData = function () {
        if (this._columnSelectData) {
            return this._columnSelectData;
        }
        var primaryCursor = this._cursors.getPrimaryCursor();
        var viewSelectionStart = primaryCursor.viewState.selectionStart.getStartPosition();
        var viewPosition = primaryCursor.viewState.position;
        return {
            isReal: false,
            fromViewLineNumber: viewSelectionStart.lineNumber,
            fromViewVisualColumn: CursorColumns.visibleColumnFromColumn2(this.context.cursorConfig, this._viewModel, viewSelectionStart),
            toViewLineNumber: viewPosition.lineNumber,
            toViewVisualColumn: CursorColumns.visibleColumnFromColumn2(this.context.cursorConfig, this._viewModel, viewPosition),
        };
    };
    Cursor.prototype.getSelections = function () {
        return this._cursors.getSelections();
    };
    Cursor.prototype.getPosition = function () {
        return this._cursors.getPrimaryCursor().modelState.position;
    };
    Cursor.prototype.setSelections = function (eventsCollector, source, selections, reason) {
        this.setStates(eventsCollector, source, reason, CursorState.fromModelSelections(selections));
    };
    Cursor.prototype.getPrevEditOperationType = function () {
        return this._prevEditOperationType;
    };
    Cursor.prototype.setPrevEditOperationType = function (type) {
        this._prevEditOperationType = type;
    };
    // ------ auxiliary handling logic
    Cursor.prototype._pushAutoClosedAction = function (autoClosedCharactersRanges, autoClosedEnclosingRanges) {
        var autoClosedCharactersDeltaDecorations = [];
        var autoClosedEnclosingDeltaDecorations = [];
        for (var i = 0, len = autoClosedCharactersRanges.length; i < len; i++) {
            autoClosedCharactersDeltaDecorations.push({
                range: autoClosedCharactersRanges[i],
                options: {
                    inlineClassName: 'auto-closed-character',
                    stickiness: 1 /* NeverGrowsWhenTypingAtEdges */
                }
            });
            autoClosedEnclosingDeltaDecorations.push({
                range: autoClosedEnclosingRanges[i],
                options: {
                    stickiness: 1 /* NeverGrowsWhenTypingAtEdges */
                }
            });
        }
        var autoClosedCharactersDecorations = this._model.deltaDecorations([], autoClosedCharactersDeltaDecorations);
        var autoClosedEnclosingDecorations = this._model.deltaDecorations([], autoClosedEnclosingDeltaDecorations);
        this._autoClosedActions.push(new AutoClosedAction(this._model, autoClosedCharactersDecorations, autoClosedEnclosingDecorations));
    };
    Cursor.prototype._executeEditOperation = function (opResult) {
        if (!opResult) {
            // Nothing to execute
            return;
        }
        if (opResult.shouldPushStackElementBefore) {
            this._model.pushStackElement();
        }
        var result = CommandExecutor.executeCommands(this._model, this._cursors.getSelections(), opResult.commands);
        if (result) {
            // The commands were applied correctly
            this._interpretCommandResult(result);
            // Check for auto-closing closed characters
            var autoClosedCharactersRanges = [];
            var autoClosedEnclosingRanges = [];
            for (var i = 0; i < opResult.commands.length; i++) {
                var command = opResult.commands[i];
                if (command instanceof TypeWithAutoClosingCommand && command.enclosingRange && command.closeCharacterRange) {
                    autoClosedCharactersRanges.push(command.closeCharacterRange);
                    autoClosedEnclosingRanges.push(command.enclosingRange);
                }
            }
            if (autoClosedCharactersRanges.length > 0) {
                this._pushAutoClosedAction(autoClosedCharactersRanges, autoClosedEnclosingRanges);
            }
            this._prevEditOperationType = opResult.type;
        }
        if (opResult.shouldPushStackElementAfter) {
            this._model.pushStackElement();
        }
    };
    Cursor.prototype._interpretCommandResult = function (cursorState) {
        if (!cursorState || cursorState.length === 0) {
            cursorState = this._cursors.readSelectionFromMarkers();
        }
        this._columnSelectData = null;
        this._cursors.setSelections(cursorState);
        this._cursors.normalize();
    };
    // -----------------------------------------------------------------------------------------------------------
    // ----- emitting events
    Cursor.prototype._emitStateChangedIfNecessary = function (eventsCollector, source, reason, oldState, reachedMaxCursorCount) {
        var newState = new CursorModelState(this._model, this);
        if (newState.equals(oldState)) {
            return false;
        }
        var selections = this._cursors.getSelections();
        var viewSelections = this._cursors.getViewSelections();
        // Let the view get the event first.
        eventsCollector.emitViewEvent(new ViewCursorStateChangedEvent(viewSelections, selections));
        // Only after the view has been notified, let the rest of the world know...
        if (!oldState
            || oldState.cursorState.length !== newState.cursorState.length
            || newState.cursorState.some(function (newCursorState, i) { return !newCursorState.modelState.equals(oldState.cursorState[i].modelState); })) {
            var oldSelections = oldState ? oldState.cursorState.map(function (s) { return s.modelState.selection; }) : null;
            var oldModelVersionId = oldState ? oldState.modelVersionId : 0;
            eventsCollector.emitOutgoingEvent(new CursorStateChangedEvent(oldSelections, selections, oldModelVersionId, newState.modelVersionId, source || 'keyboard', reason, reachedMaxCursorCount));
        }
        return true;
    };
    // -----------------------------------------------------------------------------------------------------------
    // ----- handlers beyond this point
    Cursor.prototype._findAutoClosingPairs = function (edits) {
        if (!edits.length) {
            return null;
        }
        var indices = [];
        for (var i = 0, len = edits.length; i < len; i++) {
            var edit = edits[i];
            if (!edit.text || edit.text.indexOf('\n') >= 0) {
                return null;
            }
            var m = edit.text.match(/([)\]}>'"`])([^)\]}>'"`]*)$/);
            if (!m) {
                return null;
            }
            var closeChar = m[1];
            var autoClosingPairsCandidates = this.context.cursorConfig.autoClosingPairs.autoClosingPairsCloseSingleChar.get(closeChar);
            if (!autoClosingPairsCandidates || autoClosingPairsCandidates.length !== 1) {
                return null;
            }
            var openChar = autoClosingPairsCandidates[0].open;
            var closeCharIndex = edit.text.length - m[2].length - 1;
            var openCharIndex = edit.text.lastIndexOf(openChar, closeCharIndex - 1);
            if (openCharIndex === -1) {
                return null;
            }
            indices.push([openCharIndex, closeCharIndex]);
        }
        return indices;
    };
    Cursor.prototype.executeEdits = function (eventsCollector, source, edits, cursorStateComputer) {
        var _this = this;
        var autoClosingIndices = null;
        if (source === 'snippet') {
            autoClosingIndices = this._findAutoClosingPairs(edits);
        }
        if (autoClosingIndices) {
            edits[0]._isTracked = true;
        }
        var autoClosedCharactersRanges = [];
        var autoClosedEnclosingRanges = [];
        var selections = this._model.pushEditOperations(this.getSelections(), edits, function (undoEdits) {
            if (autoClosingIndices) {
                for (var i = 0, len = autoClosingIndices.length; i < len; i++) {
                    var _a = __read(autoClosingIndices[i], 2), openCharInnerIndex = _a[0], closeCharInnerIndex = _a[1];
                    var undoEdit = undoEdits[i];
                    var lineNumber = undoEdit.range.startLineNumber;
                    var openCharIndex = undoEdit.range.startColumn - 1 + openCharInnerIndex;
                    var closeCharIndex = undoEdit.range.startColumn - 1 + closeCharInnerIndex;
                    autoClosedCharactersRanges.push(new Range(lineNumber, closeCharIndex + 1, lineNumber, closeCharIndex + 2));
                    autoClosedEnclosingRanges.push(new Range(lineNumber, openCharIndex + 1, lineNumber, closeCharIndex + 2));
                }
            }
            var selections = cursorStateComputer(undoEdits);
            if (selections) {
                // Don't recover the selection from markers because
                // we know what it should be.
                _this._isHandling = true;
            }
            return selections;
        });
        if (selections) {
            this._isHandling = false;
            this.setSelections(eventsCollector, source, selections, 0 /* NotSet */);
        }
        if (autoClosedCharactersRanges.length > 0) {
            this._pushAutoClosedAction(autoClosedCharactersRanges, autoClosedEnclosingRanges);
        }
    };
    Cursor.prototype._executeEdit = function (callback, eventsCollector, source, cursorChangeReason) {
        if (cursorChangeReason === void 0) { cursorChangeReason = 0 /* NotSet */; }
        if (this.context.cursorConfig.readOnly) {
            // we cannot edit when read only...
            return;
        }
        var oldState = new CursorModelState(this._model, this);
        this._cursors.stopTrackingSelections();
        this._isHandling = true;
        try {
            this._cursors.ensureValidState();
            callback();
        }
        catch (err) {
            onUnexpectedError(err);
        }
        this._isHandling = false;
        this._cursors.startTrackingSelections();
        this._validateAutoClosedActions();
        if (this._emitStateChangedIfNecessary(eventsCollector, source, cursorChangeReason, oldState, false)) {
            this._revealPrimaryCursor(eventsCollector, source, 0 /* Simple */, true, 0 /* Smooth */);
        }
    };
    Cursor.prototype.setIsDoingComposition = function (isDoingComposition) {
        this._isDoingComposition = isDoingComposition;
    };
    Cursor.prototype.startComposition = function (eventsCollector) {
        this._selectionsWhenCompositionStarted = this.getSelections().slice(0);
    };
    Cursor.prototype.endComposition = function (eventsCollector, source) {
        var _this = this;
        this._executeEdit(function () {
            if (source === 'keyboard') {
                // composition finishes, let's check if we need to auto complete if necessary.
                var autoClosedCharacters = AutoClosedAction.getAllAutoClosedCharacters(_this._autoClosedActions);
                _this._executeEditOperation(TypeOperations.compositionEndWithInterceptors(_this._prevEditOperationType, _this.context.cursorConfig, _this._model, _this._selectionsWhenCompositionStarted, _this.getSelections(), autoClosedCharacters));
                _this._selectionsWhenCompositionStarted = null;
            }
        }, eventsCollector, source);
    };
    Cursor.prototype.type = function (eventsCollector, text, source) {
        var _this = this;
        this._executeEdit(function () {
            if (source === 'keyboard') {
                // If this event is coming straight from the keyboard, look for electric characters and enter
                var len = text.length;
                var offset = 0;
                while (offset < len) {
                    var charLength = strings.nextCharLength(text, offset);
                    var chr = text.substr(offset, charLength);
                    // Here we must interpret each typed character individually
                    var autoClosedCharacters = AutoClosedAction.getAllAutoClosedCharacters(_this._autoClosedActions);
                    _this._executeEditOperation(TypeOperations.typeWithInterceptors(_this._isDoingComposition, _this._prevEditOperationType, _this.context.cursorConfig, _this._model, _this.getSelections(), autoClosedCharacters, chr));
                    offset += charLength;
                }
            }
            else {
                _this._executeEditOperation(TypeOperations.typeWithoutInterceptors(_this._prevEditOperationType, _this.context.cursorConfig, _this._model, _this.getSelections(), text));
            }
        }, eventsCollector, source);
    };
    Cursor.prototype.compositionType = function (eventsCollector, text, replacePrevCharCnt, replaceNextCharCnt, positionDelta, source) {
        var _this = this;
        if (text.length === 0 && replacePrevCharCnt === 0 && replaceNextCharCnt === 0) {
            // this edit is a no-op
            if (positionDelta !== 0) {
                // but it still wants to move the cursor
                var newSelections = this.getSelections().map(function (selection) {
                    var position = selection.getPosition();
                    return new Selection(position.lineNumber, position.column + positionDelta, position.lineNumber, position.column + positionDelta);
                });
                this.setSelections(eventsCollector, source, newSelections, 0 /* NotSet */);
            }
            return;
        }
        this._executeEdit(function () {
            _this._executeEditOperation(TypeOperations.compositionType(_this._prevEditOperationType, _this.context.cursorConfig, _this._model, _this.getSelections(), text, replacePrevCharCnt, replaceNextCharCnt, positionDelta));
        }, eventsCollector, source);
    };
    Cursor.prototype.paste = function (eventsCollector, text, pasteOnNewLine, multicursorText, source) {
        var _this = this;
        this._executeEdit(function () {
            _this._executeEditOperation(TypeOperations.paste(_this.context.cursorConfig, _this._model, _this.getSelections(), text, pasteOnNewLine, multicursorText || []));
        }, eventsCollector, source, 4 /* Paste */);
    };
    Cursor.prototype.cut = function (eventsCollector, source) {
        var _this = this;
        this._executeEdit(function () {
            _this._executeEditOperation(DeleteOperations.cut(_this.context.cursorConfig, _this._model, _this.getSelections()));
        }, eventsCollector, source);
    };
    Cursor.prototype.executeCommand = function (eventsCollector, command, source) {
        var _this = this;
        this._executeEdit(function () {
            _this._cursors.killSecondaryCursors();
            _this._executeEditOperation(new EditOperationResult(0 /* Other */, [command], {
                shouldPushStackElementBefore: false,
                shouldPushStackElementAfter: false
            }));
        }, eventsCollector, source);
    };
    Cursor.prototype.executeCommands = function (eventsCollector, commands, source) {
        var _this = this;
        this._executeEdit(function () {
            _this._executeEditOperation(new EditOperationResult(0 /* Other */, commands, {
                shouldPushStackElementBefore: false,
                shouldPushStackElementAfter: false
            }));
        }, eventsCollector, source);
    };
    Cursor.MAX_CURSOR_COUNT = 10000;
    return Cursor;
}(Disposable));
export { Cursor };
var CommandExecutor = /** @class */ (function () {
    function CommandExecutor() {
    }
    CommandExecutor.executeCommands = function (model, selectionsBefore, commands) {
        var ctx = {
            model: model,
            selectionsBefore: selectionsBefore,
            trackedRanges: [],
            trackedRangesDirection: []
        };
        var result = this._innerExecuteCommands(ctx, commands);
        for (var i = 0, len = ctx.trackedRanges.length; i < len; i++) {
            ctx.model._setTrackedRange(ctx.trackedRanges[i], null, 0 /* AlwaysGrowsWhenTypingAtEdges */);
        }
        return result;
    };
    CommandExecutor._innerExecuteCommands = function (ctx, commands) {
        var e_2, _a;
        if (this._arrayIsEmpty(commands)) {
            return null;
        }
        var commandsData = this._getEditOperations(ctx, commands);
        if (commandsData.operations.length === 0) {
            return null;
        }
        var rawOperations = commandsData.operations;
        var loserCursorsMap = this._getLoserCursorMap(rawOperations);
        if (loserCursorsMap.hasOwnProperty('0')) {
            // These commands are very messed up
            console.warn('Ignoring commands');
            return null;
        }
        // Remove operations belonging to losing cursors
        var filteredOperations = [];
        for (var i = 0, len = rawOperations.length; i < len; i++) {
            if (!loserCursorsMap.hasOwnProperty(rawOperations[i].identifier.major.toString())) {
                filteredOperations.push(rawOperations[i]);
            }
        }
        // TODO@Alex: find a better way to do this.
        // give the hint that edit operations are tracked to the model
        if (commandsData.hadTrackedEditOperation && filteredOperations.length > 0) {
            filteredOperations[0]._isTracked = true;
        }
        var selectionsAfter = ctx.model.pushEditOperations(ctx.selectionsBefore, filteredOperations, function (inverseEditOperations) {
            var e_3, _a;
            var groupedInverseEditOperations = [];
            for (var i = 0; i < ctx.selectionsBefore.length; i++) {
                groupedInverseEditOperations[i] = [];
            }
            try {
                for (var inverseEditOperations_1 = __values(inverseEditOperations), inverseEditOperations_1_1 = inverseEditOperations_1.next(); !inverseEditOperations_1_1.done; inverseEditOperations_1_1 = inverseEditOperations_1.next()) {
                    var op = inverseEditOperations_1_1.value;
                    if (!op.identifier) {
                        // perhaps auto whitespace trim edits
                        continue;
                    }
                    groupedInverseEditOperations[op.identifier.major].push(op);
                }
            }
            catch (e_3_1) { e_3 = { error: e_3_1 }; }
            finally {
                try {
                    if (inverseEditOperations_1_1 && !inverseEditOperations_1_1.done && (_a = inverseEditOperations_1.return)) _a.call(inverseEditOperations_1);
                }
                finally { if (e_3) throw e_3.error; }
            }
            var minorBasedSorter = function (a, b) {
                return a.identifier.minor - b.identifier.minor;
            };
            var cursorSelections = [];
            var _loop_1 = function (i) {
                if (groupedInverseEditOperations[i].length > 0) {
                    groupedInverseEditOperations[i].sort(minorBasedSorter);
                    cursorSelections[i] = commands[i].computeCursorState(ctx.model, {
                        getInverseEditOperations: function () {
                            return groupedInverseEditOperations[i];
                        },
                        getTrackedSelection: function (id) {
                            var idx = parseInt(id, 10);
                            var range = ctx.model._getTrackedRange(ctx.trackedRanges[idx]);
                            if (ctx.trackedRangesDirection[idx] === 0 /* LTR */) {
                                return new Selection(range.startLineNumber, range.startColumn, range.endLineNumber, range.endColumn);
                            }
                            return new Selection(range.endLineNumber, range.endColumn, range.startLineNumber, range.startColumn);
                        }
                    });
                }
                else {
                    cursorSelections[i] = ctx.selectionsBefore[i];
                }
            };
            for (var i = 0; i < ctx.selectionsBefore.length; i++) {
                _loop_1(i);
            }
            return cursorSelections;
        });
        if (!selectionsAfter) {
            selectionsAfter = ctx.selectionsBefore;
        }
        // Extract losing cursors
        var losingCursors = [];
        for (var losingCursorIndex in loserCursorsMap) {
            if (loserCursorsMap.hasOwnProperty(losingCursorIndex)) {
                losingCursors.push(parseInt(losingCursorIndex, 10));
            }
        }
        // Sort losing cursors descending
        losingCursors.sort(function (a, b) {
            return b - a;
        });
        try {
            // Remove losing cursors
            for (var losingCursors_1 = __values(losingCursors), losingCursors_1_1 = losingCursors_1.next(); !losingCursors_1_1.done; losingCursors_1_1 = losingCursors_1.next()) {
                var losingCursor = losingCursors_1_1.value;
                selectionsAfter.splice(losingCursor, 1);
            }
        }
        catch (e_2_1) { e_2 = { error: e_2_1 }; }
        finally {
            try {
                if (losingCursors_1_1 && !losingCursors_1_1.done && (_a = losingCursors_1.return)) _a.call(losingCursors_1);
            }
            finally { if (e_2) throw e_2.error; }
        }
        return selectionsAfter;
    };
    CommandExecutor._arrayIsEmpty = function (commands) {
        for (var i = 0, len = commands.length; i < len; i++) {
            if (commands[i]) {
                return false;
            }
        }
        return true;
    };
    CommandExecutor._getEditOperations = function (ctx, commands) {
        var operations = [];
        var hadTrackedEditOperation = false;
        for (var i = 0, len = commands.length; i < len; i++) {
            var command = commands[i];
            if (command) {
                var r = this._getEditOperationsFromCommand(ctx, i, command);
                operations = operations.concat(r.operations);
                hadTrackedEditOperation = hadTrackedEditOperation || r.hadTrackedEditOperation;
            }
        }
        return {
            operations: operations,
            hadTrackedEditOperation: hadTrackedEditOperation
        };
    };
    CommandExecutor._getEditOperationsFromCommand = function (ctx, majorIdentifier, command) {
        // This method acts as a transaction, if the command fails
        // everything it has done is ignored
        var operations = [];
        var operationMinor = 0;
        var addEditOperation = function (range, text, forceMoveMarkers) {
            if (forceMoveMarkers === void 0) { forceMoveMarkers = false; }
            if (Range.isEmpty(range) && text === '') {
                // This command wants to add a no-op => no thank you
                return;
            }
            operations.push({
                identifier: {
                    major: majorIdentifier,
                    minor: operationMinor++
                },
                range: range,
                text: text,
                forceMoveMarkers: forceMoveMarkers,
                isAutoWhitespaceEdit: command.insertsAutoWhitespace
            });
        };
        var hadTrackedEditOperation = false;
        var addTrackedEditOperation = function (selection, text, forceMoveMarkers) {
            hadTrackedEditOperation = true;
            addEditOperation(selection, text, forceMoveMarkers);
        };
        var trackSelection = function (_selection, trackPreviousOnEmpty) {
            var selection = Selection.liftSelection(_selection);
            var stickiness;
            if (selection.isEmpty()) {
                if (typeof trackPreviousOnEmpty === 'boolean') {
                    if (trackPreviousOnEmpty) {
                        stickiness = 2 /* GrowsOnlyWhenTypingBefore */;
                    }
                    else {
                        stickiness = 3 /* GrowsOnlyWhenTypingAfter */;
                    }
                }
                else {
                    // Try to lock it with surrounding text
                    var maxLineColumn = ctx.model.getLineMaxColumn(selection.startLineNumber);
                    if (selection.startColumn === maxLineColumn) {
                        stickiness = 2 /* GrowsOnlyWhenTypingBefore */;
                    }
                    else {
                        stickiness = 3 /* GrowsOnlyWhenTypingAfter */;
                    }
                }
            }
            else {
                stickiness = 1 /* NeverGrowsWhenTypingAtEdges */;
            }
            var l = ctx.trackedRanges.length;
            var id = ctx.model._setTrackedRange(null, selection, stickiness);
            ctx.trackedRanges[l] = id;
            ctx.trackedRangesDirection[l] = selection.getDirection();
            return l.toString();
        };
        var editOperationBuilder = {
            addEditOperation: addEditOperation,
            addTrackedEditOperation: addTrackedEditOperation,
            trackSelection: trackSelection
        };
        try {
            command.getEditOperations(ctx.model, editOperationBuilder);
        }
        catch (e) {
            // TODO@Alex use notification service if this should be user facing
            // e.friendlyMessage = nls.localize('corrupt.commands', "Unexpected exception while executing command.");
            onUnexpectedError(e);
            return {
                operations: [],
                hadTrackedEditOperation: false
            };
        }
        return {
            operations: operations,
            hadTrackedEditOperation: hadTrackedEditOperation
        };
    };
    CommandExecutor._getLoserCursorMap = function (operations) {
        // This is destructive on the array
        operations = operations.slice(0);
        // Sort operations with last one first
        operations.sort(function (a, b) {
            // Note the minus!
            return -(Range.compareRangesUsingEnds(a.range, b.range));
        });
        // Operations can not overlap!
        var loserCursorsMap = {};
        for (var i = 1; i < operations.length; i++) {
            var previousOp = operations[i - 1];
            var currentOp = operations[i];
            if (Range.getStartPosition(previousOp.range).isBefore(Range.getEndPosition(currentOp.range))) {
                var loserMajor = void 0;
                if (previousOp.identifier.major > currentOp.identifier.major) {
                    // previousOp loses the battle
                    loserMajor = previousOp.identifier.major;
                }
                else {
                    loserMajor = currentOp.identifier.major;
                }
                loserCursorsMap[loserMajor.toString()] = true;
                for (var j = 0; j < operations.length; j++) {
                    if (operations[j].identifier.major === loserMajor) {
                        operations.splice(j, 1);
                        if (j < i) {
                            i--;
                        }
                        j--;
                    }
                }
                if (i > 0) {
                    i--;
                }
            }
        }
        return loserCursorsMap;
    };
    return CommandExecutor;
}());
