/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __extends = (this && this.__extends) || (function () {
    var extendStatics = function (d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (Object.prototype.hasOwnProperty.call(b, p)) d[p] = b[p]; };
        return extendStatics(d, b);
    };
    return function (d, b) {
        if (typeof b !== "function" && b !== null)
            throw new TypeError("Class extends value " + String(b) + " is not a constructor or null");
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
var __read = (this && this.__read) || function (o, n) {
    var m = typeof Symbol === "function" && o[Symbol.iterator];
    if (!m) return o;
    var i = m.call(o), r, ar = [], e;
    try {
        while ((n === void 0 || n-- > 0) && !(r = i.next()).done) ar.push(r.value);
    }
    catch (error) { e = { error: error }; }
    finally {
        try {
            if (r && !r.done && (m = i["return"])) m.call(i);
        }
        finally { if (e) throw e.error; }
    }
    return ar;
};
var __values = (this && this.__values) || function(o) {
    var s = typeof Symbol === "function" && Symbol.iterator, m = s && o[s], i = 0;
    if (m) return m.call(o);
    if (o && typeof o.length === "number") return {
        next: function () {
            if (o && i >= o.length) o = void 0;
            return { value: o && o[i++], done: !o };
        }
    };
    throw new TypeError(s ? "Object is not iterable." : "Symbol.iterator is not defined.");
};
import './minimap.css';
import * as dom from '../../../../base/browser/dom';
import { createFastDomNode } from '../../../../base/browser/fastDomNode';
import { GlobalMouseMoveMonitor, standardMouseMoveMerger } from '../../../../base/browser/globalMouseMoveMonitor';
import { Disposable } from '../../../../base/common/lifecycle';
import * as platform from '../../../../base/common/platform';
import * as strings from '../../../../base/common/strings';
import { RenderedLinesCollection } from '../../view/viewLayer';
import { PartFingerprints, ViewPart } from '../../view/viewPart';
import { MINIMAP_GUTTER_WIDTH, EditorLayoutInfoComputer } from '../../../common/config/editorOptions';
import { Range } from '../../../common/core/range';
import { RGBA8 } from '../../../common/core/rgba';
import { MinimapTokensColorTracker } from '../../../common/viewModel/minimapTokensColorTracker';
import { ViewModelDecoration } from '../../../common/viewModel/viewModel';
import { minimapSelection, scrollbarShadow, minimapBackground, minimapSliderBackground, minimapSliderHoverBackground, minimapSliderActiveBackground } from '../../../../platform/theme/common/colorRegistry';
import { registerThemingParticipant } from '../../../../platform/theme/common/themeService';
import { Selection } from '../../../common/core/selection';
import { EventType, Gesture } from '../../../../base/browser/touch';
import { MinimapCharRendererFactory } from './minimapCharRendererFactory';
import { MinimapPosition } from '../../../common/model';
import { once } from '../../../../base/common/functional';
/**
 * The orthogonal distance to the slider at which dragging "resets". This implements "snapping"
 */
var MOUSE_DRAG_RESET_DISTANCE = 140;
var GUTTER_DECORATION_WIDTH = 2;
var MinimapOptions = /** @class */ (function () {
    function MinimapOptions(configuration, theme, tokensColorTracker) {
        var _this = this;
        var options = configuration.options;
        var pixelRatio = options.get(122 /* pixelRatio */);
        var layoutInfo = options.get(124 /* layoutInfo */);
        var minimapLayout = layoutInfo.minimap;
        var fontInfo = options.get(38 /* fontInfo */);
        var minimapOpts = options.get(59 /* minimap */);
        this.renderMinimap = minimapLayout.renderMinimap;
        this.size = minimapOpts.size;
        this.minimapHeightIsEditorHeight = minimapLayout.minimapHeightIsEditorHeight;
        this.scrollBeyondLastLine = options.get(89 /* scrollBeyondLastLine */);
        this.showSlider = minimapOpts.showSlider;
        this.pixelRatio = pixelRatio;
        this.typicalHalfwidthCharacterWidth = fontInfo.typicalHalfwidthCharacterWidth;
        this.lineHeight = options.get(53 /* lineHeight */);
        this.minimapLeft = minimapLayout.minimapLeft;
        this.minimapWidth = minimapLayout.minimapWidth;
        this.minimapHeight = layoutInfo.height;
        this.canvasInnerWidth = minimapLayout.minimapCanvasInnerWidth;
        this.canvasInnerHeight = minimapLayout.minimapCanvasInnerHeight;
        this.canvasOuterWidth = minimapLayout.minimapCanvasOuterWidth;
        this.canvasOuterHeight = minimapLayout.minimapCanvasOuterHeight;
        this.isSampling = minimapLayout.minimapIsSampling;
        this.editorHeight = layoutInfo.height;
        this.fontScale = minimapLayout.minimapScale;
        this.minimapLineHeight = minimapLayout.minimapLineHeight;
        this.minimapCharWidth = 1 /* BASE_CHAR_WIDTH */ * this.fontScale;
        this.charRenderer = once(function () { return MinimapCharRendererFactory.create(_this.fontScale, fontInfo.fontFamily); });
        this.backgroundColor = MinimapOptions._getMinimapBackground(theme, tokensColorTracker);
    }
    MinimapOptions._getMinimapBackground = function (theme, tokensColorTracker) {
        var themeColor = theme.getColor(minimapBackground);
        if (themeColor) {
            return new RGBA8(themeColor.rgba.r, themeColor.rgba.g, themeColor.rgba.b, themeColor.rgba.a);
        }
        return tokensColorTracker.getColor(2 /* DefaultBackground */);
    };
    MinimapOptions.prototype.equals = function (other) {
        return (this.renderMinimap === other.renderMinimap
            && this.size === other.size
            && this.minimapHeightIsEditorHeight === other.minimapHeightIsEditorHeight
            && this.scrollBeyondLastLine === other.scrollBeyondLastLine
            && this.showSlider === other.showSlider
            && this.pixelRatio === other.pixelRatio
            && this.typicalHalfwidthCharacterWidth === other.typicalHalfwidthCharacterWidth
            && this.lineHeight === other.lineHeight
            && this.minimapLeft === other.minimapLeft
            && this.minimapWidth === other.minimapWidth
            && this.minimapHeight === other.minimapHeight
            && this.canvasInnerWidth === other.canvasInnerWidth
            && this.canvasInnerHeight === other.canvasInnerHeight
            && this.canvasOuterWidth === other.canvasOuterWidth
            && this.canvasOuterHeight === other.canvasOuterHeight
            && this.isSampling === other.isSampling
            && this.editorHeight === other.editorHeight
            && this.fontScale === other.fontScale
            && this.minimapLineHeight === other.minimapLineHeight
            && this.minimapCharWidth === other.minimapCharWidth
            && this.backgroundColor && this.backgroundColor.equals(other.backgroundColor));
    };
    return MinimapOptions;
}());
var MinimapLayout = /** @class */ (function () {
    function MinimapLayout(scrollTop, scrollHeight, sliderNeeded, computedSliderRatio, sliderTop, sliderHeight, startLineNumber, endLineNumber) {
        this.scrollTop = scrollTop;
        this.scrollHeight = scrollHeight;
        this.sliderNeeded = sliderNeeded;
        this._computedSliderRatio = computedSliderRatio;
        this.sliderTop = sliderTop;
        this.sliderHeight = sliderHeight;
        this.startLineNumber = startLineNumber;
        this.endLineNumber = endLineNumber;
    }
    /**
     * Compute a desired `scrollPosition` such that the slider moves by `delta`.
     */
    MinimapLayout.prototype.getDesiredScrollTopFromDelta = function (delta) {
        return Math.round(this.scrollTop + delta / this._computedSliderRatio);
    };
    MinimapLayout.prototype.getDesiredScrollTopFromTouchLocation = function (pageY) {
        return Math.round((pageY - this.sliderHeight / 2) / this._computedSliderRatio);
    };
    MinimapLayout.create = function (options, viewportStartLineNumber, viewportEndLineNumber, viewportStartLineNumberVerticalOffset, viewportHeight, viewportContainsWhitespaceGaps, lineCount, realLineCount, scrollTop, scrollHeight, previousLayout) {
        var pixelRatio = options.pixelRatio;
        var minimapLineHeight = options.minimapLineHeight;
        var minimapLinesFitting = Math.floor(options.canvasInnerHeight / minimapLineHeight);
        var lineHeight = options.lineHeight;
        if (options.minimapHeightIsEditorHeight) {
            var logicalScrollHeight = (realLineCount * options.lineHeight
                + (options.scrollBeyondLastLine ? viewportHeight - options.lineHeight : 0));
            var sliderHeight_1 = Math.max(1, Math.floor(viewportHeight * viewportHeight / logicalScrollHeight));
            var maxMinimapSliderTop_1 = Math.max(0, options.minimapHeight - sliderHeight_1);
            // The slider can move from 0 to `maxMinimapSliderTop`
            // in the same way `scrollTop` can move from 0 to `scrollHeight` - `viewportHeight`.
            var computedSliderRatio_1 = (maxMinimapSliderTop_1) / (scrollHeight - viewportHeight);
            var sliderTop_1 = (scrollTop * computedSliderRatio_1);
            var sliderNeeded = (maxMinimapSliderTop_1 > 0);
            var maxLinesFitting = Math.floor(options.canvasInnerHeight / options.minimapLineHeight);
            return new MinimapLayout(scrollTop, scrollHeight, sliderNeeded, computedSliderRatio_1, sliderTop_1, sliderHeight_1, 1, Math.min(lineCount, maxLinesFitting));
        }
        // The visible line count in a viewport can change due to a number of reasons:
        //  a) with the same viewport width, different scroll positions can result in partial lines being visible:
        //    e.g. for a line height of 20, and a viewport height of 600
        //          * scrollTop = 0  => visible lines are [1, 30]
        //          * scrollTop = 10 => visible lines are [1, 31] (with lines 1 and 31 partially visible)
        //          * scrollTop = 20 => visible lines are [2, 31]
        //  b) whitespace gaps might make their way in the viewport (which results in a decrease in the visible line count)
        //  c) we could be in the scroll beyond last line case (which also results in a decrease in the visible line count, down to possibly only one line being visible)
        // We must first establish a desirable slider height.
        var sliderHeight;
        if (viewportContainsWhitespaceGaps && viewportEndLineNumber !== lineCount) {
            // case b) from above: there are whitespace gaps in the viewport.
            // In this case, the height of the slider directly reflects the visible line count.
            var viewportLineCount = viewportEndLineNumber - viewportStartLineNumber + 1;
            sliderHeight = Math.floor(viewportLineCount * minimapLineHeight / pixelRatio);
        }
        else {
            // The slider has a stable height
            var expectedViewportLineCount = viewportHeight / lineHeight;
            sliderHeight = Math.floor(expectedViewportLineCount * minimapLineHeight / pixelRatio);
        }
        var maxMinimapSliderTop;
        if (options.scrollBeyondLastLine) {
            // The minimap slider, when dragged all the way down, will contain the last line at its top
            maxMinimapSliderTop = (lineCount - 1) * minimapLineHeight / pixelRatio;
        }
        else {
            // The minimap slider, when dragged all the way down, will contain the last line at its bottom
            maxMinimapSliderTop = Math.max(0, lineCount * minimapLineHeight / pixelRatio - sliderHeight);
        }
        maxMinimapSliderTop = Math.min(options.minimapHeight - sliderHeight, maxMinimapSliderTop);
        // The slider can move from 0 to `maxMinimapSliderTop`
        // in the same way `scrollTop` can move from 0 to `scrollHeight` - `viewportHeight`.
        var computedSliderRatio = (maxMinimapSliderTop) / (scrollHeight - viewportHeight);
        var sliderTop = (scrollTop * computedSliderRatio);
        var extraLinesAtTheBottom = 0;
        if (options.scrollBeyondLastLine) {
            var expectedViewportLineCount = viewportHeight / lineHeight;
            extraLinesAtTheBottom = expectedViewportLineCount - 1;
        }
        if (minimapLinesFitting >= lineCount + extraLinesAtTheBottom) {
            // All lines fit in the minimap
            var startLineNumber = 1;
            var endLineNumber = lineCount;
            var sliderNeeded = (maxMinimapSliderTop > 0);
            return new MinimapLayout(scrollTop, scrollHeight, sliderNeeded, computedSliderRatio, sliderTop, sliderHeight, startLineNumber, endLineNumber);
        }
        else {
            var startLineNumber = Math.max(1, Math.floor(viewportStartLineNumber - sliderTop * pixelRatio / minimapLineHeight));
            // Avoid flickering caused by a partial viewport start line
            // by being consistent w.r.t. the previous layout decision
            if (previousLayout && previousLayout.scrollHeight === scrollHeight) {
                if (previousLayout.scrollTop > scrollTop) {
                    // Scrolling up => never increase `startLineNumber`
                    startLineNumber = Math.min(startLineNumber, previousLayout.startLineNumber);
                }
                if (previousLayout.scrollTop < scrollTop) {
                    // Scrolling down => never decrease `startLineNumber`
                    startLineNumber = Math.max(startLineNumber, previousLayout.startLineNumber);
                }
            }
            var endLineNumber = Math.min(lineCount, startLineNumber + minimapLinesFitting - 1);
            var partialLine = (scrollTop - viewportStartLineNumberVerticalOffset) / lineHeight;
            var sliderTopAligned = (viewportStartLineNumber - startLineNumber + partialLine) * minimapLineHeight / pixelRatio;
            return new MinimapLayout(scrollTop, scrollHeight, true, computedSliderRatio, sliderTopAligned, sliderHeight, startLineNumber, endLineNumber);
        }
    };
    return MinimapLayout;
}());
var MinimapLine = /** @class */ (function () {
    function MinimapLine(dy) {
        this.dy = dy;
    }
    MinimapLine.prototype.onContentChanged = function () {
        this.dy = -1;
    };
    MinimapLine.prototype.onTokensChanged = function () {
        this.dy = -1;
    };
    MinimapLine.INVALID = new MinimapLine(-1);
    return MinimapLine;
}());
var RenderData = /** @class */ (function () {
    function RenderData(renderedLayout, imageData, lines) {
        this.renderedLayout = renderedLayout;
        this._imageData = imageData;
        this._renderedLines = new RenderedLinesCollection(function () { return MinimapLine.INVALID; });
        this._renderedLines._set(renderedLayout.startLineNumber, lines);
    }
    /**
     * Check if the current RenderData matches accurately the new desired layout and no painting is needed.
     */
    RenderData.prototype.linesEquals = function (layout) {
        if (!this.scrollEquals(layout)) {
            return false;
        }
        var tmp = this._renderedLines._get();
        var lines = tmp.lines;
        for (var i = 0, len = lines.length; i < len; i++) {
            if (lines[i].dy === -1) {
                // This line is invalid
                return false;
            }
        }
        return true;
    };
    /**
     * Check if the current RenderData matches the new layout's scroll position
     */
    RenderData.prototype.scrollEquals = function (layout) {
        return this.renderedLayout.startLineNumber === layout.startLineNumber
            && this.renderedLayout.endLineNumber === layout.endLineNumber;
    };
    RenderData.prototype._get = function () {
        var tmp = this._renderedLines._get();
        return {
            imageData: this._imageData,
            rendLineNumberStart: tmp.rendLineNumberStart,
            lines: tmp.lines
        };
    };
    RenderData.prototype.onLinesChanged = function (changeFromLineNumber, changeToLineNumber) {
        return this._renderedLines.onLinesChanged(changeFromLineNumber, changeToLineNumber);
    };
    RenderData.prototype.onLinesDeleted = function (deleteFromLineNumber, deleteToLineNumber) {
        this._renderedLines.onLinesDeleted(deleteFromLineNumber, deleteToLineNumber);
    };
    RenderData.prototype.onLinesInserted = function (insertFromLineNumber, insertToLineNumber) {
        this._renderedLines.onLinesInserted(insertFromLineNumber, insertToLineNumber);
    };
    RenderData.prototype.onTokensChanged = function (ranges) {
        return this._renderedLines.onTokensChanged(ranges);
    };
    return RenderData;
}());
/**
 * Some sort of double buffering.
 *
 * Keeps two buffers around that will be rotated for painting.
 * Always gives a buffer that is filled with the background color.
 */
var MinimapBuffers = /** @class */ (function () {
    function MinimapBuffers(ctx, WIDTH, HEIGHT, background) {
        this._backgroundFillData = MinimapBuffers._createBackgroundFillData(WIDTH, HEIGHT, background);
        this._buffers = [
            ctx.createImageData(WIDTH, HEIGHT),
            ctx.createImageData(WIDTH, HEIGHT)
        ];
        this._lastUsedBuffer = 0;
    }
    MinimapBuffers.prototype.getBuffer = function () {
        // rotate buffers
        this._lastUsedBuffer = 1 - this._lastUsedBuffer;
        var result = this._buffers[this._lastUsedBuffer];
        // fill with background color
        result.data.set(this._backgroundFillData);
        return result;
    };
    MinimapBuffers._createBackgroundFillData = function (WIDTH, HEIGHT, background) {
        var backgroundR = background.r;
        var backgroundG = background.g;
        var backgroundB = background.b;
        var result = new Uint8ClampedArray(WIDTH * HEIGHT * 4);
        var offset = 0;
        for (var i = 0; i < HEIGHT; i++) {
            for (var j = 0; j < WIDTH; j++) {
                result[offset] = backgroundR;
                result[offset + 1] = backgroundG;
                result[offset + 2] = backgroundB;
                result[offset + 3] = 255;
                offset += 4;
            }
        }
        return result;
    };
    return MinimapBuffers;
}());
var MinimapSamplingState = /** @class */ (function () {
    function MinimapSamplingState(samplingRatio, minimapLines) {
        this.samplingRatio = samplingRatio;
        this.minimapLines = minimapLines;
    }
    MinimapSamplingState.compute = function (options, viewLineCount, oldSamplingState) {
        if (options.renderMinimap === 0 /* None */ || !options.isSampling) {
            return [null, []];
        }
        // ratio is intentionally not part of the layout to avoid the layout changing all the time
        // so we need to recompute it again...
        var pixelRatio = options.pixelRatio;
        var lineHeight = options.lineHeight;
        var scrollBeyondLastLine = options.scrollBeyondLastLine;
        var minimapLineCount = EditorLayoutInfoComputer.computeContainedMinimapLineCount({
            viewLineCount: viewLineCount,
            scrollBeyondLastLine: scrollBeyondLastLine,
            height: options.editorHeight,
            lineHeight: lineHeight,
            pixelRatio: pixelRatio
        }).minimapLineCount;
        var ratio = viewLineCount / minimapLineCount;
        var halfRatio = ratio / 2;
        if (!oldSamplingState || oldSamplingState.minimapLines.length === 0) {
            var result_1 = [];
            result_1[0] = 1;
            if (minimapLineCount > 1) {
                for (var i = 0, lastIndex = minimapLineCount - 1; i < lastIndex; i++) {
                    result_1[i] = Math.round(i * ratio + halfRatio);
                }
                result_1[minimapLineCount - 1] = viewLineCount;
            }
            return [new MinimapSamplingState(ratio, result_1), []];
        }
        var oldMinimapLines = oldSamplingState.minimapLines;
        var oldLength = oldMinimapLines.length;
        var result = [];
        var oldIndex = 0;
        var oldDeltaLineCount = 0;
        var minViewLineNumber = 1;
        var MAX_EVENT_COUNT = 10; // generate at most 10 events, if there are more than 10 changes, just flush all previous data
        var events = [];
        var lastEvent = null;
        for (var i = 0; i < minimapLineCount; i++) {
            var fromViewLineNumber = Math.max(minViewLineNumber, Math.round(i * ratio));
            var toViewLineNumber = Math.max(fromViewLineNumber, Math.round((i + 1) * ratio));
            while (oldIndex < oldLength && oldMinimapLines[oldIndex] < fromViewLineNumber) {
                if (events.length < MAX_EVENT_COUNT) {
                    var oldMinimapLineNumber = oldIndex + 1 + oldDeltaLineCount;
                    if (lastEvent && lastEvent.type === 'deleted' && lastEvent._oldIndex === oldIndex - 1) {
                        lastEvent.deleteToLineNumber++;
                    }
                    else {
                        lastEvent = { type: 'deleted', _oldIndex: oldIndex, deleteFromLineNumber: oldMinimapLineNumber, deleteToLineNumber: oldMinimapLineNumber };
                        events.push(lastEvent);
                    }
                    oldDeltaLineCount--;
                }
                oldIndex++;
            }
            var selectedViewLineNumber = void 0;
            if (oldIndex < oldLength && oldMinimapLines[oldIndex] <= toViewLineNumber) {
                // reuse the old sampled line
                selectedViewLineNumber = oldMinimapLines[oldIndex];
                oldIndex++;
            }
            else {
                if (i === 0) {
                    selectedViewLineNumber = 1;
                }
                else if (i + 1 === minimapLineCount) {
                    selectedViewLineNumber = viewLineCount;
                }
                else {
                    selectedViewLineNumber = Math.round(i * ratio + halfRatio);
                }
                if (events.length < MAX_EVENT_COUNT) {
                    var oldMinimapLineNumber = oldIndex + 1 + oldDeltaLineCount;
                    if (lastEvent && lastEvent.type === 'inserted' && lastEvent._i === i - 1) {
                        lastEvent.insertToLineNumber++;
                    }
                    else {
                        lastEvent = { type: 'inserted', _i: i, insertFromLineNumber: oldMinimapLineNumber, insertToLineNumber: oldMinimapLineNumber };
                        events.push(lastEvent);
                    }
                    oldDeltaLineCount++;
                }
            }
            result[i] = selectedViewLineNumber;
            minViewLineNumber = selectedViewLineNumber;
        }
        if (events.length < MAX_EVENT_COUNT) {
            while (oldIndex < oldLength) {
                var oldMinimapLineNumber = oldIndex + 1 + oldDeltaLineCount;
                if (lastEvent && lastEvent.type === 'deleted' && lastEvent._oldIndex === oldIndex - 1) {
                    lastEvent.deleteToLineNumber++;
                }
                else {
                    lastEvent = { type: 'deleted', _oldIndex: oldIndex, deleteFromLineNumber: oldMinimapLineNumber, deleteToLineNumber: oldMinimapLineNumber };
                    events.push(lastEvent);
                }
                oldDeltaLineCount--;
                oldIndex++;
            }
        }
        else {
            // too many events, just give up
            events = [{ type: 'flush' }];
        }
        return [new MinimapSamplingState(ratio, result), events];
    };
    MinimapSamplingState.prototype.modelLineToMinimapLine = function (lineNumber) {
        return Math.min(this.minimapLines.length, Math.max(1, Math.round(lineNumber / this.samplingRatio)));
    };
    /**
     * Will return null if the model line ranges are not intersecting with a sampled model line.
     */
    MinimapSamplingState.prototype.modelLineRangeToMinimapLineRange = function (fromLineNumber, toLineNumber) {
        var fromLineIndex = this.modelLineToMinimapLine(fromLineNumber) - 1;
        while (fromLineIndex > 0 && this.minimapLines[fromLineIndex - 1] >= fromLineNumber) {
            fromLineIndex--;
        }
        var toLineIndex = this.modelLineToMinimapLine(toLineNumber) - 1;
        while (toLineIndex + 1 < this.minimapLines.length && this.minimapLines[toLineIndex + 1] <= toLineNumber) {
            toLineIndex++;
        }
        if (fromLineIndex === toLineIndex) {
            var sampledLineNumber = this.minimapLines[fromLineIndex];
            if (sampledLineNumber < fromLineNumber || sampledLineNumber > toLineNumber) {
                // This line is not part of the sampled lines ==> nothing to do
                return null;
            }
        }
        return [fromLineIndex + 1, toLineIndex + 1];
    };
    /**
     * Will always return a range, even if it is not intersecting with a sampled model line.
     */
    MinimapSamplingState.prototype.decorationLineRangeToMinimapLineRange = function (startLineNumber, endLineNumber) {
        var minimapLineStart = this.modelLineToMinimapLine(startLineNumber);
        var minimapLineEnd = this.modelLineToMinimapLine(endLineNumber);
        if (startLineNumber !== endLineNumber && minimapLineEnd === minimapLineStart) {
            if (minimapLineEnd === this.minimapLines.length) {
                if (minimapLineStart > 1) {
                    minimapLineStart--;
                }
            }
            else {
                minimapLineEnd++;
            }
        }
        return [minimapLineStart, minimapLineEnd];
    };
    MinimapSamplingState.prototype.onLinesDeleted = function (e) {
        // have the mapping be sticky
        var deletedLineCount = e.toLineNumber - e.fromLineNumber + 1;
        var changeStartIndex = this.minimapLines.length;
        var changeEndIndex = 0;
        for (var i = this.minimapLines.length - 1; i >= 0; i--) {
            if (this.minimapLines[i] < e.fromLineNumber) {
                break;
            }
            if (this.minimapLines[i] <= e.toLineNumber) {
                // this line got deleted => move to previous available
                this.minimapLines[i] = Math.max(1, e.fromLineNumber - 1);
                changeStartIndex = Math.min(changeStartIndex, i);
                changeEndIndex = Math.max(changeEndIndex, i);
            }
            else {
                this.minimapLines[i] -= deletedLineCount;
            }
        }
        return [changeStartIndex, changeEndIndex];
    };
    MinimapSamplingState.prototype.onLinesInserted = function (e) {
        // have the mapping be sticky
        var insertedLineCount = e.toLineNumber - e.fromLineNumber + 1;
        for (var i = this.minimapLines.length - 1; i >= 0; i--) {
            if (this.minimapLines[i] < e.fromLineNumber) {
                break;
            }
            this.minimapLines[i] += insertedLineCount;
        }
    };
    return MinimapSamplingState;
}());
var Minimap = /** @class */ (function (_super) {
    __extends(Minimap, _super);
    function Minimap(context) {
        var _this = _super.call(this, context) || this;
        _this.tokensColorTracker = MinimapTokensColorTracker.getInstance();
        _this._selections = [];
        _this._minimapSelections = null;
        _this.options = new MinimapOptions(_this._context.configuration, _this._context.theme, _this.tokensColorTracker);
        var _a = __read(MinimapSamplingState.compute(_this.options, _this._context.model.getLineCount(), null), 1), samplingState = _a[0];
        _this._samplingState = samplingState;
        _this._shouldCheckSampling = false;
        _this._actual = new InnerMinimap(context.theme, _this);
        return _this;
    }
    Minimap.prototype.dispose = function () {
        this._actual.dispose();
        _super.prototype.dispose.call(this);
    };
    Minimap.prototype.getDomNode = function () {
        return this._actual.getDomNode();
    };
    Minimap.prototype._onOptionsMaybeChanged = function () {
        var opts = new MinimapOptions(this._context.configuration, this._context.theme, this.tokensColorTracker);
        if (this.options.equals(opts)) {
            return false;
        }
        this.options = opts;
        this._recreateLineSampling();
        this._actual.onDidChangeOptions();
        return true;
    };
    // ---- begin view event handlers
    Minimap.prototype.onConfigurationChanged = function (e) {
        return this._onOptionsMaybeChanged();
    };
    Minimap.prototype.onCursorStateChanged = function (e) {
        this._selections = e.selections;
        this._minimapSelections = null;
        return this._actual.onSelectionChanged();
    };
    Minimap.prototype.onDecorationsChanged = function (e) {
        if (e.affectsMinimap) {
            return this._actual.onDecorationsChanged();
        }
        return false;
    };
    Minimap.prototype.onFlushed = function (e) {
        if (this._samplingState) {
            this._shouldCheckSampling = true;
        }
        return this._actual.onFlushed();
    };
    Minimap.prototype.onLinesChanged = function (e) {
        if (this._samplingState) {
            var minimapLineRange = this._samplingState.modelLineRangeToMinimapLineRange(e.fromLineNumber, e.toLineNumber);
            if (minimapLineRange) {
                return this._actual.onLinesChanged(minimapLineRange[0], minimapLineRange[1]);
            }
            else {
                return false;
            }
        }
        else {
            return this._actual.onLinesChanged(e.fromLineNumber, e.toLineNumber);
        }
    };
    Minimap.prototype.onLinesDeleted = function (e) {
        if (this._samplingState) {
            var _a = __read(this._samplingState.onLinesDeleted(e), 2), changeStartIndex = _a[0], changeEndIndex = _a[1];
            if (changeStartIndex <= changeEndIndex) {
                this._actual.onLinesChanged(changeStartIndex + 1, changeEndIndex + 1);
            }
            this._shouldCheckSampling = true;
            return true;
        }
        else {
            return this._actual.onLinesDeleted(e.fromLineNumber, e.toLineNumber);
        }
    };
    Minimap.prototype.onLinesInserted = function (e) {
        if (this._samplingState) {
            this._samplingState.onLinesInserted(e);
            this._shouldCheckSampling = true;
            return true;
        }
        else {
            return this._actual.onLinesInserted(e.fromLineNumber, e.toLineNumber);
        }
    };
    Minimap.prototype.onScrollChanged = function (e) {
        return this._actual.onScrollChanged();
    };
    Minimap.prototype.onThemeChanged = function (e) {
        this._context.model.invalidateMinimapColorCache();
        this._actual.onThemeChanged();
        this._onOptionsMaybeChanged();
        return true;
    };
    Minimap.prototype.onTokensChanged = function (e) {
        var e_1, _a;
        if (this._samplingState) {
            var ranges = [];
            try {
                for (var _b = __values(e.ranges), _c = _b.next(); !_c.done; _c = _b.next()) {
                    var range = _c.value;
                    var minimapLineRange = this._samplingState.modelLineRangeToMinimapLineRange(range.fromLineNumber, range.toLineNumber);
                    if (minimapLineRange) {
                        ranges.push({ fromLineNumber: minimapLineRange[0], toLineNumber: minimapLineRange[1] });
                    }
                }
            }
            catch (e_1_1) { e_1 = { error: e_1_1 }; }
            finally {
                try {
                    if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
                }
                finally { if (e_1) throw e_1.error; }
            }
            if (ranges.length) {
                return this._actual.onTokensChanged(ranges);
            }
            else {
                return false;
            }
        }
        else {
            return this._actual.onTokensChanged(e.ranges);
        }
    };
    Minimap.prototype.onTokensColorsChanged = function (e) {
        this._onOptionsMaybeChanged();
        return this._actual.onTokensColorsChanged();
    };
    Minimap.prototype.onZonesChanged = function (e) {
        return this._actual.onZonesChanged();
    };
    // --- end event handlers
    Minimap.prototype.prepareRender = function (ctx) {
        if (this._shouldCheckSampling) {
            this._shouldCheckSampling = false;
            this._recreateLineSampling();
        }
    };
    Minimap.prototype.render = function (ctx) {
        var viewportStartLineNumber = ctx.visibleRange.startLineNumber;
        var viewportEndLineNumber = ctx.visibleRange.endLineNumber;
        if (this._samplingState) {
            viewportStartLineNumber = this._samplingState.modelLineToMinimapLine(viewportStartLineNumber);
            viewportEndLineNumber = this._samplingState.modelLineToMinimapLine(viewportEndLineNumber);
        }
        var minimapCtx = {
            viewportContainsWhitespaceGaps: (ctx.viewportData.whitespaceViewportData.length > 0),
            scrollWidth: ctx.scrollWidth,
            scrollHeight: ctx.scrollHeight,
            viewportStartLineNumber: viewportStartLineNumber,
            viewportEndLineNumber: viewportEndLineNumber,
            viewportStartLineNumberVerticalOffset: ctx.getVerticalOffsetForLineNumber(viewportStartLineNumber),
            scrollTop: ctx.scrollTop,
            scrollLeft: ctx.scrollLeft,
            viewportWidth: ctx.viewportWidth,
            viewportHeight: ctx.viewportHeight,
        };
        this._actual.render(minimapCtx);
    };
    //#region IMinimapModel
    Minimap.prototype._recreateLineSampling = function () {
        var e_2, _a;
        this._minimapSelections = null;
        var wasSampling = Boolean(this._samplingState);
        var _b = __read(MinimapSamplingState.compute(this.options, this._context.model.getLineCount(), this._samplingState), 2), samplingState = _b[0], events = _b[1];
        this._samplingState = samplingState;
        if (wasSampling && this._samplingState) {
            try {
                // was sampling, is sampling
                for (var events_1 = __values(events), events_1_1 = events_1.next(); !events_1_1.done; events_1_1 = events_1.next()) {
                    var event_1 = events_1_1.value;
                    switch (event_1.type) {
                        case 'deleted':
                            this._actual.onLinesDeleted(event_1.deleteFromLineNumber, event_1.deleteToLineNumber);
                            break;
                        case 'inserted':
                            this._actual.onLinesInserted(event_1.insertFromLineNumber, event_1.insertToLineNumber);
                            break;
                        case 'flush':
                            this._actual.onFlushed();
                            break;
                    }
                }
            }
            catch (e_2_1) { e_2 = { error: e_2_1 }; }
            finally {
                try {
                    if (events_1_1 && !events_1_1.done && (_a = events_1.return)) _a.call(events_1);
                }
                finally { if (e_2) throw e_2.error; }
            }
        }
    };
    Minimap.prototype.getLineCount = function () {
        if (this._samplingState) {
            return this._samplingState.minimapLines.length;
        }
        return this._context.model.getLineCount();
    };
    Minimap.prototype.getRealLineCount = function () {
        return this._context.model.getLineCount();
    };
    Minimap.prototype.getLineContent = function (lineNumber) {
        if (this._samplingState) {
            return this._context.model.getLineContent(this._samplingState.minimapLines[lineNumber - 1]);
        }
        return this._context.model.getLineContent(lineNumber);
    };
    Minimap.prototype.getMinimapLinesRenderingData = function (startLineNumber, endLineNumber, needed) {
        if (this._samplingState) {
            var result = [];
            for (var lineIndex = 0, lineCount = endLineNumber - startLineNumber + 1; lineIndex < lineCount; lineIndex++) {
                if (needed[lineIndex]) {
                    result[lineIndex] = this._context.model.getViewLineData(this._samplingState.minimapLines[startLineNumber + lineIndex - 1]);
                }
                else {
                    result[lineIndex] = null;
                }
            }
            return result;
        }
        return this._context.model.getMinimapLinesRenderingData(startLineNumber, endLineNumber, needed).data;
    };
    Minimap.prototype.getSelections = function () {
        var e_3, _a;
        if (this._minimapSelections === null) {
            if (this._samplingState) {
                this._minimapSelections = [];
                try {
                    for (var _b = __values(this._selections), _c = _b.next(); !_c.done; _c = _b.next()) {
                        var selection = _c.value;
                        var _d = __read(this._samplingState.decorationLineRangeToMinimapLineRange(selection.startLineNumber, selection.endLineNumber), 2), minimapLineStart = _d[0], minimapLineEnd = _d[1];
                        this._minimapSelections.push(new Selection(minimapLineStart, selection.startColumn, minimapLineEnd, selection.endColumn));
                    }
                }
                catch (e_3_1) { e_3 = { error: e_3_1 }; }
                finally {
                    try {
                        if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
                    }
                    finally { if (e_3) throw e_3.error; }
                }
            }
            else {
                this._minimapSelections = this._selections;
            }
        }
        return this._minimapSelections;
    };
    Minimap.prototype.getMinimapDecorationsInViewport = function (startLineNumber, endLineNumber) {
        var e_4, _a;
        var visibleRange;
        if (this._samplingState) {
            var modelStartLineNumber = this._samplingState.minimapLines[startLineNumber - 1];
            var modelEndLineNumber = this._samplingState.minimapLines[endLineNumber - 1];
            visibleRange = new Range(modelStartLineNumber, 1, modelEndLineNumber, this._context.model.getLineMaxColumn(modelEndLineNumber));
        }
        else {
            visibleRange = new Range(startLineNumber, 1, endLineNumber, this._context.model.getLineMaxColumn(endLineNumber));
        }
        var decorations = this._context.model.getDecorationsInViewport(visibleRange);
        if (this._samplingState) {
            var result = [];
            try {
                for (var decorations_1 = __values(decorations), decorations_1_1 = decorations_1.next(); !decorations_1_1.done; decorations_1_1 = decorations_1.next()) {
                    var decoration = decorations_1_1.value;
                    if (!decoration.options.minimap) {
                        continue;
                    }
                    var range = decoration.range;
                    var minimapStartLineNumber = this._samplingState.modelLineToMinimapLine(range.startLineNumber);
                    var minimapEndLineNumber = this._samplingState.modelLineToMinimapLine(range.endLineNumber);
                    result.push(new ViewModelDecoration(new Range(minimapStartLineNumber, range.startColumn, minimapEndLineNumber, range.endColumn), decoration.options));
                }
            }
            catch (e_4_1) { e_4 = { error: e_4_1 }; }
            finally {
                try {
                    if (decorations_1_1 && !decorations_1_1.done && (_a = decorations_1.return)) _a.call(decorations_1);
                }
                finally { if (e_4) throw e_4.error; }
            }
            return result;
        }
        return decorations;
    };
    Minimap.prototype.getOptions = function () {
        return this._context.model.getTextModelOptions();
    };
    Minimap.prototype.revealLineNumber = function (lineNumber) {
        if (this._samplingState) {
            lineNumber = this._samplingState.minimapLines[lineNumber - 1];
        }
        this._context.model.revealRange('mouse', false, new Range(lineNumber, 1, lineNumber, 1), 1 /* Center */, 0 /* Smooth */);
    };
    Minimap.prototype.setScrollTop = function (scrollTop) {
        this._context.model.setScrollPosition({
            scrollTop: scrollTop
        }, 1 /* Immediate */);
    };
    return Minimap;
}(ViewPart));
export { Minimap };
var InnerMinimap = /** @class */ (function (_super) {
    __extends(InnerMinimap, _super);
    function InnerMinimap(theme, model) {
        var _this = _super.call(this) || this;
        _this._renderDecorations = false;
        _this._gestureInProgress = false;
        _this._theme = theme;
        _this._model = model;
        _this._lastRenderData = null;
        _this._buffers = null;
        _this._selectionColor = _this._theme.getColor(minimapSelection);
        _this._domNode = createFastDomNode(document.createElement('div'));
        PartFingerprints.write(_this._domNode, 8 /* Minimap */);
        _this._domNode.setClassName(_this._getMinimapDomNodeClassName());
        _this._domNode.setPosition('absolute');
        _this._domNode.setAttribute('role', 'presentation');
        _this._domNode.setAttribute('aria-hidden', 'true');
        _this._shadow = createFastDomNode(document.createElement('div'));
        _this._shadow.setClassName('minimap-shadow-hidden');
        _this._domNode.appendChild(_this._shadow);
        _this._canvas = createFastDomNode(document.createElement('canvas'));
        _this._canvas.setPosition('absolute');
        _this._canvas.setLeft(0);
        _this._domNode.appendChild(_this._canvas);
        _this._decorationsCanvas = createFastDomNode(document.createElement('canvas'));
        _this._decorationsCanvas.setPosition('absolute');
        _this._decorationsCanvas.setClassName('minimap-decorations-layer');
        _this._decorationsCanvas.setLeft(0);
        _this._domNode.appendChild(_this._decorationsCanvas);
        _this._slider = createFastDomNode(document.createElement('div'));
        _this._slider.setPosition('absolute');
        _this._slider.setClassName('minimap-slider');
        _this._slider.setLayerHinting(true);
        _this._slider.setContain('strict');
        _this._domNode.appendChild(_this._slider);
        _this._sliderHorizontal = createFastDomNode(document.createElement('div'));
        _this._sliderHorizontal.setPosition('absolute');
        _this._sliderHorizontal.setClassName('minimap-slider-horizontal');
        _this._slider.appendChild(_this._sliderHorizontal);
        _this._applyLayout();
        _this._mouseDownListener = dom.addStandardDisposableListener(_this._domNode.domNode, 'mousedown', function (e) {
            e.preventDefault();
            var renderMinimap = _this._model.options.renderMinimap;
            if (renderMinimap === 0 /* None */) {
                return;
            }
            if (!_this._lastRenderData) {
                return;
            }
            if (_this._model.options.size !== 'proportional') {
                if (e.leftButton && _this._lastRenderData) {
                    // pretend the click occured in the center of the slider
                    var position = dom.getDomNodePagePosition(_this._slider.domNode);
                    var initialPosY = position.top + position.height / 2;
                    _this._startSliderDragging(e.buttons, e.posx, initialPosY, e.posy, _this._lastRenderData.renderedLayout);
                }
                return;
            }
            var minimapLineHeight = _this._model.options.minimapLineHeight;
            var internalOffsetY = (_this._model.options.canvasInnerHeight / _this._model.options.canvasOuterHeight) * e.browserEvent.offsetY;
            var lineIndex = Math.floor(internalOffsetY / minimapLineHeight);
            var lineNumber = lineIndex + _this._lastRenderData.renderedLayout.startLineNumber;
            lineNumber = Math.min(lineNumber, _this._model.getLineCount());
            _this._model.revealLineNumber(lineNumber);
        });
        _this._sliderMouseMoveMonitor = new GlobalMouseMoveMonitor();
        _this._sliderMouseDownListener = dom.addStandardDisposableListener(_this._slider.domNode, 'mousedown', function (e) {
            e.preventDefault();
            e.stopPropagation();
            if (e.leftButton && _this._lastRenderData) {
                _this._startSliderDragging(e.buttons, e.posx, e.posy, e.posy, _this._lastRenderData.renderedLayout);
            }
        });
        _this._gestureDisposable = Gesture.addTarget(_this._domNode.domNode);
        _this._sliderTouchStartListener = dom.addDisposableListener(_this._domNode.domNode, EventType.Start, function (e) {
            e.preventDefault();
            e.stopPropagation();
            if (_this._lastRenderData) {
                _this._slider.toggleClassName('active', true);
                _this._gestureInProgress = true;
                _this.scrollDueToTouchEvent(e);
            }
        }, { passive: false });
        _this._sliderTouchMoveListener = dom.addDisposableListener(_this._domNode.domNode, EventType.Change, function (e) {
            e.preventDefault();
            e.stopPropagation();
            if (_this._lastRenderData && _this._gestureInProgress) {
                _this.scrollDueToTouchEvent(e);
            }
        }, { passive: false });
        _this._sliderTouchEndListener = dom.addStandardDisposableListener(_this._domNode.domNode, EventType.End, function (e) {
            e.preventDefault();
            e.stopPropagation();
            _this._gestureInProgress = false;
            _this._slider.toggleClassName('active', false);
        });
        return _this;
    }
    InnerMinimap.prototype._startSliderDragging = function (initialButtons, initialPosX, initialPosY, posy, initialSliderState) {
        var _this = this;
        this._slider.toggleClassName('active', true);
        var handleMouseMove = function (posy, posx) {
            var mouseOrthogonalDelta = Math.abs(posx - initialPosX);
            if (platform.isWindows && mouseOrthogonalDelta > MOUSE_DRAG_RESET_DISTANCE) {
                // The mouse has wondered away from the scrollbar => reset dragging
                _this._model.setScrollTop(initialSliderState.scrollTop);
                return;
            }
            var mouseDelta = posy - initialPosY;
            _this._model.setScrollTop(initialSliderState.getDesiredScrollTopFromDelta(mouseDelta));
        };
        if (posy !== initialPosY) {
            handleMouseMove(posy, initialPosX);
        }
        this._sliderMouseMoveMonitor.startMonitoring(this._slider.domNode, initialButtons, standardMouseMoveMerger, function (mouseMoveData) { return handleMouseMove(mouseMoveData.posy, mouseMoveData.posx); }, function () {
            _this._slider.toggleClassName('active', false);
        });
    };
    InnerMinimap.prototype.scrollDueToTouchEvent = function (touch) {
        var startY = this._domNode.domNode.getBoundingClientRect().top;
        var scrollTop = this._lastRenderData.renderedLayout.getDesiredScrollTopFromTouchLocation(touch.pageY - startY);
        this._model.setScrollTop(scrollTop);
    };
    InnerMinimap.prototype.dispose = function () {
        this._mouseDownListener.dispose();
        this._sliderMouseMoveMonitor.dispose();
        this._sliderMouseDownListener.dispose();
        this._gestureDisposable.dispose();
        this._sliderTouchStartListener.dispose();
        this._sliderTouchMoveListener.dispose();
        this._sliderTouchEndListener.dispose();
        _super.prototype.dispose.call(this);
    };
    InnerMinimap.prototype._getMinimapDomNodeClassName = function () {
        if (this._model.options.showSlider === 'always') {
            return 'minimap slider-always';
        }
        return 'minimap slider-mouseover';
    };
    InnerMinimap.prototype.getDomNode = function () {
        return this._domNode;
    };
    InnerMinimap.prototype._applyLayout = function () {
        this._domNode.setLeft(this._model.options.minimapLeft);
        this._domNode.setWidth(this._model.options.minimapWidth);
        this._domNode.setHeight(this._model.options.minimapHeight);
        this._shadow.setHeight(this._model.options.minimapHeight);
        this._canvas.setWidth(this._model.options.canvasOuterWidth);
        this._canvas.setHeight(this._model.options.canvasOuterHeight);
        this._canvas.domNode.width = this._model.options.canvasInnerWidth;
        this._canvas.domNode.height = this._model.options.canvasInnerHeight;
        this._decorationsCanvas.setWidth(this._model.options.canvasOuterWidth);
        this._decorationsCanvas.setHeight(this._model.options.canvasOuterHeight);
        this._decorationsCanvas.domNode.width = this._model.options.canvasInnerWidth;
        this._decorationsCanvas.domNode.height = this._model.options.canvasInnerHeight;
        this._slider.setWidth(this._model.options.minimapWidth);
    };
    InnerMinimap.prototype._getBuffer = function () {
        if (!this._buffers) {
            if (this._model.options.canvasInnerWidth > 0 && this._model.options.canvasInnerHeight > 0) {
                this._buffers = new MinimapBuffers(this._canvas.domNode.getContext('2d'), this._model.options.canvasInnerWidth, this._model.options.canvasInnerHeight, this._model.options.backgroundColor);
            }
        }
        return this._buffers ? this._buffers.getBuffer() : null;
    };
    // ---- begin view event handlers
    InnerMinimap.prototype.onDidChangeOptions = function () {
        this._lastRenderData = null;
        this._buffers = null;
        this._applyLayout();
        this._domNode.setClassName(this._getMinimapDomNodeClassName());
    };
    InnerMinimap.prototype.onSelectionChanged = function () {
        this._renderDecorations = true;
        return true;
    };
    InnerMinimap.prototype.onDecorationsChanged = function () {
        this._renderDecorations = true;
        return true;
    };
    InnerMinimap.prototype.onFlushed = function () {
        this._lastRenderData = null;
        return true;
    };
    InnerMinimap.prototype.onLinesChanged = function (changeFromLineNumber, changeToLineNumber) {
        if (this._lastRenderData) {
            return this._lastRenderData.onLinesChanged(changeFromLineNumber, changeToLineNumber);
        }
        return false;
    };
    InnerMinimap.prototype.onLinesDeleted = function (deleteFromLineNumber, deleteToLineNumber) {
        if (this._lastRenderData) {
            this._lastRenderData.onLinesDeleted(deleteFromLineNumber, deleteToLineNumber);
        }
        return true;
    };
    InnerMinimap.prototype.onLinesInserted = function (insertFromLineNumber, insertToLineNumber) {
        if (this._lastRenderData) {
            this._lastRenderData.onLinesInserted(insertFromLineNumber, insertToLineNumber);
        }
        return true;
    };
    InnerMinimap.prototype.onScrollChanged = function () {
        this._renderDecorations = true;
        return true;
    };
    InnerMinimap.prototype.onThemeChanged = function () {
        this._selectionColor = this._theme.getColor(minimapSelection);
        this._renderDecorations = true;
        return true;
    };
    InnerMinimap.prototype.onTokensChanged = function (ranges) {
        if (this._lastRenderData) {
            return this._lastRenderData.onTokensChanged(ranges);
        }
        return false;
    };
    InnerMinimap.prototype.onTokensColorsChanged = function () {
        this._lastRenderData = null;
        this._buffers = null;
        return true;
    };
    InnerMinimap.prototype.onZonesChanged = function () {
        this._lastRenderData = null;
        return true;
    };
    // --- end event handlers
    InnerMinimap.prototype.render = function (renderingCtx) {
        var renderMinimap = this._model.options.renderMinimap;
        if (renderMinimap === 0 /* None */) {
            this._shadow.setClassName('minimap-shadow-hidden');
            this._sliderHorizontal.setWidth(0);
            this._sliderHorizontal.setHeight(0);
            return;
        }
        if (renderingCtx.scrollLeft + renderingCtx.viewportWidth >= renderingCtx.scrollWidth) {
            this._shadow.setClassName('minimap-shadow-hidden');
        }
        else {
            this._shadow.setClassName('minimap-shadow-visible');
        }
        var layout = MinimapLayout.create(this._model.options, renderingCtx.viewportStartLineNumber, renderingCtx.viewportEndLineNumber, renderingCtx.viewportStartLineNumberVerticalOffset, renderingCtx.viewportHeight, renderingCtx.viewportContainsWhitespaceGaps, this._model.getLineCount(), this._model.getRealLineCount(), renderingCtx.scrollTop, renderingCtx.scrollHeight, this._lastRenderData ? this._lastRenderData.renderedLayout : null);
        this._slider.setDisplay(layout.sliderNeeded ? 'block' : 'none');
        this._slider.setTop(layout.sliderTop);
        this._slider.setHeight(layout.sliderHeight);
        // Compute horizontal slider coordinates
        var scrollLeftChars = renderingCtx.scrollLeft / this._model.options.typicalHalfwidthCharacterWidth;
        var horizontalSliderLeft = Math.min(this._model.options.minimapWidth, Math.round(scrollLeftChars * this._model.options.minimapCharWidth / this._model.options.pixelRatio));
        this._sliderHorizontal.setLeft(horizontalSliderLeft);
        this._sliderHorizontal.setWidth(this._model.options.minimapWidth - horizontalSliderLeft);
        this._sliderHorizontal.setTop(0);
        this._sliderHorizontal.setHeight(layout.sliderHeight);
        this.renderDecorations(layout);
        this._lastRenderData = this.renderLines(layout);
    };
    InnerMinimap.prototype.renderDecorations = function (layout) {
        if (this._renderDecorations) {
            this._renderDecorations = false;
            var selections = this._model.getSelections();
            var decorations = this._model.getMinimapDecorationsInViewport(layout.startLineNumber, layout.endLineNumber);
            var _a = this._model.options, canvasInnerWidth = _a.canvasInnerWidth, canvasInnerHeight = _a.canvasInnerHeight;
            var lineHeight = this._model.options.minimapLineHeight;
            var characterWidth = this._model.options.minimapCharWidth;
            var tabSize = this._model.getOptions().tabSize;
            var canvasContext = this._decorationsCanvas.domNode.getContext('2d');
            canvasContext.clearRect(0, 0, canvasInnerWidth, canvasInnerHeight);
            var lineOffsetMap = new Map();
            for (var i = 0; i < selections.length; i++) {
                var selection = selections[i];
                for (var line = selection.startLineNumber; line <= selection.endLineNumber; line++) {
                    this.renderDecorationOnLine(canvasContext, lineOffsetMap, selection, this._selectionColor, layout, line, lineHeight, lineHeight, tabSize, characterWidth);
                }
            }
            // Loop over decorations, ignoring those that don't have the minimap property set and rendering rectangles for each line the decoration spans
            for (var i = 0; i < decorations.length; i++) {
                var decoration = decorations[i];
                if (!decoration.options.minimap) {
                    continue;
                }
                var decorationColor = decoration.options.minimap.getColor(this._theme);
                for (var line = decoration.range.startLineNumber; line <= decoration.range.endLineNumber; line++) {
                    switch (decoration.options.minimap.position) {
                        case MinimapPosition.Inline:
                            this.renderDecorationOnLine(canvasContext, lineOffsetMap, decoration.range, decorationColor, layout, line, lineHeight, lineHeight, tabSize, characterWidth);
                            continue;
                        case MinimapPosition.Gutter:
                            var y = (line - layout.startLineNumber) * lineHeight;
                            var x = 2;
                            this.renderDecoration(canvasContext, decorationColor, x, y, GUTTER_DECORATION_WIDTH, lineHeight);
                            continue;
                    }
                }
            }
        }
    };
    InnerMinimap.prototype.renderDecorationOnLine = function (canvasContext, lineOffsetMap, decorationRange, decorationColor, layout, lineNumber, height, lineHeight, tabSize, charWidth) {
        var y = (lineNumber - layout.startLineNumber) * lineHeight;
        // Skip rendering the line if it's vertically outside our viewport
        if (y + height < 0 || y > this._model.options.canvasInnerHeight) {
            return;
        }
        // Cache line offset data so that it is only read once per line
        var lineIndexToXOffset = lineOffsetMap.get(lineNumber);
        var isFirstDecorationForLine = !lineIndexToXOffset;
        if (!lineIndexToXOffset) {
            var lineData = this._model.getLineContent(lineNumber);
            lineIndexToXOffset = [MINIMAP_GUTTER_WIDTH];
            for (var i = 1; i < lineData.length + 1; i++) {
                var charCode = lineData.charCodeAt(i - 1);
                var dx = charCode === 9 /* Tab */
                    ? tabSize * charWidth
                    : strings.isFullWidthCharacter(charCode)
                        ? 2 * charWidth
                        : charWidth;
                lineIndexToXOffset[i] = lineIndexToXOffset[i - 1] + dx;
            }
            lineOffsetMap.set(lineNumber, lineIndexToXOffset);
        }
        var startColumn = decorationRange.startColumn, endColumn = decorationRange.endColumn, startLineNumber = decorationRange.startLineNumber, endLineNumber = decorationRange.endLineNumber;
        var x = startLineNumber === lineNumber ? lineIndexToXOffset[startColumn - 1] : MINIMAP_GUTTER_WIDTH;
        var endColumnForLine = endLineNumber > lineNumber ? lineIndexToXOffset.length - 1 : endColumn - 1;
        if (endColumnForLine > 0) {
            // If the decoration starts at the last character of the column and spans over it, ensure it has a width
            var width = lineIndexToXOffset[endColumnForLine] - x || 2;
            this.renderDecoration(canvasContext, decorationColor, x, y, width, height);
        }
        if (isFirstDecorationForLine) {
            this.renderLineHighlight(canvasContext, decorationColor, y, height);
        }
    };
    InnerMinimap.prototype.renderLineHighlight = function (canvasContext, decorationColor, y, height) {
        canvasContext.fillStyle = decorationColor && decorationColor.transparent(0.5).toString() || '';
        canvasContext.fillRect(MINIMAP_GUTTER_WIDTH, y, canvasContext.canvas.width, height);
    };
    InnerMinimap.prototype.renderDecoration = function (canvasContext, decorationColor, x, y, width, height) {
        canvasContext.fillStyle = decorationColor && decorationColor.toString() || '';
        canvasContext.fillRect(x, y, width, height);
    };
    InnerMinimap.prototype.renderLines = function (layout) {
        var startLineNumber = layout.startLineNumber;
        var endLineNumber = layout.endLineNumber;
        var minimapLineHeight = this._model.options.minimapLineHeight;
        // Check if nothing changed w.r.t. lines from last frame
        if (this._lastRenderData && this._lastRenderData.linesEquals(layout)) {
            var _lastData = this._lastRenderData._get();
            // Nice!! Nothing changed from last frame
            return new RenderData(layout, _lastData.imageData, _lastData.lines);
        }
        // Oh well!! We need to repaint some lines...
        var imageData = this._getBuffer();
        if (!imageData) {
            // 0 width or 0 height canvas, nothing to do
            return null;
        }
        // Render untouched lines by using last rendered data.
        var _a = __read(InnerMinimap._renderUntouchedLines(imageData, startLineNumber, endLineNumber, minimapLineHeight, this._lastRenderData), 3), _dirtyY1 = _a[0], _dirtyY2 = _a[1], needed = _a[2];
        // Fetch rendering info from view model for rest of lines that need rendering.
        var lineInfo = this._model.getMinimapLinesRenderingData(startLineNumber, endLineNumber, needed);
        var tabSize = this._model.getOptions().tabSize;
        var background = this._model.options.backgroundColor;
        var tokensColorTracker = this._model.tokensColorTracker;
        var useLighterFont = tokensColorTracker.backgroundIsLight();
        var renderMinimap = this._model.options.renderMinimap;
        var charRenderer = this._model.options.charRenderer();
        var fontScale = this._model.options.fontScale;
        var minimapCharWidth = this._model.options.minimapCharWidth;
        var baseCharHeight = (renderMinimap === 1 /* Text */ ? 2 /* BASE_CHAR_HEIGHT */ : 2 /* BASE_CHAR_HEIGHT */ + 1);
        var renderMinimapLineHeight = baseCharHeight * fontScale;
        var innerLinePadding = (minimapLineHeight > renderMinimapLineHeight ? Math.floor((minimapLineHeight - renderMinimapLineHeight) / 2) : 0);
        // Render the rest of lines
        var dy = 0;
        var renderedLines = [];
        for (var lineIndex = 0, lineCount = endLineNumber - startLineNumber + 1; lineIndex < lineCount; lineIndex++) {
            if (needed[lineIndex]) {
                InnerMinimap._renderLine(imageData, background, useLighterFont, renderMinimap, minimapCharWidth, tokensColorTracker, charRenderer, dy, innerLinePadding, tabSize, lineInfo[lineIndex], fontScale, minimapLineHeight);
            }
            renderedLines[lineIndex] = new MinimapLine(dy);
            dy += minimapLineHeight;
        }
        var dirtyY1 = (_dirtyY1 === -1 ? 0 : _dirtyY1);
        var dirtyY2 = (_dirtyY2 === -1 ? imageData.height : _dirtyY2);
        var dirtyHeight = dirtyY2 - dirtyY1;
        // Finally, paint to the canvas
        var ctx = this._canvas.domNode.getContext('2d');
        ctx.putImageData(imageData, 0, 0, 0, dirtyY1, imageData.width, dirtyHeight);
        // Save rendered data for reuse on next frame if possible
        return new RenderData(layout, imageData, renderedLines);
    };
    InnerMinimap._renderUntouchedLines = function (target, startLineNumber, endLineNumber, minimapLineHeight, lastRenderData) {
        var needed = [];
        if (!lastRenderData) {
            for (var i = 0, len = endLineNumber - startLineNumber + 1; i < len; i++) {
                needed[i] = true;
            }
            return [-1, -1, needed];
        }
        var _lastData = lastRenderData._get();
        var lastTargetData = _lastData.imageData.data;
        var lastStartLineNumber = _lastData.rendLineNumberStart;
        var lastLines = _lastData.lines;
        var lastLinesLength = lastLines.length;
        var WIDTH = target.width;
        var targetData = target.data;
        var maxDestPixel = (endLineNumber - startLineNumber + 1) * minimapLineHeight * WIDTH * 4;
        var dirtyPixel1 = -1; // the pixel offset up to which all the data is equal to the prev frame
        var dirtyPixel2 = -1; // the pixel offset after which all the data is equal to the prev frame
        var copySourceStart = -1;
        var copySourceEnd = -1;
        var copyDestStart = -1;
        var copyDestEnd = -1;
        var dest_dy = 0;
        for (var lineNumber = startLineNumber; lineNumber <= endLineNumber; lineNumber++) {
            var lineIndex = lineNumber - startLineNumber;
            var lastLineIndex = lineNumber - lastStartLineNumber;
            var source_dy = (lastLineIndex >= 0 && lastLineIndex < lastLinesLength ? lastLines[lastLineIndex].dy : -1);
            if (source_dy === -1) {
                needed[lineIndex] = true;
                dest_dy += minimapLineHeight;
                continue;
            }
            var sourceStart = source_dy * WIDTH * 4;
            var sourceEnd = (source_dy + minimapLineHeight) * WIDTH * 4;
            var destStart = dest_dy * WIDTH * 4;
            var destEnd = (dest_dy + minimapLineHeight) * WIDTH * 4;
            if (copySourceEnd === sourceStart && copyDestEnd === destStart) {
                // contiguous zone => extend copy request
                copySourceEnd = sourceEnd;
                copyDestEnd = destEnd;
            }
            else {
                if (copySourceStart !== -1) {
                    // flush existing copy request
                    targetData.set(lastTargetData.subarray(copySourceStart, copySourceEnd), copyDestStart);
                    if (dirtyPixel1 === -1 && copySourceStart === 0 && copySourceStart === copyDestStart) {
                        dirtyPixel1 = copySourceEnd;
                    }
                    if (dirtyPixel2 === -1 && copySourceEnd === maxDestPixel && copySourceStart === copyDestStart) {
                        dirtyPixel2 = copySourceStart;
                    }
                }
                copySourceStart = sourceStart;
                copySourceEnd = sourceEnd;
                copyDestStart = destStart;
                copyDestEnd = destEnd;
            }
            needed[lineIndex] = false;
            dest_dy += minimapLineHeight;
        }
        if (copySourceStart !== -1) {
            // flush existing copy request
            targetData.set(lastTargetData.subarray(copySourceStart, copySourceEnd), copyDestStart);
            if (dirtyPixel1 === -1 && copySourceStart === 0 && copySourceStart === copyDestStart) {
                dirtyPixel1 = copySourceEnd;
            }
            if (dirtyPixel2 === -1 && copySourceEnd === maxDestPixel && copySourceStart === copyDestStart) {
                dirtyPixel2 = copySourceStart;
            }
        }
        var dirtyY1 = (dirtyPixel1 === -1 ? -1 : dirtyPixel1 / (WIDTH * 4));
        var dirtyY2 = (dirtyPixel2 === -1 ? -1 : dirtyPixel2 / (WIDTH * 4));
        return [dirtyY1, dirtyY2, needed];
    };
    InnerMinimap._renderLine = function (target, backgroundColor, useLighterFont, renderMinimap, charWidth, colorTracker, minimapCharRenderer, dy, innerLinePadding, tabSize, lineData, fontScale, minimapLineHeight) {
        var content = lineData.content;
        var tokens = lineData.tokens;
        var maxDx = target.width - charWidth;
        var force1pxHeight = (minimapLineHeight === 1);
        var dx = MINIMAP_GUTTER_WIDTH;
        var charIndex = 0;
        var tabsCharDelta = 0;
        for (var tokenIndex = 0, tokensLen = tokens.getCount(); tokenIndex < tokensLen; tokenIndex++) {
            var tokenEndIndex = tokens.getEndOffset(tokenIndex);
            var tokenColorId = tokens.getForeground(tokenIndex);
            var tokenColor = colorTracker.getColor(tokenColorId);
            for (; charIndex < tokenEndIndex; charIndex++) {
                if (dx > maxDx) {
                    // hit edge of minimap
                    return;
                }
                var charCode = content.charCodeAt(charIndex);
                if (charCode === 9 /* Tab */) {
                    var insertSpacesCount = tabSize - (charIndex + tabsCharDelta) % tabSize;
                    tabsCharDelta += insertSpacesCount - 1;
                    // No need to render anything since tab is invisible
                    dx += insertSpacesCount * charWidth;
                }
                else if (charCode === 32 /* Space */) {
                    // No need to render anything since space is invisible
                    dx += charWidth;
                }
                else {
                    // Render twice for a full width character
                    var count = strings.isFullWidthCharacter(charCode) ? 2 : 1;
                    for (var i = 0; i < count; i++) {
                        if (renderMinimap === 2 /* Blocks */) {
                            minimapCharRenderer.blockRenderChar(target, dx, dy + innerLinePadding, tokenColor, backgroundColor, useLighterFont, force1pxHeight);
                        }
                        else { // RenderMinimap.Text
                            minimapCharRenderer.renderChar(target, dx, dy + innerLinePadding, charCode, tokenColor, backgroundColor, fontScale, useLighterFont, force1pxHeight);
                        }
                        dx += charWidth;
                        if (dx > maxDx) {
                            // hit edge of minimap
                            return;
                        }
                    }
                }
            }
        }
    };
    return InnerMinimap;
}(Disposable));
registerThemingParticipant(function (theme, collector) {
    var minimapBackgroundValue = theme.getColor(minimapBackground);
    if (minimapBackgroundValue) {
        collector.addRule(".monaco-editor .minimap > canvas { opacity: " + minimapBackgroundValue.rgba.a + "; will-change: opacity; }");
    }
    var sliderBackground = theme.getColor(minimapSliderBackground);
    if (sliderBackground) {
        collector.addRule(".monaco-editor .minimap-slider .minimap-slider-horizontal { background: " + sliderBackground + "; }");
    }
    var sliderHoverBackground = theme.getColor(minimapSliderHoverBackground);
    if (sliderHoverBackground) {
        collector.addRule(".monaco-editor .minimap-slider:hover .minimap-slider-horizontal { background: " + sliderHoverBackground + "; }");
    }
    var sliderActiveBackground = theme.getColor(minimapSliderActiveBackground);
    if (sliderActiveBackground) {
        collector.addRule(".monaco-editor .minimap-slider.active .minimap-slider-horizontal { background: " + sliderActiveBackground + "; }");
    }
    var shadow = theme.getColor(scrollbarShadow);
    if (shadow) {
        collector.addRule(".monaco-editor .minimap-shadow-visible { box-shadow: " + shadow + " -6px 0 6px -6px inset; }");
    }
});
