/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __extends = (this && this.__extends) || (function () {
    var extendStatics = function (d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (Object.prototype.hasOwnProperty.call(b, p)) d[p] = b[p]; };
        return extendStatics(d, b);
    };
    return function (d, b) {
        if (typeof b !== "function" && b !== null)
            throw new TypeError("Class extends value " + String(b) + " is not a constructor or null");
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
var __values = (this && this.__values) || function(o) {
    var s = typeof Symbol === "function" && Symbol.iterator, m = s && o[s], i = 0;
    if (m) return m.call(o);
    if (o && typeof o.length === "number") return {
        next: function () {
            if (o && i >= o.length) o = void 0;
            return { value: o && o[i++], done: !o };
        }
    };
    throw new TypeError(s ? "Object is not iterable." : "Symbol.iterator is not defined.");
};
var __read = (this && this.__read) || function (o, n) {
    var m = typeof Symbol === "function" && o[Symbol.iterator];
    if (!m) return o;
    var i = m.call(o), r, ar = [], e;
    try {
        while ((n === void 0 || n-- > 0) && !(r = i.next()).done) ar.push(r.value);
    }
    catch (error) { e = { error: error }; }
    finally {
        try {
            if (r && !r.done && (m = i["return"])) m.call(i);
        }
        finally { if (e) throw e.error; }
    }
    return ar;
};
var __spreadArray = (this && this.__spreadArray) || function (to, from) {
    for (var i = 0, il = from.length, j = to.length; i < il; i++, j++)
        to[j] = from[i];
    return to;
};
import * as nls from '../../nls';
import { URI } from '../../base/common/uri';
import { ICodeEditorService } from './services/codeEditorService';
import { Position } from '../common/core/position';
import { IModelService } from '../common/services/modelService';
import { ITextModelService } from '../common/services/resolverService';
import { MenuId, MenuRegistry, Action2 } from '../../platform/actions/common/actions';
import { CommandsRegistry } from '../../platform/commands/common/commands';
import { ContextKeyExpr, IContextKeyService } from '../../platform/contextkey/common/contextkey';
import { KeybindingsRegistry } from '../../platform/keybinding/common/keybindingsRegistry';
import { Registry } from '../../platform/registry/common/platform';
import { ITelemetryService } from '../../platform/telemetry/common/telemetry';
import { withNullAsUndefined, assertType } from '../../base/common/types';
var Command = /** @class */ (function () {
    function Command(opts) {
        this.id = opts.id;
        this.precondition = opts.precondition;
        this._kbOpts = opts.kbOpts;
        this._menuOpts = opts.menuOpts;
        this._description = opts.description;
    }
    Command.prototype.register = function () {
        var _this = this;
        if (Array.isArray(this._menuOpts)) {
            this._menuOpts.forEach(this._registerMenuItem, this);
        }
        else if (this._menuOpts) {
            this._registerMenuItem(this._menuOpts);
        }
        if (this._kbOpts) {
            var kbWhen = this._kbOpts.kbExpr;
            if (this.precondition) {
                if (kbWhen) {
                    kbWhen = ContextKeyExpr.and(kbWhen, this.precondition);
                }
                else {
                    kbWhen = this.precondition;
                }
            }
            KeybindingsRegistry.registerCommandAndKeybindingRule({
                id: this.id,
                handler: function (accessor, args) { return _this.runCommand(accessor, args); },
                weight: this._kbOpts.weight,
                args: this._kbOpts.args,
                when: kbWhen,
                primary: this._kbOpts.primary,
                secondary: this._kbOpts.secondary,
                win: this._kbOpts.win,
                linux: this._kbOpts.linux,
                mac: this._kbOpts.mac,
                description: this._description
            });
        }
        else {
            CommandsRegistry.registerCommand({
                id: this.id,
                handler: function (accessor, args) { return _this.runCommand(accessor, args); },
                description: this._description
            });
        }
    };
    Command.prototype._registerMenuItem = function (item) {
        MenuRegistry.appendMenuItem(item.menuId, {
            group: item.group,
            command: {
                id: this.id,
                title: item.title,
                icon: item.icon,
                precondition: this.precondition
            },
            when: item.when,
            order: item.order
        });
    };
    return Command;
}());
export { Command };
var MultiCommand = /** @class */ (function (_super) {
    __extends(MultiCommand, _super);
    function MultiCommand() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        _this._implementations = [];
        return _this;
    }
    /**
     * A higher priority gets to be looked at first
     */
    MultiCommand.prototype.addImplementation = function (priority, implementation) {
        var _this = this;
        this._implementations.push([priority, implementation]);
        this._implementations.sort(function (a, b) { return b[0] - a[0]; });
        return {
            dispose: function () {
                for (var i = 0; i < _this._implementations.length; i++) {
                    if (_this._implementations[i][1] === implementation) {
                        _this._implementations.splice(i, 1);
                        return;
                    }
                }
            }
        };
    };
    MultiCommand.prototype.runCommand = function (accessor, args) {
        var e_1, _a;
        try {
            for (var _b = __values(this._implementations), _c = _b.next(); !_c.done; _c = _b.next()) {
                var impl = _c.value;
                var result = impl[1](accessor, args);
                if (result) {
                    if (typeof result === 'boolean') {
                        return;
                    }
                    return result;
                }
            }
        }
        catch (e_1_1) { e_1 = { error: e_1_1 }; }
        finally {
            try {
                if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
            }
            finally { if (e_1) throw e_1.error; }
        }
    };
    return MultiCommand;
}(Command));
export { MultiCommand };
//#endregion
/**
 * A command that delegates to another command's implementation.
 *
 * This lets different commands be registered but share the same implementation
 */
var ProxyCommand = /** @class */ (function (_super) {
    __extends(ProxyCommand, _super);
    function ProxyCommand(command, opts) {
        var _this = _super.call(this, opts) || this;
        _this.command = command;
        return _this;
    }
    ProxyCommand.prototype.runCommand = function (accessor, args) {
        return this.command.runCommand(accessor, args);
    };
    return ProxyCommand;
}(Command));
export { ProxyCommand };
var EditorCommand = /** @class */ (function (_super) {
    __extends(EditorCommand, _super);
    function EditorCommand() {
        return _super !== null && _super.apply(this, arguments) || this;
    }
    /**
     * Create a command class that is bound to a certain editor contribution.
     */
    EditorCommand.bindToContribution = function (controllerGetter) {
        return /** @class */ (function (_super) {
            __extends(EditorControllerCommandImpl, _super);
            function EditorControllerCommandImpl(opts) {
                var _this = _super.call(this, opts) || this;
                _this._callback = opts.handler;
                return _this;
            }
            EditorControllerCommandImpl.prototype.runEditorCommand = function (accessor, editor, args) {
                var controller = controllerGetter(editor);
                if (controller) {
                    this._callback(controllerGetter(editor), args);
                }
            };
            return EditorControllerCommandImpl;
        }(EditorCommand));
    };
    EditorCommand.prototype.runCommand = function (accessor, args) {
        var _this = this;
        var codeEditorService = accessor.get(ICodeEditorService);
        // Find the editor with text focus or active
        var editor = codeEditorService.getFocusedCodeEditor() || codeEditorService.getActiveCodeEditor();
        if (!editor) {
            // well, at least we tried...
            return;
        }
        return editor.invokeWithinContext(function (editorAccessor) {
            var kbService = editorAccessor.get(IContextKeyService);
            if (!kbService.contextMatchesRules(withNullAsUndefined(_this.precondition))) {
                // precondition does not hold
                return;
            }
            return _this.runEditorCommand(editorAccessor, editor, args);
        });
    };
    return EditorCommand;
}(Command));
export { EditorCommand };
var EditorAction = /** @class */ (function (_super) {
    __extends(EditorAction, _super);
    function EditorAction(opts) {
        var _this = _super.call(this, EditorAction.convertOptions(opts)) || this;
        _this.label = opts.label;
        _this.alias = opts.alias;
        return _this;
    }
    EditorAction.convertOptions = function (opts) {
        var menuOpts;
        if (Array.isArray(opts.menuOpts)) {
            menuOpts = opts.menuOpts;
        }
        else if (opts.menuOpts) {
            menuOpts = [opts.menuOpts];
        }
        else {
            menuOpts = [];
        }
        function withDefaults(item) {
            if (!item.menuId) {
                item.menuId = MenuId.EditorContext;
            }
            if (!item.title) {
                item.title = opts.label;
            }
            item.when = ContextKeyExpr.and(opts.precondition, item.when);
            return item;
        }
        if (Array.isArray(opts.contextMenuOpts)) {
            menuOpts.push.apply(menuOpts, __spreadArray([], __read(opts.contextMenuOpts.map(withDefaults))));
        }
        else if (opts.contextMenuOpts) {
            menuOpts.push(withDefaults(opts.contextMenuOpts));
        }
        opts.menuOpts = menuOpts;
        return opts;
    };
    EditorAction.prototype.runEditorCommand = function (accessor, editor, args) {
        this.reportTelemetry(accessor, editor);
        return this.run(accessor, editor, args || {});
    };
    EditorAction.prototype.reportTelemetry = function (accessor, editor) {
        accessor.get(ITelemetryService).publicLog2('editorActionInvoked', { name: this.label, id: this.id });
    };
    return EditorAction;
}(EditorCommand));
export { EditorAction };
var MultiEditorAction = /** @class */ (function (_super) {
    __extends(MultiEditorAction, _super);
    function MultiEditorAction() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        _this._implementations = [];
        return _this;
    }
    /**
     * A higher priority gets to be looked at first
     */
    MultiEditorAction.prototype.addImplementation = function (priority, implementation) {
        var _this = this;
        this._implementations.push([priority, implementation]);
        this._implementations.sort(function (a, b) { return b[0] - a[0]; });
        return {
            dispose: function () {
                for (var i = 0; i < _this._implementations.length; i++) {
                    if (_this._implementations[i][1] === implementation) {
                        _this._implementations.splice(i, 1);
                        return;
                    }
                }
            }
        };
    };
    MultiEditorAction.prototype.run = function (accessor, editor, args) {
        var e_2, _a;
        try {
            for (var _b = __values(this._implementations), _c = _b.next(); !_c.done; _c = _b.next()) {
                var impl = _c.value;
                var result = impl[1](accessor, args);
                if (result) {
                    if (typeof result === 'boolean') {
                        return;
                    }
                    return result;
                }
            }
        }
        catch (e_2_1) { e_2 = { error: e_2_1 }; }
        finally {
            try {
                if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
            }
            finally { if (e_2) throw e_2.error; }
        }
    };
    return MultiEditorAction;
}(EditorAction));
export { MultiEditorAction };
//#endregion EditorAction
//#region EditorAction2
var EditorAction2 = /** @class */ (function (_super) {
    __extends(EditorAction2, _super);
    function EditorAction2() {
        return _super !== null && _super.apply(this, arguments) || this;
    }
    EditorAction2.prototype.run = function (accessor) {
        var _this = this;
        var args = [];
        for (var _i = 1; _i < arguments.length; _i++) {
            args[_i - 1] = arguments[_i];
        }
        // Find the editor with text focus or active
        var codeEditorService = accessor.get(ICodeEditorService);
        var editor = codeEditorService.getFocusedCodeEditor() || codeEditorService.getActiveCodeEditor();
        if (!editor) {
            // well, at least we tried...
            return;
        }
        // precondition does hold
        return editor.invokeWithinContext(function (editorAccessor) {
            var kbService = editorAccessor.get(IContextKeyService);
            if (kbService.contextMatchesRules(withNullAsUndefined(_this.desc.precondition))) {
                return _this.runEditorCommand(editorAccessor, editor, args);
            }
        });
    };
    return EditorAction2;
}(Action2));
export { EditorAction2 };
//#endregion
// --- Registration of commands and actions
export function registerModelAndPositionCommand(id, handler) {
    CommandsRegistry.registerCommand(id, function (accessor) {
        var args = [];
        for (var _i = 1; _i < arguments.length; _i++) {
            args[_i - 1] = arguments[_i];
        }
        var _a = __read(args, 2), resource = _a[0], position = _a[1];
        assertType(URI.isUri(resource));
        assertType(Position.isIPosition(position));
        var model = accessor.get(IModelService).getModel(resource);
        if (model) {
            var editorPosition = Position.lift(position);
            return handler.apply(void 0, __spreadArray([model, editorPosition], __read(args.slice(2))));
        }
        return accessor.get(ITextModelService).createModelReference(resource).then(function (reference) {
            return new Promise(function (resolve, reject) {
                try {
                    var result = handler(reference.object.textEditorModel, Position.lift(position), args.slice(2));
                    resolve(result);
                }
                catch (err) {
                    reject(err);
                }
            }).finally(function () {
                reference.dispose();
            });
        });
    });
}
export function registerModelCommand(id, handler) {
    CommandsRegistry.registerCommand(id, function (accessor) {
        var args = [];
        for (var _i = 1; _i < arguments.length; _i++) {
            args[_i - 1] = arguments[_i];
        }
        var _a = __read(args, 1), resource = _a[0];
        assertType(URI.isUri(resource));
        var model = accessor.get(IModelService).getModel(resource);
        if (model) {
            return handler.apply(void 0, __spreadArray([model], __read(args.slice(1))));
        }
        return accessor.get(ITextModelService).createModelReference(resource).then(function (reference) {
            return new Promise(function (resolve, reject) {
                try {
                    var result = handler(reference.object.textEditorModel, args.slice(1));
                    resolve(result);
                }
                catch (err) {
                    reject(err);
                }
            }).finally(function () {
                reference.dispose();
            });
        });
    });
}
export function registerEditorCommand(editorCommand) {
    EditorContributionRegistry.INSTANCE.registerEditorCommand(editorCommand);
    return editorCommand;
}
export function registerEditorAction(ctor) {
    var action = new ctor();
    EditorContributionRegistry.INSTANCE.registerEditorAction(action);
    return action;
}
export function registerMultiEditorAction(action) {
    EditorContributionRegistry.INSTANCE.registerEditorAction(action);
    return action;
}
export function registerInstantiatedEditorAction(editorAction) {
    EditorContributionRegistry.INSTANCE.registerEditorAction(editorAction);
}
export function registerEditorContribution(id, ctor) {
    EditorContributionRegistry.INSTANCE.registerEditorContribution(id, ctor);
}
export function registerDiffEditorContribution(id, ctor) {
    EditorContributionRegistry.INSTANCE.registerDiffEditorContribution(id, ctor);
}
export var EditorExtensionsRegistry;
(function (EditorExtensionsRegistry) {
    function getEditorCommand(commandId) {
        return EditorContributionRegistry.INSTANCE.getEditorCommand(commandId);
    }
    EditorExtensionsRegistry.getEditorCommand = getEditorCommand;
    function getEditorActions() {
        return EditorContributionRegistry.INSTANCE.getEditorActions();
    }
    EditorExtensionsRegistry.getEditorActions = getEditorActions;
    function getEditorContributions() {
        return EditorContributionRegistry.INSTANCE.getEditorContributions();
    }
    EditorExtensionsRegistry.getEditorContributions = getEditorContributions;
    function getSomeEditorContributions(ids) {
        return EditorContributionRegistry.INSTANCE.getEditorContributions().filter(function (c) { return ids.indexOf(c.id) >= 0; });
    }
    EditorExtensionsRegistry.getSomeEditorContributions = getSomeEditorContributions;
    function getDiffEditorContributions() {
        return EditorContributionRegistry.INSTANCE.getDiffEditorContributions();
    }
    EditorExtensionsRegistry.getDiffEditorContributions = getDiffEditorContributions;
})(EditorExtensionsRegistry || (EditorExtensionsRegistry = {}));
// Editor extension points
var Extensions = {
    EditorCommonContributions: 'editor.contributions'
};
var EditorContributionRegistry = /** @class */ (function () {
    function EditorContributionRegistry() {
        this.editorContributions = [];
        this.diffEditorContributions = [];
        this.editorActions = [];
        this.editorCommands = Object.create(null);
    }
    EditorContributionRegistry.prototype.registerEditorContribution = function (id, ctor) {
        this.editorContributions.push({ id: id, ctor: ctor });
    };
    EditorContributionRegistry.prototype.getEditorContributions = function () {
        return this.editorContributions.slice(0);
    };
    EditorContributionRegistry.prototype.registerDiffEditorContribution = function (id, ctor) {
        this.diffEditorContributions.push({ id: id, ctor: ctor });
    };
    EditorContributionRegistry.prototype.getDiffEditorContributions = function () {
        return this.diffEditorContributions.slice(0);
    };
    EditorContributionRegistry.prototype.registerEditorAction = function (action) {
        action.register();
        this.editorActions.push(action);
    };
    EditorContributionRegistry.prototype.getEditorActions = function () {
        return this.editorActions.slice(0);
    };
    EditorContributionRegistry.prototype.registerEditorCommand = function (editorCommand) {
        editorCommand.register();
        this.editorCommands[editorCommand.id] = editorCommand;
    };
    EditorContributionRegistry.prototype.getEditorCommand = function (commandId) {
        return (this.editorCommands[commandId] || null);
    };
    EditorContributionRegistry.INSTANCE = new EditorContributionRegistry();
    return EditorContributionRegistry;
}());
Registry.add(Extensions.EditorCommonContributions, EditorContributionRegistry.INSTANCE);
function registerCommand(command) {
    command.register();
    return command;
}
export var UndoCommand = registerCommand(new MultiCommand({
    id: 'undo',
    precondition: undefined,
    kbOpts: {
        weight: 0 /* EditorCore */,
        primary: 2048 /* CtrlCmd */ | 56 /* KEY_Z */
    },
    menuOpts: [{
            menuId: MenuId.MenubarEditMenu,
            group: '1_do',
            title: nls.localize({ key: 'miUndo', comment: ['&& denotes a mnemonic'] }, "&&Undo"),
            order: 1
        }, {
            menuId: MenuId.CommandPalette,
            group: '',
            title: nls.localize('undo', "Undo"),
            order: 1
        }]
}));
registerCommand(new ProxyCommand(UndoCommand, { id: 'default:undo', precondition: undefined }));
export var RedoCommand = registerCommand(new MultiCommand({
    id: 'redo',
    precondition: undefined,
    kbOpts: {
        weight: 0 /* EditorCore */,
        primary: 2048 /* CtrlCmd */ | 55 /* KEY_Y */,
        secondary: [2048 /* CtrlCmd */ | 1024 /* Shift */ | 56 /* KEY_Z */],
        mac: { primary: 2048 /* CtrlCmd */ | 1024 /* Shift */ | 56 /* KEY_Z */ }
    },
    menuOpts: [{
            menuId: MenuId.MenubarEditMenu,
            group: '1_do',
            title: nls.localize({ key: 'miRedo', comment: ['&& denotes a mnemonic'] }, "&&Redo"),
            order: 2
        }, {
            menuId: MenuId.CommandPalette,
            group: '',
            title: nls.localize('redo', "Redo"),
            order: 1
        }]
}));
registerCommand(new ProxyCommand(RedoCommand, { id: 'default:redo', precondition: undefined }));
export var SelectAllCommand = registerCommand(new MultiCommand({
    id: 'editor.action.selectAll',
    precondition: undefined,
    kbOpts: {
        weight: 0 /* EditorCore */,
        kbExpr: null,
        primary: 2048 /* CtrlCmd */ | 31 /* KEY_A */
    },
    menuOpts: [{
            menuId: MenuId.MenubarSelectionMenu,
            group: '1_basic',
            title: nls.localize({ key: 'miSelectAll', comment: ['&& denotes a mnemonic'] }, "&&Select All"),
            order: 1
        }, {
            menuId: MenuId.CommandPalette,
            group: '',
            title: nls.localize('selectAll', "Select All"),
            order: 1
        }]
}));
