/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __read = (this && this.__read) || function (o, n) {
    var m = typeof Symbol === "function" && o[Symbol.iterator];
    if (!m) return o;
    var i = m.call(o), r, ar = [], e;
    try {
        while ((n === void 0 || n-- > 0) && !(r = i.next()).done) ar.push(r.value);
    }
    catch (error) { e = { error: error }; }
    finally {
        try {
            if (r && !r.done && (m = i["return"])) m.call(i);
        }
        finally { if (e) throw e.error; }
    }
    return ar;
};
var __spreadArray = (this && this.__spreadArray) || function (to, from) {
    for (var i = 0, il = from.length, j = to.length; i < il; i++, j++)
        to[j] = from[i];
    return to;
};
var __values = (this && this.__values) || function(o) {
    var s = typeof Symbol === "function" && Symbol.iterator, m = s && o[s], i = 0;
    if (m) return m.call(o);
    if (o && typeof o.length === "number") return {
        next: function () {
            if (o && i >= o.length) o = void 0;
            return { value: o && o[i++], done: !o };
        }
    };
    throw new TypeError(s ? "Object is not iterable." : "Symbol.iterator is not defined.");
};
import { groupBy } from '../../../base/common/arrays';
import { dispose } from '../../../base/common/lifecycle';
import { getLeadingWhitespace } from '../../../base/common/strings';
import './snippetSession.css';
import { EditOperation } from '../../common/core/editOperation';
import { Range } from '../../common/core/range';
import { Selection } from '../../common/core/selection';
import { ModelDecorationOptions } from '../../common/model/textModel';
import { IWorkspaceContextService } from '../../../platform/workspace/common/workspace';
import { optional } from '../../../platform/instantiation/common/instantiation';
import { Choice, Placeholder, SnippetParser, Text } from './snippetParser';
import { ClipboardBasedVariableResolver, CompositeSnippetVariableResolver, ModelBasedVariableResolver, SelectionBasedVariableResolver, TimeBasedVariableResolver, CommentBasedVariableResolver, WorkspaceBasedVariableResolver, RandomBasedVariableResolver } from './snippetVariables';
import { registerThemingParticipant } from '../../../platform/theme/common/themeService';
import * as colors from '../../../platform/theme/common/colorRegistry';
import { withNullAsUndefined } from '../../../base/common/types';
import { ILabelService } from '../../../platform/label/common/label';
registerThemingParticipant(function (theme, collector) {
    function getColorGraceful(name) {
        var color = theme.getColor(name);
        return color ? color.toString() : 'transparent';
    }
    collector.addRule(".monaco-editor .snippet-placeholder { background-color: " + getColorGraceful(colors.snippetTabstopHighlightBackground) + "; outline-color: " + getColorGraceful(colors.snippetTabstopHighlightBorder) + "; }");
    collector.addRule(".monaco-editor .finish-snippet-placeholder { background-color: " + getColorGraceful(colors.snippetFinalTabstopHighlightBackground) + "; outline-color: " + getColorGraceful(colors.snippetFinalTabstopHighlightBorder) + "; }");
});
var OneSnippet = /** @class */ (function () {
    function OneSnippet(_editor, _snippet, _offset, _snippetLineLeadingWhitespace) {
        this._editor = _editor;
        this._snippet = _snippet;
        this._offset = _offset;
        this._snippetLineLeadingWhitespace = _snippetLineLeadingWhitespace;
        this._nestingLevel = 1;
        this._placeholderGroups = groupBy(_snippet.placeholders, Placeholder.compareByIndex);
        this._placeholderGroupsIdx = -1;
    }
    OneSnippet.prototype.dispose = function () {
        if (this._placeholderDecorations) {
            this._editor.deltaDecorations(__spreadArray([], __read(this._placeholderDecorations.values())), []);
        }
        this._placeholderGroups.length = 0;
    };
    OneSnippet.prototype._initDecorations = function () {
        var _this = this;
        if (this._placeholderDecorations) {
            // already initialized
            return;
        }
        this._placeholderDecorations = new Map();
        var model = this._editor.getModel();
        this._editor.changeDecorations(function (accessor) {
            var e_1, _a;
            try {
                // create a decoration for each placeholder
                for (var _b = __values(_this._snippet.placeholders), _c = _b.next(); !_c.done; _c = _b.next()) {
                    var placeholder = _c.value;
                    var placeholderOffset = _this._snippet.offset(placeholder);
                    var placeholderLen = _this._snippet.fullLen(placeholder);
                    var range = Range.fromPositions(model.getPositionAt(_this._offset + placeholderOffset), model.getPositionAt(_this._offset + placeholderOffset + placeholderLen));
                    var options = placeholder.isFinalTabstop ? OneSnippet._decor.inactiveFinal : OneSnippet._decor.inactive;
                    var handle = accessor.addDecoration(range, options);
                    _this._placeholderDecorations.set(placeholder, handle);
                }
            }
            catch (e_1_1) { e_1 = { error: e_1_1 }; }
            finally {
                try {
                    if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
                }
                finally { if (e_1) throw e_1.error; }
            }
        });
    };
    OneSnippet.prototype.move = function (fwd) {
        var e_2, _a;
        var _this = this;
        if (!this._editor.hasModel()) {
            return [];
        }
        this._initDecorations();
        // Transform placeholder text if necessary
        if (this._placeholderGroupsIdx >= 0) {
            var operations = [];
            try {
                for (var _b = __values(this._placeholderGroups[this._placeholderGroupsIdx]), _c = _b.next(); !_c.done; _c = _b.next()) {
                    var placeholder = _c.value;
                    // Check if the placeholder has a transformation
                    if (placeholder.transform) {
                        var id = this._placeholderDecorations.get(placeholder);
                        var range = this._editor.getModel().getDecorationRange(id);
                        var currentValue = this._editor.getModel().getValueInRange(range);
                        var transformedValueLines = placeholder.transform.resolve(currentValue).split(/\r\n|\r|\n/);
                        // fix indentation for transformed lines
                        for (var i = 1; i < transformedValueLines.length; i++) {
                            transformedValueLines[i] = this._editor.getModel().normalizeIndentation(this._snippetLineLeadingWhitespace + transformedValueLines[i]);
                        }
                        operations.push(EditOperation.replace(range, transformedValueLines.join(this._editor.getModel().getEOL())));
                    }
                }
            }
            catch (e_2_1) { e_2 = { error: e_2_1 }; }
            finally {
                try {
                    if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
                }
                finally { if (e_2) throw e_2.error; }
            }
            if (operations.length > 0) {
                this._editor.executeEdits('snippet.placeholderTransform', operations);
            }
        }
        var couldSkipThisPlaceholder = false;
        if (fwd === true && this._placeholderGroupsIdx < this._placeholderGroups.length - 1) {
            this._placeholderGroupsIdx += 1;
            couldSkipThisPlaceholder = true;
        }
        else if (fwd === false && this._placeholderGroupsIdx > 0) {
            this._placeholderGroupsIdx -= 1;
            couldSkipThisPlaceholder = true;
        }
        else {
            // the selection of the current placeholder might
            // not acurate any more -> simply restore it
        }
        var newSelections = this._editor.getModel().changeDecorations(function (accessor) {
            var e_3, _a, e_4, _b, e_5, _c;
            var activePlaceholders = new Set();
            // change stickiness to always grow when typing at its edges
            // because these decorations represent the currently active
            // tabstop.
            // Special case #1: reaching the final tabstop
            // Special case #2: placeholders enclosing active placeholders
            var selections = [];
            try {
                for (var _d = __values(_this._placeholderGroups[_this._placeholderGroupsIdx]), _e = _d.next(); !_e.done; _e = _d.next()) {
                    var placeholder = _e.value;
                    var id = _this._placeholderDecorations.get(placeholder);
                    var range = _this._editor.getModel().getDecorationRange(id);
                    selections.push(new Selection(range.startLineNumber, range.startColumn, range.endLineNumber, range.endColumn));
                    // consider to skip this placeholder index when the decoration
                    // range is empty but when the placeholder wasn't. that's a strong
                    // hint that the placeholder has been deleted. (all placeholder must match this)
                    couldSkipThisPlaceholder = couldSkipThisPlaceholder && _this._hasPlaceholderBeenCollapsed(placeholder);
                    accessor.changeDecorationOptions(id, placeholder.isFinalTabstop ? OneSnippet._decor.activeFinal : OneSnippet._decor.active);
                    activePlaceholders.add(placeholder);
                    try {
                        for (var _f = (e_4 = void 0, __values(_this._snippet.enclosingPlaceholders(placeholder))), _g = _f.next(); !_g.done; _g = _f.next()) {
                            var enclosingPlaceholder = _g.value;
                            var id_1 = _this._placeholderDecorations.get(enclosingPlaceholder);
                            accessor.changeDecorationOptions(id_1, enclosingPlaceholder.isFinalTabstop ? OneSnippet._decor.activeFinal : OneSnippet._decor.active);
                            activePlaceholders.add(enclosingPlaceholder);
                        }
                    }
                    catch (e_4_1) { e_4 = { error: e_4_1 }; }
                    finally {
                        try {
                            if (_g && !_g.done && (_b = _f.return)) _b.call(_f);
                        }
                        finally { if (e_4) throw e_4.error; }
                    }
                }
            }
            catch (e_3_1) { e_3 = { error: e_3_1 }; }
            finally {
                try {
                    if (_e && !_e.done && (_a = _d.return)) _a.call(_d);
                }
                finally { if (e_3) throw e_3.error; }
            }
            try {
                // change stickness to never grow when typing at its edges
                // so that in-active tabstops never grow
                for (var _h = __values(_this._placeholderDecorations), _j = _h.next(); !_j.done; _j = _h.next()) {
                    var _k = __read(_j.value, 2), placeholder = _k[0], id = _k[1];
                    if (!activePlaceholders.has(placeholder)) {
                        accessor.changeDecorationOptions(id, placeholder.isFinalTabstop ? OneSnippet._decor.inactiveFinal : OneSnippet._decor.inactive);
                    }
                }
            }
            catch (e_5_1) { e_5 = { error: e_5_1 }; }
            finally {
                try {
                    if (_j && !_j.done && (_c = _h.return)) _c.call(_h);
                }
                finally { if (e_5) throw e_5.error; }
            }
            return selections;
        });
        return !couldSkipThisPlaceholder ? newSelections !== null && newSelections !== void 0 ? newSelections : [] : this.move(fwd);
    };
    OneSnippet.prototype._hasPlaceholderBeenCollapsed = function (placeholder) {
        // A placeholder is empty when it wasn't empty when authored but
        // when its tracking decoration is empty. This also applies to all
        // potential parent placeholders
        var marker = placeholder;
        while (marker) {
            if (marker instanceof Placeholder) {
                var id = this._placeholderDecorations.get(marker);
                var range = this._editor.getModel().getDecorationRange(id);
                if (range.isEmpty() && marker.toString().length > 0) {
                    return true;
                }
            }
            marker = marker.parent;
        }
        return false;
    };
    Object.defineProperty(OneSnippet.prototype, "isAtFirstPlaceholder", {
        get: function () {
            return this._placeholderGroupsIdx <= 0 || this._placeholderGroups.length === 0;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(OneSnippet.prototype, "isAtLastPlaceholder", {
        get: function () {
            return this._placeholderGroupsIdx === this._placeholderGroups.length - 1;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(OneSnippet.prototype, "hasPlaceholder", {
        get: function () {
            return this._snippet.placeholders.length > 0;
        },
        enumerable: false,
        configurable: true
    });
    OneSnippet.prototype.computePossibleSelections = function () {
        var e_6, _a, e_7, _b;
        var result = new Map();
        try {
            for (var _c = __values(this._placeholderGroups), _d = _c.next(); !_d.done; _d = _c.next()) {
                var placeholdersWithEqualIndex = _d.value;
                var ranges = void 0;
                try {
                    for (var placeholdersWithEqualIndex_1 = (e_7 = void 0, __values(placeholdersWithEqualIndex)), placeholdersWithEqualIndex_1_1 = placeholdersWithEqualIndex_1.next(); !placeholdersWithEqualIndex_1_1.done; placeholdersWithEqualIndex_1_1 = placeholdersWithEqualIndex_1.next()) {
                        var placeholder = placeholdersWithEqualIndex_1_1.value;
                        if (placeholder.isFinalTabstop) {
                            // ignore those
                            break;
                        }
                        if (!ranges) {
                            ranges = [];
                            result.set(placeholder.index, ranges);
                        }
                        var id = this._placeholderDecorations.get(placeholder);
                        var range = this._editor.getModel().getDecorationRange(id);
                        if (!range) {
                            // one of the placeholder lost its decoration and
                            // therefore we bail out and pretend the placeholder
                            // (with its mirrors) doesn't exist anymore.
                            result.delete(placeholder.index);
                            break;
                        }
                        ranges.push(range);
                    }
                }
                catch (e_7_1) { e_7 = { error: e_7_1 }; }
                finally {
                    try {
                        if (placeholdersWithEqualIndex_1_1 && !placeholdersWithEqualIndex_1_1.done && (_b = placeholdersWithEqualIndex_1.return)) _b.call(placeholdersWithEqualIndex_1);
                    }
                    finally { if (e_7) throw e_7.error; }
                }
            }
        }
        catch (e_6_1) { e_6 = { error: e_6_1 }; }
        finally {
            try {
                if (_d && !_d.done && (_a = _c.return)) _a.call(_c);
            }
            finally { if (e_6) throw e_6.error; }
        }
        return result;
    };
    Object.defineProperty(OneSnippet.prototype, "choice", {
        get: function () {
            return this._placeholderGroups[this._placeholderGroupsIdx][0].choice;
        },
        enumerable: false,
        configurable: true
    });
    OneSnippet.prototype.merge = function (others) {
        var _this = this;
        var model = this._editor.getModel();
        this._nestingLevel *= 10;
        this._editor.changeDecorations(function (accessor) {
            var e_8, _a, e_9, _b, e_10, _c;
            try {
                // For each active placeholder take one snippet and merge it
                // in that the placeholder (can be many for `$1foo$1foo`). Because
                // everything is sorted by editor selection we can simply remove
                // elements from the beginning of the array
                for (var _d = __values(_this._placeholderGroups[_this._placeholderGroupsIdx]), _e = _d.next(); !_e.done; _e = _d.next()) {
                    var placeholder = _e.value;
                    var nested = others.shift();
                    console.assert(!nested._placeholderDecorations);
                    // Massage placeholder-indicies of the nested snippet to be
                    // sorted right after the insertion point. This ensures we move
                    // through the placeholders in the correct order
                    var indexLastPlaceholder = nested._snippet.placeholderInfo.last.index;
                    try {
                        for (var _f = (e_9 = void 0, __values(nested._snippet.placeholderInfo.all)), _g = _f.next(); !_g.done; _g = _f.next()) {
                            var nestedPlaceholder = _g.value;
                            if (nestedPlaceholder.isFinalTabstop) {
                                nestedPlaceholder.index = placeholder.index + ((indexLastPlaceholder + 1) / _this._nestingLevel);
                            }
                            else {
                                nestedPlaceholder.index = placeholder.index + (nestedPlaceholder.index / _this._nestingLevel);
                            }
                        }
                    }
                    catch (e_9_1) { e_9 = { error: e_9_1 }; }
                    finally {
                        try {
                            if (_g && !_g.done && (_b = _f.return)) _b.call(_f);
                        }
                        finally { if (e_9) throw e_9.error; }
                    }
                    _this._snippet.replace(placeholder, nested._snippet.children);
                    // Remove the placeholder at which position are inserting
                    // the snippet and also remove its decoration.
                    var id = _this._placeholderDecorations.get(placeholder);
                    accessor.removeDecoration(id);
                    _this._placeholderDecorations.delete(placeholder);
                    try {
                        // For each *new* placeholder we create decoration to monitor
                        // how and if it grows/shrinks.
                        for (var _h = (e_10 = void 0, __values(nested._snippet.placeholders)), _j = _h.next(); !_j.done; _j = _h.next()) {
                            var placeholder_1 = _j.value;
                            var placeholderOffset = nested._snippet.offset(placeholder_1);
                            var placeholderLen = nested._snippet.fullLen(placeholder_1);
                            var range = Range.fromPositions(model.getPositionAt(nested._offset + placeholderOffset), model.getPositionAt(nested._offset + placeholderOffset + placeholderLen));
                            var handle = accessor.addDecoration(range, OneSnippet._decor.inactive);
                            _this._placeholderDecorations.set(placeholder_1, handle);
                        }
                    }
                    catch (e_10_1) { e_10 = { error: e_10_1 }; }
                    finally {
                        try {
                            if (_j && !_j.done && (_c = _h.return)) _c.call(_h);
                        }
                        finally { if (e_10) throw e_10.error; }
                    }
                }
            }
            catch (e_8_1) { e_8 = { error: e_8_1 }; }
            finally {
                try {
                    if (_e && !_e.done && (_a = _d.return)) _a.call(_d);
                }
                finally { if (e_8) throw e_8.error; }
            }
            // Last, re-create the placeholder groups by sorting placeholders by their index.
            _this._placeholderGroups = groupBy(_this._snippet.placeholders, Placeholder.compareByIndex);
        });
    };
    OneSnippet.prototype.getEnclosingRange = function () {
        var e_11, _a;
        var result;
        var model = this._editor.getModel();
        try {
            for (var _b = __values(this._placeholderDecorations.values()), _c = _b.next(); !_c.done; _c = _b.next()) {
                var decorationId = _c.value;
                var placeholderRange = withNullAsUndefined(model.getDecorationRange(decorationId));
                if (!result) {
                    result = placeholderRange;
                }
                else {
                    result = result.plusRange(placeholderRange);
                }
            }
        }
        catch (e_11_1) { e_11 = { error: e_11_1 }; }
        finally {
            try {
                if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
            }
            finally { if (e_11) throw e_11.error; }
        }
        return result;
    };
    OneSnippet._decor = {
        active: ModelDecorationOptions.register({ stickiness: 0 /* AlwaysGrowsWhenTypingAtEdges */, className: 'snippet-placeholder' }),
        inactive: ModelDecorationOptions.register({ stickiness: 1 /* NeverGrowsWhenTypingAtEdges */, className: 'snippet-placeholder' }),
        activeFinal: ModelDecorationOptions.register({ stickiness: 1 /* NeverGrowsWhenTypingAtEdges */, className: 'finish-snippet-placeholder' }),
        inactiveFinal: ModelDecorationOptions.register({ stickiness: 1 /* NeverGrowsWhenTypingAtEdges */, className: 'finish-snippet-placeholder' }),
    };
    return OneSnippet;
}());
export { OneSnippet };
var _defaultOptions = {
    overwriteBefore: 0,
    overwriteAfter: 0,
    adjustWhitespace: true,
    clipboardText: undefined,
    overtypingCapturer: undefined
};
var SnippetSession = /** @class */ (function () {
    function SnippetSession(editor, template, options) {
        if (options === void 0) { options = _defaultOptions; }
        this._templateMerges = [];
        this._snippets = [];
        this._editor = editor;
        this._template = template;
        this._options = options;
    }
    SnippetSession.adjustWhitespace = function (model, position, snippet, adjustIndentation, adjustNewlines) {
        var line = model.getLineContent(position.lineNumber);
        var lineLeadingWhitespace = getLeadingWhitespace(line, 0, position.column - 1);
        // the snippet as inserted
        var snippetTextString;
        snippet.walk(function (marker) {
            // all text elements that are not inside choice
            if (!(marker instanceof Text) || marker.parent instanceof Choice) {
                return true;
            }
            var lines = marker.value.split(/\r\n|\r|\n/);
            if (adjustIndentation) {
                // adjust indentation of snippet test
                // -the snippet-start doesn't get extra-indented (lineLeadingWhitespace), only normalized
                // -all N+1 lines get extra-indented and normalized
                // -the text start get extra-indented and normalized when following a linebreak
                var offset = snippet.offset(marker);
                if (offset === 0) {
                    // snippet start
                    lines[0] = model.normalizeIndentation(lines[0]);
                }
                else {
                    // check if text start is after a linebreak
                    snippetTextString = snippetTextString !== null && snippetTextString !== void 0 ? snippetTextString : snippet.toString();
                    var prevChar = snippetTextString.charCodeAt(offset - 1);
                    if (prevChar === 10 /* LineFeed */ || prevChar === 13 /* CarriageReturn */) {
                        lines[0] = model.normalizeIndentation(lineLeadingWhitespace + lines[0]);
                    }
                }
                for (var i = 1; i < lines.length; i++) {
                    lines[i] = model.normalizeIndentation(lineLeadingWhitespace + lines[i]);
                }
            }
            var newValue = lines.join(model.getEOL());
            if (newValue !== marker.value) {
                marker.parent.replace(marker, [new Text(newValue)]);
                snippetTextString = undefined;
            }
            return true;
        });
        return lineLeadingWhitespace;
    };
    SnippetSession.adjustSelection = function (model, selection, overwriteBefore, overwriteAfter) {
        if (overwriteBefore !== 0 || overwriteAfter !== 0) {
            // overwrite[Before|After] is compute using the position, not the whole
            // selection. therefore we adjust the selection around that position
            var positionLineNumber = selection.positionLineNumber, positionColumn = selection.positionColumn;
            var positionColumnBefore = positionColumn - overwriteBefore;
            var positionColumnAfter = positionColumn + overwriteAfter;
            var range = model.validateRange({
                startLineNumber: positionLineNumber,
                startColumn: positionColumnBefore,
                endLineNumber: positionLineNumber,
                endColumn: positionColumnAfter
            });
            selection = Selection.createWithDirection(range.startLineNumber, range.startColumn, range.endLineNumber, range.endColumn, selection.getDirection());
        }
        return selection;
    };
    SnippetSession.createEditsAndSnippets = function (editor, template, overwriteBefore, overwriteAfter, enforceFinalTabstop, adjustWhitespace, clipboardText, overtypingCapturer) {
        var e_12, _a;
        var edits = [];
        var snippets = [];
        if (!editor.hasModel()) {
            return { edits: edits, snippets: snippets };
        }
        var model = editor.getModel();
        var workspaceService = editor.invokeWithinContext(function (accessor) { return accessor.get(IWorkspaceContextService, optional); });
        var modelBasedVariableResolver = editor.invokeWithinContext(function (accessor) { return new ModelBasedVariableResolver(accessor.get(ILabelService, optional), model); });
        var readClipboardText = function () { return clipboardText; };
        var delta = 0;
        // know what text the overwrite[Before|After] extensions
        // of the primary curser have selected because only when
        // secondary selections extend to the same text we can grow them
        var firstBeforeText = model.getValueInRange(SnippetSession.adjustSelection(model, editor.getSelection(), overwriteBefore, 0));
        var firstAfterText = model.getValueInRange(SnippetSession.adjustSelection(model, editor.getSelection(), 0, overwriteAfter));
        // remember the first non-whitespace column to decide if
        // `keepWhitespace` should be overruled for secondary selections
        var firstLineFirstNonWhitespace = model.getLineFirstNonWhitespaceColumn(editor.getSelection().positionLineNumber);
        // sort selections by their start position but remeber
        // the original index. that allows you to create correct
        // offset-based selection logic without changing the
        // primary selection
        var indexedSelections = editor.getSelections()
            .map(function (selection, idx) { return ({ selection: selection, idx: idx }); })
            .sort(function (a, b) { return Range.compareRangesUsingStarts(a.selection, b.selection); });
        try {
            for (var indexedSelections_1 = __values(indexedSelections), indexedSelections_1_1 = indexedSelections_1.next(); !indexedSelections_1_1.done; indexedSelections_1_1 = indexedSelections_1.next()) {
                var _b = indexedSelections_1_1.value, selection = _b.selection, idx = _b.idx;
                // extend selection with the `overwriteBefore` and `overwriteAfter` and then
                // compare if this matches the extensions of the primary selection
                var extensionBefore = SnippetSession.adjustSelection(model, selection, overwriteBefore, 0);
                var extensionAfter = SnippetSession.adjustSelection(model, selection, 0, overwriteAfter);
                if (firstBeforeText !== model.getValueInRange(extensionBefore)) {
                    extensionBefore = selection;
                }
                if (firstAfterText !== model.getValueInRange(extensionAfter)) {
                    extensionAfter = selection;
                }
                // merge the before and after selection into one
                var snippetSelection = selection
                    .setStartPosition(extensionBefore.startLineNumber, extensionBefore.startColumn)
                    .setEndPosition(extensionAfter.endLineNumber, extensionAfter.endColumn);
                var snippet = new SnippetParser().parse(template, true, enforceFinalTabstop);
                // adjust the template string to match the indentation and
                // whitespace rules of this insert location (can be different for each cursor)
                // happens when being asked for (default) or when this is a secondary
                // cursor and the leading whitespace is different
                var start = snippetSelection.getStartPosition();
                var snippetLineLeadingWhitespace = SnippetSession.adjustWhitespace(model, start, snippet, adjustWhitespace || (idx > 0 && firstLineFirstNonWhitespace !== model.getLineFirstNonWhitespaceColumn(selection.positionLineNumber)), true);
                snippet.resolveVariables(new CompositeSnippetVariableResolver([
                    modelBasedVariableResolver,
                    new ClipboardBasedVariableResolver(readClipboardText, idx, indexedSelections.length, editor.getOption(65 /* multiCursorPaste */) === 'spread'),
                    new SelectionBasedVariableResolver(model, selection, idx, overtypingCapturer),
                    new CommentBasedVariableResolver(model, selection),
                    new TimeBasedVariableResolver,
                    new WorkspaceBasedVariableResolver(workspaceService),
                    new RandomBasedVariableResolver,
                ]));
                var offset = model.getOffsetAt(start) + delta;
                delta += snippet.toString().length - model.getValueLengthInRange(snippetSelection);
                // store snippets with the index of their originating selection.
                // that ensures the primiary cursor stays primary despite not being
                // the one with lowest start position
                edits[idx] = EditOperation.replace(snippetSelection, snippet.toString());
                edits[idx].identifier = { major: idx, minor: 0 }; // mark the edit so only our undo edits will be used to generate end cursors
                snippets[idx] = new OneSnippet(editor, snippet, offset, snippetLineLeadingWhitespace);
            }
        }
        catch (e_12_1) { e_12 = { error: e_12_1 }; }
        finally {
            try {
                if (indexedSelections_1_1 && !indexedSelections_1_1.done && (_a = indexedSelections_1.return)) _a.call(indexedSelections_1);
            }
            finally { if (e_12) throw e_12.error; }
        }
        return { edits: edits, snippets: snippets };
    };
    SnippetSession.prototype.dispose = function () {
        dispose(this._snippets);
    };
    SnippetSession.prototype._logInfo = function () {
        return "template=\"" + this._template + "\", merged_templates=\"" + this._templateMerges.join(' -> ') + "\"";
    };
    SnippetSession.prototype.insert = function () {
        var _this = this;
        if (!this._editor.hasModel()) {
            return;
        }
        // make insert edit and start with first selections
        var _a = SnippetSession.createEditsAndSnippets(this._editor, this._template, this._options.overwriteBefore, this._options.overwriteAfter, false, this._options.adjustWhitespace, this._options.clipboardText, this._options.overtypingCapturer), edits = _a.edits, snippets = _a.snippets;
        this._snippets = snippets;
        this._editor.executeEdits('snippet', edits, function (undoEdits) {
            if (_this._snippets[0].hasPlaceholder) {
                return _this._move(true);
            }
            else {
                return undoEdits
                    .filter(function (edit) { return !!edit.identifier; }) // only use our undo edits
                    .map(function (edit) { return Selection.fromPositions(edit.range.getEndPosition()); });
            }
        });
        this._editor.revealRange(this._editor.getSelections()[0]);
    };
    SnippetSession.prototype.merge = function (template, options) {
        var _this = this;
        if (options === void 0) { options = _defaultOptions; }
        if (!this._editor.hasModel()) {
            return;
        }
        this._templateMerges.push([this._snippets[0]._nestingLevel, this._snippets[0]._placeholderGroupsIdx, template]);
        var _a = SnippetSession.createEditsAndSnippets(this._editor, template, options.overwriteBefore, options.overwriteAfter, true, options.adjustWhitespace, options.clipboardText, options.overtypingCapturer), edits = _a.edits, snippets = _a.snippets;
        this._editor.executeEdits('snippet', edits, function (undoEdits) {
            var e_13, _a;
            try {
                for (var _b = __values(_this._snippets), _c = _b.next(); !_c.done; _c = _b.next()) {
                    var snippet = _c.value;
                    snippet.merge(snippets);
                }
            }
            catch (e_13_1) { e_13 = { error: e_13_1 }; }
            finally {
                try {
                    if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
                }
                finally { if (e_13) throw e_13.error; }
            }
            console.assert(snippets.length === 0);
            if (_this._snippets[0].hasPlaceholder) {
                return _this._move(undefined);
            }
            else {
                return (undoEdits
                    .filter(function (edit) { return !!edit.identifier; }) // only use our undo edits
                    .map(function (edit) { return Selection.fromPositions(edit.range.getEndPosition()); }));
            }
        });
    };
    SnippetSession.prototype.next = function () {
        var newSelections = this._move(true);
        this._editor.setSelections(newSelections);
        this._editor.revealPositionInCenterIfOutsideViewport(newSelections[0].getPosition());
    };
    SnippetSession.prototype.prev = function () {
        var newSelections = this._move(false);
        this._editor.setSelections(newSelections);
        this._editor.revealPositionInCenterIfOutsideViewport(newSelections[0].getPosition());
    };
    SnippetSession.prototype._move = function (fwd) {
        var e_14, _a;
        var selections = [];
        try {
            for (var _b = __values(this._snippets), _c = _b.next(); !_c.done; _c = _b.next()) {
                var snippet = _c.value;
                var oneSelection = snippet.move(fwd);
                selections.push.apply(selections, __spreadArray([], __read(oneSelection)));
            }
        }
        catch (e_14_1) { e_14 = { error: e_14_1 }; }
        finally {
            try {
                if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
            }
            finally { if (e_14) throw e_14.error; }
        }
        return selections;
    };
    Object.defineProperty(SnippetSession.prototype, "isAtFirstPlaceholder", {
        get: function () {
            return this._snippets[0].isAtFirstPlaceholder;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(SnippetSession.prototype, "isAtLastPlaceholder", {
        get: function () {
            return this._snippets[0].isAtLastPlaceholder;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(SnippetSession.prototype, "hasPlaceholder", {
        get: function () {
            return this._snippets[0].hasPlaceholder;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(SnippetSession.prototype, "choice", {
        get: function () {
            return this._snippets[0].choice;
        },
        enumerable: false,
        configurable: true
    });
    SnippetSession.prototype.isSelectionWithinPlaceholders = function () {
        var e_15, _a, e_16, _b;
        if (!this.hasPlaceholder) {
            return false;
        }
        var selections = this._editor.getSelections();
        if (selections.length < this._snippets.length) {
            // this means we started snippet mode with N
            // selections and have M (N > M) selections.
            // So one snippet is without selection -> cancel
            return false;
        }
        var allPossibleSelections = new Map();
        var _loop_1 = function (snippet) {
            var e_17, _f, e_18, _g;
            var possibleSelections = snippet.computePossibleSelections();
            // for the first snippet find the placeholder (and its ranges)
            // that contain at least one selection. for all remaining snippets
            // the same placeholder (and their ranges) must be used.
            if (allPossibleSelections.size === 0) {
                try {
                    for (var possibleSelections_1 = (e_17 = void 0, __values(possibleSelections)), possibleSelections_1_1 = possibleSelections_1.next(); !possibleSelections_1_1.done; possibleSelections_1_1 = possibleSelections_1.next()) {
                        var _h = __read(possibleSelections_1_1.value, 2), index = _h[0], ranges = _h[1];
                        ranges.sort(Range.compareRangesUsingStarts);
                        try {
                            for (var selections_1 = (e_18 = void 0, __values(selections)), selections_1_1 = selections_1.next(); !selections_1_1.done; selections_1_1 = selections_1.next()) {
                                var selection = selections_1_1.value;
                                if (ranges[0].containsRange(selection)) {
                                    allPossibleSelections.set(index, []);
                                    break;
                                }
                            }
                        }
                        catch (e_18_1) { e_18 = { error: e_18_1 }; }
                        finally {
                            try {
                                if (selections_1_1 && !selections_1_1.done && (_g = selections_1.return)) _g.call(selections_1);
                            }
                            finally { if (e_18) throw e_18.error; }
                        }
                    }
                }
                catch (e_17_1) { e_17 = { error: e_17_1 }; }
                finally {
                    try {
                        if (possibleSelections_1_1 && !possibleSelections_1_1.done && (_f = possibleSelections_1.return)) _f.call(possibleSelections_1);
                    }
                    finally { if (e_17) throw e_17.error; }
                }
            }
            if (allPossibleSelections.size === 0) {
                return { value: false };
            }
            // add selections from 'this' snippet so that we know all
            // selections for this placeholder
            allPossibleSelections.forEach(function (array, index) {
                array.push.apply(array, __spreadArray([], __read(possibleSelections.get(index))));
            });
        };
        try {
            for (var _c = __values(this._snippets), _d = _c.next(); !_d.done; _d = _c.next()) {
                var snippet = _d.value;
                var state_1 = _loop_1(snippet);
                if (typeof state_1 === "object")
                    return state_1.value;
            }
        }
        catch (e_15_1) { e_15 = { error: e_15_1 }; }
        finally {
            try {
                if (_d && !_d.done && (_a = _c.return)) _a.call(_c);
            }
            finally { if (e_15) throw e_15.error; }
        }
        // sort selections (and later placeholder-ranges). then walk both
        // arrays and make sure the placeholder-ranges contain the corresponding
        // selection
        selections.sort(Range.compareRangesUsingStarts);
        try {
            for (var allPossibleSelections_1 = __values(allPossibleSelections), allPossibleSelections_1_1 = allPossibleSelections_1.next(); !allPossibleSelections_1_1.done; allPossibleSelections_1_1 = allPossibleSelections_1.next()) {
                var _e = __read(allPossibleSelections_1_1.value, 2), index = _e[0], ranges = _e[1];
                if (ranges.length !== selections.length) {
                    allPossibleSelections.delete(index);
                    continue;
                }
                ranges.sort(Range.compareRangesUsingStarts);
                for (var i = 0; i < ranges.length; i++) {
                    if (!ranges[i].containsRange(selections[i])) {
                        allPossibleSelections.delete(index);
                        continue;
                    }
                }
            }
        }
        catch (e_16_1) { e_16 = { error: e_16_1 }; }
        finally {
            try {
                if (allPossibleSelections_1_1 && !allPossibleSelections_1_1.done && (_b = allPossibleSelections_1.return)) _b.call(allPossibleSelections_1);
            }
            finally { if (e_16) throw e_16.error; }
        }
        // from all possible selections we have deleted those
        // that don't match with the current selection. if we don't
        // have any left, we don't have a selection anymore
        return allPossibleSelections.size > 0;
    };
    SnippetSession.prototype.getEnclosingRange = function () {
        var e_19, _a;
        var result;
        try {
            for (var _b = __values(this._snippets), _c = _b.next(); !_c.done; _c = _b.next()) {
                var snippet = _c.value;
                var snippetRange = snippet.getEnclosingRange();
                if (!result) {
                    result = snippetRange;
                }
                else {
                    result = result.plusRange(snippetRange);
                }
            }
        }
        catch (e_19_1) { e_19 = { error: e_19_1 }; }
        finally {
            try {
                if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
            }
            finally { if (e_19) throw e_19.error; }
        }
        return result;
    };
    return SnippetSession;
}());
export { SnippetSession };
