/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __extends = (this && this.__extends) || (function () {
    var extendStatics = function (d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (Object.prototype.hasOwnProperty.call(b, p)) d[p] = b[p]; };
        return extendStatics(d, b);
    };
    return function (d, b) {
        if (typeof b !== "function" && b !== null)
            throw new TypeError("Class extends value " + String(b) + " is not a constructor or null");
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __param = (this && this.__param) || function (paramIndex, decorator) {
    return function (target, key) { decorator(target, key, paramIndex); }
};
var __values = (this && this.__values) || function(o) {
    var s = typeof Symbol === "function" && Symbol.iterator, m = s && o[s], i = 0;
    if (m) return m.call(o);
    if (o && typeof o.length === "number") return {
        next: function () {
            if (o && i >= o.length) o = void 0;
            return { value: o && o[i++], done: !o };
        }
    };
    throw new TypeError(s ? "Object is not iterable." : "Symbol.iterator is not defined.");
};
import { Emitter } from '../../../base/common/event';
import { Disposable, DisposableStore, dispose } from '../../../base/common/lifecycle';
import * as platform from '../../../base/common/platform';
import * as errors from '../../../base/common/errors';
import { EDITOR_MODEL_DEFAULTS } from '../config/editorOptions';
import { EditOperation } from '../core/editOperation';
import { Range } from '../core/range';
import { TextModel, createTextBuffer } from '../model/textModel';
import { DocumentSemanticTokensProviderRegistry } from '../modes';
import { PLAINTEXT_LANGUAGE_IDENTIFIER } from '../modes/modesRegistry';
import { ITextResourcePropertiesService } from './textResourceConfigurationService';
import { IConfigurationService } from '../../../platform/configuration/common/configuration';
import { RunOnceScheduler } from '../../../base/common/async';
import { CancellationTokenSource } from '../../../base/common/cancellation';
import { IThemeService } from '../../../platform/theme/common/themeService';
import { ILogService } from '../../../platform/log/common/log';
import { IUndoRedoService } from '../../../platform/undoRedo/common/undoRedo';
import { StringSHA1 } from '../../../base/common/hash';
import { isEditStackElement } from '../model/editStack';
import { Schemas } from '../../../base/common/network';
import { SemanticTokensProviderStyling, toMultilineTokens2 } from './semanticTokensProviderStyling';
import { getDocumentSemanticTokens, isSemanticTokens, isSemanticTokensEdits } from './getSemanticTokens';
function MODEL_ID(resource) {
    return resource.toString();
}
function computeModelSha1(model) {
    // compute the sha1
    var shaComputer = new StringSHA1();
    var snapshot = model.createSnapshot();
    var text;
    while ((text = snapshot.read())) {
        shaComputer.update(text);
    }
    return shaComputer.digest();
}
var ModelData = /** @class */ (function () {
    function ModelData(model, onWillDispose, onDidChangeLanguage) {
        this._modelEventListeners = new DisposableStore();
        this.model = model;
        this._languageSelection = null;
        this._languageSelectionListener = null;
        this._modelEventListeners.add(model.onWillDispose(function () { return onWillDispose(model); }));
        this._modelEventListeners.add(model.onDidChangeLanguage(function (e) { return onDidChangeLanguage(model, e); }));
    }
    ModelData.prototype._disposeLanguageSelection = function () {
        if (this._languageSelectionListener) {
            this._languageSelectionListener.dispose();
            this._languageSelectionListener = null;
        }
        if (this._languageSelection) {
            this._languageSelection.dispose();
            this._languageSelection = null;
        }
    };
    ModelData.prototype.dispose = function () {
        this._modelEventListeners.dispose();
        this._disposeLanguageSelection();
    };
    ModelData.prototype.setLanguage = function (languageSelection) {
        var _this = this;
        this._disposeLanguageSelection();
        this._languageSelection = languageSelection;
        this._languageSelectionListener = this._languageSelection.onDidChange(function () { return _this.model.setMode(languageSelection.languageIdentifier); });
        this.model.setMode(languageSelection.languageIdentifier);
    };
    return ModelData;
}());
var DEFAULT_EOL = (platform.isLinux || platform.isMacintosh) ? 1 /* LF */ : 2 /* CRLF */;
var DisposedModelInfo = /** @class */ (function () {
    function DisposedModelInfo(uri, initialUndoRedoSnapshot, time, sharesUndoRedoStack, heapSize, sha1, versionId, alternativeVersionId) {
        this.uri = uri;
        this.initialUndoRedoSnapshot = initialUndoRedoSnapshot;
        this.time = time;
        this.sharesUndoRedoStack = sharesUndoRedoStack;
        this.heapSize = heapSize;
        this.sha1 = sha1;
        this.versionId = versionId;
        this.alternativeVersionId = alternativeVersionId;
    }
    return DisposedModelInfo;
}());
function schemaShouldMaintainUndoRedoElements(resource) {
    return (resource.scheme === Schemas.file
        || resource.scheme === Schemas.vscodeRemote
        || resource.scheme === Schemas.userData
        || resource.scheme === 'fake-fs' // for tests
    );
}
var ModelServiceImpl = /** @class */ (function (_super) {
    __extends(ModelServiceImpl, _super);
    function ModelServiceImpl(_configurationService, _resourcePropertiesService, _themeService, _logService, _undoRedoService) {
        var _this = _super.call(this) || this;
        _this._configurationService = _configurationService;
        _this._resourcePropertiesService = _resourcePropertiesService;
        _this._themeService = _themeService;
        _this._logService = _logService;
        _this._undoRedoService = _undoRedoService;
        _this._onModelAdded = _this._register(new Emitter());
        _this.onModelAdded = _this._onModelAdded.event;
        _this._onModelRemoved = _this._register(new Emitter());
        _this.onModelRemoved = _this._onModelRemoved.event;
        _this._onModelModeChanged = _this._register(new Emitter());
        _this.onModelModeChanged = _this._onModelModeChanged.event;
        _this._modelCreationOptionsByLanguageAndResource = Object.create(null);
        _this._models = {};
        _this._disposedModels = new Map();
        _this._disposedModelsHeapSize = 0;
        _this._semanticStyling = _this._register(new SemanticStyling(_this._themeService, _this._logService));
        _this._register(_this._configurationService.onDidChangeConfiguration(function () { return _this._updateModelOptions(); }));
        _this._updateModelOptions();
        _this._register(new SemanticColoringFeature(_this, _this._themeService, _this._configurationService, _this._semanticStyling));
        return _this;
    }
    ModelServiceImpl._readModelOptions = function (config, isForSimpleWidget) {
        var tabSize = EDITOR_MODEL_DEFAULTS.tabSize;
        if (config.editor && typeof config.editor.tabSize !== 'undefined') {
            var parsedTabSize = parseInt(config.editor.tabSize, 10);
            if (!isNaN(parsedTabSize)) {
                tabSize = parsedTabSize;
            }
            if (tabSize < 1) {
                tabSize = 1;
            }
        }
        var indentSize = tabSize;
        if (config.editor && typeof config.editor.indentSize !== 'undefined' && config.editor.indentSize !== 'tabSize') {
            var parsedIndentSize = parseInt(config.editor.indentSize, 10);
            if (!isNaN(parsedIndentSize)) {
                indentSize = parsedIndentSize;
            }
            if (indentSize < 1) {
                indentSize = 1;
            }
        }
        var insertSpaces = EDITOR_MODEL_DEFAULTS.insertSpaces;
        if (config.editor && typeof config.editor.insertSpaces !== 'undefined') {
            insertSpaces = (config.editor.insertSpaces === 'false' ? false : Boolean(config.editor.insertSpaces));
        }
        var newDefaultEOL = DEFAULT_EOL;
        var eol = config.eol;
        if (eol === '\r\n') {
            newDefaultEOL = 2 /* CRLF */;
        }
        else if (eol === '\n') {
            newDefaultEOL = 1 /* LF */;
        }
        var trimAutoWhitespace = EDITOR_MODEL_DEFAULTS.trimAutoWhitespace;
        if (config.editor && typeof config.editor.trimAutoWhitespace !== 'undefined') {
            trimAutoWhitespace = (config.editor.trimAutoWhitespace === 'false' ? false : Boolean(config.editor.trimAutoWhitespace));
        }
        var detectIndentation = EDITOR_MODEL_DEFAULTS.detectIndentation;
        if (config.editor && typeof config.editor.detectIndentation !== 'undefined') {
            detectIndentation = (config.editor.detectIndentation === 'false' ? false : Boolean(config.editor.detectIndentation));
        }
        var largeFileOptimizations = EDITOR_MODEL_DEFAULTS.largeFileOptimizations;
        if (config.editor && typeof config.editor.largeFileOptimizations !== 'undefined') {
            largeFileOptimizations = (config.editor.largeFileOptimizations === 'false' ? false : Boolean(config.editor.largeFileOptimizations));
        }
        return {
            isForSimpleWidget: isForSimpleWidget,
            tabSize: tabSize,
            indentSize: indentSize,
            insertSpaces: insertSpaces,
            detectIndentation: detectIndentation,
            defaultEOL: newDefaultEOL,
            trimAutoWhitespace: trimAutoWhitespace,
            largeFileOptimizations: largeFileOptimizations
        };
    };
    ModelServiceImpl.prototype._getEOL = function (resource, language) {
        if (resource) {
            return this._resourcePropertiesService.getEOL(resource, language);
        }
        var eol = this._configurationService.getValue('files.eol', { overrideIdentifier: language });
        if (eol && eol !== 'auto') {
            return eol;
        }
        return platform.OS === 3 /* Linux */ || platform.OS === 2 /* Macintosh */ ? '\n' : '\r\n';
    };
    ModelServiceImpl.prototype._shouldRestoreUndoStack = function () {
        var result = this._configurationService.getValue('files.restoreUndoStack');
        if (typeof result === 'boolean') {
            return result;
        }
        return true;
    };
    ModelServiceImpl.prototype.getCreationOptions = function (language, resource, isForSimpleWidget) {
        var creationOptions = this._modelCreationOptionsByLanguageAndResource[language + resource];
        if (!creationOptions) {
            var editor = this._configurationService.getValue('editor', { overrideIdentifier: language, resource: resource });
            var eol = this._getEOL(resource, language);
            creationOptions = ModelServiceImpl._readModelOptions({ editor: editor, eol: eol }, isForSimpleWidget);
            this._modelCreationOptionsByLanguageAndResource[language + resource] = creationOptions;
        }
        return creationOptions;
    };
    ModelServiceImpl.prototype._updateModelOptions = function () {
        var oldOptionsByLanguageAndResource = this._modelCreationOptionsByLanguageAndResource;
        this._modelCreationOptionsByLanguageAndResource = Object.create(null);
        // Update options on all models
        var keys = Object.keys(this._models);
        for (var i = 0, len = keys.length; i < len; i++) {
            var modelId = keys[i];
            var modelData = this._models[modelId];
            var language = modelData.model.getLanguageIdentifier().language;
            var uri = modelData.model.uri;
            var oldOptions = oldOptionsByLanguageAndResource[language + uri];
            var newOptions = this.getCreationOptions(language, uri, modelData.model.isForSimpleWidget);
            ModelServiceImpl._setModelOptionsForModel(modelData.model, newOptions, oldOptions);
        }
    };
    ModelServiceImpl._setModelOptionsForModel = function (model, newOptions, currentOptions) {
        if (currentOptions && currentOptions.defaultEOL !== newOptions.defaultEOL && model.getLineCount() === 1) {
            model.setEOL(newOptions.defaultEOL === 1 /* LF */ ? 0 /* LF */ : 1 /* CRLF */);
        }
        if (currentOptions
            && (currentOptions.detectIndentation === newOptions.detectIndentation)
            && (currentOptions.insertSpaces === newOptions.insertSpaces)
            && (currentOptions.tabSize === newOptions.tabSize)
            && (currentOptions.indentSize === newOptions.indentSize)
            && (currentOptions.trimAutoWhitespace === newOptions.trimAutoWhitespace)) {
            // Same indent opts, no need to touch the model
            return;
        }
        if (newOptions.detectIndentation) {
            model.detectIndentation(newOptions.insertSpaces, newOptions.tabSize);
            model.updateOptions({
                trimAutoWhitespace: newOptions.trimAutoWhitespace
            });
        }
        else {
            model.updateOptions({
                insertSpaces: newOptions.insertSpaces,
                tabSize: newOptions.tabSize,
                indentSize: newOptions.indentSize,
                trimAutoWhitespace: newOptions.trimAutoWhitespace
            });
        }
    };
    // --- begin IModelService
    ModelServiceImpl.prototype._insertDisposedModel = function (disposedModelData) {
        this._disposedModels.set(MODEL_ID(disposedModelData.uri), disposedModelData);
        this._disposedModelsHeapSize += disposedModelData.heapSize;
    };
    ModelServiceImpl.prototype._removeDisposedModel = function (resource) {
        var disposedModelData = this._disposedModels.get(MODEL_ID(resource));
        if (disposedModelData) {
            this._disposedModelsHeapSize -= disposedModelData.heapSize;
        }
        this._disposedModels.delete(MODEL_ID(resource));
        return disposedModelData;
    };
    ModelServiceImpl.prototype._ensureDisposedModelsHeapSize = function (maxModelsHeapSize) {
        if (this._disposedModelsHeapSize > maxModelsHeapSize) {
            // we must remove some old undo stack elements to free up some memory
            var disposedModels_1 = [];
            this._disposedModels.forEach(function (entry) {
                if (!entry.sharesUndoRedoStack) {
                    disposedModels_1.push(entry);
                }
            });
            disposedModels_1.sort(function (a, b) { return a.time - b.time; });
            while (disposedModels_1.length > 0 && this._disposedModelsHeapSize > maxModelsHeapSize) {
                var disposedModel = disposedModels_1.shift();
                this._removeDisposedModel(disposedModel.uri);
                if (disposedModel.initialUndoRedoSnapshot !== null) {
                    this._undoRedoService.restoreSnapshot(disposedModel.initialUndoRedoSnapshot);
                }
            }
        }
    };
    ModelServiceImpl.prototype._createModelData = function (value, languageIdentifier, resource, isForSimpleWidget) {
        var e_1, _a, e_2, _b;
        var _this = this;
        // create & save the model
        var options = this.getCreationOptions(languageIdentifier.language, resource, isForSimpleWidget);
        var model = new TextModel(value, options, languageIdentifier, resource, this._undoRedoService);
        if (resource && this._disposedModels.has(MODEL_ID(resource))) {
            var disposedModelData = this._removeDisposedModel(resource);
            var elements = this._undoRedoService.getElements(resource);
            var sha1IsEqual = (computeModelSha1(model) === disposedModelData.sha1);
            if (sha1IsEqual || disposedModelData.sharesUndoRedoStack) {
                try {
                    for (var _c = __values(elements.past), _d = _c.next(); !_d.done; _d = _c.next()) {
                        var element = _d.value;
                        if (isEditStackElement(element) && element.matchesResource(resource)) {
                            element.setModel(model);
                        }
                    }
                }
                catch (e_1_1) { e_1 = { error: e_1_1 }; }
                finally {
                    try {
                        if (_d && !_d.done && (_a = _c.return)) _a.call(_c);
                    }
                    finally { if (e_1) throw e_1.error; }
                }
                try {
                    for (var _e = __values(elements.future), _f = _e.next(); !_f.done; _f = _e.next()) {
                        var element = _f.value;
                        if (isEditStackElement(element) && element.matchesResource(resource)) {
                            element.setModel(model);
                        }
                    }
                }
                catch (e_2_1) { e_2 = { error: e_2_1 }; }
                finally {
                    try {
                        if (_f && !_f.done && (_b = _e.return)) _b.call(_e);
                    }
                    finally { if (e_2) throw e_2.error; }
                }
                this._undoRedoService.setElementsValidFlag(resource, true, function (element) { return (isEditStackElement(element) && element.matchesResource(resource)); });
                if (sha1IsEqual) {
                    model._overwriteVersionId(disposedModelData.versionId);
                    model._overwriteAlternativeVersionId(disposedModelData.alternativeVersionId);
                    model._overwriteInitialUndoRedoSnapshot(disposedModelData.initialUndoRedoSnapshot);
                }
            }
            else {
                if (disposedModelData.initialUndoRedoSnapshot !== null) {
                    this._undoRedoService.restoreSnapshot(disposedModelData.initialUndoRedoSnapshot);
                }
            }
        }
        var modelId = MODEL_ID(model.uri);
        if (this._models[modelId]) {
            // There already exists a model with this id => this is a programmer error
            throw new Error('ModelService: Cannot add model because it already exists!');
        }
        var modelData = new ModelData(model, function (model) { return _this._onWillDispose(model); }, function (model, e) { return _this._onDidChangeLanguage(model, e); });
        this._models[modelId] = modelData;
        return modelData;
    };
    ModelServiceImpl.prototype.updateModel = function (model, value) {
        var options = this.getCreationOptions(model.getLanguageIdentifier().language, model.uri, model.isForSimpleWidget);
        var _a = createTextBuffer(value, options.defaultEOL), textBuffer = _a.textBuffer, disposable = _a.disposable;
        // Return early if the text is already set in that form
        if (model.equalsTextBuffer(textBuffer)) {
            disposable.dispose();
            return;
        }
        // Otherwise find a diff between the values and update model
        model.pushStackElement();
        model.pushEOL(textBuffer.getEOL() === '\r\n' ? 1 /* CRLF */ : 0 /* LF */);
        model.pushEditOperations([], ModelServiceImpl._computeEdits(model, textBuffer), function () { return []; });
        model.pushStackElement();
        disposable.dispose();
    };
    ModelServiceImpl._commonPrefix = function (a, aLen, aDelta, b, bLen, bDelta) {
        var maxResult = Math.min(aLen, bLen);
        var result = 0;
        for (var i = 0; i < maxResult && a.getLineContent(aDelta + i) === b.getLineContent(bDelta + i); i++) {
            result++;
        }
        return result;
    };
    ModelServiceImpl._commonSuffix = function (a, aLen, aDelta, b, bLen, bDelta) {
        var maxResult = Math.min(aLen, bLen);
        var result = 0;
        for (var i = 0; i < maxResult && a.getLineContent(aDelta + aLen - i) === b.getLineContent(bDelta + bLen - i); i++) {
            result++;
        }
        return result;
    };
    /**
     * Compute edits to bring `model` to the state of `textSource`.
     */
    ModelServiceImpl._computeEdits = function (model, textBuffer) {
        var modelLineCount = model.getLineCount();
        var textBufferLineCount = textBuffer.getLineCount();
        var commonPrefix = this._commonPrefix(model, modelLineCount, 1, textBuffer, textBufferLineCount, 1);
        if (modelLineCount === textBufferLineCount && commonPrefix === modelLineCount) {
            // equality case
            return [];
        }
        var commonSuffix = this._commonSuffix(model, modelLineCount - commonPrefix, commonPrefix, textBuffer, textBufferLineCount - commonPrefix, commonPrefix);
        var oldRange;
        var newRange;
        if (commonSuffix > 0) {
            oldRange = new Range(commonPrefix + 1, 1, modelLineCount - commonSuffix + 1, 1);
            newRange = new Range(commonPrefix + 1, 1, textBufferLineCount - commonSuffix + 1, 1);
        }
        else if (commonPrefix > 0) {
            oldRange = new Range(commonPrefix, model.getLineMaxColumn(commonPrefix), modelLineCount, model.getLineMaxColumn(modelLineCount));
            newRange = new Range(commonPrefix, 1 + textBuffer.getLineLength(commonPrefix), textBufferLineCount, 1 + textBuffer.getLineLength(textBufferLineCount));
        }
        else {
            oldRange = new Range(1, 1, modelLineCount, model.getLineMaxColumn(modelLineCount));
            newRange = new Range(1, 1, textBufferLineCount, 1 + textBuffer.getLineLength(textBufferLineCount));
        }
        return [EditOperation.replaceMove(oldRange, textBuffer.getValueInRange(newRange, 0 /* TextDefined */))];
    };
    ModelServiceImpl.prototype.createModel = function (value, languageSelection, resource, isForSimpleWidget) {
        if (isForSimpleWidget === void 0) { isForSimpleWidget = false; }
        var modelData;
        if (languageSelection) {
            modelData = this._createModelData(value, languageSelection.languageIdentifier, resource, isForSimpleWidget);
            this.setMode(modelData.model, languageSelection);
        }
        else {
            modelData = this._createModelData(value, PLAINTEXT_LANGUAGE_IDENTIFIER, resource, isForSimpleWidget);
        }
        this._onModelAdded.fire(modelData.model);
        return modelData.model;
    };
    ModelServiceImpl.prototype.setMode = function (model, languageSelection) {
        if (!languageSelection) {
            return;
        }
        var modelData = this._models[MODEL_ID(model.uri)];
        if (!modelData) {
            return;
        }
        modelData.setLanguage(languageSelection);
    };
    ModelServiceImpl.prototype.destroyModel = function (resource) {
        // We need to support that not all models get disposed through this service (i.e. model.dispose() should work!)
        var modelData = this._models[MODEL_ID(resource)];
        if (!modelData) {
            return;
        }
        modelData.model.dispose();
    };
    ModelServiceImpl.prototype.getModels = function () {
        var ret = [];
        var keys = Object.keys(this._models);
        for (var i = 0, len = keys.length; i < len; i++) {
            var modelId = keys[i];
            ret.push(this._models[modelId].model);
        }
        return ret;
    };
    ModelServiceImpl.prototype.getModel = function (resource) {
        var modelId = MODEL_ID(resource);
        var modelData = this._models[modelId];
        if (!modelData) {
            return null;
        }
        return modelData.model;
    };
    ModelServiceImpl.prototype.getSemanticTokensProviderStyling = function (provider) {
        return this._semanticStyling.get(provider);
    };
    // --- end IModelService
    ModelServiceImpl.prototype._onWillDispose = function (model) {
        var e_3, _a, e_4, _b;
        var modelId = MODEL_ID(model.uri);
        var modelData = this._models[modelId];
        var sharesUndoRedoStack = (this._undoRedoService.getUriComparisonKey(model.uri) !== model.uri.toString());
        var maintainUndoRedoStack = false;
        var heapSize = 0;
        if (sharesUndoRedoStack || (this._shouldRestoreUndoStack() && schemaShouldMaintainUndoRedoElements(model.uri))) {
            var elements = this._undoRedoService.getElements(model.uri);
            if (elements.past.length > 0 || elements.future.length > 0) {
                try {
                    for (var _c = __values(elements.past), _d = _c.next(); !_d.done; _d = _c.next()) {
                        var element = _d.value;
                        if (isEditStackElement(element) && element.matchesResource(model.uri)) {
                            maintainUndoRedoStack = true;
                            heapSize += element.heapSize(model.uri);
                            element.setModel(model.uri); // remove reference from text buffer instance
                        }
                    }
                }
                catch (e_3_1) { e_3 = { error: e_3_1 }; }
                finally {
                    try {
                        if (_d && !_d.done && (_a = _c.return)) _a.call(_c);
                    }
                    finally { if (e_3) throw e_3.error; }
                }
                try {
                    for (var _e = __values(elements.future), _f = _e.next(); !_f.done; _f = _e.next()) {
                        var element = _f.value;
                        if (isEditStackElement(element) && element.matchesResource(model.uri)) {
                            maintainUndoRedoStack = true;
                            heapSize += element.heapSize(model.uri);
                            element.setModel(model.uri); // remove reference from text buffer instance
                        }
                    }
                }
                catch (e_4_1) { e_4 = { error: e_4_1 }; }
                finally {
                    try {
                        if (_f && !_f.done && (_b = _e.return)) _b.call(_e);
                    }
                    finally { if (e_4) throw e_4.error; }
                }
            }
        }
        var maxMemory = ModelServiceImpl.MAX_MEMORY_FOR_CLOSED_FILES_UNDO_STACK;
        if (!maintainUndoRedoStack) {
            if (!sharesUndoRedoStack) {
                var initialUndoRedoSnapshot = modelData.model.getInitialUndoRedoSnapshot();
                if (initialUndoRedoSnapshot !== null) {
                    this._undoRedoService.restoreSnapshot(initialUndoRedoSnapshot);
                }
            }
        }
        else if (!sharesUndoRedoStack && heapSize > maxMemory) {
            // the undo stack for this file would never fit in the configured memory, so don't bother with it.
            var initialUndoRedoSnapshot = modelData.model.getInitialUndoRedoSnapshot();
            if (initialUndoRedoSnapshot !== null) {
                this._undoRedoService.restoreSnapshot(initialUndoRedoSnapshot);
            }
        }
        else {
            this._ensureDisposedModelsHeapSize(maxMemory - heapSize);
            // We only invalidate the elements, but they remain in the undo-redo service.
            this._undoRedoService.setElementsValidFlag(model.uri, false, function (element) { return (isEditStackElement(element) && element.matchesResource(model.uri)); });
            this._insertDisposedModel(new DisposedModelInfo(model.uri, modelData.model.getInitialUndoRedoSnapshot(), Date.now(), sharesUndoRedoStack, heapSize, computeModelSha1(model), model.getVersionId(), model.getAlternativeVersionId()));
        }
        delete this._models[modelId];
        modelData.dispose();
        // clean up cache
        delete this._modelCreationOptionsByLanguageAndResource[model.getLanguageIdentifier().language + model.uri];
        this._onModelRemoved.fire(model);
    };
    ModelServiceImpl.prototype._onDidChangeLanguage = function (model, e) {
        var oldModeId = e.oldLanguage;
        var newModeId = model.getLanguageIdentifier().language;
        var oldOptions = this.getCreationOptions(oldModeId, model.uri, model.isForSimpleWidget);
        var newOptions = this.getCreationOptions(newModeId, model.uri, model.isForSimpleWidget);
        ModelServiceImpl._setModelOptionsForModel(model, newOptions, oldOptions);
        this._onModelModeChanged.fire({ model: model, oldModeId: oldModeId });
    };
    ModelServiceImpl.MAX_MEMORY_FOR_CLOSED_FILES_UNDO_STACK = 20 * 1024 * 1024;
    ModelServiceImpl = __decorate([
        __param(0, IConfigurationService),
        __param(1, ITextResourcePropertiesService),
        __param(2, IThemeService),
        __param(3, ILogService),
        __param(4, IUndoRedoService)
    ], ModelServiceImpl);
    return ModelServiceImpl;
}(Disposable));
export { ModelServiceImpl };
export var SEMANTIC_HIGHLIGHTING_SETTING_ID = 'editor.semanticHighlighting';
export function isSemanticColoringEnabled(model, themeService, configurationService) {
    var _a;
    var setting = (_a = configurationService.getValue(SEMANTIC_HIGHLIGHTING_SETTING_ID, { overrideIdentifier: model.getLanguageIdentifier().language, resource: model.uri })) === null || _a === void 0 ? void 0 : _a.enabled;
    if (typeof setting === 'boolean') {
        return setting;
    }
    return themeService.getColorTheme().semanticHighlighting;
}
var SemanticColoringFeature = /** @class */ (function (_super) {
    __extends(SemanticColoringFeature, _super);
    function SemanticColoringFeature(modelService, themeService, configurationService, semanticStyling) {
        var _this = _super.call(this) || this;
        _this._watchers = Object.create(null);
        _this._semanticStyling = semanticStyling;
        var register = function (model) {
            _this._watchers[model.uri.toString()] = new ModelSemanticColoring(model, themeService, _this._semanticStyling);
        };
        var deregister = function (model, modelSemanticColoring) {
            modelSemanticColoring.dispose();
            delete _this._watchers[model.uri.toString()];
        };
        var handleSettingOrThemeChange = function () {
            var e_5, _a;
            try {
                for (var _b = __values(modelService.getModels()), _c = _b.next(); !_c.done; _c = _b.next()) {
                    var model = _c.value;
                    var curr = _this._watchers[model.uri.toString()];
                    if (isSemanticColoringEnabled(model, themeService, configurationService)) {
                        if (!curr) {
                            register(model);
                        }
                    }
                    else {
                        if (curr) {
                            deregister(model, curr);
                        }
                    }
                }
            }
            catch (e_5_1) { e_5 = { error: e_5_1 }; }
            finally {
                try {
                    if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
                }
                finally { if (e_5) throw e_5.error; }
            }
        };
        _this._register(modelService.onModelAdded(function (model) {
            if (isSemanticColoringEnabled(model, themeService, configurationService)) {
                register(model);
            }
        }));
        _this._register(modelService.onModelRemoved(function (model) {
            var curr = _this._watchers[model.uri.toString()];
            if (curr) {
                deregister(model, curr);
            }
        }));
        _this._register(configurationService.onDidChangeConfiguration(function (e) {
            if (e.affectsConfiguration(SEMANTIC_HIGHLIGHTING_SETTING_ID)) {
                handleSettingOrThemeChange();
            }
        }));
        _this._register(themeService.onDidColorThemeChange(handleSettingOrThemeChange));
        return _this;
    }
    return SemanticColoringFeature;
}(Disposable));
var SemanticStyling = /** @class */ (function (_super) {
    __extends(SemanticStyling, _super);
    function SemanticStyling(_themeService, _logService) {
        var _this = _super.call(this) || this;
        _this._themeService = _themeService;
        _this._logService = _logService;
        _this._caches = new WeakMap();
        _this._register(_this._themeService.onDidColorThemeChange(function () {
            _this._caches = new WeakMap();
        }));
        return _this;
    }
    SemanticStyling.prototype.get = function (provider) {
        if (!this._caches.has(provider)) {
            this._caches.set(provider, new SemanticTokensProviderStyling(provider.getLegend(), this._themeService, this._logService));
        }
        return this._caches.get(provider);
    };
    return SemanticStyling;
}(Disposable));
var SemanticTokensResponse = /** @class */ (function () {
    function SemanticTokensResponse(_provider, resultId, data) {
        this._provider = _provider;
        this.resultId = resultId;
        this.data = data;
    }
    SemanticTokensResponse.prototype.dispose = function () {
        this._provider.releaseDocumentSemanticTokens(this.resultId);
    };
    return SemanticTokensResponse;
}());
var ModelSemanticColoring = /** @class */ (function (_super) {
    __extends(ModelSemanticColoring, _super);
    function ModelSemanticColoring(model, themeService, stylingProvider) {
        var _this = _super.call(this) || this;
        _this._isDisposed = false;
        _this._model = model;
        _this._semanticStyling = stylingProvider;
        _this._fetchDocumentSemanticTokens = _this._register(new RunOnceScheduler(function () { return _this._fetchDocumentSemanticTokensNow(); }, ModelSemanticColoring.FETCH_DOCUMENT_SEMANTIC_TOKENS_DELAY));
        _this._currentDocumentResponse = null;
        _this._currentDocumentRequestCancellationTokenSource = null;
        _this._documentProvidersChangeListeners = [];
        _this._register(_this._model.onDidChangeContent(function () {
            if (!_this._fetchDocumentSemanticTokens.isScheduled()) {
                _this._fetchDocumentSemanticTokens.schedule();
            }
        }));
        var bindDocumentChangeListeners = function () {
            var e_6, _a;
            dispose(_this._documentProvidersChangeListeners);
            _this._documentProvidersChangeListeners = [];
            try {
                for (var _b = __values(DocumentSemanticTokensProviderRegistry.all(model)), _c = _b.next(); !_c.done; _c = _b.next()) {
                    var provider = _c.value;
                    if (typeof provider.onDidChange === 'function') {
                        _this._documentProvidersChangeListeners.push(provider.onDidChange(function () { return _this._fetchDocumentSemanticTokens.schedule(0); }));
                    }
                }
            }
            catch (e_6_1) { e_6 = { error: e_6_1 }; }
            finally {
                try {
                    if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
                }
                finally { if (e_6) throw e_6.error; }
            }
        };
        bindDocumentChangeListeners();
        _this._register(DocumentSemanticTokensProviderRegistry.onDidChange(function () {
            bindDocumentChangeListeners();
            _this._fetchDocumentSemanticTokens.schedule();
        }));
        _this._register(themeService.onDidColorThemeChange(function (_) {
            // clear out existing tokens
            _this._setDocumentSemanticTokens(null, null, null, []);
            _this._fetchDocumentSemanticTokens.schedule();
        }));
        _this._fetchDocumentSemanticTokens.schedule(0);
        return _this;
    }
    ModelSemanticColoring.prototype.dispose = function () {
        if (this._currentDocumentResponse) {
            this._currentDocumentResponse.dispose();
            this._currentDocumentResponse = null;
        }
        if (this._currentDocumentRequestCancellationTokenSource) {
            this._currentDocumentRequestCancellationTokenSource.cancel();
            this._currentDocumentRequestCancellationTokenSource = null;
        }
        this._setDocumentSemanticTokens(null, null, null, []);
        this._isDisposed = true;
        _super.prototype.dispose.call(this);
    };
    ModelSemanticColoring.prototype._fetchDocumentSemanticTokensNow = function () {
        var _this = this;
        if (this._currentDocumentRequestCancellationTokenSource) {
            // there is already a request running, let it finish...
            return;
        }
        var cancellationTokenSource = new CancellationTokenSource();
        var lastResultId = this._currentDocumentResponse ? this._currentDocumentResponse.resultId || null : null;
        var r = getDocumentSemanticTokens(this._model, lastResultId, cancellationTokenSource.token);
        if (!r) {
            // there is no provider
            if (this._currentDocumentResponse) {
                // there are semantic tokens set
                this._model.setSemanticTokens(null, false);
            }
            return;
        }
        var provider = r.provider, request = r.request;
        this._currentDocumentRequestCancellationTokenSource = cancellationTokenSource;
        var pendingChanges = [];
        var contentChangeListener = this._model.onDidChangeContent(function (e) {
            pendingChanges.push(e);
        });
        var styling = this._semanticStyling.get(provider);
        request.then(function (res) {
            _this._currentDocumentRequestCancellationTokenSource = null;
            contentChangeListener.dispose();
            _this._setDocumentSemanticTokens(provider, res || null, styling, pendingChanges);
        }, function (err) {
            var isExpectedError = err && (errors.isPromiseCanceledError(err) || (typeof err.message === 'string' && err.message.indexOf('busy') !== -1));
            if (!isExpectedError) {
                errors.onUnexpectedError(err);
            }
            // Semantic tokens eats up all errors and considers errors to mean that the result is temporarily not available
            // The API does not have a special error kind to express this...
            _this._currentDocumentRequestCancellationTokenSource = null;
            contentChangeListener.dispose();
            if (pendingChanges.length > 0) {
                // More changes occurred while the request was running
                if (!_this._fetchDocumentSemanticTokens.isScheduled()) {
                    _this._fetchDocumentSemanticTokens.schedule();
                }
            }
        });
    };
    ModelSemanticColoring._copy = function (src, srcOffset, dest, destOffset, length) {
        for (var i = 0; i < length; i++) {
            dest[destOffset + i] = src[srcOffset + i];
        }
    };
    ModelSemanticColoring.prototype._setDocumentSemanticTokens = function (provider, tokens, styling, pendingChanges) {
        var e_7, _a, e_8, _b, e_9, _c, e_10, _d;
        var _this = this;
        var currentResponse = this._currentDocumentResponse;
        var rescheduleIfNeeded = function () {
            if (pendingChanges.length > 0 && !_this._fetchDocumentSemanticTokens.isScheduled()) {
                _this._fetchDocumentSemanticTokens.schedule();
            }
        };
        if (this._currentDocumentResponse) {
            this._currentDocumentResponse.dispose();
            this._currentDocumentResponse = null;
        }
        if (this._isDisposed) {
            // disposed!
            if (provider && tokens) {
                provider.releaseDocumentSemanticTokens(tokens.resultId);
            }
            return;
        }
        if (!provider || !styling) {
            this._model.setSemanticTokens(null, false);
            return;
        }
        if (!tokens) {
            this._model.setSemanticTokens(null, true);
            rescheduleIfNeeded();
            return;
        }
        if (isSemanticTokensEdits(tokens)) {
            if (!currentResponse) {
                // not possible!
                this._model.setSemanticTokens(null, true);
                return;
            }
            if (tokens.edits.length === 0) {
                // nothing to do!
                tokens = {
                    resultId: tokens.resultId,
                    data: currentResponse.data
                };
            }
            else {
                var deltaLength = 0;
                try {
                    for (var _e = __values(tokens.edits), _f = _e.next(); !_f.done; _f = _e.next()) {
                        var edit = _f.value;
                        deltaLength += (edit.data ? edit.data.length : 0) - edit.deleteCount;
                    }
                }
                catch (e_7_1) { e_7 = { error: e_7_1 }; }
                finally {
                    try {
                        if (_f && !_f.done && (_a = _e.return)) _a.call(_e);
                    }
                    finally { if (e_7) throw e_7.error; }
                }
                var srcData = currentResponse.data;
                var destData = new Uint32Array(srcData.length + deltaLength);
                var srcLastStart = srcData.length;
                var destLastStart = destData.length;
                for (var i = tokens.edits.length - 1; i >= 0; i--) {
                    var edit = tokens.edits[i];
                    var copyCount = srcLastStart - (edit.start + edit.deleteCount);
                    if (copyCount > 0) {
                        ModelSemanticColoring._copy(srcData, srcLastStart - copyCount, destData, destLastStart - copyCount, copyCount);
                        destLastStart -= copyCount;
                    }
                    if (edit.data) {
                        ModelSemanticColoring._copy(edit.data, 0, destData, destLastStart - edit.data.length, edit.data.length);
                        destLastStart -= edit.data.length;
                    }
                    srcLastStart = edit.start;
                }
                if (srcLastStart > 0) {
                    ModelSemanticColoring._copy(srcData, 0, destData, 0, srcLastStart);
                }
                tokens = {
                    resultId: tokens.resultId,
                    data: destData
                };
            }
        }
        if (isSemanticTokens(tokens)) {
            this._currentDocumentResponse = new SemanticTokensResponse(provider, tokens.resultId, tokens.data);
            var result = toMultilineTokens2(tokens, styling, this._model.getLanguageIdentifier());
            // Adjust incoming semantic tokens
            if (pendingChanges.length > 0) {
                try {
                    // More changes occurred while the request was running
                    // We need to:
                    // 1. Adjust incoming semantic tokens
                    // 2. Request them again
                    for (var pendingChanges_1 = __values(pendingChanges), pendingChanges_1_1 = pendingChanges_1.next(); !pendingChanges_1_1.done; pendingChanges_1_1 = pendingChanges_1.next()) {
                        var change = pendingChanges_1_1.value;
                        try {
                            for (var result_1 = (e_9 = void 0, __values(result)), result_1_1 = result_1.next(); !result_1_1.done; result_1_1 = result_1.next()) {
                                var area = result_1_1.value;
                                try {
                                    for (var _g = (e_10 = void 0, __values(change.changes)), _h = _g.next(); !_h.done; _h = _g.next()) {
                                        var singleChange = _h.value;
                                        area.applyEdit(singleChange.range, singleChange.text);
                                    }
                                }
                                catch (e_10_1) { e_10 = { error: e_10_1 }; }
                                finally {
                                    try {
                                        if (_h && !_h.done && (_d = _g.return)) _d.call(_g);
                                    }
                                    finally { if (e_10) throw e_10.error; }
                                }
                            }
                        }
                        catch (e_9_1) { e_9 = { error: e_9_1 }; }
                        finally {
                            try {
                                if (result_1_1 && !result_1_1.done && (_c = result_1.return)) _c.call(result_1);
                            }
                            finally { if (e_9) throw e_9.error; }
                        }
                    }
                }
                catch (e_8_1) { e_8 = { error: e_8_1 }; }
                finally {
                    try {
                        if (pendingChanges_1_1 && !pendingChanges_1_1.done && (_b = pendingChanges_1.return)) _b.call(pendingChanges_1);
                    }
                    finally { if (e_8) throw e_8.error; }
                }
            }
            this._model.setSemanticTokens(result, true);
        }
        else {
            this._model.setSemanticTokens(null, true);
        }
        rescheduleIfNeeded();
    };
    ModelSemanticColoring.FETCH_DOCUMENT_SEMANTIC_TOKENS_DELAY = 300;
    return ModelSemanticColoring;
}(Disposable));
export { ModelSemanticColoring };
