/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __extends = (this && this.__extends) || (function () {
    var extendStatics = function (d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (Object.prototype.hasOwnProperty.call(b, p)) d[p] = b[p]; };
        return extendStatics(d, b);
    };
    return function (d, b) {
        if (typeof b !== "function" && b !== null)
            throw new TypeError("Class extends value " + String(b) + " is not a constructor or null");
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
var __assign = (this && this.__assign) || function () {
    __assign = Object.assign || function(t) {
        for (var s, i = 1, n = arguments.length; i < n; i++) {
            s = arguments[i];
            for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                t[p] = s[p];
        }
        return t;
    };
    return __assign.apply(this, arguments);
};
var __read = (this && this.__read) || function (o, n) {
    var m = typeof Symbol === "function" && o[Symbol.iterator];
    if (!m) return o;
    var i = m.call(o), r, ar = [], e;
    try {
        while ((n === void 0 || n-- > 0) && !(r = i.next()).done) ar.push(r.value);
    }
    catch (error) { e = { error: error }; }
    finally {
        try {
            if (r && !r.done && (m = i["return"])) m.call(i);
        }
        finally { if (e) throw e.error; }
    }
    return ar;
};
var __spreadArray = (this && this.__spreadArray) || function (to, from) {
    for (var i = 0, il = from.length, j = to.length; i < il; i++, j++)
        to[j] = from[i];
    return to;
};
import * as nls from '../../../../nls';
import * as strings from '../../../common/strings';
import { SubmenuAction, Separator, EmptySubmenuAction } from '../../../common/actions';
import { ActionBar } from '../actionbar/actionbar';
import { EventType, EventHelper, isAncestor, addDisposableListener, append, $, clearNode, createStyleSheet, isInShadowDOM, getActiveElement, Dimension } from '../../dom';
import { StandardKeyboardEvent } from '../../keyboardEvent';
import { RunOnceScheduler } from '../../../common/async';
import { DisposableStore } from '../../../common/lifecycle';
import { DomScrollableElement } from '../scrollbar/scrollableElement';
import { layout } from '../contextview/contextview';
import { isLinux, isMacintosh } from '../../../common/platform';
import { Codicon, registerCodicon } from '../../../common/codicons';
import { BaseActionViewItem, ActionViewItem } from '../actionbar/actionViewItems';
import { formatRule } from '../codicons/codiconStyles';
import { isFirefox } from '../../browser';
import { StandardMouseEvent } from '../../mouseEvent';
import { stripIcons } from '../../../common/iconLabels';
export var MENU_MNEMONIC_REGEX = /\(&([^\s&])\)|(^|[^&])&([^\s&])/;
export var MENU_ESCAPED_MNEMONIC_REGEX = /(&amp;)?(&amp;)([^\s&])/g;
var menuSelectionIcon = registerCodicon('menu-selection', Codicon.check);
var menuSubmenuIcon = registerCodicon('menu-submenu', Codicon.chevronRight);
export var Direction;
(function (Direction) {
    Direction[Direction["Right"] = 0] = "Right";
    Direction[Direction["Left"] = 1] = "Left";
})(Direction || (Direction = {}));
var Menu = /** @class */ (function (_super) {
    __extends(Menu, _super);
    function Menu(container, actions, options) {
        if (options === void 0) { options = {}; }
        var _this = this;
        container.classList.add('monaco-menu-container');
        container.setAttribute('role', 'presentation');
        var menuElement = document.createElement('div');
        menuElement.classList.add('monaco-menu');
        menuElement.setAttribute('role', 'presentation');
        _this = _super.call(this, menuElement, {
            orientation: 2 /* VERTICAL */,
            actionViewItemProvider: function (action) { return _this.doGetActionViewItem(action, options, parentData); },
            context: options.context,
            actionRunner: options.actionRunner,
            ariaLabel: options.ariaLabel,
            focusOnlyEnabledItems: true,
            triggerKeys: { keys: __spreadArray([3 /* Enter */], __read((isMacintosh || isLinux ? [10 /* Space */] : []))), keyDown: true }
        }) || this;
        _this.menuElement = menuElement;
        _this.actionsList.setAttribute('role', 'menu');
        _this.actionsList.tabIndex = 0;
        _this.menuDisposables = _this._register(new DisposableStore());
        _this.initializeStyleSheet(container);
        addDisposableListener(menuElement, EventType.KEY_DOWN, function (e) {
            var event = new StandardKeyboardEvent(e);
            // Stop tab navigation of menus
            if (event.equals(2 /* Tab */)) {
                e.preventDefault();
            }
        });
        if (options.enableMnemonics) {
            _this.menuDisposables.add(addDisposableListener(menuElement, EventType.KEY_DOWN, function (e) {
                var key = e.key.toLocaleLowerCase();
                if (_this.mnemonics.has(key)) {
                    EventHelper.stop(e, true);
                    var actions_1 = _this.mnemonics.get(key);
                    if (actions_1.length === 1) {
                        if (actions_1[0] instanceof SubmenuMenuActionViewItem && actions_1[0].container) {
                            _this.focusItemByElement(actions_1[0].container);
                        }
                        actions_1[0].onClick(e);
                    }
                    if (actions_1.length > 1) {
                        var action = actions_1.shift();
                        if (action && action.container) {
                            _this.focusItemByElement(action.container);
                            actions_1.push(action);
                        }
                        _this.mnemonics.set(key, actions_1);
                    }
                }
            }));
        }
        if (isLinux) {
            _this._register(addDisposableListener(menuElement, EventType.KEY_DOWN, function (e) {
                var event = new StandardKeyboardEvent(e);
                if (event.equals(14 /* Home */) || event.equals(11 /* PageUp */)) {
                    _this.focusedItem = _this.viewItems.length - 1;
                    _this.focusNext();
                    EventHelper.stop(e, true);
                }
                else if (event.equals(13 /* End */) || event.equals(12 /* PageDown */)) {
                    _this.focusedItem = 0;
                    _this.focusPrevious();
                    EventHelper.stop(e, true);
                }
            }));
        }
        _this._register(addDisposableListener(_this.domNode, EventType.MOUSE_OUT, function (e) {
            var relatedTarget = e.relatedTarget;
            if (!isAncestor(relatedTarget, _this.domNode)) {
                _this.focusedItem = undefined;
                _this.updateFocus();
                e.stopPropagation();
            }
        }));
        _this._register(addDisposableListener(_this.actionsList, EventType.MOUSE_OVER, function (e) {
            var target = e.target;
            if (!target || !isAncestor(target, _this.actionsList) || target === _this.actionsList) {
                return;
            }
            while (target.parentElement !== _this.actionsList && target.parentElement !== null) {
                target = target.parentElement;
            }
            if (target.classList.contains('action-item')) {
                var lastFocusedItem = _this.focusedItem;
                _this.setFocusedItem(target);
                if (lastFocusedItem !== _this.focusedItem) {
                    _this.updateFocus();
                }
            }
        }));
        var parentData = {
            parent: _this
        };
        _this.mnemonics = new Map();
        // Scroll Logic
        _this.scrollableElement = _this._register(new DomScrollableElement(menuElement, {
            alwaysConsumeMouseWheel: true,
            horizontal: 2 /* Hidden */,
            vertical: 3 /* Visible */,
            verticalScrollbarSize: 7,
            handleMouseWheel: true,
            useShadows: true
        }));
        var scrollElement = _this.scrollableElement.getDomNode();
        scrollElement.style.position = '';
        _this._register(addDisposableListener(scrollElement, EventType.MOUSE_UP, function (e) {
            // Absorb clicks in menu dead space https://github.com/microsoft/vscode/issues/63575
            // We do this on the scroll element so the scroll bar doesn't dismiss the menu either
            e.preventDefault();
        }));
        menuElement.style.maxHeight = Math.max(10, window.innerHeight - container.getBoundingClientRect().top - 35) + "px";
        actions = actions.filter(function (a) {
            var _a;
            if ((_a = options.submenuIds) === null || _a === void 0 ? void 0 : _a.has(a.id)) {
                console.warn("Found submenu cycle: " + a.id);
                return false;
            }
            return true;
        });
        _this.push(actions, { icon: true, label: true, isMenu: true });
        container.appendChild(_this.scrollableElement.getDomNode());
        _this.scrollableElement.scanDomNode();
        _this.viewItems.filter(function (item) { return !(item instanceof MenuSeparatorActionViewItem); }).forEach(function (item, index, array) {
            item.updatePositionInSet(index + 1, array.length);
        });
        return _this;
    }
    Menu.prototype.initializeStyleSheet = function (container) {
        if (isInShadowDOM(container)) {
            this.styleSheet = createStyleSheet(container);
            this.styleSheet.textContent = MENU_WIDGET_CSS;
        }
        else {
            if (!Menu.globalStyleSheet) {
                Menu.globalStyleSheet = createStyleSheet();
                Menu.globalStyleSheet.textContent = MENU_WIDGET_CSS;
            }
            this.styleSheet = Menu.globalStyleSheet;
        }
    };
    Menu.prototype.style = function (style) {
        var container = this.getContainer();
        var fgColor = style.foregroundColor ? "" + style.foregroundColor : '';
        var bgColor = style.backgroundColor ? "" + style.backgroundColor : '';
        var border = style.borderColor ? "1px solid " + style.borderColor : '';
        var shadow = style.shadowColor ? "0 2px 4px " + style.shadowColor : '';
        container.style.border = border;
        this.domNode.style.color = fgColor;
        this.domNode.style.backgroundColor = bgColor;
        container.style.boxShadow = shadow;
        if (this.viewItems) {
            this.viewItems.forEach(function (item) {
                if (item instanceof BaseMenuActionViewItem || item instanceof MenuSeparatorActionViewItem) {
                    item.style(style);
                }
            });
        }
    };
    Menu.prototype.getContainer = function () {
        return this.scrollableElement.getDomNode();
    };
    Object.defineProperty(Menu.prototype, "onScroll", {
        get: function () {
            return this.scrollableElement.onScroll;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(Menu.prototype, "scrollOffset", {
        get: function () {
            return this.menuElement.scrollTop;
        },
        enumerable: false,
        configurable: true
    });
    Menu.prototype.trigger = function (index) {
        if (index <= this.viewItems.length && index >= 0) {
            var item = this.viewItems[index];
            if (item instanceof SubmenuMenuActionViewItem) {
                _super.prototype.focus.call(this, index);
                item.open(true);
            }
            else if (item instanceof BaseMenuActionViewItem) {
                _super.prototype.run.call(this, item._action, item._context);
            }
            else {
                return;
            }
        }
    };
    Menu.prototype.focusItemByElement = function (element) {
        var lastFocusedItem = this.focusedItem;
        this.setFocusedItem(element);
        if (lastFocusedItem !== this.focusedItem) {
            this.updateFocus();
        }
    };
    Menu.prototype.setFocusedItem = function (element) {
        for (var i = 0; i < this.actionsList.children.length; i++) {
            var elem = this.actionsList.children[i];
            if (element === elem) {
                this.focusedItem = i;
                break;
            }
        }
    };
    Menu.prototype.updateFocus = function (fromRight) {
        _super.prototype.updateFocus.call(this, fromRight, true);
        if (typeof this.focusedItem !== 'undefined') {
            // Workaround for #80047 caused by an issue in chromium
            // https://bugs.chromium.org/p/chromium/issues/detail?id=414283
            // When that's fixed, just call this.scrollableElement.scanDomNode()
            this.scrollableElement.setScrollPosition({
                scrollTop: Math.round(this.menuElement.scrollTop)
            });
        }
    };
    Menu.prototype.doGetActionViewItem = function (action, options, parentData) {
        if (action instanceof Separator) {
            return new MenuSeparatorActionViewItem(options.context, action, { icon: true });
        }
        else if (action instanceof SubmenuAction) {
            var menuActionViewItem = new SubmenuMenuActionViewItem(action, action.actions, parentData, __assign(__assign({}, options), { submenuIds: new Set(__spreadArray(__spreadArray([], __read((options.submenuIds || []))), [action.id])) }));
            if (options.enableMnemonics) {
                var mnemonic = menuActionViewItem.getMnemonic();
                if (mnemonic && menuActionViewItem.isEnabled()) {
                    var actionViewItems = [];
                    if (this.mnemonics.has(mnemonic)) {
                        actionViewItems = this.mnemonics.get(mnemonic);
                    }
                    actionViewItems.push(menuActionViewItem);
                    this.mnemonics.set(mnemonic, actionViewItems);
                }
            }
            return menuActionViewItem;
        }
        else {
            var menuItemOptions = { enableMnemonics: options.enableMnemonics, useEventAsContext: options.useEventAsContext };
            if (options.getKeyBinding) {
                var keybinding = options.getKeyBinding(action);
                if (keybinding) {
                    var keybindingLabel = keybinding.getLabel();
                    if (keybindingLabel) {
                        menuItemOptions.keybinding = keybindingLabel;
                    }
                }
            }
            var menuActionViewItem = new BaseMenuActionViewItem(options.context, action, menuItemOptions);
            if (options.enableMnemonics) {
                var mnemonic = menuActionViewItem.getMnemonic();
                if (mnemonic && menuActionViewItem.isEnabled()) {
                    var actionViewItems = [];
                    if (this.mnemonics.has(mnemonic)) {
                        actionViewItems = this.mnemonics.get(mnemonic);
                    }
                    actionViewItems.push(menuActionViewItem);
                    this.mnemonics.set(mnemonic, actionViewItems);
                }
            }
            return menuActionViewItem;
        }
    };
    return Menu;
}(ActionBar));
export { Menu };
var BaseMenuActionViewItem = /** @class */ (function (_super) {
    __extends(BaseMenuActionViewItem, _super);
    function BaseMenuActionViewItem(ctx, action, options) {
        if (options === void 0) { options = {}; }
        var _this = this;
        options.isMenu = true;
        _this = _super.call(this, action, action, options) || this;
        _this.options = options;
        _this.options.icon = options.icon !== undefined ? options.icon : false;
        _this.options.label = options.label !== undefined ? options.label : true;
        _this.cssClass = '';
        // Set mnemonic
        if (_this.options.label && options.enableMnemonics) {
            var label = _this.getAction().label;
            if (label) {
                var matches = MENU_MNEMONIC_REGEX.exec(label);
                if (matches) {
                    _this.mnemonic = (!!matches[1] ? matches[1] : matches[3]).toLocaleLowerCase();
                }
            }
        }
        // Add mouse up listener later to avoid accidental clicks
        _this.runOnceToEnableMouseUp = new RunOnceScheduler(function () {
            if (!_this.element) {
                return;
            }
            _this._register(addDisposableListener(_this.element, EventType.MOUSE_UP, function (e) {
                // removed default prevention as it conflicts
                // with BaseActionViewItem #101537
                // add back if issues arise and link new issue
                EventHelper.stop(e, true);
                // See https://developer.mozilla.org/en-US/Add-ons/WebExtensions/Interact_with_the_clipboard
                // > Writing to the clipboard
                // > You can use the "cut" and "copy" commands without any special
                // permission if you are using them in a short-lived event handler
                // for a user action (for example, a click handler).
                // => to get the Copy and Paste context menu actions working on Firefox,
                // there should be no timeout here
                if (isFirefox) {
                    var mouseEvent = new StandardMouseEvent(e);
                    // Allowing right click to trigger the event causes the issue described below,
                    // but since the solution below does not work in FF, we must disable right click
                    if (mouseEvent.rightButton) {
                        return;
                    }
                    _this.onClick(e);
                }
                // In all other cases, set timout to allow context menu cancellation to trigger
                // otherwise the action will destroy the menu and a second context menu
                // will still trigger for right click.
                else {
                    setTimeout(function () {
                        _this.onClick(e);
                    }, 0);
                }
            }));
            _this._register(addDisposableListener(_this.element, EventType.CONTEXT_MENU, function (e) {
                EventHelper.stop(e, true);
            }));
        }, 100);
        _this._register(_this.runOnceToEnableMouseUp);
        return _this;
    }
    BaseMenuActionViewItem.prototype.render = function (container) {
        _super.prototype.render.call(this, container);
        if (!this.element) {
            return;
        }
        this.container = container;
        this.item = append(this.element, $('a.action-menu-item'));
        if (this._action.id === Separator.ID) {
            // A separator is a presentation item
            this.item.setAttribute('role', 'presentation');
        }
        else {
            this.item.setAttribute('role', 'menuitem');
            if (this.mnemonic) {
                this.item.setAttribute('aria-keyshortcuts', "" + this.mnemonic);
            }
        }
        this.check = append(this.item, $('span.menu-item-check' + menuSelectionIcon.cssSelector));
        this.check.setAttribute('role', 'none');
        this.label = append(this.item, $('span.action-label'));
        if (this.options.label && this.options.keybinding) {
            append(this.item, $('span.keybinding')).textContent = this.options.keybinding;
        }
        // Adds mouse up listener to actually run the action
        this.runOnceToEnableMouseUp.schedule();
        this.updateClass();
        this.updateLabel();
        this.updateTooltip();
        this.updateEnabled();
        this.updateChecked();
    };
    BaseMenuActionViewItem.prototype.blur = function () {
        _super.prototype.blur.call(this);
        this.applyStyle();
    };
    BaseMenuActionViewItem.prototype.focus = function () {
        _super.prototype.focus.call(this);
        if (this.item) {
            this.item.focus();
        }
        this.applyStyle();
    };
    BaseMenuActionViewItem.prototype.updatePositionInSet = function (pos, setSize) {
        if (this.item) {
            this.item.setAttribute('aria-posinset', "" + pos);
            this.item.setAttribute('aria-setsize', "" + setSize);
        }
    };
    BaseMenuActionViewItem.prototype.updateLabel = function () {
        if (!this.label) {
            return;
        }
        if (this.options.label) {
            clearNode(this.label);
            var label = stripIcons(this.getAction().label);
            if (label) {
                var cleanLabel = cleanMnemonic(label);
                if (!this.options.enableMnemonics) {
                    label = cleanLabel;
                }
                this.label.setAttribute('aria-label', cleanLabel.replace(/&&/g, '&'));
                var matches = MENU_MNEMONIC_REGEX.exec(label);
                if (matches) {
                    label = strings.escape(label);
                    // This is global, reset it
                    MENU_ESCAPED_MNEMONIC_REGEX.lastIndex = 0;
                    var escMatch = MENU_ESCAPED_MNEMONIC_REGEX.exec(label);
                    // We can't use negative lookbehind so if we match our negative and skip
                    while (escMatch && escMatch[1]) {
                        escMatch = MENU_ESCAPED_MNEMONIC_REGEX.exec(label);
                    }
                    var replaceDoubleEscapes = function (str) { return str.replace(/&amp;&amp;/g, '&amp;'); };
                    if (escMatch) {
                        this.label.append(strings.ltrim(replaceDoubleEscapes(label.substr(0, escMatch.index)), ' '), $('u', { 'aria-hidden': 'true' }, escMatch[3]), strings.rtrim(replaceDoubleEscapes(label.substr(escMatch.index + escMatch[0].length)), ' '));
                    }
                    else {
                        this.label.innerText = replaceDoubleEscapes(label).trim();
                    }
                    if (this.item) {
                        this.item.setAttribute('aria-keyshortcuts', (!!matches[1] ? matches[1] : matches[3]).toLocaleLowerCase());
                    }
                }
                else {
                    this.label.innerText = label.replace(/&&/g, '&').trim();
                }
            }
        }
    };
    BaseMenuActionViewItem.prototype.updateTooltip = function () {
        var title = null;
        if (this.getAction().tooltip) {
            title = this.getAction().tooltip;
        }
        else if (!this.options.label && this.getAction().label && this.options.icon) {
            title = this.getAction().label;
            if (this.options.keybinding) {
                title = nls.localize({ key: 'titleLabel', comment: ['action title', 'action keybinding'] }, "{0} ({1})", title, this.options.keybinding);
            }
        }
        if (title && this.item) {
            this.item.title = title;
        }
    };
    BaseMenuActionViewItem.prototype.updateClass = function () {
        var _a, _b;
        if (this.cssClass && this.item) {
            (_a = this.item.classList).remove.apply(_a, __spreadArray([], __read(this.cssClass.split(' '))));
        }
        if (this.options.icon && this.label) {
            this.cssClass = this.getAction().class || '';
            this.label.classList.add('icon');
            if (this.cssClass) {
                (_b = this.label.classList).add.apply(_b, __spreadArray([], __read(this.cssClass.split(' '))));
            }
            this.updateEnabled();
        }
        else if (this.label) {
            this.label.classList.remove('icon');
        }
    };
    BaseMenuActionViewItem.prototype.updateEnabled = function () {
        if (this.getAction().enabled) {
            if (this.element) {
                this.element.classList.remove('disabled');
                this.element.removeAttribute('aria-disabled');
            }
            if (this.item) {
                this.item.classList.remove('disabled');
                this.item.removeAttribute('aria-disabled');
                this.item.tabIndex = 0;
            }
        }
        else {
            if (this.element) {
                this.element.classList.add('disabled');
                this.element.setAttribute('aria-disabled', 'true');
            }
            if (this.item) {
                this.item.classList.add('disabled');
                this.item.setAttribute('aria-disabled', 'true');
            }
        }
    };
    BaseMenuActionViewItem.prototype.updateChecked = function () {
        if (!this.item) {
            return;
        }
        if (this.getAction().checked) {
            this.item.classList.add('checked');
            this.item.setAttribute('role', 'menuitemcheckbox');
            this.item.setAttribute('aria-checked', 'true');
        }
        else {
            this.item.classList.remove('checked');
            this.item.setAttribute('role', 'menuitem');
            this.item.setAttribute('aria-checked', 'false');
        }
    };
    BaseMenuActionViewItem.prototype.getMnemonic = function () {
        return this.mnemonic;
    };
    BaseMenuActionViewItem.prototype.applyStyle = function () {
        if (!this.menuStyle) {
            return;
        }
        var isSelected = this.element && this.element.classList.contains('focused');
        var fgColor = isSelected && this.menuStyle.selectionForegroundColor ? this.menuStyle.selectionForegroundColor : this.menuStyle.foregroundColor;
        var bgColor = isSelected && this.menuStyle.selectionBackgroundColor ? this.menuStyle.selectionBackgroundColor : undefined;
        var border = isSelected && this.menuStyle.selectionBorderColor ? "thin solid " + this.menuStyle.selectionBorderColor : '';
        if (this.item) {
            this.item.style.color = fgColor ? fgColor.toString() : '';
            this.item.style.backgroundColor = bgColor ? bgColor.toString() : '';
        }
        if (this.check) {
            this.check.style.color = fgColor ? fgColor.toString() : '';
        }
        if (this.container) {
            this.container.style.border = border;
        }
    };
    BaseMenuActionViewItem.prototype.style = function (style) {
        this.menuStyle = style;
        this.applyStyle();
    };
    return BaseMenuActionViewItem;
}(BaseActionViewItem));
var SubmenuMenuActionViewItem = /** @class */ (function (_super) {
    __extends(SubmenuMenuActionViewItem, _super);
    function SubmenuMenuActionViewItem(action, submenuActions, parentData, submenuOptions) {
        var _this = _super.call(this, action, action, submenuOptions) || this;
        _this.submenuActions = submenuActions;
        _this.parentData = parentData;
        _this.submenuOptions = submenuOptions;
        _this.mysubmenu = null;
        _this.submenuDisposables = _this._register(new DisposableStore());
        _this.mouseOver = false;
        _this.expandDirection = submenuOptions && submenuOptions.expandDirection !== undefined ? submenuOptions.expandDirection : Direction.Right;
        _this.showScheduler = new RunOnceScheduler(function () {
            if (_this.mouseOver) {
                _this.cleanupExistingSubmenu(false);
                _this.createSubmenu(false);
            }
        }, 250);
        _this.hideScheduler = new RunOnceScheduler(function () {
            if (_this.element && (!isAncestor(getActiveElement(), _this.element) && _this.parentData.submenu === _this.mysubmenu)) {
                _this.parentData.parent.focus(false);
                _this.cleanupExistingSubmenu(true);
            }
        }, 750);
        return _this;
    }
    SubmenuMenuActionViewItem.prototype.render = function (container) {
        var _this = this;
        _super.prototype.render.call(this, container);
        if (!this.element) {
            return;
        }
        if (this.item) {
            this.item.classList.add('monaco-submenu-item');
            this.item.tabIndex = 0;
            this.item.setAttribute('aria-haspopup', 'true');
            this.updateAriaExpanded('false');
            this.submenuIndicator = append(this.item, $('span.submenu-indicator' + menuSubmenuIcon.cssSelector));
            this.submenuIndicator.setAttribute('aria-hidden', 'true');
        }
        this._register(addDisposableListener(this.element, EventType.KEY_UP, function (e) {
            var event = new StandardKeyboardEvent(e);
            if (event.equals(17 /* RightArrow */) || event.equals(3 /* Enter */)) {
                EventHelper.stop(e, true);
                _this.createSubmenu(true);
            }
        }));
        this._register(addDisposableListener(this.element, EventType.KEY_DOWN, function (e) {
            var event = new StandardKeyboardEvent(e);
            if (getActiveElement() === _this.item) {
                if (event.equals(17 /* RightArrow */) || event.equals(3 /* Enter */)) {
                    EventHelper.stop(e, true);
                }
            }
        }));
        this._register(addDisposableListener(this.element, EventType.MOUSE_OVER, function (e) {
            if (!_this.mouseOver) {
                _this.mouseOver = true;
                _this.showScheduler.schedule();
            }
        }));
        this._register(addDisposableListener(this.element, EventType.MOUSE_LEAVE, function (e) {
            _this.mouseOver = false;
        }));
        this._register(addDisposableListener(this.element, EventType.FOCUS_OUT, function (e) {
            if (_this.element && !isAncestor(getActiveElement(), _this.element)) {
                _this.hideScheduler.schedule();
            }
        }));
        this._register(this.parentData.parent.onScroll(function () {
            _this.parentData.parent.focus(false);
            _this.cleanupExistingSubmenu(false);
        }));
    };
    SubmenuMenuActionViewItem.prototype.updateEnabled = function () {
        // override on submenu entry
        // native menus do not observe enablement on sumbenus
        // we mimic that behavior
    };
    SubmenuMenuActionViewItem.prototype.open = function (selectFirst) {
        this.cleanupExistingSubmenu(false);
        this.createSubmenu(selectFirst);
    };
    SubmenuMenuActionViewItem.prototype.onClick = function (e) {
        // stop clicking from trying to run an action
        EventHelper.stop(e, true);
        this.cleanupExistingSubmenu(false);
        this.createSubmenu(true);
    };
    SubmenuMenuActionViewItem.prototype.cleanupExistingSubmenu = function (force) {
        if (this.parentData.submenu && (force || (this.parentData.submenu !== this.mysubmenu))) {
            // disposal may throw if the submenu has already been removed
            try {
                this.parentData.submenu.dispose();
            }
            catch (_a) { }
            this.parentData.submenu = undefined;
            this.updateAriaExpanded('false');
            if (this.submenuContainer) {
                this.submenuDisposables.clear();
                this.submenuContainer = undefined;
            }
        }
    };
    SubmenuMenuActionViewItem.prototype.calculateSubmenuMenuLayout = function (windowDimensions, submenu, entry, expandDirection) {
        var ret = { top: 0, left: 0 };
        // Start with horizontal
        ret.left = layout(windowDimensions.width, submenu.width, { position: expandDirection === Direction.Right ? 0 /* Before */ : 1 /* After */, offset: entry.left, size: entry.width });
        // We don't have enough room to layout the menu fully, so we are overlapping the menu
        if (ret.left >= entry.left && ret.left < entry.left + entry.width) {
            if (entry.left + 10 + submenu.width <= windowDimensions.width) {
                ret.left = entry.left + 10;
            }
            entry.top += 10;
            entry.height = 0;
        }
        // Now that we have a horizontal position, try layout vertically
        ret.top = layout(windowDimensions.height, submenu.height, { position: 0 /* Before */, offset: entry.top, size: 0 });
        // We didn't have enough room below, but we did above, so we shift down to align the menu
        if (ret.top + submenu.height === entry.top && ret.top + entry.height + submenu.height <= windowDimensions.height) {
            ret.top += entry.height;
        }
        return ret;
    };
    SubmenuMenuActionViewItem.prototype.createSubmenu = function (selectFirstItem) {
        var _this = this;
        if (selectFirstItem === void 0) { selectFirstItem = true; }
        if (!this.element) {
            return;
        }
        if (!this.parentData.submenu) {
            this.updateAriaExpanded('true');
            this.submenuContainer = append(this.element, $('div.monaco-submenu'));
            this.submenuContainer.classList.add('menubar-menu-items-holder', 'context-view');
            // Set the top value of the menu container before construction
            // This allows the menu constructor to calculate the proper max height
            var computedStyles = getComputedStyle(this.parentData.parent.domNode);
            var paddingTop = parseFloat(computedStyles.paddingTop || '0') || 0;
            // this.submenuContainer.style.top = `${this.element.offsetTop - this.parentData.parent.scrollOffset - paddingTop}px`;
            this.submenuContainer.style.zIndex = '1';
            this.submenuContainer.style.position = 'fixed';
            this.submenuContainer.style.top = '0';
            this.submenuContainer.style.left = '0';
            this.parentData.submenu = new Menu(this.submenuContainer, this.submenuActions.length ? this.submenuActions : [new EmptySubmenuAction()], this.submenuOptions);
            if (this.menuStyle) {
                this.parentData.submenu.style(this.menuStyle);
            }
            // layout submenu
            var entryBox = this.element.getBoundingClientRect();
            var entryBoxUpdated = {
                top: entryBox.top - paddingTop,
                left: entryBox.left,
                height: entryBox.height + 2 * paddingTop,
                width: entryBox.width
            };
            var viewBox = this.submenuContainer.getBoundingClientRect();
            var _a = this.calculateSubmenuMenuLayout(new Dimension(window.innerWidth, window.innerHeight), Dimension.lift(viewBox), entryBoxUpdated, this.expandDirection), top_1 = _a.top, left = _a.left;
            this.submenuContainer.style.left = left + "px";
            this.submenuContainer.style.top = top_1 + "px";
            this.submenuDisposables.add(addDisposableListener(this.submenuContainer, EventType.KEY_UP, function (e) {
                var event = new StandardKeyboardEvent(e);
                if (event.equals(15 /* LeftArrow */)) {
                    EventHelper.stop(e, true);
                    _this.parentData.parent.focus();
                    _this.cleanupExistingSubmenu(true);
                }
            }));
            this.submenuDisposables.add(addDisposableListener(this.submenuContainer, EventType.KEY_DOWN, function (e) {
                var event = new StandardKeyboardEvent(e);
                if (event.equals(15 /* LeftArrow */)) {
                    EventHelper.stop(e, true);
                }
            }));
            this.submenuDisposables.add(this.parentData.submenu.onDidCancel(function () {
                _this.parentData.parent.focus();
                _this.cleanupExistingSubmenu(true);
            }));
            this.parentData.submenu.focus(selectFirstItem);
            this.mysubmenu = this.parentData.submenu;
        }
        else {
            this.parentData.submenu.focus(false);
        }
    };
    SubmenuMenuActionViewItem.prototype.updateAriaExpanded = function (value) {
        var _a;
        if (this.item) {
            (_a = this.item) === null || _a === void 0 ? void 0 : _a.setAttribute('aria-expanded', value);
        }
    };
    SubmenuMenuActionViewItem.prototype.applyStyle = function () {
        _super.prototype.applyStyle.call(this);
        if (!this.menuStyle) {
            return;
        }
        var isSelected = this.element && this.element.classList.contains('focused');
        var fgColor = isSelected && this.menuStyle.selectionForegroundColor ? this.menuStyle.selectionForegroundColor : this.menuStyle.foregroundColor;
        if (this.submenuIndicator) {
            this.submenuIndicator.style.color = fgColor ? "" + fgColor : '';
        }
        if (this.parentData.submenu) {
            this.parentData.submenu.style(this.menuStyle);
        }
    };
    SubmenuMenuActionViewItem.prototype.dispose = function () {
        _super.prototype.dispose.call(this);
        this.hideScheduler.dispose();
        if (this.mysubmenu) {
            this.mysubmenu.dispose();
            this.mysubmenu = null;
        }
        if (this.submenuContainer) {
            this.submenuContainer = undefined;
        }
    };
    return SubmenuMenuActionViewItem;
}(BaseMenuActionViewItem));
var MenuSeparatorActionViewItem = /** @class */ (function (_super) {
    __extends(MenuSeparatorActionViewItem, _super);
    function MenuSeparatorActionViewItem() {
        return _super !== null && _super.apply(this, arguments) || this;
    }
    MenuSeparatorActionViewItem.prototype.style = function (style) {
        if (this.label) {
            this.label.style.borderBottomColor = style.separatorColor ? "" + style.separatorColor : '';
        }
    };
    return MenuSeparatorActionViewItem;
}(ActionViewItem));
export function cleanMnemonic(label) {
    var regex = MENU_MNEMONIC_REGEX;
    var matches = regex.exec(label);
    if (!matches) {
        return label;
    }
    var mnemonicInText = !matches[1];
    return label.replace(regex, mnemonicInText ? '$2$3' : '').trim();
}
var MENU_WIDGET_CSS = "\n.monaco-menu {\n\tfont-size: 13px;\n\n}\n\n" + formatRule(menuSelectionIcon) + "\n" + formatRule(menuSubmenuIcon) + "\n\n.monaco-menu .monaco-action-bar {\n\ttext-align: right;\n\toverflow: hidden;\n\twhite-space: nowrap;\n}\n\n.monaco-menu .monaco-action-bar .actions-container {\n\tdisplay: flex;\n\tmargin: 0 auto;\n\tpadding: 0;\n\twidth: 100%;\n\tjustify-content: flex-end;\n}\n\n.monaco-menu .monaco-action-bar.vertical .actions-container {\n\tdisplay: inline-block;\n}\n\n.monaco-menu .monaco-action-bar.reverse .actions-container {\n\tflex-direction: row-reverse;\n}\n\n.monaco-menu .monaco-action-bar .action-item {\n\tcursor: pointer;\n\tdisplay: inline-block;\n\ttransition: transform 50ms ease;\n\tposition: relative;  /* DO NOT REMOVE - this is the key to preventing the ghosting icon bug in Chrome 42 */\n}\n\n.monaco-menu .monaco-action-bar .action-item.disabled {\n\tcursor: default;\n}\n\n.monaco-menu .monaco-action-bar.animated .action-item.active {\n\ttransform: scale(1.272019649, 1.272019649); /* 1.272019649 = \u221A\u03C6 */\n}\n\n.monaco-menu .monaco-action-bar .action-item .icon,\n.monaco-menu .monaco-action-bar .action-item .codicon {\n\tdisplay: inline-block;\n}\n\n.monaco-menu .monaco-action-bar .action-item .codicon {\n\tdisplay: flex;\n\talign-items: center;\n}\n\n.monaco-menu .monaco-action-bar .action-label {\n\tfont-size: 11px;\n\tmargin-right: 4px;\n}\n\n.monaco-menu .monaco-action-bar .action-item.disabled .action-label,\n.monaco-menu .monaco-action-bar .action-item.disabled .action-label:hover {\n\topacity: 0.4;\n}\n\n/* Vertical actions */\n\n.monaco-menu .monaco-action-bar.vertical {\n\ttext-align: left;\n}\n\n.monaco-menu .monaco-action-bar.vertical .action-item {\n\tdisplay: block;\n}\n\n.monaco-menu .monaco-action-bar.vertical .action-label.separator {\n\tdisplay: block;\n\tborder-bottom: 1px solid #bbb;\n\tpadding-top: 1px;\n\tmargin-left: .8em;\n\tmargin-right: .8em;\n}\n\n.monaco-menu .secondary-actions .monaco-action-bar .action-label {\n\tmargin-left: 6px;\n}\n\n/* Action Items */\n.monaco-menu .monaco-action-bar .action-item.select-container {\n\toverflow: hidden; /* somehow the dropdown overflows its container, we prevent it here to not push */\n\tflex: 1;\n\tmax-width: 170px;\n\tmin-width: 60px;\n\tdisplay: flex;\n\talign-items: center;\n\tjustify-content: center;\n\tmargin-right: 10px;\n}\n\n.monaco-menu .monaco-action-bar.vertical {\n\tmargin-left: 0;\n\toverflow: visible;\n}\n\n.monaco-menu .monaco-action-bar.vertical .actions-container {\n\tdisplay: block;\n}\n\n.monaco-menu .monaco-action-bar.vertical .action-item {\n\tpadding: 0;\n\ttransform: none;\n\tdisplay: flex;\n}\n\n.monaco-menu .monaco-action-bar.vertical .action-item.active {\n\ttransform: none;\n}\n\n.monaco-menu .monaco-action-bar.vertical .action-menu-item {\n\tflex: 1 1 auto;\n\tdisplay: flex;\n\theight: 2em;\n\talign-items: center;\n\tposition: relative;\n}\n\n.monaco-menu .monaco-action-bar.vertical .action-label {\n\tflex: 1 1 auto;\n\ttext-decoration: none;\n\tpadding: 0 1em;\n\tbackground: none;\n\tfont-size: 12px;\n\tline-height: 1;\n}\n\n.monaco-menu .monaco-action-bar.vertical .keybinding,\n.monaco-menu .monaco-action-bar.vertical .submenu-indicator {\n\tdisplay: inline-block;\n\tflex: 2 1 auto;\n\tpadding: 0 1em;\n\ttext-align: right;\n\tfont-size: 12px;\n\tline-height: 1;\n}\n\n.monaco-menu .monaco-action-bar.vertical .submenu-indicator {\n\theight: 100%;\n}\n\n.monaco-menu .monaco-action-bar.vertical .submenu-indicator.codicon {\n\tfont-size: 16px !important;\n\tdisplay: flex;\n\talign-items: center;\n}\n\n.monaco-menu .monaco-action-bar.vertical .submenu-indicator.codicon::before {\n\tmargin-left: auto;\n\tmargin-right: -20px;\n}\n\n.monaco-menu .monaco-action-bar.vertical .action-item.disabled .keybinding,\n.monaco-menu .monaco-action-bar.vertical .action-item.disabled .submenu-indicator {\n\topacity: 0.4;\n}\n\n.monaco-menu .monaco-action-bar.vertical .action-label:not(.separator) {\n\tdisplay: inline-block;\n\tbox-sizing: border-box;\n\tmargin: 0;\n}\n\n.monaco-menu .monaco-action-bar.vertical .action-item {\n\tposition: static;\n\toverflow: visible;\n}\n\n.monaco-menu .monaco-action-bar.vertical .action-item .monaco-submenu {\n\tposition: absolute;\n}\n\n.monaco-menu .monaco-action-bar.vertical .action-label.separator {\n\tpadding: 0.5em 0 0 0;\n\tmargin-bottom: 0.5em;\n\twidth: 100%;\n\theight: 0px !important;\n\tmargin-left: .8em !important;\n\tmargin-right: .8em !important;\n}\n\n.monaco-menu .monaco-action-bar.vertical .action-label.separator.text {\n\tpadding: 0.7em 1em 0.1em 1em;\n\tfont-weight: bold;\n\topacity: 1;\n}\n\n.monaco-menu .monaco-action-bar.vertical .action-label:hover {\n\tcolor: inherit;\n}\n\n.monaco-menu .monaco-action-bar.vertical .menu-item-check {\n\tposition: absolute;\n\tvisibility: hidden;\n\twidth: 1em;\n\theight: 100%;\n}\n\n.monaco-menu .monaco-action-bar.vertical .action-menu-item.checked .menu-item-check {\n\tvisibility: visible;\n\tdisplay: flex;\n\talign-items: center;\n\tjustify-content: center;\n}\n\n/* Context Menu */\n\n.context-view.monaco-menu-container {\n\toutline: 0;\n\tborder: none;\n\tanimation: fadeIn 0.083s linear;\n\t-webkit-app-region: no-drag;\n}\n\n.context-view.monaco-menu-container :focus,\n.context-view.monaco-menu-container .monaco-action-bar.vertical:focus,\n.context-view.monaco-menu-container .monaco-action-bar.vertical :focus {\n\toutline: 0;\n}\n\n.monaco-menu .monaco-action-bar.vertical .action-item {\n\tborder: thin solid transparent; /* prevents jumping behaviour on hover or focus */\n}\n\n\n/* High Contrast Theming */\n:host-context(.hc-black) .context-view.monaco-menu-container {\n\tbox-shadow: none;\n}\n\n:host-context(.hc-black) .monaco-menu .monaco-action-bar.vertical .action-item.focused {\n\tbackground: none;\n}\n\n/* Vertical Action Bar Styles */\n\n.monaco-menu .monaco-action-bar.vertical {\n\tpadding: .5em 0;\n}\n\n.monaco-menu .monaco-action-bar.vertical .action-menu-item {\n\theight: 1.8em;\n}\n\n.monaco-menu .monaco-action-bar.vertical .action-label:not(.separator),\n.monaco-menu .monaco-action-bar.vertical .keybinding {\n\tfont-size: inherit;\n\tpadding: 0 2em;\n}\n\n.monaco-menu .monaco-action-bar.vertical .menu-item-check {\n\tfont-size: inherit;\n\twidth: 2em;\n}\n\n.monaco-menu .monaco-action-bar.vertical .action-label.separator {\n\tfont-size: inherit;\n\tpadding: 0.2em 0 0 0;\n\tmargin-bottom: 0.2em;\n}\n\n:host-context(.linux) .monaco-menu .monaco-action-bar.vertical .action-label.separator {\n\tmargin-left: 0;\n\tmargin-right: 0;\n}\n\n.monaco-menu .monaco-action-bar.vertical .submenu-indicator {\n\tfont-size: 60%;\n\tpadding: 0 1.8em;\n}\n\n:host-context(.linux) .monaco-menu .monaco-action-bar.vertical .submenu-indicator {\n\theight: 100%;\n\tmask-size: 10px 10px;\n\t-webkit-mask-size: 10px 10px;\n}\n\n.monaco-menu .action-item {\n\tcursor: default;\n}\n\n/* Arrows */\n.monaco-scrollable-element > .scrollbar > .scra {\n\tcursor: pointer;\n\tfont-size: 11px !important;\n}\n\n.monaco-scrollable-element > .visible {\n\topacity: 1;\n\n\t/* Background rule added for IE9 - to allow clicks on dom node */\n\tbackground:rgba(0,0,0,0);\n\n\ttransition: opacity 100ms linear;\n}\n.monaco-scrollable-element > .invisible {\n\topacity: 0;\n\tpointer-events: none;\n}\n.monaco-scrollable-element > .invisible.fade {\n\ttransition: opacity 800ms linear;\n}\n\n/* Scrollable Content Inset Shadow */\n.monaco-scrollable-element > .shadow {\n\tposition: absolute;\n\tdisplay: none;\n}\n.monaco-scrollable-element > .shadow.top {\n\tdisplay: block;\n\ttop: 0;\n\tleft: 3px;\n\theight: 3px;\n\twidth: 100%;\n\tbox-shadow: #DDD 0 6px 6px -6px inset;\n}\n.monaco-scrollable-element > .shadow.left {\n\tdisplay: block;\n\ttop: 3px;\n\tleft: 0;\n\theight: 100%;\n\twidth: 3px;\n\tbox-shadow: #DDD 6px 0 6px -6px inset;\n}\n.monaco-scrollable-element > .shadow.top-left-corner {\n\tdisplay: block;\n\ttop: 0;\n\tleft: 0;\n\theight: 3px;\n\twidth: 3px;\n}\n.monaco-scrollable-element > .shadow.top.left {\n\tbox-shadow: #DDD 6px 6px 6px -6px inset;\n}\n\n/* ---------- Default Style ---------- */\n\n:host-context(.vs) .monaco-scrollable-element > .scrollbar > .slider {\n\tbackground: rgba(100, 100, 100, .4);\n}\n:host-context(.vs-dark) .monaco-scrollable-element > .scrollbar > .slider {\n\tbackground: rgba(121, 121, 121, .4);\n}\n:host-context(.hc-black) .monaco-scrollable-element > .scrollbar > .slider {\n\tbackground: rgba(111, 195, 223, .6);\n}\n\n.monaco-scrollable-element > .scrollbar > .slider:hover {\n\tbackground: rgba(100, 100, 100, .7);\n}\n:host-context(.hc-black) .monaco-scrollable-element > .scrollbar > .slider:hover {\n\tbackground: rgba(111, 195, 223, .8);\n}\n\n.monaco-scrollable-element > .scrollbar > .slider.active {\n\tbackground: rgba(0, 0, 0, .6);\n}\n:host-context(.vs-dark) .monaco-scrollable-element > .scrollbar > .slider.active {\n\tbackground: rgba(191, 191, 191, .4);\n}\n:host-context(.hc-black) .monaco-scrollable-element > .scrollbar > .slider.active {\n\tbackground: rgba(111, 195, 223, 1);\n}\n\n:host-context(.vs-dark) .monaco-scrollable-element .shadow.top {\n\tbox-shadow: none;\n}\n\n:host-context(.vs-dark) .monaco-scrollable-element .shadow.left {\n\tbox-shadow: #000 6px 0 6px -6px inset;\n}\n\n:host-context(.vs-dark) .monaco-scrollable-element .shadow.top.left {\n\tbox-shadow: #000 6px 6px 6px -6px inset;\n}\n\n:host-context(.hc-black) .monaco-scrollable-element .shadow.top {\n\tbox-shadow: none;\n}\n\n:host-context(.hc-black) .monaco-scrollable-element .shadow.left {\n\tbox-shadow: none;\n}\n\n:host-context(.hc-black) .monaco-scrollable-element .shadow.top.left {\n\tbox-shadow: none;\n}\n";
