/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __extends = (this && this.__extends) || (function () {
    var extendStatics = function (d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (Object.prototype.hasOwnProperty.call(b, p)) d[p] = b[p]; };
        return extendStatics(d, b);
    };
    return function (d, b) {
        if (typeof b !== "function" && b !== null)
            throw new TypeError("Class extends value " + String(b) + " is not a constructor or null");
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
var __assign = (this && this.__assign) || function () {
    __assign = Object.assign || function(t) {
        for (var s, i = 1, n = arguments.length; i < n; i++) {
            s = arguments[i];
            for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                t[p] = s[p];
        }
        return t;
    };
    return __assign.apply(this, arguments);
};
var __read = (this && this.__read) || function (o, n) {
    var m = typeof Symbol === "function" && o[Symbol.iterator];
    if (!m) return o;
    var i = m.call(o), r, ar = [], e;
    try {
        while ((n === void 0 || n-- > 0) && !(r = i.next()).done) ar.push(r.value);
    }
    catch (error) { e = { error: error }; }
    finally {
        try {
            if (r && !r.done && (m = i["return"])) m.call(i);
        }
        finally { if (e) throw e.error; }
    }
    return ar;
};
var __spreadArray = (this && this.__spreadArray) || function (to, from) {
    for (var i = 0, il = from.length, j = to.length; i < il; i++, j++)
        to[j] = from[i];
    return to;
};
var __values = (this && this.__values) || function(o) {
    var s = typeof Symbol === "function" && Symbol.iterator, m = s && o[s], i = 0;
    if (m) return m.call(o);
    if (o && typeof o.length === "number") return {
        next: function () {
            if (o && i >= o.length) o = void 0;
            return { value: o && o[i++], done: !o };
        }
    };
    throw new TypeError(s ? "Object is not iterable." : "Symbol.iterator is not defined.");
};
var _a;
import * as browser from './browser';
import { domEvent } from './event';
import { StandardKeyboardEvent } from './keyboardEvent';
import { StandardMouseEvent } from './mouseEvent';
import { TimeoutTimer } from '../common/async';
import { onUnexpectedError } from '../common/errors';
import { Emitter } from '../common/event';
import { Disposable, DisposableStore, toDisposable } from '../common/lifecycle';
import * as platform from '../common/platform';
import { URI } from '../common/uri';
import { FileAccess, RemoteAuthorities } from '../common/network';
import { BrowserFeatures } from './canIUse';
import { insane } from '../common/insane/insane';
export function clearNode(node) {
    while (node.firstChild) {
        node.firstChild.remove();
    }
}
/**
 * @deprecated Use node.isConnected directly
 */
export function isInDOM(node) {
    var _a;
    return (_a = node === null || node === void 0 ? void 0 : node.isConnected) !== null && _a !== void 0 ? _a : false;
}
var DomListener = /** @class */ (function () {
    function DomListener(node, type, handler, options) {
        this._node = node;
        this._type = type;
        this._handler = handler;
        this._options = (options || false);
        this._node.addEventListener(this._type, this._handler, this._options);
    }
    DomListener.prototype.dispose = function () {
        if (!this._handler) {
            // Already disposed
            return;
        }
        this._node.removeEventListener(this._type, this._handler, this._options);
        // Prevent leakers from holding on to the dom or handler func
        this._node = null;
        this._handler = null;
    };
    return DomListener;
}());
export function addDisposableListener(node, type, handler, useCaptureOrOptions) {
    return new DomListener(node, type, handler, useCaptureOrOptions);
}
function _wrapAsStandardMouseEvent(handler) {
    return function (e) {
        return handler(new StandardMouseEvent(e));
    };
}
function _wrapAsStandardKeyboardEvent(handler) {
    return function (e) {
        return handler(new StandardKeyboardEvent(e));
    };
}
export var addStandardDisposableListener = function addStandardDisposableListener(node, type, handler, useCapture) {
    var wrapHandler = handler;
    if (type === 'click' || type === 'mousedown') {
        wrapHandler = _wrapAsStandardMouseEvent(handler);
    }
    else if (type === 'keydown' || type === 'keypress' || type === 'keyup') {
        wrapHandler = _wrapAsStandardKeyboardEvent(handler);
    }
    return addDisposableListener(node, type, wrapHandler, useCapture);
};
export var addStandardDisposableGenericMouseDownListner = function addStandardDisposableListener(node, handler, useCapture) {
    var wrapHandler = _wrapAsStandardMouseEvent(handler);
    return addDisposableGenericMouseDownListner(node, wrapHandler, useCapture);
};
export var addStandardDisposableGenericMouseUpListner = function addStandardDisposableListener(node, handler, useCapture) {
    var wrapHandler = _wrapAsStandardMouseEvent(handler);
    return addDisposableGenericMouseUpListner(node, wrapHandler, useCapture);
};
export function addDisposableGenericMouseDownListner(node, handler, useCapture) {
    return addDisposableListener(node, platform.isIOS && BrowserFeatures.pointerEvents ? EventType.POINTER_DOWN : EventType.MOUSE_DOWN, handler, useCapture);
}
export function addDisposableGenericMouseMoveListner(node, handler, useCapture) {
    return addDisposableListener(node, platform.isIOS && BrowserFeatures.pointerEvents ? EventType.POINTER_MOVE : EventType.MOUSE_MOVE, handler, useCapture);
}
export function addDisposableGenericMouseUpListner(node, handler, useCapture) {
    return addDisposableListener(node, platform.isIOS && BrowserFeatures.pointerEvents ? EventType.POINTER_UP : EventType.MOUSE_UP, handler, useCapture);
}
export function addDisposableNonBubblingMouseOutListener(node, handler) {
    return addDisposableListener(node, 'mouseout', function (e) {
        // Mouse out bubbles, so this is an attempt to ignore faux mouse outs coming from children elements
        var toElement = (e.relatedTarget);
        while (toElement && toElement !== node) {
            toElement = toElement.parentNode;
        }
        if (toElement === node) {
            return;
        }
        handler(e);
    });
}
export function addDisposableNonBubblingPointerOutListener(node, handler) {
    return addDisposableListener(node, 'pointerout', function (e) {
        // Mouse out bubbles, so this is an attempt to ignore faux mouse outs coming from children elements
        var toElement = (e.relatedTarget);
        while (toElement && toElement !== node) {
            toElement = toElement.parentNode;
        }
        if (toElement === node) {
            return;
        }
        handler(e);
    });
}
var _animationFrame = null;
function doRequestAnimationFrame(callback) {
    if (!_animationFrame) {
        var emulatedRequestAnimationFrame = function (callback) {
            return setTimeout(function () { return callback(new Date().getTime()); }, 0);
        };
        _animationFrame = (self.requestAnimationFrame
            || self.msRequestAnimationFrame
            || self.webkitRequestAnimationFrame
            || self.mozRequestAnimationFrame
            || self.oRequestAnimationFrame
            || emulatedRequestAnimationFrame);
    }
    return _animationFrame.call(self, callback);
}
/**
 * Schedule a callback to be run at the next animation frame.
 * This allows multiple parties to register callbacks that should run at the next animation frame.
 * If currently in an animation frame, `runner` will be executed immediately.
 * @return token that can be used to cancel the scheduled runner (only if `runner` was not executed immediately).
 */
export var runAtThisOrScheduleAtNextAnimationFrame;
/**
 * Schedule a callback to be run at the next animation frame.
 * This allows multiple parties to register callbacks that should run at the next animation frame.
 * If currently in an animation frame, `runner` will be executed at the next animation frame.
 * @return token that can be used to cancel the scheduled runner.
 */
export var scheduleAtNextAnimationFrame;
var AnimationFrameQueueItem = /** @class */ (function () {
    function AnimationFrameQueueItem(runner, priority) {
        if (priority === void 0) { priority = 0; }
        this._runner = runner;
        this.priority = priority;
        this._canceled = false;
    }
    AnimationFrameQueueItem.prototype.dispose = function () {
        this._canceled = true;
    };
    AnimationFrameQueueItem.prototype.execute = function () {
        if (this._canceled) {
            return;
        }
        try {
            this._runner();
        }
        catch (e) {
            onUnexpectedError(e);
        }
    };
    // Sort by priority (largest to lowest)
    AnimationFrameQueueItem.sort = function (a, b) {
        return b.priority - a.priority;
    };
    return AnimationFrameQueueItem;
}());
(function () {
    /**
     * The runners scheduled at the next animation frame
     */
    var NEXT_QUEUE = [];
    /**
     * The runners scheduled at the current animation frame
     */
    var CURRENT_QUEUE = null;
    /**
     * A flag to keep track if the native requestAnimationFrame was already called
     */
    var animFrameRequested = false;
    /**
     * A flag to indicate if currently handling a native requestAnimationFrame callback
     */
    var inAnimationFrameRunner = false;
    var animationFrameRunner = function () {
        animFrameRequested = false;
        CURRENT_QUEUE = NEXT_QUEUE;
        NEXT_QUEUE = [];
        inAnimationFrameRunner = true;
        while (CURRENT_QUEUE.length > 0) {
            CURRENT_QUEUE.sort(AnimationFrameQueueItem.sort);
            var top_1 = CURRENT_QUEUE.shift();
            top_1.execute();
        }
        inAnimationFrameRunner = false;
    };
    scheduleAtNextAnimationFrame = function (runner, priority) {
        if (priority === void 0) { priority = 0; }
        var item = new AnimationFrameQueueItem(runner, priority);
        NEXT_QUEUE.push(item);
        if (!animFrameRequested) {
            animFrameRequested = true;
            doRequestAnimationFrame(animationFrameRunner);
        }
        return item;
    };
    runAtThisOrScheduleAtNextAnimationFrame = function (runner, priority) {
        if (inAnimationFrameRunner) {
            var item = new AnimationFrameQueueItem(runner, priority);
            CURRENT_QUEUE.push(item);
            return item;
        }
        else {
            return scheduleAtNextAnimationFrame(runner, priority);
        }
    };
})();
export function measure(callback) {
    return scheduleAtNextAnimationFrame(callback, 10000 /* must be early */);
}
export function modify(callback) {
    return scheduleAtNextAnimationFrame(callback, -10000 /* must be late */);
}
var MINIMUM_TIME_MS = 8;
var DEFAULT_EVENT_MERGER = function (lastEvent, currentEvent) {
    return currentEvent;
};
var TimeoutThrottledDomListener = /** @class */ (function (_super) {
    __extends(TimeoutThrottledDomListener, _super);
    function TimeoutThrottledDomListener(node, type, handler, eventMerger, minimumTimeMs) {
        if (eventMerger === void 0) { eventMerger = DEFAULT_EVENT_MERGER; }
        if (minimumTimeMs === void 0) { minimumTimeMs = MINIMUM_TIME_MS; }
        var _this = _super.call(this) || this;
        var lastEvent = null;
        var lastHandlerTime = 0;
        var timeout = _this._register(new TimeoutTimer());
        var invokeHandler = function () {
            lastHandlerTime = (new Date()).getTime();
            handler(lastEvent);
            lastEvent = null;
        };
        _this._register(addDisposableListener(node, type, function (e) {
            lastEvent = eventMerger(lastEvent, e);
            var elapsedTime = (new Date()).getTime() - lastHandlerTime;
            if (elapsedTime >= minimumTimeMs) {
                timeout.cancel();
                invokeHandler();
            }
            else {
                timeout.setIfNotSet(invokeHandler, minimumTimeMs - elapsedTime);
            }
        }));
        return _this;
    }
    return TimeoutThrottledDomListener;
}(Disposable));
export function addDisposableThrottledListener(node, type, handler, eventMerger, minimumTimeMs) {
    return new TimeoutThrottledDomListener(node, type, handler, eventMerger, minimumTimeMs);
}
export function getComputedStyle(el) {
    return document.defaultView.getComputedStyle(el, null);
}
export function getClientArea(element) {
    // Try with DOM clientWidth / clientHeight
    if (element !== document.body) {
        return new Dimension(element.clientWidth, element.clientHeight);
    }
    // If visual view port exits and it's on mobile, it should be used instead of window innerWidth / innerHeight, or document.body.clientWidth / document.body.clientHeight
    if (platform.isIOS && window.visualViewport) {
        var width = window.visualViewport.width;
        var height = window.visualViewport.height - (browser.isStandalone
            // in PWA mode, the visual viewport always includes the safe-area-inset-bottom (which is for the home indicator)
            // even when you are using the onscreen monitor, the visual viewport will include the area between system statusbar and the onscreen keyboard
            // plus the area between onscreen keyboard and the bottom bezel, which is 20px on iOS.
            ? (20 + 4) // + 4px for body margin
            : 0);
        return new Dimension(width, height);
    }
    // Try innerWidth / innerHeight
    if (window.innerWidth && window.innerHeight) {
        return new Dimension(window.innerWidth, window.innerHeight);
    }
    // Try with document.body.clientWidth / document.body.clientHeight
    if (document.body && document.body.clientWidth && document.body.clientHeight) {
        return new Dimension(document.body.clientWidth, document.body.clientHeight);
    }
    // Try with document.documentElement.clientWidth / document.documentElement.clientHeight
    if (document.documentElement && document.documentElement.clientWidth && document.documentElement.clientHeight) {
        return new Dimension(document.documentElement.clientWidth, document.documentElement.clientHeight);
    }
    throw new Error('Unable to figure out browser width and height');
}
var SizeUtils = /** @class */ (function () {
    function SizeUtils() {
    }
    // Adapted from WinJS
    // Converts a CSS positioning string for the specified element to pixels.
    SizeUtils.convertToPixels = function (element, value) {
        return parseFloat(value) || 0;
    };
    SizeUtils.getDimension = function (element, cssPropertyName, jsPropertyName) {
        var computedStyle = getComputedStyle(element);
        var value = '0';
        if (computedStyle) {
            if (computedStyle.getPropertyValue) {
                value = computedStyle.getPropertyValue(cssPropertyName);
            }
            else {
                // IE8
                value = computedStyle.getAttribute(jsPropertyName);
            }
        }
        return SizeUtils.convertToPixels(element, value);
    };
    SizeUtils.getBorderLeftWidth = function (element) {
        return SizeUtils.getDimension(element, 'border-left-width', 'borderLeftWidth');
    };
    SizeUtils.getBorderRightWidth = function (element) {
        return SizeUtils.getDimension(element, 'border-right-width', 'borderRightWidth');
    };
    SizeUtils.getBorderTopWidth = function (element) {
        return SizeUtils.getDimension(element, 'border-top-width', 'borderTopWidth');
    };
    SizeUtils.getBorderBottomWidth = function (element) {
        return SizeUtils.getDimension(element, 'border-bottom-width', 'borderBottomWidth');
    };
    SizeUtils.getPaddingLeft = function (element) {
        return SizeUtils.getDimension(element, 'padding-left', 'paddingLeft');
    };
    SizeUtils.getPaddingRight = function (element) {
        return SizeUtils.getDimension(element, 'padding-right', 'paddingRight');
    };
    SizeUtils.getPaddingTop = function (element) {
        return SizeUtils.getDimension(element, 'padding-top', 'paddingTop');
    };
    SizeUtils.getPaddingBottom = function (element) {
        return SizeUtils.getDimension(element, 'padding-bottom', 'paddingBottom');
    };
    SizeUtils.getMarginLeft = function (element) {
        return SizeUtils.getDimension(element, 'margin-left', 'marginLeft');
    };
    SizeUtils.getMarginTop = function (element) {
        return SizeUtils.getDimension(element, 'margin-top', 'marginTop');
    };
    SizeUtils.getMarginRight = function (element) {
        return SizeUtils.getDimension(element, 'margin-right', 'marginRight');
    };
    SizeUtils.getMarginBottom = function (element) {
        return SizeUtils.getDimension(element, 'margin-bottom', 'marginBottom');
    };
    return SizeUtils;
}());
var Dimension = /** @class */ (function () {
    function Dimension(width, height) {
        this.width = width;
        this.height = height;
    }
    Dimension.prototype.with = function (width, height) {
        if (width === void 0) { width = this.width; }
        if (height === void 0) { height = this.height; }
        if (width !== this.width || height !== this.height) {
            return new Dimension(width, height);
        }
        else {
            return this;
        }
    };
    Dimension.is = function (obj) {
        return typeof obj === 'object' && typeof obj.height === 'number' && typeof obj.width === 'number';
    };
    Dimension.lift = function (obj) {
        if (obj instanceof Dimension) {
            return obj;
        }
        else {
            return new Dimension(obj.width, obj.height);
        }
    };
    Dimension.equals = function (a, b) {
        if (a === b) {
            return true;
        }
        if (!a || !b) {
            return false;
        }
        return a.width === b.width && a.height === b.height;
    };
    Dimension.None = new Dimension(0, 0);
    return Dimension;
}());
export { Dimension };
export function getTopLeftOffset(element) {
    // Adapted from WinJS.Utilities.getPosition
    // and added borders to the mix
    var offsetParent = element.offsetParent;
    var top = element.offsetTop;
    var left = element.offsetLeft;
    while ((element = element.parentNode) !== null
        && element !== document.body
        && element !== document.documentElement) {
        top -= element.scrollTop;
        var c = isShadowRoot(element) ? null : getComputedStyle(element);
        if (c) {
            left -= c.direction !== 'rtl' ? element.scrollLeft : -element.scrollLeft;
        }
        if (element === offsetParent) {
            left += SizeUtils.getBorderLeftWidth(element);
            top += SizeUtils.getBorderTopWidth(element);
            top += element.offsetTop;
            left += element.offsetLeft;
            offsetParent = element.offsetParent;
        }
    }
    return {
        left: left,
        top: top
    };
}
export function size(element, width, height) {
    if (typeof width === 'number') {
        element.style.width = width + "px";
    }
    if (typeof height === 'number') {
        element.style.height = height + "px";
    }
}
export function position(element, top, right, bottom, left, position) {
    if (position === void 0) { position = 'absolute'; }
    if (typeof top === 'number') {
        element.style.top = top + "px";
    }
    if (typeof right === 'number') {
        element.style.right = right + "px";
    }
    if (typeof bottom === 'number') {
        element.style.bottom = bottom + "px";
    }
    if (typeof left === 'number') {
        element.style.left = left + "px";
    }
    element.style.position = position;
}
/**
 * Returns the position of a dom node relative to the entire page.
 */
export function getDomNodePagePosition(domNode) {
    var bb = domNode.getBoundingClientRect();
    return {
        left: bb.left + StandardWindow.scrollX,
        top: bb.top + StandardWindow.scrollY,
        width: bb.width,
        height: bb.height
    };
}
export var StandardWindow = new /** @class */ (function () {
    function class_1() {
    }
    Object.defineProperty(class_1.prototype, "scrollX", {
        get: function () {
            if (typeof window.scrollX === 'number') {
                // modern browsers
                return window.scrollX;
            }
            else {
                return document.body.scrollLeft + document.documentElement.scrollLeft;
            }
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(class_1.prototype, "scrollY", {
        get: function () {
            if (typeof window.scrollY === 'number') {
                // modern browsers
                return window.scrollY;
            }
            else {
                return document.body.scrollTop + document.documentElement.scrollTop;
            }
        },
        enumerable: false,
        configurable: true
    });
    return class_1;
}());
// Adapted from WinJS
// Gets the width of the element, including margins.
export function getTotalWidth(element) {
    var margin = SizeUtils.getMarginLeft(element) + SizeUtils.getMarginRight(element);
    return element.offsetWidth + margin;
}
export function getContentWidth(element) {
    var border = SizeUtils.getBorderLeftWidth(element) + SizeUtils.getBorderRightWidth(element);
    var padding = SizeUtils.getPaddingLeft(element) + SizeUtils.getPaddingRight(element);
    return element.offsetWidth - border - padding;
}
export function getTotalScrollWidth(element) {
    var margin = SizeUtils.getMarginLeft(element) + SizeUtils.getMarginRight(element);
    return element.scrollWidth + margin;
}
// Adapted from WinJS
// Gets the height of the content of the specified element. The content height does not include borders or padding.
export function getContentHeight(element) {
    var border = SizeUtils.getBorderTopWidth(element) + SizeUtils.getBorderBottomWidth(element);
    var padding = SizeUtils.getPaddingTop(element) + SizeUtils.getPaddingBottom(element);
    return element.offsetHeight - border - padding;
}
// Adapted from WinJS
// Gets the height of the element, including its margins.
export function getTotalHeight(element) {
    var margin = SizeUtils.getMarginTop(element) + SizeUtils.getMarginBottom(element);
    return element.offsetHeight + margin;
}
// Gets the left coordinate of the specified element relative to the specified parent.
function getRelativeLeft(element, parent) {
    if (element === null) {
        return 0;
    }
    var elementPosition = getTopLeftOffset(element);
    var parentPosition = getTopLeftOffset(parent);
    return elementPosition.left - parentPosition.left;
}
export function getLargestChildWidth(parent, children) {
    var childWidths = children.map(function (child) {
        return Math.max(getTotalScrollWidth(child), getTotalWidth(child)) + getRelativeLeft(child, parent) || 0;
    });
    var maxWidth = Math.max.apply(Math, __spreadArray([], __read(childWidths)));
    return maxWidth;
}
// ----------------------------------------------------------------------------------------
export function isAncestor(testChild, testAncestor) {
    while (testChild) {
        if (testChild === testAncestor) {
            return true;
        }
        testChild = testChild.parentNode;
    }
    return false;
}
var parentFlowToDataKey = 'parentFlowToElementId';
/**
 * Set an explicit parent to use for nodes that are not part of the
 * regular dom structure.
 */
export function setParentFlowTo(fromChildElement, toParentElement) {
    fromChildElement.dataset[parentFlowToDataKey] = toParentElement.id;
}
function getParentFlowToElement(node) {
    var flowToParentId = node.dataset[parentFlowToDataKey];
    if (typeof flowToParentId === 'string') {
        return document.getElementById(flowToParentId);
    }
    return null;
}
/**
 * Check if `testAncestor` is an ancessor of `testChild`, observing the explicit
 * parents set by `setParentFlowTo`.
 */
export function isAncestorUsingFlowTo(testChild, testAncestor) {
    var node = testChild;
    while (node) {
        if (node === testAncestor) {
            return true;
        }
        if (node instanceof HTMLElement) {
            var flowToParentElement = getParentFlowToElement(node);
            if (flowToParentElement) {
                node = flowToParentElement;
                continue;
            }
        }
        node = node.parentNode;
    }
    return false;
}
export function findParentWithClass(node, clazz, stopAtClazzOrNode) {
    while (node && node.nodeType === node.ELEMENT_NODE) {
        if (node.classList.contains(clazz)) {
            return node;
        }
        if (stopAtClazzOrNode) {
            if (typeof stopAtClazzOrNode === 'string') {
                if (node.classList.contains(stopAtClazzOrNode)) {
                    return null;
                }
            }
            else {
                if (node === stopAtClazzOrNode) {
                    return null;
                }
            }
        }
        node = node.parentNode;
    }
    return null;
}
export function hasParentWithClass(node, clazz, stopAtClazzOrNode) {
    return !!findParentWithClass(node, clazz, stopAtClazzOrNode);
}
export function isShadowRoot(node) {
    return (node && !!node.host && !!node.mode);
}
export function isInShadowDOM(domNode) {
    return !!getShadowRoot(domNode);
}
export function getShadowRoot(domNode) {
    while (domNode.parentNode) {
        if (domNode === document.body) {
            // reached the body
            return null;
        }
        domNode = domNode.parentNode;
    }
    return isShadowRoot(domNode) ? domNode : null;
}
export function getActiveElement() {
    var result = document.activeElement;
    while (result === null || result === void 0 ? void 0 : result.shadowRoot) {
        result = result.shadowRoot.activeElement;
    }
    return result;
}
export function createStyleSheet(container) {
    if (container === void 0) { container = document.getElementsByTagName('head')[0]; }
    var style = document.createElement('style');
    style.type = 'text/css';
    style.media = 'screen';
    container.appendChild(style);
    return style;
}
export function createMetaElement(container) {
    if (container === void 0) { container = document.getElementsByTagName('head')[0]; }
    var meta = document.createElement('meta');
    container.appendChild(meta);
    return meta;
}
var _sharedStyleSheet = null;
function getSharedStyleSheet() {
    if (!_sharedStyleSheet) {
        _sharedStyleSheet = createStyleSheet();
    }
    return _sharedStyleSheet;
}
function getDynamicStyleSheetRules(style) {
    var _a, _b;
    if ((_a = style === null || style === void 0 ? void 0 : style.sheet) === null || _a === void 0 ? void 0 : _a.rules) {
        // Chrome, IE
        return style.sheet.rules;
    }
    if ((_b = style === null || style === void 0 ? void 0 : style.sheet) === null || _b === void 0 ? void 0 : _b.cssRules) {
        // FF
        return style.sheet.cssRules;
    }
    return [];
}
export function createCSSRule(selector, cssText, style) {
    if (style === void 0) { style = getSharedStyleSheet(); }
    if (!style || !cssText) {
        return;
    }
    style.sheet.insertRule(selector + '{' + cssText + '}', 0);
}
export function removeCSSRulesContainingSelector(ruleName, style) {
    if (style === void 0) { style = getSharedStyleSheet(); }
    if (!style) {
        return;
    }
    var rules = getDynamicStyleSheetRules(style);
    var toDelete = [];
    for (var i = 0; i < rules.length; i++) {
        var rule = rules[i];
        if (rule.selectorText.indexOf(ruleName) !== -1) {
            toDelete.push(i);
        }
    }
    for (var i = toDelete.length - 1; i >= 0; i--) {
        style.sheet.deleteRule(toDelete[i]);
    }
}
export function isHTMLElement(o) {
    if (typeof HTMLElement === 'object') {
        return o instanceof HTMLElement;
    }
    return o && typeof o === 'object' && o.nodeType === 1 && typeof o.nodeName === 'string';
}
export var EventType = {
    // Mouse
    CLICK: 'click',
    AUXCLICK: 'auxclick',
    DBLCLICK: 'dblclick',
    MOUSE_UP: 'mouseup',
    MOUSE_DOWN: 'mousedown',
    MOUSE_OVER: 'mouseover',
    MOUSE_MOVE: 'mousemove',
    MOUSE_OUT: 'mouseout',
    MOUSE_ENTER: 'mouseenter',
    MOUSE_LEAVE: 'mouseleave',
    MOUSE_WHEEL: 'wheel',
    POINTER_UP: 'pointerup',
    POINTER_DOWN: 'pointerdown',
    POINTER_MOVE: 'pointermove',
    CONTEXT_MENU: 'contextmenu',
    WHEEL: 'wheel',
    // Keyboard
    KEY_DOWN: 'keydown',
    KEY_PRESS: 'keypress',
    KEY_UP: 'keyup',
    // HTML Document
    LOAD: 'load',
    BEFORE_UNLOAD: 'beforeunload',
    UNLOAD: 'unload',
    ABORT: 'abort',
    ERROR: 'error',
    RESIZE: 'resize',
    SCROLL: 'scroll',
    FULLSCREEN_CHANGE: 'fullscreenchange',
    WK_FULLSCREEN_CHANGE: 'webkitfullscreenchange',
    // Form
    SELECT: 'select',
    CHANGE: 'change',
    SUBMIT: 'submit',
    RESET: 'reset',
    FOCUS: 'focus',
    FOCUS_IN: 'focusin',
    FOCUS_OUT: 'focusout',
    BLUR: 'blur',
    INPUT: 'input',
    // Local Storage
    STORAGE: 'storage',
    // Drag
    DRAG_START: 'dragstart',
    DRAG: 'drag',
    DRAG_ENTER: 'dragenter',
    DRAG_LEAVE: 'dragleave',
    DRAG_OVER: 'dragover',
    DROP: 'drop',
    DRAG_END: 'dragend',
    // Animation
    ANIMATION_START: browser.isWebKit ? 'webkitAnimationStart' : 'animationstart',
    ANIMATION_END: browser.isWebKit ? 'webkitAnimationEnd' : 'animationend',
    ANIMATION_ITERATION: browser.isWebKit ? 'webkitAnimationIteration' : 'animationiteration'
};
export var EventHelper = {
    stop: function (e, cancelBubble) {
        if (e.preventDefault) {
            e.preventDefault();
        }
        else {
            // IE8
            e.returnValue = false;
        }
        if (cancelBubble) {
            if (e.stopPropagation) {
                e.stopPropagation();
            }
            else {
                // IE8
                e.cancelBubble = true;
            }
        }
    }
};
export function saveParentsScrollTop(node) {
    var r = [];
    for (var i = 0; node && node.nodeType === node.ELEMENT_NODE; i++) {
        r[i] = node.scrollTop;
        node = node.parentNode;
    }
    return r;
}
export function restoreParentsScrollTop(node, state) {
    for (var i = 0; node && node.nodeType === node.ELEMENT_NODE; i++) {
        if (node.scrollTop !== state[i]) {
            node.scrollTop = state[i];
        }
        node = node.parentNode;
    }
}
var FocusTracker = /** @class */ (function (_super) {
    __extends(FocusTracker, _super);
    function FocusTracker(element) {
        var _this = _super.call(this) || this;
        _this._onDidFocus = _this._register(new Emitter());
        _this.onDidFocus = _this._onDidFocus.event;
        _this._onDidBlur = _this._register(new Emitter());
        _this.onDidBlur = _this._onDidBlur.event;
        var hasFocus = isAncestor(document.activeElement, element);
        var loosingFocus = false;
        var onFocus = function () {
            loosingFocus = false;
            if (!hasFocus) {
                hasFocus = true;
                _this._onDidFocus.fire();
            }
        };
        var onBlur = function () {
            if (hasFocus) {
                loosingFocus = true;
                window.setTimeout(function () {
                    if (loosingFocus) {
                        loosingFocus = false;
                        hasFocus = false;
                        _this._onDidBlur.fire();
                    }
                }, 0);
            }
        };
        _this._refreshStateHandler = function () {
            var currentNodeHasFocus = isAncestor(document.activeElement, element);
            if (currentNodeHasFocus !== hasFocus) {
                if (hasFocus) {
                    onBlur();
                }
                else {
                    onFocus();
                }
            }
        };
        _this._register(domEvent(element, EventType.FOCUS, true)(onFocus));
        _this._register(domEvent(element, EventType.BLUR, true)(onBlur));
        return _this;
    }
    FocusTracker.prototype.refreshState = function () {
        this._refreshStateHandler();
    };
    return FocusTracker;
}(Disposable));
export function trackFocus(element) {
    return new FocusTracker(element);
}
export function after(sibling, child) {
    sibling.after(child);
    return child;
}
export function append(parent) {
    var children = [];
    for (var _i = 1; _i < arguments.length; _i++) {
        children[_i - 1] = arguments[_i];
    }
    parent.append.apply(parent, __spreadArray([], __read(children)));
    if (children.length === 1 && typeof children[0] !== 'string') {
        return children[0];
    }
}
export function prepend(parent, child) {
    parent.insertBefore(child, parent.firstChild);
    return child;
}
/**
 * Removes all children from `parent` and appends `children`
 */
export function reset(parent) {
    var children = [];
    for (var _i = 1; _i < arguments.length; _i++) {
        children[_i - 1] = arguments[_i];
    }
    parent.innerText = '';
    append.apply(void 0, __spreadArray([parent], __read(children)));
}
var SELECTOR_REGEX = /([\w\-]+)?(#([\w\-]+))?((\.([\w\-]+))*)/;
export var Namespace;
(function (Namespace) {
    Namespace["HTML"] = "http://www.w3.org/1999/xhtml";
    Namespace["SVG"] = "http://www.w3.org/2000/svg";
})(Namespace || (Namespace = {}));
function _$(namespace, description, attrs) {
    var children = [];
    for (var _i = 3; _i < arguments.length; _i++) {
        children[_i - 3] = arguments[_i];
    }
    var match = SELECTOR_REGEX.exec(description);
    if (!match) {
        throw new Error('Bad use of emmet');
    }
    attrs = __assign({}, (attrs || {}));
    var tagName = match[1] || 'div';
    var result;
    if (namespace !== Namespace.HTML) {
        result = document.createElementNS(namespace, tagName);
    }
    else {
        result = document.createElement(tagName);
    }
    if (match[3]) {
        result.id = match[3];
    }
    if (match[4]) {
        result.className = match[4].replace(/\./g, ' ').trim();
    }
    Object.keys(attrs).forEach(function (name) {
        var value = attrs[name];
        if (typeof value === 'undefined') {
            return;
        }
        if (/^on\w+$/.test(name)) {
            result[name] = value;
        }
        else if (name === 'selected') {
            if (value) {
                result.setAttribute(name, 'true');
            }
        }
        else {
            result.setAttribute(name, value);
        }
    });
    result.append.apply(result, __spreadArray([], __read(children)));
    return result;
}
export function $(description, attrs) {
    var children = [];
    for (var _i = 2; _i < arguments.length; _i++) {
        children[_i - 2] = arguments[_i];
    }
    return _$.apply(void 0, __spreadArray([Namespace.HTML, description, attrs], __read(children)));
}
$.SVG = function (description, attrs) {
    var children = [];
    for (var _i = 2; _i < arguments.length; _i++) {
        children[_i - 2] = arguments[_i];
    }
    return _$.apply(void 0, __spreadArray([Namespace.SVG, description, attrs], __read(children)));
};
export function join(nodes, separator) {
    var result = [];
    nodes.forEach(function (node, index) {
        if (index > 0) {
            if (separator instanceof Node) {
                result.push(separator.cloneNode());
            }
            else {
                result.push(document.createTextNode(separator));
            }
        }
        result.push(node);
    });
    return result;
}
export function show() {
    var e_1, _a;
    var elements = [];
    for (var _i = 0; _i < arguments.length; _i++) {
        elements[_i] = arguments[_i];
    }
    try {
        for (var elements_1 = __values(elements), elements_1_1 = elements_1.next(); !elements_1_1.done; elements_1_1 = elements_1.next()) {
            var element = elements_1_1.value;
            element.style.display = '';
            element.removeAttribute('aria-hidden');
        }
    }
    catch (e_1_1) { e_1 = { error: e_1_1 }; }
    finally {
        try {
            if (elements_1_1 && !elements_1_1.done && (_a = elements_1.return)) _a.call(elements_1);
        }
        finally { if (e_1) throw e_1.error; }
    }
}
export function hide() {
    var e_2, _a;
    var elements = [];
    for (var _i = 0; _i < arguments.length; _i++) {
        elements[_i] = arguments[_i];
    }
    try {
        for (var elements_2 = __values(elements), elements_2_1 = elements_2.next(); !elements_2_1.done; elements_2_1 = elements_2.next()) {
            var element = elements_2_1.value;
            element.style.display = 'none';
            element.setAttribute('aria-hidden', 'true');
        }
    }
    catch (e_2_1) { e_2 = { error: e_2_1 }; }
    finally {
        try {
            if (elements_2_1 && !elements_2_1.done && (_a = elements_2.return)) _a.call(elements_2);
        }
        finally { if (e_2) throw e_2.error; }
    }
}
function findParentWithAttribute(node, attribute) {
    while (node && node.nodeType === node.ELEMENT_NODE) {
        if (node instanceof HTMLElement && node.hasAttribute(attribute)) {
            return node;
        }
        node = node.parentNode;
    }
    return null;
}
export function removeTabIndexAndUpdateFocus(node) {
    if (!node || !node.hasAttribute('tabIndex')) {
        return;
    }
    // If we are the currently focused element and tabIndex is removed,
    // standard DOM behavior is to move focus to the <body> element. We
    // typically never want that, rather put focus to the closest element
    // in the hierarchy of the parent DOM nodes.
    if (document.activeElement === node) {
        var parentFocusable = findParentWithAttribute(node.parentElement, 'tabIndex');
        if (parentFocusable) {
            parentFocusable.focus();
        }
    }
    node.removeAttribute('tabindex');
}
export function getElementsByTagName(tag) {
    return Array.prototype.slice.call(document.getElementsByTagName(tag), 0);
}
export function finalHandler(fn) {
    return function (e) {
        e.preventDefault();
        e.stopPropagation();
        fn(e);
    };
}
export function domContentLoaded() {
    return new Promise(function (resolve) {
        var readyState = document.readyState;
        if (readyState === 'complete' || (document && document.body !== null)) {
            platform.setImmediate(resolve);
        }
        else {
            window.addEventListener('DOMContentLoaded', resolve, false);
        }
    });
}
/**
 * Find a value usable for a dom node size such that the likelihood that it would be
 * displayed with constant screen pixels size is as high as possible.
 *
 * e.g. We would desire for the cursors to be 2px (CSS px) wide. Under a devicePixelRatio
 * of 1.25, the cursor will be 2.5 screen pixels wide. Depending on how the dom node aligns/"snaps"
 * with the screen pixels, it will sometimes be rendered with 2 screen pixels, and sometimes with 3 screen pixels.
 */
export function computeScreenAwareSize(cssPx) {
    var screenPx = window.devicePixelRatio * cssPx;
    return Math.max(1, Math.floor(screenPx)) / window.devicePixelRatio;
}
/**
 * See https://github.com/microsoft/monaco-editor/issues/601
 * To protect against malicious code in the linked site, particularly phishing attempts,
 * the window.opener should be set to null to prevent the linked site from having access
 * to change the location of the current page.
 * See https://mathiasbynens.github.io/rel-noopener/
 */
export function windowOpenNoOpener(url) {
    if (browser.isElectron || browser.isEdgeLegacyWebView) {
        // In VSCode, window.open() always returns null...
        // The same is true for a WebView (see https://github.com/microsoft/monaco-editor/issues/628)
        // Also call directly window.open in sandboxed Electron (see https://github.com/microsoft/monaco-editor/issues/2220)
        window.open(url);
    }
    else {
        var newTab = window.open();
        if (newTab) {
            newTab.opener = null;
            newTab.location.href = url;
        }
    }
}
export function animate(fn) {
    var step = function () {
        fn();
        stepDisposable = scheduleAtNextAnimationFrame(step);
    };
    var stepDisposable = scheduleAtNextAnimationFrame(step);
    return toDisposable(function () { return stepDisposable.dispose(); });
}
RemoteAuthorities.setPreferredWebSchema(/^https:/.test(window.location.href) ? 'https' : 'http');
/**
 * returns url('...')
 */
export function asCSSUrl(uri) {
    if (!uri) {
        return "url('')";
    }
    return "url('" + FileAccess.asBrowserUri(uri).toString(true).replace(/'/g, '%27') + "')";
}
export function asCSSPropertyValue(value) {
    return "'" + value.replace(/'/g, '%27') + "'";
}
export function triggerDownload(dataOrUri, name) {
    // If the data is provided as Buffer, we create a
    // blob URL out of it to produce a valid link
    var url;
    if (URI.isUri(dataOrUri)) {
        url = dataOrUri.toString(true);
    }
    else {
        var blob = new Blob([dataOrUri]);
        url = URL.createObjectURL(blob);
        // Ensure to free the data from DOM eventually
        setTimeout(function () { return URL.revokeObjectURL(url); });
    }
    // In order to download from the browser, the only way seems
    // to be creating a <a> element with download attribute that
    // points to the file to download.
    // See also https://developers.google.com/web/updates/2011/08/Downloading-resources-in-HTML5-a-download
    var anchor = document.createElement('a');
    document.body.appendChild(anchor);
    anchor.download = name;
    anchor.href = url;
    anchor.click();
    // Ensure to remove the element from DOM eventually
    setTimeout(function () { return document.body.removeChild(anchor); });
}
export var DetectedFullscreenMode;
(function (DetectedFullscreenMode) {
    /**
     * The document is fullscreen, e.g. because an element
     * in the document requested to be fullscreen.
     */
    DetectedFullscreenMode[DetectedFullscreenMode["DOCUMENT"] = 1] = "DOCUMENT";
    /**
     * The browser is fullsreen, e.g. because the user enabled
     * native window fullscreen for it.
     */
    DetectedFullscreenMode[DetectedFullscreenMode["BROWSER"] = 2] = "BROWSER";
})(DetectedFullscreenMode || (DetectedFullscreenMode = {}));
export function detectFullscreen() {
    // Browser fullscreen: use DOM APIs to detect
    if (document.fullscreenElement || document.webkitFullscreenElement || document.webkitIsFullScreen) {
        return { mode: DetectedFullscreenMode.DOCUMENT, guess: false };
    }
    // There is no standard way to figure out if the browser
    // is using native fullscreen. Via checking on screen
    // height and comparing that to window height, we can guess
    // it though.
    if (window.innerHeight === screen.height) {
        // if the height of the window matches the screen height, we can
        // safely assume that the browser is fullscreen because no browser
        // chrome is taking height away (e.g. like toolbars).
        return { mode: DetectedFullscreenMode.BROWSER, guess: false };
    }
    if (platform.isMacintosh || platform.isLinux) {
        // macOS and Linux do not properly report `innerHeight`, only Windows does
        if (window.outerHeight === screen.height && window.outerWidth === screen.width) {
            // if the height of the browser matches the screen height, we can
            // only guess that we are in fullscreen. It is also possible that
            // the user has turned off taskbars in the OS and the browser is
            // simply able to span the entire size of the screen.
            return { mode: DetectedFullscreenMode.BROWSER, guess: true };
        }
    }
    // Not in fullscreen
    return null;
}
// -- sanitize and trusted html
function _extInsaneOptions(opts, allowedAttributesForAll) {
    var e_3, _a;
    var _b;
    var allowedAttributes = (_b = opts.allowedAttributes) !== null && _b !== void 0 ? _b : {};
    if (opts.allowedTags) {
        try {
            for (var _c = __values(opts.allowedTags), _d = _c.next(); !_d.done; _d = _c.next()) {
                var tag = _d.value;
                var array = allowedAttributes[tag];
                if (!array) {
                    array = allowedAttributesForAll;
                }
                else {
                    array = array.concat(allowedAttributesForAll);
                }
                allowedAttributes[tag] = array;
            }
        }
        catch (e_3_1) { e_3 = { error: e_3_1 }; }
        finally {
            try {
                if (_d && !_d.done && (_a = _c.return)) _a.call(_c);
            }
            finally { if (e_3) throw e_3.error; }
        }
    }
    return __assign(__assign({}, opts), { allowedAttributes: allowedAttributes });
}
var _ttpSafeInnerHtml = (_a = window.trustedTypes) === null || _a === void 0 ? void 0 : _a.createPolicy('safeInnerHtml', {
    createHTML: function (value, options) {
        return insane(value, options);
    }
});
/**
 * Sanitizes the given `value` and reset the given `node` with it.
 */
export function safeInnerHtml(node, value) {
    var _a;
    var options = _extInsaneOptions({
        allowedTags: ['a', 'button', 'blockquote', 'code', 'div', 'h1', 'h2', 'h3', 'input', 'label', 'li', 'p', 'pre', 'select', 'small', 'span', 'strong', 'textarea', 'ul', 'ol'],
        allowedAttributes: {
            'a': ['href', 'x-dispatch'],
            'button': ['data-href', 'x-dispatch'],
            'input': ['type', 'placeholder', 'checked', 'required'],
            'label': ['for'],
            'select': ['required'],
            'span': ['data-command', 'role'],
            'textarea': ['name', 'placeholder', 'required'],
        },
        allowedSchemes: ['http', 'https', 'command']
    }, ['class', 'id', 'role', 'tabindex']);
    var html = (_a = _ttpSafeInnerHtml === null || _ttpSafeInnerHtml === void 0 ? void 0 : _ttpSafeInnerHtml.createHTML(value, options)) !== null && _a !== void 0 ? _a : insane(value, options);
    node.innerHTML = html;
}
/**
 * Convert a Unicode string to a string in which each 16-bit unit occupies only one byte
 *
 * From https://developer.mozilla.org/en-US/docs/Web/API/WindowOrWorkerGlobalScope/btoa
 */
function toBinary(str) {
    var codeUnits = new Uint16Array(str.length);
    for (var i = 0; i < codeUnits.length; i++) {
        codeUnits[i] = str.charCodeAt(i);
    }
    var binary = '';
    var uint8array = new Uint8Array(codeUnits.buffer);
    for (var i = 0; i < uint8array.length; i++) {
        binary += String.fromCharCode(uint8array[i]);
    }
    return binary;
}
/**
 * Version of the global `btoa` function that handles multi-byte characters instead
 * of throwing an exception.
 */
export function multibyteAwareBtoa(str) {
    return btoa(toBinary(str));
}
/**
 * Typings for the https://wicg.github.io/file-system-access
 *
 * Use `supported(window)` to find out if the browser supports this kind of API.
 */
export var WebFileSystemAccess;
(function (WebFileSystemAccess) {
    function supported(obj) {
        var candidate = obj;
        if (typeof (candidate === null || candidate === void 0 ? void 0 : candidate.showDirectoryPicker) === 'function') {
            return true;
        }
        return false;
    }
    WebFileSystemAccess.supported = supported;
})(WebFileSystemAccess || (WebFileSystemAccess = {}));
var ModifierKeyEmitter = /** @class */ (function (_super) {
    __extends(ModifierKeyEmitter, _super);
    function ModifierKeyEmitter() {
        var _this = _super.call(this) || this;
        _this._subscriptions = new DisposableStore();
        _this._keyStatus = {
            altKey: false,
            shiftKey: false,
            ctrlKey: false,
            metaKey: false
        };
        _this._subscriptions.add(domEvent(window, 'keydown', true)(function (e) {
            var event = new StandardKeyboardEvent(e);
            // If Alt-key keydown event is repeated, ignore it #112347
            // Only known to be necessary for Alt-Key at the moment #115810
            if (event.keyCode === 6 /* Alt */ && e.repeat) {
                return;
            }
            if (e.altKey && !_this._keyStatus.altKey) {
                _this._keyStatus.lastKeyPressed = 'alt';
            }
            else if (e.ctrlKey && !_this._keyStatus.ctrlKey) {
                _this._keyStatus.lastKeyPressed = 'ctrl';
            }
            else if (e.metaKey && !_this._keyStatus.metaKey) {
                _this._keyStatus.lastKeyPressed = 'meta';
            }
            else if (e.shiftKey && !_this._keyStatus.shiftKey) {
                _this._keyStatus.lastKeyPressed = 'shift';
            }
            else if (event.keyCode !== 6 /* Alt */) {
                _this._keyStatus.lastKeyPressed = undefined;
            }
            else {
                return;
            }
            _this._keyStatus.altKey = e.altKey;
            _this._keyStatus.ctrlKey = e.ctrlKey;
            _this._keyStatus.metaKey = e.metaKey;
            _this._keyStatus.shiftKey = e.shiftKey;
            if (_this._keyStatus.lastKeyPressed) {
                _this._keyStatus.event = e;
                _this.fire(_this._keyStatus);
            }
        }));
        _this._subscriptions.add(domEvent(window, 'keyup', true)(function (e) {
            if (!e.altKey && _this._keyStatus.altKey) {
                _this._keyStatus.lastKeyReleased = 'alt';
            }
            else if (!e.ctrlKey && _this._keyStatus.ctrlKey) {
                _this._keyStatus.lastKeyReleased = 'ctrl';
            }
            else if (!e.metaKey && _this._keyStatus.metaKey) {
                _this._keyStatus.lastKeyReleased = 'meta';
            }
            else if (!e.shiftKey && _this._keyStatus.shiftKey) {
                _this._keyStatus.lastKeyReleased = 'shift';
            }
            else {
                _this._keyStatus.lastKeyReleased = undefined;
            }
            if (_this._keyStatus.lastKeyPressed !== _this._keyStatus.lastKeyReleased) {
                _this._keyStatus.lastKeyPressed = undefined;
            }
            _this._keyStatus.altKey = e.altKey;
            _this._keyStatus.ctrlKey = e.ctrlKey;
            _this._keyStatus.metaKey = e.metaKey;
            _this._keyStatus.shiftKey = e.shiftKey;
            if (_this._keyStatus.lastKeyReleased) {
                _this._keyStatus.event = e;
                _this.fire(_this._keyStatus);
            }
        }));
        _this._subscriptions.add(domEvent(document.body, 'mousedown', true)(function (e) {
            _this._keyStatus.lastKeyPressed = undefined;
        }));
        _this._subscriptions.add(domEvent(document.body, 'mouseup', true)(function (e) {
            _this._keyStatus.lastKeyPressed = undefined;
        }));
        _this._subscriptions.add(domEvent(document.body, 'mousemove', true)(function (e) {
            if (e.buttons) {
                _this._keyStatus.lastKeyPressed = undefined;
            }
        }));
        _this._subscriptions.add(domEvent(window, 'blur')(function (e) {
            _this.resetKeyStatus();
        }));
        return _this;
    }
    Object.defineProperty(ModifierKeyEmitter.prototype, "keyStatus", {
        get: function () {
            return this._keyStatus;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(ModifierKeyEmitter.prototype, "isModifierPressed", {
        get: function () {
            return this._keyStatus.altKey || this._keyStatus.ctrlKey || this._keyStatus.metaKey || this._keyStatus.shiftKey;
        },
        enumerable: false,
        configurable: true
    });
    /**
     * Allows to explicitly reset the key status based on more knowledge (#109062)
     */
    ModifierKeyEmitter.prototype.resetKeyStatus = function () {
        this.doResetKeyStatus();
        this.fire(this._keyStatus);
    };
    ModifierKeyEmitter.prototype.doResetKeyStatus = function () {
        this._keyStatus = {
            altKey: false,
            shiftKey: false,
            ctrlKey: false,
            metaKey: false
        };
    };
    ModifierKeyEmitter.getInstance = function () {
        if (!ModifierKeyEmitter.instance) {
            ModifierKeyEmitter.instance = new ModifierKeyEmitter();
        }
        return ModifierKeyEmitter.instance;
    };
    ModifierKeyEmitter.prototype.dispose = function () {
        _super.prototype.dispose.call(this);
        this._subscriptions.dispose();
    };
    return ModifierKeyEmitter;
}(Emitter));
export { ModifierKeyEmitter };
export function getCookieValue(name) {
    var match = document.cookie.match('(^|[^;]+)\\s*' + name + '\\s*=\\s*([^;]+)'); // See https://stackoverflow.com/a/25490531
    return match ? match.pop() : undefined;
}
