"use strict";
/********************************************************************************
 * Copyright (C) 2018 Red Hat, Inc. and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.DebugConfigurationManager = void 0;
/*---------------------------------------------------------------------------------------------
*  Copyright (c) Microsoft Corporation. All rights reserved.
*  Licensed under the MIT License. See License.txt in the project root for license information.
*--------------------------------------------------------------------------------------------*/
const debounce = require("p-debounce");
const jsonc_parser_1 = require("jsonc-parser");
const inversify_1 = require("@theia/core/shared/inversify");
const event_1 = require("@theia/core/lib/common/event");
const browser_1 = require("@theia/editor/lib/browser");
const monaco_editor_1 = require("@theia/monaco/lib/browser/monaco-editor");
const browser_2 = require("@theia/core/lib/browser");
const quick_pick_service_1 = require("@theia/core/lib/common/quick-pick-service");
const workspace_service_1 = require("@theia/workspace/lib/browser/workspace-service");
const debug_configuration_model_1 = require("./debug-configuration-model");
const debug_service_1 = require("../common/debug-service");
const context_key_service_1 = require("@theia/core/lib/browser/context-key-service");
const workspace_variable_contribution_1 = require("@theia/workspace/lib/browser/workspace-variable-contribution");
const preference_configurations_1 = require("@theia/core/lib/browser/preferences/preference-configurations");
const monaco_text_model_service_1 = require("@theia/monaco/lib/browser/monaco-text-model-service");
let DebugConfigurationManager = class DebugConfigurationManager {
    constructor() {
        this.onDidChangeEmitter = new event_1.Emitter();
        this.onDidChange = this.onDidChangeEmitter.event;
        this.onWillProvideDebugConfigurationEmitter = new event_1.Emitter();
        this.onWillProvideDebugConfiguration = this.onWillProvideDebugConfigurationEmitter.event;
        this.models = new Map();
        this.updateModels = debounce(async () => {
            const roots = await this.workspaceService.roots;
            const toDelete = new Set(this.models.keys());
            for (const rootStat of roots) {
                const key = rootStat.resource.toString();
                toDelete.delete(key);
                if (!this.models.has(key)) {
                    const model = new debug_configuration_model_1.DebugConfigurationModel(key, this.preferences);
                    model.onDidChange(() => this.updateCurrent());
                    model.onDispose(() => this.models.delete(key));
                    this.models.set(key, model);
                }
            }
            for (const uri of toDelete) {
                const model = this.models.get(uri);
                if (model) {
                    model.dispose();
                }
            }
            this.updateCurrent();
        }, 500);
    }
    async init() {
        this.debugConfigurationTypeKey = this.contextKeyService.createKey('debugConfigurationType', undefined);
        this.initialized = this.updateModels();
        this.preferences.onPreferenceChanged(e => {
            if (e.preferenceName === 'launch') {
                this.updateModels();
            }
        });
    }
    get all() {
        return this.getAll();
    }
    *getAll() {
        for (const model of this.models.values()) {
            for (const configuration of model.configurations) {
                yield {
                    configuration,
                    workspaceFolderUri: model.workspaceFolderUri
                };
            }
        }
    }
    get supported() {
        return this.getSupported();
    }
    async getSupported() {
        await this.initialized;
        const debugTypes = await this.debug.debugTypes();
        return this.doGetSupported(new Set(debugTypes));
    }
    *doGetSupported(debugTypes) {
        for (const options of this.getAll()) {
            if (debugTypes.has(options.configuration.type)) {
                yield options;
            }
        }
    }
    get current() {
        return this._currentOptions;
    }
    set current(option) {
        this.updateCurrent(option);
    }
    updateCurrent(options = this._currentOptions) {
        this._currentOptions = options
            && this.find(options.configuration.name, options.workspaceFolderUri);
        if (!this._currentOptions) {
            const { model } = this;
            if (model) {
                const configuration = model.configurations[0];
                if (configuration) {
                    this._currentOptions = {
                        configuration,
                        workspaceFolderUri: model.workspaceFolderUri
                    };
                }
            }
        }
        this.debugConfigurationTypeKey.set(this.current && this.current.configuration.type);
        this.onDidChangeEmitter.fire(undefined);
    }
    find(name, workspaceFolderUri) {
        for (const model of this.models.values()) {
            if (model.workspaceFolderUri === workspaceFolderUri) {
                for (const configuration of model.configurations) {
                    if (configuration.name === name) {
                        return {
                            configuration,
                            workspaceFolderUri
                        };
                    }
                }
            }
        }
        return undefined;
    }
    async openConfiguration() {
        const { model } = this;
        if (model) {
            await this.doOpen(model);
        }
    }
    async addConfiguration() {
        const { model } = this;
        if (!model) {
            return;
        }
        const widget = await this.doOpen(model);
        if (!(widget.editor instanceof monaco_editor_1.MonacoEditor)) {
            return;
        }
        const editor = widget.editor.getControl();
        const { commandService } = widget.editor;
        let position;
        let depthInArray = 0;
        let lastProperty = '';
        jsonc_parser_1.visit(editor.getValue(), {
            onObjectProperty: property => {
                lastProperty = property;
            },
            onArrayBegin: offset => {
                if (lastProperty === 'configurations' && depthInArray === 0) {
                    position = editor.getModel().getPositionAt(offset + 1);
                }
                depthInArray++;
            },
            onArrayEnd: () => {
                depthInArray--;
            }
        });
        if (!position) {
            return;
        }
        // Check if there are more characters on a line after a "configurations": [, if yes enter a newline
        if (editor.getModel().getLineLastNonWhitespaceColumn(position.lineNumber) > position.column) {
            editor.setPosition(position);
            editor.trigger('debug', 'lineBreakInsert', undefined);
        }
        // Check if there is already an empty line to insert suggest, if yes just place the cursor
        if (editor.getModel().getLineLastNonWhitespaceColumn(position.lineNumber + 1) === 0) {
            editor.setPosition({ lineNumber: position.lineNumber + 1, column: 1 << 30 });
            await commandService.executeCommand('editor.action.deleteLines');
        }
        editor.setPosition(position);
        await commandService.executeCommand('editor.action.insertLineAfter');
        await commandService.executeCommand('editor.action.triggerSuggest');
    }
    get model() {
        const workspaceFolderUri = this.workspaceVariables.getWorkspaceRootUri();
        if (workspaceFolderUri) {
            const key = workspaceFolderUri.toString();
            for (const model of this.models.values()) {
                if (model.workspaceFolderUri === key) {
                    return model;
                }
            }
        }
        for (const model of this.models.values()) {
            if (model.uri) {
                return model;
            }
        }
        return this.models.values().next().value;
    }
    async doOpen(model) {
        const uri = await this.doCreate(model);
        return this.editorManager.open(uri, {
            mode: 'activate'
        });
    }
    async doCreate(model) {
        var _a;
        const uri = (_a = model.uri) !== null && _a !== void 0 ? _a : this.preferences.getConfigUri(browser_2.PreferenceScope.Folder, model.workspaceFolderUri, 'launch');
        if (!uri) { // Since we are requesting information about a known workspace folder, this should never happen.
            throw new Error('PreferenceService.getConfigUri has returned undefined when a URI was expected.');
        }
        const settingsUri = this.preferences.getConfigUri(browser_2.PreferenceScope.Folder, model.workspaceFolderUri);
        // Users may have placed their debug configurations in a `settings.json`, in which case we shouldn't modify the file.
        if (settingsUri && !uri.isEqual(settingsUri)) {
            await this.ensureContent(uri, model);
        }
        return uri;
    }
    /**
     * Checks whether a `launch.json` file contains the minimum necessary content.
     * If content not found, provides content and populates the file using Monaco.
     */
    async ensureContent(uri, model) {
        const textModel = await this.textModelService.createModelReference(uri);
        const currentContent = textModel.object.getText();
        try { // Look for the minimal well-formed launch.json content: {configurations: []}
            const parsedContent = jsonc_parser_1.parse(currentContent);
            if (Array.isArray(parsedContent.configurations)) {
                return;
            }
        }
        catch (_a) {
            // Just keep going
        }
        const debugType = await this.selectDebugType();
        const configurations = debugType ? await this.provideDebugConfigurations(debugType, model.workspaceFolderUri) : [];
        const content = this.getInitialConfigurationContent(configurations);
        textModel.object.textEditorModel.setValue(content); // Will clobber anything the user has entered!
        await textModel.object.save();
    }
    async provideDebugConfigurations(debugType, workspaceFolderUri) {
        await this.fireWillProvideDebugConfiguration();
        return this.debug.provideDebugConfigurations(debugType, workspaceFolderUri);
    }
    async fireWillProvideDebugConfiguration() {
        await event_1.WaitUntilEvent.fire(this.onWillProvideDebugConfigurationEmitter, {});
    }
    getInitialConfigurationContent(initialConfigurations) {
        return `{
  // Use IntelliSense to learn about possible attributes.
  // Hover to view descriptions of existing attributes.
  "version": "0.2.0",
  "configurations": ${JSON.stringify(initialConfigurations, undefined, '  ').split('\n').map(line => '  ' + line).join('\n').trim()}
}
`;
    }
    async selectDebugType() {
        const widget = this.editorManager.currentEditor;
        if (!widget) {
            return undefined;
        }
        const { languageId } = widget.editor.document;
        const debuggers = await this.debug.getDebuggersForLanguage(languageId);
        if (debuggers.length === 0) {
            return undefined;
        }
        const items = debuggers.map(({ label, type }) => ({ label, value: type }));
        const selectedItem = await this.quickPickService.show(items, { placeholder: 'Select Environment' });
        return selectedItem === null || selectedItem === void 0 ? void 0 : selectedItem.value;
    }
    async load() {
        await this.initialized;
        const data = await this.storage.getData('debug.configurations', {});
        if (data.current) {
            this.current = this.find(data.current.name, data.current.workspaceFolderUri);
        }
    }
    save() {
        const data = {};
        const { current } = this;
        if (current) {
            data.current = {
                name: current.configuration.name,
                workspaceFolderUri: current.workspaceFolderUri
            };
        }
        this.storage.setData('debug.configurations', data);
    }
};
__decorate([
    inversify_1.inject(workspace_service_1.WorkspaceService),
    __metadata("design:type", workspace_service_1.WorkspaceService)
], DebugConfigurationManager.prototype, "workspaceService", void 0);
__decorate([
    inversify_1.inject(browser_1.EditorManager),
    __metadata("design:type", browser_1.EditorManager)
], DebugConfigurationManager.prototype, "editorManager", void 0);
__decorate([
    inversify_1.inject(debug_service_1.DebugService),
    __metadata("design:type", Object)
], DebugConfigurationManager.prototype, "debug", void 0);
__decorate([
    inversify_1.inject(quick_pick_service_1.QuickPickService),
    __metadata("design:type", Object)
], DebugConfigurationManager.prototype, "quickPickService", void 0);
__decorate([
    inversify_1.inject(context_key_service_1.ContextKeyService),
    __metadata("design:type", context_key_service_1.ContextKeyService)
], DebugConfigurationManager.prototype, "contextKeyService", void 0);
__decorate([
    inversify_1.inject(monaco_text_model_service_1.MonacoTextModelService),
    __metadata("design:type", monaco_text_model_service_1.MonacoTextModelService)
], DebugConfigurationManager.prototype, "textModelService", void 0);
__decorate([
    inversify_1.inject(browser_2.PreferenceService),
    __metadata("design:type", Object)
], DebugConfigurationManager.prototype, "preferences", void 0);
__decorate([
    inversify_1.inject(preference_configurations_1.PreferenceConfigurations),
    __metadata("design:type", preference_configurations_1.PreferenceConfigurations)
], DebugConfigurationManager.prototype, "preferenceConfigurations", void 0);
__decorate([
    inversify_1.inject(workspace_variable_contribution_1.WorkspaceVariableContribution),
    __metadata("design:type", workspace_variable_contribution_1.WorkspaceVariableContribution)
], DebugConfigurationManager.prototype, "workspaceVariables", void 0);
__decorate([
    inversify_1.postConstruct(),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], DebugConfigurationManager.prototype, "init", null);
__decorate([
    inversify_1.inject(browser_2.StorageService),
    __metadata("design:type", Object)
], DebugConfigurationManager.prototype, "storage", void 0);
DebugConfigurationManager = __decorate([
    inversify_1.injectable()
], DebugConfigurationManager);
exports.DebugConfigurationManager = DebugConfigurationManager;
//# sourceMappingURL=debug-configuration-manager.js.map