/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.rdf4j.sail.nativerdf;

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.stream.Collectors;
import java.util.zip.CRC32C;
import org.eclipse.rdf4j.common.annotation.InternalUseOnly;
import org.eclipse.rdf4j.common.concurrent.locks.Lock;
import org.eclipse.rdf4j.common.concurrent.locks.ReadWriteLockManager;
import org.eclipse.rdf4j.common.concurrent.locks.WritePrefReadWriteLockManager;
import org.eclipse.rdf4j.common.io.ByteArrayUtil;
import org.eclipse.rdf4j.model.BNode;
import org.eclipse.rdf4j.model.IRI;
import org.eclipse.rdf4j.model.Literal;
import org.eclipse.rdf4j.model.Resource;
import org.eclipse.rdf4j.model.Value;
import org.eclipse.rdf4j.model.base.CoreDatatype;
import org.eclipse.rdf4j.model.impl.SimpleValueFactory;
import org.eclipse.rdf4j.model.util.Literals;
import org.eclipse.rdf4j.model.vocabulary.RDF;
import org.eclipse.rdf4j.model.vocabulary.XSD;
import org.eclipse.rdf4j.sail.SailException;
import org.eclipse.rdf4j.sail.nativerdf.ConcurrentCache;
import org.eclipse.rdf4j.sail.nativerdf.NativeStore;
import org.eclipse.rdf4j.sail.nativerdf.ValueStoreRevision;
import org.eclipse.rdf4j.sail.nativerdf.datastore.DataStore;
import org.eclipse.rdf4j.sail.nativerdf.datastore.RecoveredDataException;
import org.eclipse.rdf4j.sail.nativerdf.model.CorruptIRI;
import org.eclipse.rdf4j.sail.nativerdf.model.CorruptIRIOrBNode;
import org.eclipse.rdf4j.sail.nativerdf.model.CorruptLiteral;
import org.eclipse.rdf4j.sail.nativerdf.model.CorruptUnknownValue;
import org.eclipse.rdf4j.sail.nativerdf.model.CorruptValue;
import org.eclipse.rdf4j.sail.nativerdf.model.NativeBNode;
import org.eclipse.rdf4j.sail.nativerdf.model.NativeIRI;
import org.eclipse.rdf4j.sail.nativerdf.model.NativeLiteral;
import org.eclipse.rdf4j.sail.nativerdf.model.NativeResource;
import org.eclipse.rdf4j.sail.nativerdf.model.NativeValue;
import org.eclipse.rdf4j.sail.nativerdf.wal.ValueStoreWAL;
import org.eclipse.rdf4j.sail.nativerdf.wal.ValueStoreWalConfig;
import org.eclipse.rdf4j.sail.nativerdf.wal.ValueStoreWalReader;
import org.eclipse.rdf4j.sail.nativerdf.wal.ValueStoreWalRecord;
import org.eclipse.rdf4j.sail.nativerdf.wal.ValueStoreWalRecovery;
import org.eclipse.rdf4j.sail.nativerdf.wal.ValueStoreWalSearch;
import org.eclipse.rdf4j.sail.nativerdf.wal.ValueStoreWalValueKind;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InternalUseOnly
public class ValueStore
extends SimpleValueFactory
implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(ValueStore.class);
    private static final String WAL_RECOVERY_LOG_PROP = "org.eclipse.rdf4j.sail.nativerdf.valuestorewal.recoveryLog";
    private static final String WAL_RECOVERY_LOG = System.getProperty("org.eclipse.rdf4j.sail.nativerdf.valuestorewal.recoveryLog", "debug").toLowerCase();
    public static final int VALUE_CACHE_SIZE = 512;
    public static final int VALUE_ID_CACHE_SIZE = 128;
    public static final int NAMESPACE_CACHE_SIZE = 64;
    public static final int NAMESPACE_ID_CACHE_SIZE = 32;
    private static final String FILENAME_PREFIX = "values";
    private static final byte URI_VALUE = 1;
    private static final byte BNODE_VALUE = 2;
    private static final byte LITERAL_VALUE = 3;
    private final File dataDir;
    private final DataStore dataStore;
    private final ValueStoreWAL wal;
    private final ThreadLocal<Long> walPendingLsn;
    private volatile CompletableFuture<Void> walBootstrapFuture;
    private volatile ValueStoreWalSearch walSearch;
    private final ReadWriteLockManager lockManager = new WritePrefReadWriteLockManager();
    private volatile ValueStoreRevision revision;
    private final ConcurrentCache<Integer, NativeValue> valueCache;
    private final ConcurrentCache<NativeValue, Integer> valueIDCache;
    private final ConcurrentCache<Integer, String> namespaceCache;
    private final ConcurrentCache<String, Integer> namespaceIDCache;
    private static final ThreadLocal<CRC32C> CRC32C_HOLDER = ThreadLocal.withInitial(CRC32C::new);

    public ValueStore(File dataDir) throws IOException {
        this(dataDir, false);
    }

    public ValueStore(File dataDir, boolean forceSync) throws IOException {
        this(dataDir, forceSync, 512, 128, 64, 32);
    }

    public ValueStore(File dataDir, boolean forceSync, int valueCacheSize, int valueIDCacheSize, int namespaceCacheSize, int namespaceIDCacheSize) throws IOException {
        this(dataDir, forceSync, valueCacheSize, valueIDCacheSize, namespaceCacheSize, namespaceIDCacheSize, null);
    }

    public ValueStore(File dataDir, boolean forceSync, int valueCacheSize, int valueIDCacheSize, int namespaceCacheSize, int namespaceIDCacheSize, ValueStoreWAL wal) throws IOException {
        this.dataDir = dataDir;
        this.dataStore = new DataStore(dataDir, FILENAME_PREFIX, forceSync, this);
        this.valueCache = new ConcurrentCache(valueCacheSize);
        this.valueIDCache = new ConcurrentCache(valueIDCacheSize);
        this.namespaceCache = new ConcurrentCache(namespaceCacheSize);
        this.namespaceIDCache = new ConcurrentCache(namespaceIDCacheSize);
        this.wal = wal;
        this.walPendingLsn = wal != null ? ThreadLocal.withInitial(() -> -1L) : null;
        this.autoRecoverValueStoreIfConfigured();
        this.setNewRevision();
        this.maybeScheduleWalBootstrap();
    }

    private void setNewRevision() {
        this.revision = new ValueStoreRevision(this);
    }

    public ValueStoreRevision getRevision() {
        return this.revision;
    }

    public Lock getReadLock() throws InterruptedException {
        return this.lockManager.getReadLock();
    }

    public NativeValue getValue(int id) throws IOException {
        Integer cacheID = id;
        NativeValue resultValue = this.valueCache.get(cacheID);
        if (resultValue == null) {
            boolean recoveredDirectlyFromWal = false;
            try {
                byte[] data = this.dataStore.getData(id);
                if (data != null) {
                    NativeValue walValue;
                    resultValue = this.data2value(id, data);
                    if (resultValue instanceof CorruptValue) {
                        NativeValue recovered = ((CorruptValue)resultValue).getRecovered();
                        if (recovered != null) {
                            resultValue = recovered;
                        }
                    } else if (this.shouldValidateAgainstWal() && (walValue = this.recoverValueFromWal(id, false)) != null && !this.valuesMatch(resultValue, walValue)) {
                        resultValue = walValue;
                        recoveredDirectlyFromWal = true;
                    }
                } else {
                    resultValue = this.recoverValueFromWal(id, false);
                    recoveredDirectlyFromWal = resultValue != null;
                }
            }
            catch (RecoveredDataException rde) {
                byte t;
                byte[] recovered = rde.getData();
                CorruptValue corruptValue = recovered != null && recovered.length > 0 ? ((t = recovered[0]) == 1 ? new CorruptIRI(this.revision, id, null, recovered) : (t == 2 ? new CorruptIRIOrBNode(this.revision, id, recovered) : (t == 3 ? new CorruptLiteral(this.revision, id, recovered) : new CorruptUnknownValue(this.revision, id, recovered)))) : new CorruptUnknownValue(this.revision, id, recovered);
                this.tryRecoverFromWal(id, corruptValue);
                NativeValue recoveredValue = corruptValue.getRecovered();
                if (recoveredValue != null) {
                    resultValue = recoveredValue;
                    recoveredDirectlyFromWal = true;
                }
                resultValue = corruptValue;
            }
            if (recoveredDirectlyFromWal && resultValue != null) {
                this.logRecovered(id, resultValue);
                this.logWalRepairHint(id);
            }
            if (resultValue != null && !(resultValue instanceof CorruptValue)) {
                this.valueCache.put(cacheID, resultValue);
            }
        }
        return resultValue;
    }

    public <T extends NativeValue & Resource> T getResource(int id) throws IOException {
        NativeValue resultValue = this.getValue(id);
        if (resultValue != null && !(resultValue instanceof Resource)) {
            if (NativeStore.SOFT_FAIL_ON_CORRUPT_DATA_AND_REPAIR_INDEXES && resultValue instanceof CorruptValue) {
                return (T)new CorruptIRIOrBNode(this.revision, id, ((CorruptValue)resultValue).getData());
            }
            logger.warn("NativeStore is possibly corrupt. To attempt to repair or retrieve the data, read the documentation on http://rdf4j.org about the system property org.eclipse.rdf4j.sail.nativerdf.softFailOnCorruptDataAndRepairIndexes");
        }
        return (T)resultValue;
    }

    public <T extends NativeValue & IRI> T getIRI(int id) throws IOException {
        NativeValue resultValue = this.getValue(id);
        if (resultValue != null && !(resultValue instanceof IRI)) {
            if (NativeStore.SOFT_FAIL_ON_CORRUPT_DATA_AND_REPAIR_INDEXES && resultValue instanceof CorruptValue) {
                if (resultValue instanceof CorruptIRI) {
                    return (T)resultValue;
                }
                return (T)new CorruptIRI(this.revision, id, null, ((CorruptValue)resultValue).getData());
            }
            logger.warn("NativeStore is possibly corrupt. To attempt to repair or retrieve the data, read the documentation on http://rdf4j.org about the system property org.eclipse.rdf4j.sail.nativerdf.softFailOnCorruptDataAndRepairIndexes");
        }
        return (T)resultValue;
    }

    public int getID(Value value) throws IOException {
        int id;
        NativeValue nativeValue;
        boolean isOwnValue;
        if (logger.isDebugEnabled()) {
            logger.debug("getID start thread={} value={}", (Object)ValueStore.threadName(), (Object)this.describeValue(value));
        }
        if ((isOwnValue = this.isOwnValue(value)) && this.revisionIsCurrent(nativeValue = (NativeValue)value) && (id = nativeValue.getInternalID()) != -1) {
            if (logger.isDebugEnabled()) {
                logger.debug("getID returning cached internal id {} for value={} thread={}", new Object[]{id, this.describeValue(value), ValueStore.threadName()});
            }
            return id;
        }
        Integer cachedID = this.valueIDCache.get(value);
        if (cachedID != null) {
            id = cachedID;
            if (isOwnValue) {
                ((NativeValue)value).setInternalID(id, this.revision);
            }
            if (logger.isDebugEnabled()) {
                logger.debug("getID returning cache id {} for value={} thread={}", new Object[]{id, this.describeValue(value), ValueStore.threadName()});
            }
            return id;
        }
        byte[] data = this.value2data(value, false);
        if (data == null && value instanceof Literal) {
            data = this.literal2legacy((Literal)value);
        }
        if (data != null) {
            int id2;
            if (logger.isDebugEnabled()) {
                logger.debug("getID querying datastore for value={} thread={} dataSummary={}", new Object[]{this.describeValue(value), ValueStore.threadName(), ValueStore.summarize(data)});
            }
            if ((id2 = this.dataStore.getID(data)) == -1 && value instanceof Literal) {
                id2 = this.dataStore.getID(this.literal2legacy((Literal)value));
            }
            if (id2 != -1) {
                if (isOwnValue) {
                    ((NativeValue)value).setInternalID(id2, this.revision);
                } else {
                    NativeValue nv = this.getNativeValue(value);
                    nv.setInternalID(id2, this.revision);
                    this.valueIDCache.put(nv, id2);
                }
                if (logger.isDebugEnabled()) {
                    logger.debug("getID resolved value={} id={} thread={}", new Object[]{this.describeValue(value), id2, ValueStore.threadName()});
                }
            }
            return id2;
        }
        if (logger.isDebugEnabled()) {
            logger.debug("getID returning UNKNOWN for value={} thread={}", (Object)this.describeValue(value), (Object)ValueStore.threadName());
        }
        return -1;
    }

    private static String summarize(byte[] data) {
        if (data == null) {
            return "null";
        }
        return "len=" + data.length + ",hash=" + Arrays.hashCode(data);
    }

    private static String threadName() {
        return Thread.currentThread().getName();
    }

    public synchronized int storeValue(Value value) throws IOException {
        int previousMaxID;
        int id;
        NativeValue nativeValue;
        boolean isOwnValue;
        if (logger.isDebugEnabled()) {
            logger.debug("storeValue start thread={} value={}", (Object)ValueStore.threadName(), (Object)this.describeValue(value));
        }
        if ((isOwnValue = this.isOwnValue(value)) && this.revisionIsCurrent(nativeValue = (NativeValue)value) && (id = nativeValue.getInternalID()) != -1) {
            if (logger.isDebugEnabled()) {
                logger.debug("storeValue returning cached internal id {} for value={} thread={}", new Object[]{id, this.describeValue(value), ValueStore.threadName()});
            }
            return id;
        }
        Integer cachedID = this.valueIDCache.get(value);
        if (cachedID != null) {
            id = cachedID;
            if (isOwnValue) {
                ((NativeValue)value).setInternalID(id, this.revision);
            }
            if (logger.isDebugEnabled()) {
                logger.debug("storeValue returning cached id {} for value={} thread={}", new Object[]{id, this.describeValue(value), ValueStore.threadName()});
            }
            return id;
        }
        byte[] valueData = this.value2data(value, true);
        int n = previousMaxID = this.walEnabled() ? this.dataStore.getMaxID() : 0;
        if (valueData == null) {
            if (logger.isDebugEnabled()) {
                logger.debug("storeValue computed no data for value={} thread={}", (Object)this.describeValue(value), (Object)ValueStore.threadName());
            }
            return -1;
        }
        int id2 = this.dataStore.storeData(valueData);
        NativeValue nv = isOwnValue ? (NativeValue)value : this.getNativeValue(value);
        nv.setInternalID(id2, this.revision);
        this.valueIDCache.put(nv, id2);
        if (this.walEnabled() && id2 > previousMaxID) {
            this.logMintedValue(id2, nv);
        }
        if (logger.isDebugEnabled()) {
            logger.debug("storeValue stored value={} assigned id={} thread={} dataSummary={}", new Object[]{this.describeValue(nv), id2, ValueStore.threadName(), ValueStore.summarize(valueData)});
        }
        return id2;
    }

    public void clear() throws IOException {
        try {
            Lock writeLock = this.lockManager.getWriteLock();
            try {
                if (this.walEnabled()) {
                    try {
                        this.wal.purgeAllSegments();
                    }
                    catch (IOException e) {
                        logger.warn("Failed to purge ValueStore WAL during clear for {}", (Object)this.dataDir, (Object)e);
                        throw e;
                    }
                }
                this.dataStore.clear();
                this.valueCache.clear();
                this.valueIDCache.clear();
                this.namespaceCache.clear();
                this.namespaceIDCache.clear();
                this.setNewRevision();
            }
            finally {
                writeLock.release();
            }
        }
        catch (InterruptedException e) {
            throw new IOException("Failed to acquire write lock", e);
        }
    }

    public void sync() throws IOException {
        this.dataStore.sync();
    }

    @Override
    public void close() throws IOException {
        CompletableFuture<Void> bootstrap = this.walBootstrapFuture;
        if (bootstrap != null) {
            try {
                bootstrap.join();
            }
            catch (CompletionException e) {
                Throwable cause = e.getCause() == null ? e : e.getCause();
                logger.warn("ValueStore WAL bootstrap failed during close", cause);
            }
            catch (CancellationException e) {
                logger.warn("ValueStore WAL bootstrap was cancelled during close");
            }
        }
        this.dataStore.close();
    }

    public void checkConsistency() throws SailException, IOException {
        int maxID = this.dataStore.getMaxID();
        for (int id = 1; id <= maxID; ++id) {
            NativeValue value;
            try {
                byte[] data = this.dataStore.getData(id);
                if (data == null || data.length == 0) {
                    throw new SailException("Empty data array for value with id " + id);
                }
                if (this.isNamespaceData(data)) {
                    String namespace;
                    block7: {
                        namespace = this.data2namespace(data);
                        try {
                            if (id != this.getNamespaceID(namespace, false) || !URI.create(namespace + "part").isAbsolute()) break block7;
                            continue;
                        }
                        catch (IllegalArgumentException illegalArgumentException) {
                            // empty catch block
                        }
                    }
                    logger.error("Inconsistent namespace data for id {} (also id {}): {}", new Object[]{id, this.getNamespaceID(namespace, false), namespace});
                    throw new SailException("Store must be manually exported and imported to fix namespaces like " + namespace);
                }
                value = this.data2value(id, data);
                if (id == this.getID(this.copy(value))) continue;
                throw new SailException("Store must be manually exported and imported to merge values like " + String.valueOf(value));
            }
            catch (RecoveredDataException rde) {
                value = new CorruptUnknownValue(this.revision, id, rde.getData());
                if (id == this.getID(this.copy(value))) continue;
                throw new SailException("Store must be manually exported and imported to merge values like " + String.valueOf(value));
            }
        }
    }

    private Value copy(Value value) {
        if (value instanceof IRI) {
            return this.createIRI(value.stringValue());
        }
        if (value instanceof Literal) {
            Literal lit = (Literal)value;
            if (Literals.isLanguageLiteral(lit)) {
                return this.createLiteral(value.stringValue(), (String)lit.getLanguage().orElse(null));
            }
            return this.createLiteral(value.stringValue(), lit.getDatatype());
        }
        return this.createBNode(value.stringValue());
    }

    private boolean isOwnValue(Value value) {
        return value instanceof NativeValue && ((NativeValue)value).getValueStoreRevision().getValueStore() == this;
    }

    private boolean revisionIsCurrent(NativeValue value) {
        return this.revision.equals(value.getValueStoreRevision());
    }

    private byte[] value2data(Value value, boolean create) throws IOException {
        byte[] data;
        if (value instanceof IRI) {
            data = this.uri2data((IRI)value, create);
        } else if (value instanceof BNode) {
            data = this.bnode2data((BNode)value, create);
        } else if (value instanceof Literal) {
            data = this.literal2data((Literal)value, create);
        } else {
            throw new IllegalArgumentException("value parameter should be a URI, BNode or Literal");
        }
        if (logger.isDebugEnabled()) {
            logger.debug("value2data thread={} value={} create={} summary={}", new Object[]{ValueStore.threadName(), this.describeValue(value), create, ValueStore.summarize(data)});
        }
        return data;
    }

    private byte[] uri2data(IRI uri, boolean create) throws IOException {
        int nsID = this.getNamespaceID(uri.getNamespace(), create);
        if (logger.isDebugEnabled()) {
            logger.debug("uri2data thread={} namespace='{}' nsId={} create={}", new Object[]{ValueStore.threadName(), uri.getNamespace(), nsID, create});
        }
        if (nsID == -1) {
            return null;
        }
        byte[] localNameData = uri.getLocalName().getBytes(StandardCharsets.UTF_8);
        byte[] uriData = new byte[5 + localNameData.length];
        uriData[0] = 1;
        ByteArrayUtil.putInt(nsID, uriData, 1);
        ByteArrayUtil.put(localNameData, uriData, 5);
        if (logger.isDebugEnabled()) {
            logger.debug("uri2data produced len={} summary={} thread={}", new Object[]{uriData.length, ValueStore.summarize(uriData), ValueStore.threadName()});
        }
        return uriData;
    }

    private byte[] bnode2data(BNode bNode, boolean create) {
        byte[] idData = bNode.getID().getBytes(StandardCharsets.UTF_8);
        byte[] bNodeData = new byte[1 + idData.length];
        bNodeData[0] = 2;
        ByteArrayUtil.put(idData, bNodeData, 1);
        return bNodeData;
    }

    private byte[] literal2data(Literal literal, boolean create) throws IOException {
        return this.literal2data(literal.getLabel(), literal.getLanguage(), literal.getDatatype(), create);
    }

    private byte[] literal2legacy(Literal literal) throws IOException {
        IRI dt = literal.getDatatype();
        if (XSD.STRING.equals(dt) || RDF.LANGSTRING.equals(dt)) {
            return this.literal2data(literal.getLabel(), literal.getLanguage(), null, false);
        }
        return this.literal2data(literal.getLabel(), literal.getLanguage(), dt, false);
    }

    private byte[] literal2data(String label, Optional<String> lang, IRI dt, boolean create) throws IOException, UnsupportedEncodingException {
        int datatypeID = -1;
        if (create) {
            datatypeID = this.storeValue(dt);
        } else if (dt != null && (datatypeID = this.getID(dt)) == -1) {
            return null;
        }
        if (logger.isDebugEnabled()) {
            logger.debug("literal2data thread={} valueLength={} langPresent={} datatype={} datatypeId={} create={}", new Object[]{ValueStore.threadName(), label.length(), lang.isPresent(), dt, datatypeID, create});
        }
        byte[] langData = null;
        int langDataLength = 0;
        if (lang.isPresent() && (langDataLength = (langData = lang.get().getBytes(StandardCharsets.UTF_8)).length) > 255) {
            throw new IllegalArgumentException("Language tag too long (length " + langDataLength + " > maximum 255): " + lang.get());
        }
        byte[] labelData = label.getBytes(StandardCharsets.UTF_8);
        byte[] literalData = new byte[6 + langDataLength + labelData.length];
        literalData[0] = 3;
        ByteArrayUtil.putInt(datatypeID, literalData, 1);
        literalData[5] = (byte)(langDataLength & 0xFF);
        if (langData != null) {
            ByteArrayUtil.put(langData, literalData, 6);
        }
        ByteArrayUtil.put(labelData, literalData, 6 + langDataLength);
        if (logger.isDebugEnabled()) {
            logger.debug("literal2data produced len={} summary={} thread={}", new Object[]{literalData.length, ValueStore.summarize(literalData), ValueStore.threadName()});
        }
        return literalData;
    }

    private boolean isNamespaceData(byte[] data) {
        return data[0] != 1 && data[0] != 2 && data[0] != 3;
    }

    @InternalUseOnly
    public NativeValue data2value(int id, byte[] data) throws IOException {
        if (data.length == 0) {
            if (NativeStore.SOFT_FAIL_ON_CORRUPT_DATA_AND_REPAIR_INDEXES) {
                logger.error("Soft fail on corrupt data: Empty data array for value with id {}", (Object)id);
                CorruptUnknownValue v = new CorruptUnknownValue(this.revision, id, data);
                this.tryRecoverFromWal(id, v);
                return v;
            }
            throw new SailException("Empty data array for value with id " + id + " consider setting the system property org.eclipse.rdf4j.sail.nativerdf.softFailOnCorruptDataAndRepairIndexes to true");
        }
        switch (data[0]) {
            case 1: {
                return (NativeValue)this.data2uri(id, data);
            }
            case 2: {
                return this.data2bnode(id, data);
            }
            case 3: {
                return this.data2literal(id, data);
            }
        }
        if (NativeStore.SOFT_FAIL_ON_CORRUPT_DATA_AND_REPAIR_INDEXES) {
            logger.error("Soft fail on corrupt data: Invalid type {} for value with id {}", (Object)data[0], (Object)id);
            CorruptUnknownValue v = new CorruptUnknownValue(this.revision, id, data);
            this.tryRecoverFromWal(id, v);
            return v;
        }
        throw new SailException("Invalid type " + data[0] + " for value with id " + id + "  consider setting the system property org.eclipse.rdf4j.sail.nativerdf.softFailOnCorruptDataAndRepairIndexes to true");
    }

    private <T extends IRI & NativeValue> T data2uri(int id, byte[] data) throws IOException {
        String namespace = null;
        try {
            int nsID = ByteArrayUtil.getInt(data, 1);
            namespace = this.getNamespace(nsID);
            String localName = new String(data, 5, data.length - 5, StandardCharsets.UTF_8);
            return (T)new NativeIRI(this.revision, namespace, localName, id);
        }
        catch (Throwable e) {
            if (NativeStore.SOFT_FAIL_ON_CORRUPT_DATA_AND_REPAIR_INDEXES && (e instanceof Exception || e instanceof AssertionError)) {
                CorruptIRI v = new CorruptIRI(this.revision, id, namespace, data);
                this.tryRecoverFromWal(id, v);
                return (T)v;
            }
            logger.warn("NativeStore is possibly corrupt. To attempt to repair or retrieve the data, read the documentation on http://rdf4j.org about the system property org.eclipse.rdf4j.sail.nativerdf.softFailOnCorruptDataAndRepairIndexes");
            throw e;
        }
    }

    private NativeBNode data2bnode(int id, byte[] data) {
        String nodeID = new String(data, 1, data.length - 1, StandardCharsets.UTF_8);
        return new NativeBNode(this.revision, nodeID, id);
    }

    private <T extends NativeValue & Literal> T data2literal(int id, byte[] data) throws IOException {
        try {
            int datatypeID = ByteArrayUtil.getInt(data, 1);
            IRI datatype = null;
            if (datatypeID != -1) {
                datatype = (IRI)((Object)this.getValue(datatypeID));
            }
            String lang = null;
            int langLength = data[5] & 0xFF;
            if (langLength > 0) {
                lang = new String(data, 6, langLength, StandardCharsets.UTF_8);
            }
            String label = new String(data, 6 + langLength, data.length - 6 - langLength, StandardCharsets.UTF_8);
            if (lang != null) {
                return (T)new NativeLiteral(this.revision, label, lang, id);
            }
            if (datatype != null) {
                return (T)new NativeLiteral(this.revision, label, datatype, id);
            }
            return (T)new NativeLiteral(this.revision, label, CoreDatatype.XSD.STRING, id);
        }
        catch (Throwable e) {
            if (NativeStore.SOFT_FAIL_ON_CORRUPT_DATA_AND_REPAIR_INDEXES && (e instanceof Exception || e instanceof AssertionError)) {
                CorruptLiteral v = new CorruptLiteral(this.revision, id, data);
                this.tryRecoverFromWal(id, v);
                return (T)v;
            }
            throw e;
        }
    }

    private void tryRecoverFromWal(int id, CorruptValue holder) {
        NativeValue recovered = this.recoverValueFromWal(id);
        if (recovered != null) {
            holder.setRecovered(recovered);
        }
    }

    private NativeValue recoverValueFromWal(int id) {
        return this.recoverValueFromWal(id, true);
    }

    private NativeValue recoverValueFromWal(int id, boolean log) {
        ValueStoreWalSearch search = this.getOrCreateWalSearch();
        if (search == null) {
            return null;
        }
        try {
            Value v = search.findValueById(id);
            if (v == null) {
                return null;
            }
            NativeValue nv = this.getNativeValue(v);
            if (nv != null) {
                nv.setInternalID(id, this.revision);
                if (log) {
                    this.logRecovered(id, nv);
                    this.logWalRepairHint(id);
                }
                return nv;
            }
        }
        catch (IOException iOException) {
            // empty catch block
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private ValueStoreWalSearch getOrCreateWalSearch() {
        if (this.wal == null) {
            return null;
        }
        ValueStoreWalSearch search = this.walSearch;
        if (search != null) {
            return search;
        }
        ValueStore valueStore = this;
        synchronized (valueStore) {
            search = this.walSearch;
            if (search == null) {
                this.walSearch = search = ValueStoreWalSearch.open(this.wal.config());
            }
            return search;
        }
    }

    private boolean shouldValidateAgainstWal() {
        return this.walEnabled() && NativeStore.SOFT_FAIL_ON_CORRUPT_DATA_AND_REPAIR_INDEXES;
    }

    private boolean valuesMatch(NativeValue storeValue, NativeValue walValue) {
        if (storeValue == walValue) {
            return true;
        }
        if (storeValue == null || walValue == null) {
            return false;
        }
        if (storeValue instanceof Literal && walValue instanceof Literal) {
            Literal a = (Literal)((Object)storeValue);
            Literal b = (Literal)((Object)walValue);
            return Objects.equals(a.getLabel(), b.getLabel()) && Objects.equals(a.getLanguage().orElse(null), b.getLanguage().orElse(null)) && Objects.equals(this.datatypeIri(a), this.datatypeIri(b));
        }
        if (storeValue instanceof IRI && walValue instanceof IRI) {
            return Objects.equals(storeValue.stringValue(), walValue.stringValue());
        }
        if (storeValue instanceof BNode && walValue instanceof BNode) {
            return Objects.equals(storeValue.stringValue(), walValue.stringValue());
        }
        return Objects.equals(storeValue.stringValue(), walValue.stringValue());
    }

    private String datatypeIri(Literal literal) {
        return literal.getDatatype() == null ? "" : literal.getDatatype().stringValue();
    }

    private void logRecovered(int id, NativeValue nv) {
        switch (WAL_RECOVERY_LOG) {
            case "trace": {
                if (!logger.isTraceEnabled()) break;
                logger.trace("Recovered value for id {} from WAL as {}", (Object)id, (Object)nv.stringValue());
                break;
            }
            case "debug": {
                if (!logger.isDebugEnabled()) break;
                logger.debug("Recovered value for id {} from WAL as {}", (Object)id, (Object)nv.stringValue());
                break;
            }
        }
    }

    private void logWalRepairHint(int id) {
        logger.error("ValueStore {} recovered value id {} from WAL because the values.* files are corrupt. Enable NativeStore#setWalAutoRecoverOnOpen(true) (config:native.walAutoRecoverOnOpen) and restart, or run ValueStoreWalRecovery to replay the WAL and rebuild values.dat/values.id/values.hash so the on-disk data matches the WAL again.", (Object)this.dataDir, (Object)id);
    }

    private NativeValue fromWalRecord(ValueStoreWalRecord rec) {
        switch (rec.valueKind()) {
            case IRI: {
                return this.createIRI(rec.lexical());
            }
            case BNODE: {
                return this.createBNode(rec.lexical());
            }
            case LITERAL: {
                String lang = rec.language();
                String dt = rec.datatype();
                if (lang != null && !lang.isEmpty()) {
                    return this.createLiteral(rec.lexical(), lang);
                }
                if (dt != null && !dt.isEmpty()) {
                    return this.createLiteral(rec.lexical(), this.createIRI(dt));
                }
                return this.createLiteral(rec.lexical());
            }
            case NAMESPACE: {
                return null;
            }
        }
        return null;
    }

    private String data2namespace(byte[] data) {
        return new String(data, StandardCharsets.UTF_8);
    }

    private int getNamespaceID(String namespace, boolean create) throws IOException {
        int id;
        Integer cacheID;
        if (logger.isDebugEnabled()) {
            logger.debug("getNamespaceID thread={} namespace='{}' create={}", new Object[]{ValueStore.threadName(), namespace, create});
        }
        if ((cacheID = this.namespaceIDCache.get(namespace)) != null) {
            if (logger.isDebugEnabled()) {
                logger.debug("getNamespaceID cache hit namespace='{}' id={} thread={}", new Object[]{namespace, cacheID, ValueStore.threadName()});
            }
            return cacheID;
        }
        byte[] namespaceData = namespace.getBytes(StandardCharsets.UTF_8);
        if (create) {
            int previousMaxID = this.walEnabled() ? this.dataStore.getMaxID() : 0;
            id = this.dataStore.storeData(namespaceData);
            if (this.walEnabled() && id > previousMaxID) {
                this.logNamespaceMint(id, namespace);
            }
        } else {
            id = this.dataStore.getID(namespaceData);
        }
        if (id != -1) {
            this.namespaceIDCache.put(namespace, id);
            if (logger.isDebugEnabled()) {
                logger.debug("getNamespaceID resolved namespace='{}' id={} thread={}", new Object[]{namespace, id, ValueStore.threadName()});
            }
        } else if (logger.isDebugEnabled()) {
            logger.debug("getNamespaceID unresolved namespace='{}' thread={}", (Object)namespace, (Object)ValueStore.threadName());
        }
        return id;
    }

    public OptionalLong drainPendingWalHighWaterMark() {
        if (this.walPendingLsn == null) {
            return OptionalLong.empty();
        }
        long lsn = this.walPendingLsn.get();
        if (lsn <= -1L) {
            return OptionalLong.empty();
        }
        this.walPendingLsn.set(-1L);
        return OptionalLong.of(lsn);
    }

    public void awaitWalDurable(long lsn) throws IOException {
        if (!this.walEnabled() || lsn <= -1L) {
            return;
        }
        try {
            this.wal.awaitDurable(lsn);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IOException("Interrupted while awaiting WAL durability", e);
        }
    }

    private void logMintedValue(int id, Value value) throws IOException {
        ValueStoreWalDescription description = this.describeValue(value);
        int hash = this.computeWalHash(description.kind, description.lexical, description.datatype, description.language);
        long lsn = this.wal.logMint(id, description.kind, description.lexical, description.datatype, description.language, hash);
        this.recordWalLsn(lsn);
    }

    private void logNamespaceMint(int id, String namespace) throws IOException {
        int hash = this.computeWalHash(ValueStoreWalValueKind.NAMESPACE, namespace, "", "");
        long lsn = this.wal.logMint(id, ValueStoreWalValueKind.NAMESPACE, namespace, "", "", hash);
        this.recordWalLsn(lsn);
    }

    private void maybeScheduleWalBootstrap() {
        boolean needsBootstrap;
        if (!this.walEnabled()) {
            return;
        }
        int maxId = this.dataStore.getMaxID();
        if (maxId <= 0) {
            return;
        }
        boolean bl = needsBootstrap = !this.wal.hasInitialSegments() || this.walNeedsBootstrap(maxId);
        if (!needsBootstrap) {
            return;
        }
        boolean syncBootstrap = false;
        try {
            syncBootstrap = this.wal.config().syncBootstrapOnOpen();
        }
        catch (Throwable throwable2) {
            // empty catch block
        }
        if (syncBootstrap) {
            this.rebuildWalFromExistingValues(maxId);
        } else {
            if (this.walBootstrapFuture != null) {
                return;
            }
            CompletableFuture<Void> future = CompletableFuture.runAsync(() -> this.rebuildWalFromExistingValues(maxId));
            this.walBootstrapFuture = future;
            future.whenComplete((unused, throwable) -> {
                if (throwable != null) {
                    logger.warn("ValueStore WAL bootstrap failed", throwable);
                }
            });
        }
    }

    private void rebuildWalFromExistingValues(int maxId) {
        try {
            OptionalLong pending;
            for (int id = 1; id <= maxId; ++id) {
                byte[] data;
                if (Thread.currentThread().isInterrupted()) {
                    Thread.currentThread().interrupt();
                    return;
                }
                if (this.wal.isClosed()) {
                    return;
                }
                try {
                    data = this.dataStore.getData(id);
                }
                catch (IOException e) {
                    logger.warn("Failed to read value {} while rebuilding WAL", (Object)id, (Object)e);
                    continue;
                }
                if (data == null) continue;
                try {
                    if (this.isNamespaceData(data)) {
                        String namespace = this.data2namespace(data);
                        this.logNamespaceMint(id, namespace);
                        continue;
                    }
                    NativeValue value = this.data2value(id, data);
                    if (value == null) continue;
                    this.logMintedValue(id, value);
                    continue;
                }
                catch (IOException e) {
                    if (this.wal.isClosed()) {
                        return;
                    }
                    logger.warn("Failed to rebuild WAL entry for id {}", (Object)id, (Object)e);
                    continue;
                }
                catch (RuntimeException e) {
                    logger.warn("Unexpected failure while rebuilding WAL entry for id {}", (Object)id, (Object)e);
                }
            }
            if (!this.wal.isClosed() && (pending = this.drainPendingWalHighWaterMark()).isPresent()) {
                this.awaitWalDurable(pending.getAsLong());
            }
        }
        catch (Throwable t) {
            logger.warn("Error while rebuilding ValueStore WAL", t);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private boolean walNeedsBootstrap(int maxId) {
        try (ValueStoreWalReader reader = ValueStoreWalReader.open(this.wal.config());){
            ValueStoreWalRecovery recovery = new ValueStoreWalRecovery();
            ValueStoreWalRecovery.ReplayReport report = recovery.replayWithReport(reader);
            Map<Integer, ValueStoreWalRecord> dict = report.dictionary();
            if (dict.isEmpty()) {
                boolean bl = true;
                return bl;
            }
            if (!report.complete()) {
                boolean bl = true;
                return bl;
            }
            for (int id = 1; id <= maxId; ++id) {
                if (dict.containsKey(id)) continue;
                boolean bl = true;
                return bl;
            }
            boolean bl = false;
            return bl;
        }
        catch (IOException e) {
            return false;
        }
    }

    private void recordWalLsn(long lsn) {
        if (this.walPendingLsn == null || lsn <= -1L) {
            return;
        }
        long current = this.walPendingLsn.get();
        if (lsn > current) {
            this.walPendingLsn.set(lsn);
        }
    }

    private ValueStoreWalDescription describeValue(Value value) {
        if (value instanceof IRI) {
            return new ValueStoreWalDescription(ValueStoreWalValueKind.IRI, value.stringValue(), "", "");
        }
        if (value instanceof BNode) {
            return new ValueStoreWalDescription(ValueStoreWalValueKind.BNODE, value.stringValue(), "", "");
        }
        if (value instanceof Literal) {
            Literal literal = (Literal)value;
            String lang = literal.getLanguage().orElse("");
            String datatype = literal.getDatatype() != null ? literal.getDatatype().stringValue() : "";
            return new ValueStoreWalDescription(ValueStoreWalValueKind.LITERAL, literal.getLabel(), datatype, lang);
        }
        throw new IllegalArgumentException("value parameter should be a URI, BNode or Literal");
    }

    private int computeWalHash(ValueStoreWalValueKind kind, String lexical, String datatype, String language) {
        CRC32C crc32c = CRC32C_HOLDER.get();
        crc32c.reset();
        crc32c.update((byte)kind.code());
        this.updateCrc(crc32c, lexical);
        crc32c.update(0);
        this.updateCrc(crc32c, datatype);
        crc32c.update(0);
        this.updateCrc(crc32c, language);
        return (int)crc32c.getValue();
    }

    private void updateCrc(CRC32C crc32c, String value) {
        if (value == null || value.isEmpty()) {
            return;
        }
        byte[] bytes = value.getBytes(StandardCharsets.UTF_8);
        crc32c.update(bytes, 0, bytes.length);
    }

    private boolean walEnabled() {
        return this.wal != null;
    }

    private String getNamespace(int id) throws IOException {
        Integer cacheID = id;
        String namespace = this.namespaceCache.get(cacheID);
        if (namespace == null) {
            try {
                byte[] namespaceData = this.dataStore.getData(id);
                namespace = this.data2namespace(namespaceData);
            }
            catch (RecoveredDataException rde) {
                namespace = this.data2namespace(rde.getData());
            }
            this.namespaceCache.put(cacheID, namespace);
        }
        return namespace;
    }

    @Override
    public NativeIRI createIRI(String uri) {
        return new NativeIRI(this.revision, uri);
    }

    @Override
    public NativeIRI createIRI(String namespace, String localName) {
        return new NativeIRI(this.revision, namespace, localName);
    }

    @Override
    public NativeBNode createBNode(String nodeID) {
        return new NativeBNode(this.revision, nodeID);
    }

    @Override
    public NativeLiteral createLiteral(String value) {
        return new NativeLiteral(this.revision, value, (CoreDatatype)CoreDatatype.XSD.STRING);
    }

    @Override
    public NativeLiteral createLiteral(String value, String language) {
        return new NativeLiteral(this.revision, value, language);
    }

    @Override
    public NativeLiteral createLiteral(String value, IRI datatype) {
        return new NativeLiteral(this.revision, value, datatype);
    }

    public NativeValue getNativeValue(Value value) {
        if (value instanceof Resource) {
            return this.getNativeResource((Resource)value);
        }
        if (value instanceof Literal) {
            return this.getNativeLiteral((Literal)value);
        }
        throw new IllegalArgumentException("Unknown value type: " + String.valueOf(value.getClass()));
    }

    public NativeResource getNativeResource(Resource resource) {
        if (resource instanceof IRI) {
            return this.getNativeURI((IRI)resource);
        }
        if (resource instanceof BNode) {
            return this.getNativeBNode((BNode)resource);
        }
        throw new IllegalArgumentException("Unknown resource type: " + String.valueOf(resource.getClass()));
    }

    public NativeIRI getNativeURI(IRI uri) {
        if (this.isOwnValue(uri)) {
            return (NativeIRI)uri;
        }
        return new NativeIRI(this.revision, uri.toString());
    }

    public NativeBNode getNativeBNode(BNode bnode) {
        if (this.isOwnValue(bnode)) {
            return (NativeBNode)bnode;
        }
        return new NativeBNode(this.revision, bnode.getID());
    }

    public NativeLiteral getNativeLiteral(Literal l) {
        if (this.isOwnValue(l)) {
            return (NativeLiteral)l;
        }
        if (Literals.isLanguageLiteral(l)) {
            return new NativeLiteral(this.revision, l.getLabel(), l.getLanguage().get());
        }
        NativeIRI datatype = this.getNativeURI(l.getDatatype());
        return new NativeLiteral(this.revision, l.getLabel(), datatype);
    }

    public static void main(String[] args) throws Exception {
        File dataDir = new File(args[0]);
        ValueStore valueStore = new ValueStore(dataDir);
        int maxID = valueStore.dataStore.getMaxID();
        for (int id = 1; id <= maxID; ++id) {
            try {
                byte[] data = valueStore.dataStore.getData(id);
                if (valueStore.isNamespaceData(data)) {
                    String ns = valueStore.data2namespace(data);
                    System.out.println("[" + id + "] " + ns);
                    continue;
                }
                NativeValue value = valueStore.data2value(id, data);
                System.out.println("[" + id + "] " + value.toString());
                continue;
            }
            catch (RecoveredDataException rde) {
                System.out.println("[" + id + "] CorruptUnknownValue:" + String.valueOf(new CorruptUnknownValue(valueStore.revision, id, rde.getData())));
            }
        }
    }

    private void autoRecoverValueStoreIfConfigured() {
        ValueStoreWalConfig config;
        if (this.wal == null) {
            return;
        }
        try {
            config = this.wal.config();
        }
        catch (Throwable t) {
            logger.warn("ValueStore WAL configuration unavailable for {}", (Object)this.dataDir, (Object)t);
            return;
        }
        if (!config.recoverValueStoreOnOpen()) {
            return;
        }
        try {
            ValueStoreWalRecovery.ReplayReport report;
            ValueStoreWalRecovery recovery = new ValueStoreWalRecovery();
            try (ValueStoreWalReader reader = ValueStoreWalReader.open(config);){
                report = recovery.replayWithReport(reader);
            }
            Map<Integer, ValueStoreWalRecord> dictionary = report.dictionary();
            if (dictionary.isEmpty()) {
                return;
            }
            if (!report.complete()) {
                logger.warn("Skipping ValueStore WAL recovery for {}: WAL segments incomplete", (Object)this.dataDir);
                return;
            }
            if (this.hasDictionaryGaps(dictionary)) {
                logger.warn("Skipping ValueStore WAL recovery for {}: WAL dictionary has gaps", (Object)this.dataDir);
                return;
            }
            if (!this.shouldRecoverFromWalDictionary(dictionary)) {
                return;
            }
            this.recoverValueStoreFromWal(dictionary);
            this.logAutoRecovery(dictionary.size());
        }
        catch (IOException e) {
            logger.warn("ValueStore WAL recovery failed for {}", (Object)this.dataDir, (Object)e);
        }
    }

    private boolean hasDictionaryGaps(Map<Integer, ValueStoreWalRecord> dictionary) {
        int maxId = dictionary.keySet().stream().mapToInt(Integer::intValue).max().orElse(0);
        if (maxId <= 0) {
            return false;
        }
        if (dictionary.size() == maxId) {
            return false;
        }
        for (int expected = 1; expected <= maxId; ++expected) {
            if (dictionary.containsKey(expected)) continue;
            return true;
        }
        return false;
    }

    private boolean shouldRecoverFromWalDictionary(Map<Integer, ValueStoreWalRecord> dictionary) {
        int maxWalId = dictionary.keySet().stream().mapToInt(Integer::intValue).max().orElse(0);
        if (maxWalId <= 0) {
            return false;
        }
        int currentMaxId = this.dataStore.getMaxID();
        if (currentMaxId == 0 && maxWalId > 0) {
            return true;
        }
        if (currentMaxId < maxWalId) {
            return true;
        }
        ArrayList<Integer> ids = new ArrayList<Integer>(dictionary.keySet());
        if (ids.isEmpty()) {
            return false;
        }
        ids.sort(Integer::compareTo);
        for (Integer id : ids) {
            if (!this.isMissingValueData(id)) continue;
            return true;
        }
        return false;
    }

    private boolean isMissingValueData(int id) {
        if (id <= 0) {
            return false;
        }
        try {
            byte[] data = this.dataStore.getData(id);
            return data == null || data.length == 0;
        }
        catch (IOException e) {
            return true;
        }
    }

    private void recoverValueStoreFromWal(Map<Integer, ValueStoreWalRecord> dictionary) throws IOException {
        this.dataStore.clear();
        this.valueCache.clear();
        this.valueIDCache.clear();
        this.namespaceCache.clear();
        this.namespaceIDCache.clear();
        List entries = dictionary.entrySet().stream().sorted(Map.Entry.comparingByKey(Comparator.naturalOrder())).collect(Collectors.toList());
        block6: for (Map.Entry entry : entries) {
            int assigned;
            byte[] data;
            ValueStoreWalRecord record = (ValueStoreWalRecord)entry.getValue();
            switch (record.valueKind()) {
                case NAMESPACE: {
                    data = record.lexical().getBytes(StandardCharsets.UTF_8);
                    break;
                }
                case IRI: {
                    data = this.encodeIri(record.lexical(), this.dataStore);
                    break;
                }
                case BNODE: {
                    byte[] idBytes = record.lexical().getBytes(StandardCharsets.UTF_8);
                    data = new byte[1 + idBytes.length];
                    data[0] = 2;
                    ByteArrayUtil.put(idBytes, data, 1);
                    break;
                }
                case LITERAL: {
                    data = this.encodeLiteral(record.lexical(), record.datatype(), record.language(), this.dataStore);
                    break;
                }
                default: {
                    continue block6;
                }
            }
            if (data == null || (assigned = this.dataStore.storeData(data)) == record.id()) continue;
            throw new IOException("ValueStore WAL recovery produced mismatched id " + assigned + " (expected " + record.id() + ")");
        }
        this.dataStore.sync();
    }

    private void logAutoRecovery(int recoveredCount) {
        switch (WAL_RECOVERY_LOG) {
            case "trace": {
                if (!logger.isTraceEnabled()) break;
                logger.trace("Recovered {} ValueStore entries from WAL for {}", (Object)recoveredCount, (Object)this.dataDir);
                break;
            }
            case "debug": {
                if (!logger.isDebugEnabled()) break;
                logger.debug("Recovered {} ValueStore entries from WAL for {}", (Object)recoveredCount, (Object)this.dataDir);
                break;
            }
        }
    }

    private byte[] encodeIri(String lexical, DataStore ds) throws IOException {
        NativeIRI iri = this.createIRI(lexical);
        String ns = iri.getNamespace();
        String local = iri.getLocalName();
        int nsId = ds.getID(ns.getBytes(StandardCharsets.UTF_8));
        if (nsId == -1) {
            nsId = ds.storeData(ns.getBytes(StandardCharsets.UTF_8));
        }
        byte[] localBytes = local.getBytes(StandardCharsets.UTF_8);
        byte[] data = new byte[5 + localBytes.length];
        data[0] = 1;
        ByteArrayUtil.putInt(nsId, data, 1);
        ByteArrayUtil.put(localBytes, data, 5);
        return data;
    }

    private byte[] encodeLiteral(String label, String datatype, String language, DataStore ds) throws IOException {
        int dtId = -1;
        if (datatype != null && !datatype.isEmpty()) {
            byte[] dtBytes = this.encodeIri(datatype, ds);
            int id = ds.getID(dtBytes);
            dtId = id == -1 ? ds.storeData(dtBytes) : id;
        }
        byte[] langBytes = language == null ? new byte[]{} : language.getBytes(StandardCharsets.UTF_8);
        byte[] labelBytes = label.getBytes(StandardCharsets.UTF_8);
        byte[] data = new byte[6 + langBytes.length + labelBytes.length];
        data[0] = 3;
        ByteArrayUtil.putInt(dtId, data, 1);
        data[5] = (byte)(langBytes.length & 0xFF);
        if (langBytes.length > 0) {
            ByteArrayUtil.put(langBytes, data, 6);
        }
        ByteArrayUtil.put(labelBytes, data, 6 + langBytes.length);
        return data;
    }

    private static final class ValueStoreWalDescription {
        final ValueStoreWalValueKind kind;
        final String lexical;
        final String datatype;
        final String language;

        ValueStoreWalDescription(ValueStoreWalValueKind kind, String lexical, String datatype, String language) {
            this.kind = kind;
            this.lexical = lexical == null ? "" : lexical;
            this.datatype = datatype == null ? "" : datatype;
            this.language = language == null ? "" : language;
        }
    }
}

