/*
 * Copyright (c) 2021, 2026 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.lsat.timing.util;

import java.io.File;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.stream.Collectors;

import javax.swing.BoxLayout;
import javax.swing.JCheckBox;
import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;

import org.eclipse.lsat.motioncalculator.TimePosition;
import org.jfree.chart.ChartPanel;
import org.jfree.data.xy.XYDataset;

/**
 * CsvExporter handles exporting XY chart data to CSV files.
 * It allows selecting which series to export, setting a custom time step (dt),
 * and automatically interpolates values for aligning the timestamps of different axis.
 */
public class ChartCsvExporter {

    /**
     * Method to export chart data to CSV.
     *
     * @param chartPanel The ChartPanel containing the plot
     * @param datasetsMap Map of dataset indices to XYDataset objects
     * @param chartTitle Title of the chart
     */
    public void exportChart(ChartPanel chartPanel, Map<Integer, XYDataset> datasetsMap, String chartTitle) {
        if (datasetsMap == null || datasetsMap.isEmpty()) {
            JOptionPane.showMessageDialog(chartPanel, "No datasets available to export.");
            return;
        }

        // Step 1: select series
        List<String> seriesNames = getAllSeriesNames(datasetsMap);
        if (seriesNames.isEmpty()) {
            JOptionPane.showMessageDialog(chartPanel, "No series available to export.");
            return;
        }

        List<String> selectedSeries = selectSeries(chartPanel, seriesNames);
        if (selectedSeries.isEmpty()) {
            return; // user cancelled or nothing selected
        }

        // Step 2: compute automatic dt and allow user to override
        double autoDt = computeDt(selectedSeries, datasetsMap);
        double dt = promptForDt(chartPanel, autoDt);
        if (dt <= 0) {
            return; // user cancelled
        }

        // Step 3: choose file
        File file = chooseCsvFile(chartPanel, chartTitle);
        if (file == null) {
            return; // user cancelled
        }

        // Step 4: export
        try (PrintWriter pw = new PrintWriter(file)) {
            exportSelectedSeries(pw, selectedSeries, datasetsMap, dt);
            JOptionPane.showMessageDialog(chartPanel,
                    "Export completed:\n" + file.getAbsolutePath());
        } catch (Exception ex) {
            ex.printStackTrace();
            JOptionPane.showMessageDialog(chartPanel,
                    "Error exporting CSV:\n" + ex.getMessage(), "Error", JOptionPane.ERROR_MESSAGE);
        }
    }

    /** Collect all series names from all datasets */
    private List<String> getAllSeriesNames(Map<Integer, XYDataset> datasetsMap) {
        List<String> names = new ArrayList<>();
        for (XYDataset ds : datasetsMap.values()) {
            if (ds == null) {
                continue;
            }
            for (int i = 0; i < ds.getSeriesCount(); i++) {
                names.add(ds.getSeriesKey(i).toString());
            }
        }
        return names;
    }

    /** Show a selection dialog for the user to pick series to export */
    private List<String> selectSeries(ChartPanel chartPanel, List<String> seriesNames) {
        JPanel panel = new JPanel();
        panel.setLayout(new BoxLayout(panel, BoxLayout.Y_AXIS));
        List<JCheckBox> checkboxes = new ArrayList<>();
        for (String name : seriesNames) {
            JCheckBox cb = new JCheckBox(name, true);
            checkboxes.add(cb);
            panel.add(cb);
        }

        int result = JOptionPane.showConfirmDialog(
                chartPanel,
                new JScrollPane(panel),
                "Select series to export",
                JOptionPane.OK_CANCEL_OPTION,
                JOptionPane.PLAIN_MESSAGE
        );

        if (result != JOptionPane.OK_OPTION) {
            return Collections.emptyList();
        }

        return checkboxes.stream()
                .filter(JCheckBox::isSelected)
                .map(JCheckBox::getText)
                .collect(Collectors.toList());
    }

    /** Compute an dt based on minimum spacing of selected series */
    private double computeDt(List<String> selectedSeries, Map<Integer, XYDataset> datasetsMap) {
        double minDeltaT = Double.MAX_VALUE;

        // Iterate over all datasets
        for (XYDataset ds : datasetsMap.values()) {
            if (ds == null) {
                continue;
            }

            // Iterate over all series in the dataset
            for (int i = 0; i < ds.getSeriesCount(); i++) {
                String name = ds.getSeriesKey(i).toString();

                // Skip series that are not selected
                if (!selectedSeries.contains(name)) {
                    continue;
                }

                // Check spacing between consecutive points in the series
                for (int j = 1; j < ds.getItemCount(i); j++) {
                    double delta = ds.getXValue(i, j) - ds.getXValue(i, j - 1);
                    if (delta > 0) {
                        minDeltaT = Math.min(minDeltaT, delta);
                    }
                }
            }
        }

        return minDeltaT;
    }

    /** Prompt user to enter dt, defaulting to auto-calculated value */
    private double promptForDt(ChartPanel chartPanel, double autoDt) {
        String input = JOptionPane.showInputDialog(
                chartPanel,
                "Enter export time step (dt):",
                String.format(Locale.ROOT, "%.6f", autoDt)
        );

        if (input == null) {
            return -1;
        }

        try {
            double dt = Double.parseDouble(input);
            if (dt <= 0) {
                throw new NumberFormatException();
            }
            return dt;
        } catch (NumberFormatException ex) {
            JOptionPane.showMessageDialog(chartPanel, "Invalid dt value. Using recommended: " + autoDt);
            return autoDt;
        }
    }

    /** Show file chooser dialog to save CSV */
    private File chooseCsvFile(ChartPanel chartPanel, String chartTitle) {
        JFileChooser chooser = new JFileChooser();
        chooser.setDialogTitle("Save CSV file");
        chooser.setSelectedFile(new File(chartTitle + " export.csv"));

        while (true) {
            int result = chooser.showSaveDialog(chartPanel);
            if (result != JFileChooser.APPROVE_OPTION) {
                return null; // user cancelled
            }

            File file = chooser.getSelectedFile();
            if (file == null) {
                return null;
            }

            file = new File(file.getParentFile(), file.getName());

            if (!file.exists()) {
                return file;
            }

            int overwrite = JOptionPane.showConfirmDialog(
                    chartPanel,
                    "The file already exists:\n" + file.getAbsolutePath() +
                    "\n\nDo you want to overwrite it?",
                    "Confirm overwrite",
                    JOptionPane.YES_NO_CANCEL_OPTION,
                    JOptionPane.WARNING_MESSAGE
            );

            if (overwrite == JOptionPane.YES_OPTION) {
                return file; // user confirmed overwrite
            } else if (overwrite == JOptionPane.NO_OPTION) {
                // loop again to let them choose another name
                continue;
            } else {
                return null; // cancel
            }
        }
    }

    /** Export the selected series with interpolation at the chosen dt */
    private void exportSelectedSeries(PrintWriter pw,
                                      List<String> selectedSeries,
                                      Map<Integer, XYDataset> datasetsMap,
                                      double dt)
    {
        // Collect time-position data for all selected series
        Map<String, List<TimePosition>> axisData = new HashMap<>();
        double minTime = Double.MAX_VALUE;
        double maxTime = -Double.MAX_VALUE;

        for (XYDataset ds : datasetsMap.values()) {
            if (ds == null) {
                continue;
            }
            for (int i = 0; i < ds.getSeriesCount(); i++) {
                String name = ds.getSeriesKey(i).toString();
                if (!selectedSeries.contains(name)) {
                    continue;
                }

                List<TimePosition> list = new ArrayList<>();
                for (int j = 0; j < ds.getItemCount(i); j++) {
                    list.add(new TimePosition(ds.getXValue(i, j), ds.getYValue(i, j)));
                }
                list.sort(Comparator.comparingDouble(TimePosition::getTime));
                axisData.put(name, list);

                if (!list.isEmpty()) {
                    minTime = Math.min(minTime, list.get(0).getTime());
                    maxTime = Math.max(maxTime, list.get(list.size() - 1).getTime());
                }
            }
        }

        // Write CSV header
        pw.print("Time");
        selectedSeries.forEach(s -> pw.print("," + s));
        pw.println();

        // Export interpolated data
        for (double t = minTime; t <= maxTime; t += dt) {
            pw.print(String.format(Locale.ROOT, "%.6f", t));
            for (String s : selectedSeries) {
                pw.print("," + String.format(Locale.US, "%.6f", interpolate(axisData.get(s), t)));
            }
            pw.println();
        }
    }

    /**
     * Linear interpolation for a series of (time, position) points.
     * Given a target time t, this returns an interpolated position.
     */
    private double interpolate(List<TimePosition> list, double t) {

        // If the list is null or empty, nothing to interpolate
        if (list == null || list.isEmpty()) {
            return 0.0;
        }

        // If t is before or equal to the first point, return the first position
        if (t <= list.get(0).getTime()) {
            return list.get(0).getPosition();
        }

        // If t is after or equal to the last point, return the last position
        if (t >= list.get(list.size() - 1).getTime()) {
            return list.get(list.size() - 1).getPosition();
        }

        // Search for the interval prev, next such that prev.time <= t <= next.time
        for (int i = 1; i < list.size(); i++) {
            TimePosition prev = list.get(i - 1);
            TimePosition next = list.get(i);

            // Once t is within (or equal to) next.time, interpolate between prev and next
            if (t <= next.getTime()) {

                // Compute how far t is between prev.time and next.time
                double ratio = (t - prev.getTime()) / (next.getTime() - prev.getTime());

                // Linear interpolation of position
                return prev.getPosition() + ratio * (next.getPosition() - prev.getPosition());
            }
        }

        // Should never reach this point if data is valid
        return 0.0;
    }
}
