/*
 * Copyright (c) 2021, 2026 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 */
package distributions;

import org.apache.commons.math3.random.RandomGenerator;
import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.ResourceSet;

public class DistributionsAdapter extends AdapterImpl {
    private CalculationMode calculationMode;

    private RandomGenerator randomGenerator;

    private final ResourceSet resourceSet;

    /**
     * @param resourceSet
     */
    public DistributionsAdapter(ResourceSet resourceSet) {
        this.resourceSet = resourceSet;
    }

    public CalculationMode getCalculationMode() {
        return calculationMode;
    }

    public void setCalculationMode(CalculationMode mode) {
        if (mode != calculationMode) {
            if (mode != null) {
                resetDistributions();
            }
            this.calculationMode = mode;
        }
    }

    private void resetDistributions() {
        resetDistributions(resourceSet);
    }

    public RandomGenerator getRandomGenerator() {
        return randomGenerator;
    }

    public void setRandomGenerator(RandomGenerator randomGenerator) {
        if (randomGenerator != this.randomGenerator) {
            resetDistributions();
            this.randomGenerator = randomGenerator;
        }
    }

    /**
     * Get or create the adapter for the given context
     */
    public static DistributionsAdapter getAdapter(EObject context) {
        if (context == null || context.eResource() == null || context.eResource().getResourceSet() == null) {
            return null;
        }

        ResourceSet resourceSet = context.eResource().getResourceSet();
        return getAdapter(resourceSet);
    }

    /**
     * Get or create the adapter for the given resource set
     */
    public static DistributionsAdapter getAdapter(ResourceSet resourceSet) {
        if (resourceSet == null) {
            return null;
        }

        for (Adapter adapter: resourceSet.eAdapters()) {
            if (adapter instanceof DistributionsAdapter) {
                return (DistributionsAdapter)adapter;
            }
        }

        // Create and attach a new adapter
        DistributionsAdapter adapter = new DistributionsAdapter(resourceSet);
        resourceSet.eAdapters().add(adapter);
        return adapter;
    }

    public static void resetDistributions(ResourceSet root) {
        // Get a tree iterator that will traverse the entire containment hierarchy
        TreeIterator<Notifier> iterator = root.getAllContents();

        // Iterate through all objects in the model
        while (iterator.hasNext()) {
            var instance = iterator.next();

            // Check if the current object is an instance of the target class
            if (instance instanceof Distribution dist) {
                dist.setModeDistributionInstance(null);
            }
        }
    }
}
