package org.bouncycastle.asn1.x509;

import java.util.Enumeration;

import org.bouncycastle.asn1.ASN1EncodableVector;
import org.bouncycastle.asn1.ASN1Object;
import org.bouncycastle.asn1.ASN1Primitive;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.asn1.ASN1TaggedObject;
import org.bouncycastle.asn1.DERSequence;
import org.bouncycastle.asn1.DERTaggedObject;

/**
 * This class helps to support crossCerfificatePairs in a LDAP directory
 * according RFC 2587
 * 
 * <pre>
 *     crossCertificatePairATTRIBUTE::={
 *       WITH SYNTAX   CertificatePair
 *       EQUALITY MATCHING RULE certificatePairExactMatch
 *       ID joint-iso-ccitt(2) ds(5) attributeType(4) crossCertificatePair(40)}
 * </pre>
 * 
 * <blockquote> The forward elements of the crossCertificatePair attribute of a
 * CA's directory entry shall be used to store all, except self-issued
 * certificates issued to this CA. Optionally, the reverse elements of the
 * crossCertificatePair attribute, of a CA's directory entry may contain a
 * subset of certificates issued by this CA to other CAs. When both the forward
 * and the reverse elements are present in a single attribute value, issuer name
 * in one certificate shall match the subject name in the other and vice versa,
 * and the subject public key in one certificate shall be capable of verifying
 * the digital signature on the other certificate and vice versa.
 * 
 * When a reverse element is present, the forward element value and the reverse
 * element value need not be stored in the same attribute value; in other words,
 * they can be stored in either a single attribute value or two attribute
 * values. </blockquote>
 * 
 * <pre>
 *       CertificatePair ::= SEQUENCE {
 *         forward        [0]    Certificate OPTIONAL,
 *         reverse        [1]    Certificate OPTIONAL,
 *         -- at least one of the pair shall be present -- } 
 * </pre>
 */
public class CertificatePair
    extends ASN1Object
{
    private Certificate forward;

    private Certificate reverse;

    public static CertificatePair getInstance(Object obj)
    {
        if (obj == null || obj instanceof CertificatePair)
        {
            return (CertificatePair)obj;
        }

        if (obj instanceof ASN1Sequence)
        {
            return new CertificatePair((ASN1Sequence)obj);
        }

        throw new IllegalArgumentException("illegal object in getInstance: "
            + obj.getClass().getName());
    }

    /**
     * Constructor from ASN1Sequence.
     * <p>
     * The sequence is of type CertificatePair:
     * <pre>
     *       CertificatePair ::= SEQUENCE {
     *         forward        [0]    Certificate OPTIONAL,
     *         reverse        [1]    Certificate OPTIONAL,
     *         -- at least one of the pair shall be present -- }
     * </pre>
     * </p>
     * @param seq The ASN.1 sequence.
     */
    private CertificatePair(ASN1Sequence seq)
    {
        if (seq.size() != 1 && seq.size() != 2)
        {
            throw new IllegalArgumentException("Bad sequence size: "
                + seq.size());
        }

        Enumeration e = seq.getObjects();

        while (e.hasMoreElements())
        {
            ASN1TaggedObject o = ASN1TaggedObject.getInstance(e.nextElement());
            if (o.getTagNo() == 0)
            {
                forward = Certificate.getInstance(o, true);
            }
            else if (o.getTagNo() == 1)
            {
                reverse = Certificate.getInstance(o, true);
            }
            else
            {
                throw new IllegalArgumentException("Bad tag number: "
                    + o.getTagNo());
            }
        }
    }

    /**
     * Constructor from a given details.
     *
     * @param forward Certificates issued to this CA.
     * @param reverse Certificates issued by this CA to other CAs.
     */
    public CertificatePair(Certificate forward, Certificate reverse)
    {
        this.forward = forward;
        this.reverse = reverse;
    }

    /**
     * Produce an object suitable for an ASN1OutputStream.
     * <p>
     * Returns:
     * <pre>
     *       CertificatePair ::= SEQUENCE {
     *         forward        [0]    Certificate OPTIONAL,
     *         reverse        [1]    Certificate OPTIONAL,
     *         -- at least one of the pair shall be present -- }
     * </pre>
     *
     * @return a ASN1Primitive
     */
    public ASN1Primitive toASN1Primitive()
    {
        ASN1EncodableVector vec = new ASN1EncodableVector(2);

        if (forward != null)
        {
            vec.add(new DERTaggedObject(0, forward));
        }
        if (reverse != null)
        {
            vec.add(new DERTaggedObject(1, reverse));
        }

        return new DERSequence(vec);
    }

    /**
     * @return Returns the forward.
     */
    public Certificate getForward()
    {
        return forward;
    }

    /**
     * @return Returns the reverse.
     */
    public Certificate getReverse()
    {
        return reverse;
    }
}
