# frozen_string_literal: true

require 'spec_helper'

RSpec.describe Gitlab::BackgroundMigration::BackfillOwaspTopTenOfVulnerabilityReads,
  feature_category: :vulnerability_management do
  let(:namespaces) { table(:namespaces) }
  let(:projects) { table(:projects) }
  let(:users) { table(:users) }
  let(:scanners) { table(:vulnerability_scanners) }
  let(:vulnerabilities) { table(:vulnerabilities) }
  let(:vulnerability_reads) { table(:vulnerability_reads) }
  let(:vulnerability_findings) { table(:vulnerability_occurrences) }
  let(:vulnerability_occurrence_identifiers) { table(:vulnerability_occurrence_identifiers) }
  let(:vulnerability_identifiers) { table(:vulnerability_identifiers) }

  let(:namespace) { namespaces.create!(name: 'user', path: 'user') }
  let(:project) { projects.create!(namespace_id: namespace.id, project_namespace_id: namespace.id) }
  let(:user) { users.create!(username: 'john_doe', email: 'johndoe@gitlab.com', projects_limit: 10) }
  let(:scanner) { scanners.create!(project_id: project.id, external_id: 'external_id', name: 'Test Scanner') }

  shared_context 'with vulnerability data' do # rubocop:disable RSpec/MultipleMemoizedHelpers -- we need to satifsy foreign keys
    let(:external_id) { '' }
    let(:external_type) { '' }
    let(:identifier_name) { '' }

    let(:vuln_identifier) do
      create_identifier(external_id: external_id, external_type: external_type, name: identifier_name)
    end

    let(:vuln_identifier_2) { create_identifier(external_id: 'A1:2021', external_type: 'owasp', name: 'A1 2021') }
    let(:vuln_identifier_3) { create_identifier }

    let(:vuln_finding_1) do
      create_finding(primary_identifier_id: vuln_identifier.id)
    end

    let(:vuln_finding_2) do
      create_finding(primary_identifier_id: vuln_identifier_2.id)
    end

    let(:vuln_finding_3) do
      create_finding(primary_identifier_id: vuln_identifier_3.id)
    end

    let(:vulnerability_1) { create_vulnerability(title: 'vulnerability 1', finding_id: vuln_finding_1.id) }
    let(:vulnerability_2) { create_vulnerability(title: 'vulnerability 2', finding_id: vuln_finding_2.id) }
    let(:vulnerability_3) { create_vulnerability(title: 'vulnerability 3', finding_id: vuln_finding_3.id) }

    let!(:vulnerability_read_1) { create_vulnerability_read(vulnerability_id: vulnerability_1.id) }
    let!(:vulnerability_read_2) { create_vulnerability_read(vulnerability_id: vulnerability_2.id) }
    let!(:vulnerability_read_3) { create_vulnerability_read(vulnerability_id: vulnerability_3.id) }

    before do
      create_vulnerability_occurrence_identifier(occurrence_id: vuln_finding_1.id, identifier_id: vuln_identifier.id)
      vuln_finding_1.update!(vulnerability_id: vulnerability_1.id)
      vuln_finding_2.update!(vulnerability_id: vulnerability_2.id)
      vuln_finding_3.update!(vulnerability_id: vulnerability_3.id)
    end
  end

  describe '#perform' do
    subject(:perform_migration) do
      described_class.new(
        start_id: vulnerability_reads.first.vulnerability_id,
        end_id: vulnerability_reads.last.vulnerability_id,
        batch_table: :vulnerability_reads,
        batch_column: :vulnerability_id,
        sub_batch_size: vulnerability_reads.count,
        pause_ms: 0,
        connection: ActiveRecord::Base.connection
      ).perform
    end

    context 'with owasp top 10 data' do
      include_context 'with vulnerability data' do
        let(:external_id) { 'A1:2017-Injection' }
        let(:external_type) { 'owasp' }
        let(:identifier_name) { 'Injection' }
      end

      it 'updates vulnerability_reads' do
        expect { perform_migration }.to change { vulnerability_read_1.reload.owasp_top_10 }
          .from(nil).to(1)
          .and not_change { vulnerability_read_2.reload.owasp_top_10 }.from(nil)
      end

      it 'updates vulnerability_reads with correct mapping' do
        create_vulnerability_occurrence_identifier(occurrence_id: vuln_finding_2.id,
          identifier_id: vuln_identifier_2.id)
        create_vulnerability_occurrence_identifier(occurrence_id: vuln_finding_3.id,
          identifier_id: vuln_identifier_3.id)

        perform_migration

        expect(vulnerability_read_1.reload.owasp_top_10).to eq(1)
        expect(vulnerability_read_2.reload.owasp_top_10).to eq(11)
        expect(vulnerability_read_3.reload.owasp_top_10).to be_nil
      end
    end

    context 'with incorrect owasp top 10 data' do
      include_context 'with vulnerability data'

      shared_examples 'does not update vulnerability_reads' do
        it do
          perform_migration

          expect(vulnerability_read_1.reload.owasp_top_10).to be_nil
          expect(vulnerability_read_2.reload.owasp_top_10).to be_nil
          expect(vulnerability_read_3.reload.owasp_top_10).to be_nil
        end
      end

      context 'with incorrect long format external_id' do
        let(:external_id) { 'A1:2015-Injection' }
        let(:external_type) { 'owasp' }
        let(:identifier_name) { 'Injection' }

        it_behaves_like 'does not update vulnerability_reads'
      end

      context 'with incorrect short format external_id' do
        let(:external_id) { 'A1' }
        let(:external_type) { 'owasp' }
        let(:identifier_name) { 'Injection' }

        it_behaves_like 'does not update vulnerability_reads'
      end

      context 'with incorrect external_type' do
        let(:external_id) { 'A1:2017' }
        let(:external_type) { 'owasp2017' }
        let(:identifier_name) { 'Injection' }

        it_behaves_like 'does not update vulnerability_reads'
      end
    end

    context 'with no vulnerability identifiers match' do
      include_context 'with vulnerability data' do
        let(:external_id) { 'CVE-2018-1234' }
        let(:external_type) { 'CVE' }
        let(:identifier_name) { 'CVE-2018-1234' }
      end

      it 'does not update vulnerability_reads' do
        perform_migration

        expect(vulnerability_reads.where.not(owasp_top_10: nil).count).to eq(0)
      end
    end
  end

  private

  def create_vulnerability(overrides = {})
    attrs = {
      project_id: project.id,
      author_id: user.id,
      title: 'test',
      severity: 1,
      confidence: 1,
      report_type: 1
    }.merge(overrides)

    vulnerabilities.create!(attrs)
  end

  def create_vulnerability_read(overrides = {})
    attrs = {
      project_id: project.id,
      vulnerability_id: 1,
      scanner_id: scanner.id,
      severity: 1,
      report_type: 1,
      state: 1,
      uuid: SecureRandom.uuid
    }.merge(overrides)

    vulnerability_reads.create!(attrs)
  end

  def create_finding(overrides = {})
    attrs = {
      project_id: project.id,
      scanner_id: scanner.id,
      severity: 5, # medium
      confidence: 2, # unknown,
      report_type: 99, # generic
      primary_identifier_id: create_identifier.id,
      project_fingerprint: SecureRandom.hex(20),
      location_fingerprint: SecureRandom.hex(20),
      uuid: SecureRandom.uuid,
      name: "CVE-2018-1234",
      raw_metadata: "{}",
      metadata_version: "test:1.0"
    }.merge(overrides)

    vulnerability_findings.create!(attrs)
  end

  def create_identifier(overrides = {})
    attrs = {
      project_id: project.id,
      external_id: "CVE-2018-1234",
      external_type: "CVE",
      name: "CVE-2018-1234",
      fingerprint: SecureRandom.hex(20)
    }.merge(overrides)

    vulnerability_identifiers.create!(attrs)
  end

  def create_vulnerability_occurrence_identifier(overrides = {})
    time = Time.now.utc

    attrs = {
      created_at: time,
      updated_at: time,
      occurrence_id: nil,
      identifier_id: nil
    }.merge(overrides)

    vulnerability_occurrence_identifiers.create!(attrs)
  end

  def checksum(value)
    sha = Digest::SHA256.hexdigest(value)
    Gitlab::Database::ShaAttribute.new.serialize(sha)
  end
end
