# frozen_string_literal: true

module Packages
  module Npm
    class GenerateMetadataService
      include API::Helpers::RelatedResourcesHelpers
      include Gitlab::Utils::StrongMemoize

      # Allowed fields are those defined in the abbreviated form
      # defined here: https://github.com/npm/registry/blob/master/docs/responses/package-metadata.md#abbreviated-version-object
      # except: name, version, dist, dependencies and xDependencies. Those are generated by this service.
      PACKAGE_JSON_ALLOWED_FIELDS = %w[deprecated bin directories dist engines _hasShrinkwrap].freeze

      def initialize(name, packages)
        @name = name
        @packages = packages
        @dependencies = {}
        @dependency_ids = Hash.new { |h, key| h[key] = {} }
        @tags = {}
        @tags_updated_at = {}
        @versions = {}
        @latest_version = nil
      end

      def execute(only_dist_tags: false)
        ServiceResponse.success(payload: metadata(only_dist_tags))
      end

      private

      attr_reader :name, :packages, :dependencies, :dependency_ids, :tags, :tags_updated_at, :versions
      attr_accessor :latest_version

      def metadata(only_dist_tags)
        packages.each_batch do |batch|
          relation = preload_needed_relations(batch, only_dist_tags)

          relation.each do |package|
            build_tags(package)
            store_latest_version(package.version)
            next if only_dist_tags

            build_versions(package)
          end
        end

        {
          name: only_dist_tags ? nil : name,
          versions: versions,
          dist_tags: tags.tap { |t| t['latest'] ||= latest_version }
        }.compact_blank
      end

      def preload_needed_relations(batch, only_dist_tags)
        relation = batch.preload_tags
        return relation if only_dist_tags

        load_dependencies(relation)
        load_dependency_ids(relation)

        relation.preload_files.preload_npm_metadatum
      end

      def load_dependencies(packages)
        Packages::Dependency
          .id_in(
            Packages::DependencyLink
              .for_packages(packages)
              .select_dependency_id
          )
          .id_not_in(dependencies.keys)
          .each_batch do |relation|
            relation.each do |dependency|
              dependencies[dependency.id] = { dependency.name => dependency.version_pattern }
            end
          end
      end

      def load_dependency_ids(packages)
        Packages::DependencyLink
          .dependency_ids_grouped_by_type(packages)
          .each_batch(column: :package_id) do |relation|
            relation.each do |dependency_link|
              dependency_ids[dependency_link.package_id] = dependency_link.dependency_ids_by_type
            end
          end
      end

      def build_tags(package)
        package.tags.each do |tag|
          next if tags.key?(tag.name) && tags_updated_at[tag.name] > tag.updated_at

          tags[tag.name] = package.version
          tags_updated_at[tag.name] = tag.updated_at
        end
      end

      def store_latest_version(version)
        self.latest_version = version if latest_version.blank? || VersionSorter.compare(version, latest_version) == 1
      end

      def build_versions(package)
        package_file = package.installable_package_files.last

        return unless package_file

        versions[package.version] = build_package_version(package, package_file)
      end

      def build_package_version(package, package_file)
        abbreviated_package_json(package).merge(
          name: package.name,
          version: package.version,
          dist: {
            shasum: package_file.file_sha1,
            tarball: tarball_url(package, package_file)
          }
        ).tap do |package_version|
          package_version.merge!(build_package_dependencies(package))
        end
      end

      def abbreviated_package_json(package)
        json = package.npm_metadatum&.package_json || {}
        json.slice(*PACKAGE_JSON_ALLOWED_FIELDS)
      end

      def tarball_url(package, package_file)
        expose_url api_v4_projects_packages_npm_package_name___file_name_path(
          { id: package.project_id, package_name: package.name, file_name: package_file.file_name }, true
        )
      end

      def build_package_dependencies(package)
        dependency_ids[package.id].each_with_object(Hash.new { |h, key| h[key] = {} }) do |(type, ids), memo|
          ids.each do |id|
            memo[inverted_dependency_types[type]].merge!(dependencies[id])
          end
        end
      end

      def inverted_dependency_types
        Packages::DependencyLink.dependency_types.invert.stringify_keys
      end
      strong_memoize_attr :inverted_dependency_types
    end
  end
end
