## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
    collapse = TRUE,
    comment = "#>"
)

options(
  rmarkdown.html_vignette.check_title = FALSE
)

## ----setup--------------------------------------------------------------------
library(tidytof)
library(dplyr)
library(stringr)

## ----eval = FALSE-------------------------------------------------------------
# if (!requireNamespace("BiocManager", quietly = TRUE)) {
#     install.packages("BiocManager")
# }
# 
# BiocManager::install("HDCytoData")

## ----message = FALSE, warning = FALSE-----------------------------------------
citrus_raw <- HDCytoData::Bodenmiller_BCR_XL_flowSet()

citrus_data <-
    citrus_raw |>
    as_tof_tbl(sep = "_")

## -----------------------------------------------------------------------------
citrus_metadata <-
    tibble(
        file_name = as.character(flowCore::pData(citrus_raw)[[1]]),
        sample_id = 1:length(file_name),
        patient = stringr::str_extract(file_name, "patient[:digit:]"),
        stimulation = stringr::str_extract(file_name, "(BCR-XL)|Reference")
    ) |>
    mutate(
        stimulation = if_else(stimulation == "Reference", "Basal", stimulation)
    )

citrus_metadata |>
    head()

## -----------------------------------------------------------------------------
citrus_data <-
    citrus_data |>
    left_join(citrus_metadata, by = "sample_id")

## -----------------------------------------------------------------------------
# preprocess the numeric columns in the citrus dataset
citrus_data <-
    citrus_data |>
    mutate(cluster = str_c("cluster", population_id)) |>
    tof_preprocess()

citrus_data |>
    tof_extract_proportion(
        cluster_col = cluster,
        group_cols = c(patient, stimulation)
    ) |>
    head()

## -----------------------------------------------------------------------------
citrus_data |>
    tof_extract_proportion(
        cluster_col = cluster,
        group_cols = c(patient, stimulation),
        format = "long"
    ) |>
    head()

## -----------------------------------------------------------------------------
citrus_data |>
    tof_extract_central_tendency(
        cluster_col = cluster,
        group_cols = c(patient, stimulation),
        marker_cols = any_of(c("CD45_In115", "CD4_Nd145", "CD20_Sm147")),
        central_tendency_function = mean
    ) |>
    head()

## -----------------------------------------------------------------------------
citrus_data |>
    tof_extract_central_tendency(
        cluster_col = cluster,
        group_cols = c(patient, stimulation),
        marker_cols = any_of(c("CD45_In115", "CD4_Nd145", "CD20_Sm147")),
        central_tendency_function = function(x) quantile(x = x, probs = 0.75)
    ) |>
    head()

## -----------------------------------------------------------------------------
citrus_data |>
    tof_extract_threshold(
        cluster_col = cluster,
        group_cols = c(patient, stimulation),
        marker_cols = any_of(c("CD45_In115", "CD4_Nd145", "CD20_Sm147")),
        threshold = 5
    ) |>
    head()

## -----------------------------------------------------------------------------
# Earth-mover's distance
citrus_data |>
    tof_extract_emd(
        cluster_col = cluster,
        group_cols = patient,
        marker_cols = any_of(c("CD45_In115", "CD4_Nd145", "CD20_Sm147")),
        emd_col = stimulation,
        reference_level = "Basal"
    ) |>
    head()

## -----------------------------------------------------------------------------
# Jensen-Shannon Divergence
citrus_data |>
    tof_extract_jsd(
        cluster_col = cluster,
        group_cols = patient,
        marker_cols = any_of(c("CD45_In115", "CD4_Nd145", "CD20_Sm147")),
        jsd_col = stimulation,
        reference_level = "Basal"
    ) |>
    head()

## -----------------------------------------------------------------------------
signaling_markers <-
    c(
        "pNFkB_Nd142", "pStat5_Nd150", "pAkt_Sm152", "pStat1_Eu153", "pStat3_Gd158",
        "pSlp76_Dy164", "pBtk_Er166", "pErk_Er168", "pS6_Yb172", "pZap70_Gd156"
    )

citrus_data |>
    tof_extract_features(
        cluster_col = cluster,
        group_cols = patient,
        stimulation_col = stimulation,
        lineage_cols = any_of(c("CD45_In115", "CD20_Sm147", "CD33_Nd148")),
        signaling_cols = any_of(signaling_markers),
        signaling_method = "emd",
        basal_level = "Basal"
    ) |>
    head()

## -----------------------------------------------------------------------------
sessionInfo()

