## ----intro, include = FALSE---------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>", 
  message = FALSE, 
  warning = FALSE,
  cache = TRUE
)

## ----eval = FALSE-------------------------------------------------------------
# if (!require("BiocManager", quietly = TRUE))
#     install.packages("BiocManager")
# BiocManager::install("cytofQC")

## ----setup--------------------------------------------------------------------
library(cytofQC)
library(CATALYST)
library(SingleCellExperiment)
library(ggplot2)
library(gridExtra)

## ----data1--------------------------------------------------------------------
f <- system.file("extdata", "raw_cytof.fcs", package = "cytofQC")
x <- prepData(f)
names(int_colData(x))
rownames(x)

x <- readCytof(f, 
                 beads = c("Bead"),
                 dna = c("DNA1", "DNA2"),
                 event_length = "Event_length",
                 viability = "Live_Dead",
                 gaussian = c("Center", "Offset", "Width", "Residual"))
x

## ----viewData-----------------------------------------------------------------
head(tech(x))
head(label(x))
head(scores(x))
head(initial(x))
head(probs(x))

## ----labelQC------------------------------------------------------------------
x <- labelQC(x)
table(label(x))
head(scores(x))
head(initial(x))
head(probs(x))

## ----cytofHist----------------------------------------------------------------
bead <- cytofHist(scores(x, type = 'bead'), label(x), title = "Bead score")
debris <- cytofHist(scores(x, type = 'debris'), label(x), title = "Debris score")
doublet <- cytofHist(scores(x, type = 'doublet'), label(x), title = "Doublet score")
grid.arrange(bead, debris, doublet, ncol = 1)

## ----cytofHist_density--------------------------------------------------------
bead <- cytofHist(scores(x, type = 'bead'), label(x), type = "density", title = "Bead score")
debris <- cytofHist(scores(x, type = 'debris'), label(x), type = "density", title = "Debris score")
doublet <- cytofHist(scores(x, type = 'doublet'), label(x), type = "density", title = "Doublet score")
grid.arrange(bead, debris, doublet, ncol = 1)

## ----rf_gbm, eval = FALSE-----------------------------------------------------
# x.rf <- labelQC(x, model = "rf", loss = "class")
# x.gbm <- labelQC(x, model = "gbm", loss = "class")

## ----label_beads--------------------------------------------------------------
x <- readCytof(f, 
                 beads = c("Bead"),
                 dna = c("DNA1", "DNA2"),
                 event_length = "Event_length",
                 viability = "Live_Dead",
                 gaussian = c("Center", "Offset", "Width", "Residual"))

x <- initialBead(x)
x <- svmLabel(x, type = "bead", n = 500)


## ----label_debris-------------------------------------------------------------
x <- initialDebris(x)
x <- svmLabel(x, type = "debris", n = 500)


## ----label_doublets-----------------------------------------------------------
x <- initialDoublet(x)
x <- svmLabel(x, type = "doublet", n = 500)


## ----label_dead---------------------------------------------------------------
x <- initialDead(x)
x <- svmLabel(x, type = "dead", n = 500)


## ----umap---------------------------------------------------------------------
library(uwot)
lab.umap <- umap(scale(x$tech), ret_model = TRUE)
lab.umapD <- data.frame(x = lab.umap$embedding[, 1], 
                        y = lab.umap$embedding[, 2], 
                        labs = x$label)

## ----plot_umap----------------------------------------------------------------
library(RColorBrewer)
ggplot(lab.umapD, aes(x = x, y = y)) +
    geom_point(aes(color = labs), size = 0.5, alpha = 0.5) +
    scale_color_manual(name = NULL, values = brewer.pal(5, "Dark2")) +
    guides(colour = guide_legend(override.aes = list(size = 2))) +
    labs(x = NULL, y = NULL) +
    theme_bw()

## ----session_info-------------------------------------------------------------
library(utils)
sessionInfo()

