## ----setup, include = FALSE-----------------------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  error = FALSE,
  warning = FALSE,
  eval = TRUE,
  message = FALSE,
  fig.width = 8
)
options(width = 100)

## ----install, eval=FALSE--------------------------------------------------------------------------
# if (!requireNamespace("BiocManager", quietly = TRUE)) {
#   install.packages("BiocManager")
# }
# 
# BiocManager::install("GeDi")

## ----loadlib--------------------------------------------------------------------------------------
library("GeDi")

## ----launchapp, eval=FALSE------------------------------------------------------------------------
# GeDi()

## ----launchappwithData, eval=FALSE----------------------------------------------------------------
# GeDi(
#   genesets = geneset_df,
#   ppi = ppi_df,
#   distance_scores = distance_scores_df
# )

## ----examplerun, eval=FALSE-----------------------------------------------------------------------
# GeDi()

## ----create_dds-----------------------------------------------------------------------------------
# Load required libraries
library("macrophage")
library("DESeq2")

# Load the example dataset "gse" from the "macrophage" package
data("gse", package = "macrophage")

# Create a DESeqDataSet object using the "gse" dataset and define the 
# experimental design.
# We use the condition as part of the experimental design, because we are 
# interested in the differentially expressed genes between treatments. We also 
# add the line to the design to account for the inherent differences between 
# the donors.
dds_macrophage <- DESeqDataSet(gse, design = ~ line + condition)

# Change the row names of the DESeqDataSet object to Ensembl IDs
rownames(dds_macrophage) <- gsub("\\..*", "", rownames(dds_macrophage))

# Have a look at the resulting DESeqDataSet object
dds_macrophage

## ----create_resde1--------------------------------------------------------------------------------
# Filter genes based on read counts
# Calculate the number of genes with at least 10 counts in at least 6 samples
keep <- rowSums(counts(dds_macrophage) >= 10) >= 6

# Subset the DESeqDataSet object to keep only the selected genes
dds_macrophage <- dds_macrophage[keep, ]

# Have a look at the resulting DESeqDataSet object
dds_macrophage

## ----create_resde2--------------------------------------------------------------------------------
# Perform differential expression analysis using DESeq2
dds_macrophage <- DESeq(dds_macrophage)

# Extract differentially expressed genes
# Perform contrast analysis comparing "IFNg" condition to "naive" condition
# Set a log2 fold change threshold of 1 and a significance level (alpha) of 0.05
res_macrophage_IFNg_vs_naive <- results(dds_macrophage,
  contrast = c("condition", "IFNg", "naive"),
  lfcThreshold = 1, alpha = 0.05
)

# Add gene symbols to the results in a column "SYMBOL"
res_macrophage_IFNg_vs_naive$SYMBOL <- rowData(dds_macrophage)$SYMBOL

## ----create_resenrich1, eval=TRUE-----------------------------------------------------------------
# Load required packages for analysis
library("pcaExplorer")
library("GeneTonic")
library("AnnotationDbi")

# Extract gene symbols from the DESeq2 results object where FDR is below 0.05
# The function deseqresult2df is used to convert the DESeq2 results to a 
# dataframe format
# FDR is set to 0.05 to filter significant results
de_symbols_IFNg_vs_naive <- deseqresult2df(res_macrophage_IFNg_vs_naive,
                                           FDR = 0.05)$SYMBOL

# Extract gene symbols for background using the DESeq2 results object
# Filter genes that have nonzero counts
bg_ids <- rowData(dds_macrophage)$SYMBOL[rowSums(counts(dds_macrophage)) > 0]

## ----create_resenrich2, eval=TRUE-----------------------------------------------------------------
# Load required package for analysis
library("topGO")
library("org.Hs.eg.db")

# Perform Gene Ontology enrichment analysis using the topGOtable function from 
# the "pcaExplorer" package
macrophage_topGO_example <-
  pcaExplorer::topGOtable(de_symbols_IFNg_vs_naive,
    bg_ids,
    ontology = "BP",
    mapping = "org.Hs.eg.db",
    geneID = "symbol",
    topTablerows = 500
  )

## ----renamecolumns, eval=TRUE---------------------------------------------------------------------
# Rename columns in the macrophage_topGO_example dataframe
# Change the column name "GO.ID" to "Genesets"
names(macrophage_topGO_example)[names(macrophage_topGO_example) == "GO.ID"] <- "Genesets"

# Change the column name "genes" to "Genes"
names(macrophage_topGO_example)[names(macrophage_topGO_example) == "genes"] <- "Genes"

## ----dryrun, eval=FALSE---------------------------------------------------------------------------
# GeDi()
# 
# GeDi(genesets = macrophage_topGO_example)

## ----welcome-page2, fig.align = "center", fig.cap = "The Welcome panel of GeDi", echo = FALSE-----
knitr::include_graphics("Welcome_page.png")

## ----data-input-step1, fig.align = "center", fig.cap = "The Data input panel - Step 1", echo = FALSE----
knitr::include_graphics("Data_Input_panel_Step1.png")

## ----optional-filtering, fig.align = "center", fig.cap = "Optional Filtering Step", echo = FALSE----
knitr::include_graphics("Optional_Filtering.png")

## ----species-selection, fig.align = "center", fig.cap = "Species Selection", echo = FALSE---------
knitr::include_graphics("Species_Selection.png")

## ----download-ppi, fig.align = "center", fig.cap = "Downloading the PPI", echo = FALSE------------
knitr::include_graphics("Downloading_PPI.png")

## ----distance-score, fig.align = "center", fig.cap = "The Distance Score panel", echo = FALSE-----
knitr::include_graphics("Distance_Score_panel.png")

## ----clustering-graph, fig.align = "center", fig.cap = "The Clustering Graph panel", echo = FALSE----
knitr::include_graphics("Clustering_Graph_Panel.png")

## ----report-panel, fig.align = "center", fig.cap = "The Report panel", echo = FALSE---------------
knitr::include_graphics("Report_panel.png")

## ----withbiomart, eval = FALSE--------------------------------------------------------------------
# # Load the "biomaRt" package to access the BioMart database
# library("biomaRt")
# 
# # Set up a connection to the ENSEMBL BioMart database for human genes
# mart <-
#   useMart(biomart = "ENSEMBL_MART_ENSEMBL", dataset = "hsapiens_gene_ensembl")
# 
# # Retrieve gene annotations using the BioMart database
# anns <- getBM(
#   attributes = c(
#     "ensembl_gene_id",
#     "external_gene_name",
#     "entrezgene_id",
#     "description"
#   ),
#   filters = "ensembl_gene_id",
#   values = rownames(dds_macrophage),
#   mart = mart
# )
# 
# # Match the retrieved annotations to the genes in dds_macrophage
# anns <- anns[match(rownames(dds_macrophage), anns[, 1]), ]

## ----enrichKegg, eval = FALSE---------------------------------------------------------------------
# # Load the "clusterProfiler" package for functional enrichment analysis
# library("clusterProfiler")
# 
# # Retrieve Entrez gene IDs from the annotations data frame based on matching
# # Ensembl gene IDs from the DE results
# genes <- anns$entrezgene_id[match(rownames(res_macrophage_IFNg_vs_naive),
#                                   anns$ensembl_gene_id)]
# 
# # Perform KEGG pathway enrichment analysis using the retrieved gene IDs
# res_enrich <- enrichKEGG(genes,
#   organism = "hsa",
#   pvalueCutoff = 0.05
# )

## ----GeDi_Kegg, eval = FALSE----------------------------------------------------------------------
# # Load the "macrophage_KEGG_example" dataset from the "GeDi" package
# data("macrophage_KEGG_example", package = "GeDi")
# 
# # Start the GeDi app with the loaded data
# # The "genesets" parameter is set to the loaded "macrophage_KEGG_example"
# # dataset
# GeDi(genesets = macrophage_KEGG_example)

## ----enrichReactome, eval = FALSE-----------------------------------------------------------------
# # Load the "ReactomePA" package for pathway enrichment analysis
# library("ReactomePA")
# 
# # Perform pathway enrichment analysis using the "enrichPathway" function
# reactome <- enrichPathway(genes,
#   organism = "human",
#   pvalueCutoff = 0.05,
#   readable = TRUE
# )

## ----GeDi_Reactome, eval = FALSE------------------------------------------------------------------
# # Load the "macrophage_Reactome_example" dataset from the "GeDi" package
# data("macrophage_Reactome_example", package = "GeDi")
# 
# # Start the GeDi app with the loaded data
# # The "genesets" parameter is set to the loaded "macrophage_Reactome_example"
# # dataset
# GeDi(genesets = macrophage_Reactome_example)

## ----sessioninfo----------------------------------------------------------------------------------
utils::sessionInfo()

