## ----setup, include=FALSE-----------------------------------------------------
knitr::opts_chunk$set(echo = TRUE)

## ----message=FALSE, warning=FALSE---------------------------------------------
library(ggspavis)
library(STexampleData)
library(patchwork)
library(scater)

## ----message=FALSE------------------------------------------------------------
# load data in SpatialExperiment format
spe <- Visium_mouseCoronal()
rownames(spe) <- rowData(spe)$gene_name
colData(spe)$sum <- colSums(counts(spe))

## ----message=FALSE, warning=FALSE, fig.width=8, fig.height=3.5----------------
p1 <- plotVisium(spe, annotate = "sum", highlight = "in_tissue", 
                 legend_position = "none")
p2 <- plotVisium(spe, annotate = "sum", highlight = "in_tissue", 
                 pal = "darkred") + 
  guides(fill = guide_colorbar(title = "Libsize"))

# display panels using patchwork
p1 | p2

## ----fig.width=8, fig.height=3.5----------------------------------------------
p1 <- plotVisium(spe, annotate = "Gapdh", highlight = "in_tissue")
p2 <- plotVisium(spe, annotate = "Mbp", highlight = "in_tissue")

# display panels using patchwork
p1 | p2

## ----fig.width=8, fig.height=3.5----------------------------------------------
p1 <- plotVisium(spe, annotate = "Mbp", highlight = "in_tissue", image = FALSE)
p2 <- plotVisium(spe, annotate = "Mbp", highlight = "in_tissue", spots = FALSE)

# display panels using patchwork
p1 | p2

## ----fig.width=8, fig.height=3.5----------------------------------------------
p1 <- plotCoords(spe, annotate = "Gapdh")
p2 <- plotCoords(spe, annotate = "Mbp", pal = "viridis")

# display panels using patchwork
p1 | p2

## ----message=FALSE------------------------------------------------------------
# load data in SpatialExperiment format
spe <- Visium_humanDLPFC()
rownames(spe) <- rowData(spe)$gene_name
colData(spe)$libsize <- colSums(counts(spe))

## ----message=FALSE, out.width="60%"-------------------------------------------
plotVisium(spe, annotate = "ground_truth", highlight = "in_tissue", 
           pal = "libd_layer_colors")

## ----fig.width=8, fig.height=4------------------------------------------------
p1 <- plotCoords(spe, annotate = "ground_truth") + 
  ggtitle("Reference")
p2 <- plotCoords(spe, annotate = "libsize") + 
  ggtitle("Library size")

# display panels using patchwork
p1 | p2

## -----------------------------------------------------------------------------
# calculate QC metrics using scater
spe <- addPerCellQCMetrics(spe, 
  subsets = list(mito = grepl("(^MT-)|(^mt-)", rowData(spe)$gene_name)))

# apply QC thresholds
colData(spe)$low_libsize <- colData(spe)$sum < 400 | colData(spe)$detected < 400
colData(spe)$high_mito <- colData(spe)$subsets_mito_percent > 30

## ----fig.width=8, fig.height=4.5----------------------------------------------
p1 <- plotObsQC(spe, plot_type = "histogram", 
                x_metric = "sum", annotate = "low_libsize", )
p2 <- plotObsQC(spe, plot_type = "violin", 
                x_metric = "sum", annotate = "low_libsize", point_size = 0.1)
p3 <- plotObsQC(spe, plot_type = "spot", in_tissue = "in_tissue", 
                annotate = "low_libsize", point_size = 0.2)

# display panels using patchwork
p1 | p2 | p3

## ----fig.width=10, fig.height=4.5---------------------------------------------
p1 <- plotObsQC(spe, plot_type = "histogram", 
                x_metric = "subsets_mito_percent", annotate = "high_mito", )
p2 <- plotObsQC(spe, plot_type = "violin", 
                x_metric = "subsets_mito_percent", annotate = "high_mito", point_size = 0.1)
p3 <- plotObsQC(spe, plot_type = "spot", in_tissue = "in_tissue", 
                annotate = "high_mito", point_size = 0.2)

# display panels using patchwork
p1 | p2 | p3

## ----out.width="60%", warning=FALSE, message=FALSE----------------------------
plotObsQC(spe, plot_type = "scatter", 
          x_metric = "subsets_mito_percent", y_metric = "sum", 
          x_threshold = 30, y_threshold = 400)

## ----warning=FALSE, message=FALSE, fig.width=8, fig.height=4------------------
rowData(spe)$feature_sum <- rowSums(counts(spe))
rowData(spe)$low_abundance <- rowSums(counts(spe) > 0) < 20

p1 <- plotFeatureQC(spe, plot_type = "histogram", 
                    x_metric = "feature_sum", annotate = "low_abundance")
p2 <- plotFeatureQC(spe, plot_type = "violin", 
                    x_metric = "feature_sum", annotate = "low_abundance")

# display panels using patchwork
p1 | p2

## -----------------------------------------------------------------------------
sessionInfo()

