## ----setup, echo = FALSE------------------------------------------------------
knitr::opts_chunk$set(tidy = FALSE,
                      dpi = 72,
                      fig.width = 6,
                      fig.height = 6,
                      width=80,
                      cache = FALSE,
                      dev = "png",
                      message = TRUE,
                      error = TRUE,
                      warning = TRUE)

## ----installation, eval = FALSE-----------------------------------------------
# devtools::install_github("dunbarlabNIH/barcodetrackR")

## ----load packages required, eval = TRUE, warning=FALSE, message=FALSE--------

require("magrittr")
require("barcodetrackR")
require("SummarizedExperiment")


## ----load data, eval = TRUE, results="hold"-----------------------------------

system.file("extdata", "/WuC_etal_appdata/sample_data_ZJ31.txt", package = "barcodetrackR") %>%
  read.delim(row.names = 1) -> wu_dataframe

system.file("extdata", "/WuC_etal_appdata/sample_metadata_ZJ31.txt", package = "barcodetrackR") %>%
  read.delim() -> wu_metadata

wu_SE <- create_SE(your_data = wu_dataframe,
                   meta_data = wu_metadata,
                   threshold = 0)


system.file("extdata", "/BelderbosME_etal/count_matrix_mouse_C21.txt", package = "barcodetrackR") %>%
  read.delim(row.names = 1) -> belderbos_dataframe

system.file("extdata", "/BelderbosME_etal/metadata_mouse_C21.txt", package = "barcodetrackR") %>%
  read.delim() -> belderbos_metadata

belderbos_metadata$weeks <- factor(belderbos_metadata$weeks, levels = c("9", "14", "20", "22", "sac"))

belderbos_SE <- create_SE(your_data = belderbos_dataframe,
                          meta_data = belderbos_metadata,
                          threshold = 0)

system.file("extdata", "/SixE_etal/WAS5_reads.txt", package = "barcodetrackR") %>%
  read.delim(row.names = 1) -> six_dataframe

system.file("extdata", "/SixE_etal/WAS5_metadata.txt", package = "barcodetrackR") %>%
  read.delim() -> six_metadata

six_SE <- create_SE(your_data = six_dataframe,
                    meta_data = six_metadata,
                    threshold = 0)


## ----list assays, eval = TRUE-------------------------------------------------
assays(six_SE)

## ----estimate threshold, eval = TRUE------------------------------------------
my_thresh <- estimate_barcode_threshold(capture_efficiency = 0.4,
                                        population_size = 500000,
                                        proportion_labeled = 0.3,
                                        confidence_level = 0.95,
                                        verbose = TRUE)

## ----apply thresholding, eval = TRUE------------------------------------------
wu_thresh <- threshold_SE(your_SE = wu_SE,
                          threshold_value = 0.005,
                          threshold_type = "relative",
                          verbose = TRUE)

## ----launch_app, eval=FALSE---------------------------------------------------
# barcodetrackR::launchApp()

## ----scatter_plot, eval=TRUE, fig.width = 7, fig.height=4---------------------
Gr_B_20 <- c("ZJ31_20m_Gr", "ZJ31_20m_B")
Gr_T_20 <- c("ZJ31_20m_Gr", "ZJ31_20m_T")
wu_scatterplot_1 <- scatter_plot(wu_SE[,Gr_B_20], your_title = "Gr vs B", assay = "proportions")
wu_scatterplot_2 <- scatter_plot(wu_SE[,Gr_T_20], your_title = "Gr vs T", assay = "proportions")
cowplot::plot_grid(wu_scatterplot_1, wu_scatterplot_2, ncol = 2)

## ----cor_plot 1, eval = TRUE, fig.width = 6, fig.height = 5-------------------
wu_cor_plot_sample_selection <- colData(wu_SE)$SAMPLENAME[1:20]
cor_plot(wu_SE[,wu_cor_plot_sample_selection],
         method_corr = "pearson",
         plot_type = "color",
         assay = "proportions")

## ----cor_plot 2, eval = TRUE, fig.width = 6, fig.height = 5-------------------
wu_cor_plot_sample_selection <- colData(wu_SE)$SAMPLENAME[1:20]
cor_plot(wu_SE[,wu_cor_plot_sample_selection],
         method_corr = "pearson",
         plot_type = "color",
         assay = "logs")

## ----cor_plot 3, eval = TRUE--------------------------------------------------
cor_plot(wu_SE[,wu_cor_plot_sample_selection],
         method_corr = "pearson",
         assay = "proportions",
         return_table = TRUE) %>% head

## ----cor_plot 4, eval = TRUE, fig.width = 4, fig.height = 3, warning=FALSE, message=FALSE----
belderbos_wk_samples_PB <- c("wk9_U", "wk14_U", "wk20_U", "wk22_U")
cor_plot(your_SE = belderbos_SE[,belderbos_wk_samples_PB],
         method_corr = "pearson",
         plot_type = "number",
         assay = "proportions",
         number_size = 3)

## ----summary proxy, eval = TRUE-----------------------------------------------
summary(proxy::pr_DB)

## ----dist plot 1, eval = TRUE, fig.width = 7, fig.height = 5------------------
dist_plot(wu_SE[,wu_cor_plot_sample_selection],
          dist_method =  "manhattan",
          plot_type = "color",
          assay = "logs")

## ----dist plot 2, eval = TRUE, fig.width = 7, fig.height = 5------------------
dist_plot(wu_SE[,wu_cor_plot_sample_selection],
          dist_method =  "cosine",
          plot_type = "color",
          assay = "logs",
          color_pal = "Greens",
          cluster_tree = TRUE)

## ----barcode_ggheatmap wu_SE 1, echo=TRUE, warning = FALSE, fig.width = 8, fig.height=8, eval = TRUE----
barcode_ggheatmap(your_SE = wu_SE[,wu_cor_plot_sample_selection],
                  n_clones = 10,
                  grid = FALSE,
                  label_size = 14,
                  cellnote_size = 4)

## ----barcode_ggheatmap six_SE, echo=TRUE, warning = FALSE, fig.width = 8, fig.height=6, eval = TRUE----
six_celltype_order <- c("m13_TCELLS", "m36_TCELLS", "m43_TCELLS",
                        "m55_TCELLS","m13_BCELLS", "m36_BCELLS",
                        "m43_BCELLS", "m55_BCELLS", "m13_NKCELLS",
                        "m36_NKCELLS", "m43_NKCELLS","m55_NKCELLS",
                        "m13_GRANULOCYTES", "m36_GRANULOCYTES",
                        "m43_GRANULOCYTES", "m55_GRANULOCYTES",
                        "m13_MONOCYTES", "m36_MONOCYTES",
                        "m43_MONOCYTES","m55_MONOCYTES")

barcode_ggheatmap(your_SE = six_SE[,six_celltype_order],
                  n_clones = 5,
                  cellnote_assay = "stars",
                  cellnote_size = 3,
                  label_size = 14,
                  dendro = TRUE,
                  grid = FALSE,
                  clusters = 4,
                  distance_method = "euclidean")

## ----barcode_ggheatmap_stat wu_SE, echo=TRUE, warning = FALSE, fig.width = 8, fig.height=6, eval = TRUE----
wu_CD16_NK_order <- colnames(wu_SE)[17:20]

barcode_ggheatmap_stat(your_SE = wu_SE[,wu_CD16_NK_order],
                       sample_size = rep(40000,length(wu_CD16_NK_order)),
                       stat_test = "chi-squared",
                       stat_option = "subsequent",
                       p_threshold = 0.05,
                       n_clones = 10,
                       cellnote_assay = "stars",
                       cellnote_size = 6)

## ----barcode_stat_test wu_SE, eval = TRUE, echo=TRUE, warning = FALSE---------

sample_size <- rep(40000,length(wu_CD16_NK_order))
wu_CD16_NK_statistics <- barcode_stat_test(your_SE = wu_SE[,wu_CD16_NK_order],
                                           sample_size = sample_size,
                                           stat_test = "chi-squared",
                                           stat_option = "subsequent",
                                           bc_threshold = 0.01)

head(wu_CD16_NK_statistics$p_val)

## ----binary_heat_map, echo=TRUE, warning = FALSE, eval = TRUE-----------------
barcode_binary_heatmap(your_SE = belderbos_SE[,belderbos_wk_samples_PB],
                       label_size = 12,
                       threshold = 0.01)

## ----clonal_contribution wu_SE, echo=TRUE, warning = FALSE, eval = TRUE-------
clonal_contribution(your_SE = wu_SE,
                    SAMPLENAME_choice = "ZJ31_20m_NK_CD56n_CD16p",
                    n_clones = 10,
                    graph_type = "line",
                    plot_over = "months",
                    filter_by = "celltype",
                    filter_selection = "NK_16",
                    plot_non_selected = FALSE)

## ----bias_histogram wu_SE, echo=TRUE, warning = FALSE, eval = TRUE------------
wu_bias_plot_sample_selection <- colData(wu_SE)$SAMPLENAME[1:20]

bias_histogram(your_SE = wu_SE[,wu_bias_plot_sample_selection],
               split_bias_on = "celltype",
               bias_1 = "B",
               bias_2 = "Gr",
               split_bias_over = "months",
               ncols = 2)

## ----bias_histogram six_SE, echo=TRUE, warning = FALSE, eval = TRUE-----------
bias_histogram(your_SE = six_SE,
               split_bias_on = "celltype",
               bias_1 = "B",
               bias_2 = "T",
               split_bias_over = "months",
               ncols = 2)

## ----bias_ridge_plot wu_SE 1, echo=TRUE, warning = FALSE, message=FALSE, eval = TRUE----
bias_ridge_plot(your_SE = wu_SE,
                split_bias_on = "celltype",
                bias_1 = "B",
                bias_2 = "Gr",
                split_bias_over = "months",
                bias_over = c("6","9.5","12","20"),
                weighted = TRUE,
                add_dots = TRUE)

## ----bias_ridge_plot wu_SE 2, echo=TRUE, warning = FALSE, message=FALSE, eval = TRUE----
bias_ridge_plot(your_SE = wu_SE,
                split_bias_on = "celltype",
                bias_1 = "B",
                bias_2 = "Gr",
                split_bias_over = "months",
                bias_over = c("6","9.5","12","20"),
                weighted = FALSE,
                add_dots = TRUE)

## ----bias_lineplot six_SE, echo=TRUE, warning = FALSE, eval = TRUE------------
bias_lineplot(your_SE = six_SE,
              split_bias_on = "celltype",
              bias_1 = "Gr",
              bias_2 = "Mo",
              split_bias_over = "months",
              remove_unique = TRUE)

## ----clone counts 1, eval = TRUE----------------------------------------------
clonal_count(your_SE = six_SE,
             plot_over = "months",
             group_by = "celltype")

## ----clone counts cumulative, eval = TRUE-------------------------------------
clonal_count(your_SE = six_SE,
             plot_over = "months",
             group_by = "celltype",
             cumulative = TRUE)

## ----belderbos rank_abundance, eval = TRUE------------------------------------
rank_abundance_plot(belderbos_SE[,1:8],
                    scale_rank = TRUE,
                    point_size = 2)

## ----belderbos rank_abundance stat_test, eval = TRUE, warning = FALSE---------
stat_result <- rank_abundance_stat_test(belderbos_SE[,1:8])
print(stat_result[["p_value"]])

## ----wu shannon, eval = TRUE--------------------------------------------------
clonal_diversity(wu_SE[,1:20],
                 plot_over = "months",
                 group_by = "celltype",
                 index_type = "shannon")

## ----belderbos shannon, eval = TRUE-------------------------------------------
clonal_diversity(belderbos_SE[,1:6],
                 plot_over = "weeks",
                 group_by = "celltype", 
                 index_type = "shannon")

## ----six simpson, eval = TRUE-------------------------------------------------
clonal_diversity(six_SE,
                 plot_over = "months",
                 group_by = "celltype",
                 index_type = "simpson")

## ----wu bray diversity, eval = TRUE-------------------------------------------
mds_plot(wu_SE, group_by = "celltype",
         method_dist = "bray", assay = "proportions")

## ----six bray diversity, eval = TRUE------------------------------------------
mds_plot(six_SE, group_by = "celltype",
         method_dist = "bray", assay = "proportions")

## ----chord_diagram wu_SE 1, echo=TRUE, warning = FALSE, eval = TRUE-----------
wu_circos_selection <- c("ZJ31_12m_T","ZJ31_12m_B","ZJ31_12m_Gr")
chord_diagram(wu_SE[,wu_circos_selection], plot_label = "celltype")

## ----chord_diagram wu_SE 2, echo=TRUE, warning = FALSE, eval = TRUE-----------
chord_diagram(wu_SE[,wu_circos_selection],
              plot_label = "celltype",
              weighted = TRUE)

## ----chord_diagram wu_SE 3, echo=TRUE, warning = FALSE, eval = TRUE-----------
wu_circos_selection2 <- c("ZJ31_12m_T",
                          "ZJ31_12m_B",
                          "ZJ31_12m_Gr",
                          "ZJ31_12m_NK_CD56n_CD16p")
chord_diagram(wu_SE[,wu_circos_selection2],
              plot_label = "celltype",
              alpha = 0.9)

## ----chord_diagram belderbos_SE 1, echo=TRUE, warning = FALSE, eval = TRUE----
belderbos_wk_samples_PB <- c("wk9_U", "wk14_U", "wk20_U", "wk22_U")
chord_diagram(belderbos_SE[,belderbos_wk_samples_PB])

## ----chord_diagram belderbos_SE 2, echo=TRUE, warning = FALSE, eval=TRUE------
chord_diagram(belderbos_SE[,belderbos_wk_samples_PB],
            weighted = TRUE)

## ----session info, eval=TRUE, width=6-----------------------------------------
sessionInfo()

