# SPDX-FileCopyrightText: Copyright (c) 2025 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
# SPDX-License-Identifier: LicenseRef-NvidiaProprietary
#
# NVIDIA CORPORATION, its affiliates and licensors retain all intellectual
# property and proprietary rights in and to this material, related
# documentation and any modifications thereto. Any use, reproduction,
# disclosure or distribution of this material and related documentation
# without an express license agreement from NVIDIA CORPORATION or
# its affiliates is strictly prohibited.


from enum import IntEnum
from typing import Optional


class ExtendedIntEnum(IntEnum):
    @classmethod
    def from_name(cls, name):
        return cls._member_map_[name]


class PerfMetricType(ExtendedIntEnum):
    IPC = 0
    RETIRING = 1
    BACKEND_STALLED_CYCLES = 2
    FRONTEND_STALLED_CYCLES = 3
    BACKEND_BOUND = 4
    FRONTEND_BOUND = 5
    SVE_GFLOPS = 6
    NON_SVE_GFLOPS = 7
    GFLOPS = 8
    LOAD_OP_PERCENT = 9
    STORE_OP_PERCENT = 10
    BRANCH_OP_PERCENT = 11
    INT_OP_PERCENT = 12
    SCALAR_FLOP_PERCENT = 13
    SYNC_INST_PERCENT = 14
    CRYPTO_OP_PERCENT = 15
    SVE_OP_PERCENT = 16
    ASE_OP_PERCENT = 17
    SIMD_PERCENT = 18
    FP16_PERCENT = 19
    FP32_PERCENT = 20
    FP64_PERCENT = 21
    FULL_SVE_INST = 22
    PARTIAL_SVE_INST = 23
    EMPTY_SVE_INST = 24
    L1D_CACHE_MISS_RATIO = 25
    L1D_CACHE_MPKI = 26
    L1I_CACHE_MISS_RATIO = 27
    L1I_CACHE_MPKI = 28
    L2_CACHE_MISS_RATIO = 29
    L2_CACHE_MPKI = 30
    LL_CACHE_RD_HIT_RATIO = 31
    LL_CACHE_RD_MISS_RATIO = 32
    LL_CACHE_RD_MPKI = 33
    L1D_TLB_MISS_RATIO = 34
    L1D_TLB_MPKI = 35
    L1I_TLB_MISS_RATIO = 36
    L1I_TLB_MPKI = 37
    BR_MISPRED_RATIO = 38
    BR_MISPRED_MPKI = 39
    BAD_SPECULATION = 40
    DTLB_MPKI = 41
    DTLB_WALK_RATIO = 42
    ITLB_MPKI = 43
    ITLB_WALK_RATIO = 44
    L2_UNIFIED_TLB_MISS_RATIO = 45
    L2_UNIFIED_TLB_MPKI = 46
    LAST = 47


class TopDownL1Type(ExtendedIntEnum):
    FRONTEND_BOUND = 0
    BACKEND_BOUND = 1
    BAD_SPECULATION = 2
    RETIRING = 3
    LAST = 4


class TopDownL2Type(ExtendedIntEnum):
    BRANCH_EFFECTIVENESS = 0
    ITLB_EFFECTIVENESS = 1
    DTLB_EFFECTIVENESS = 2
    L1I_CACHE_EFFECTIVENESS = 3
    L1D_CACHE_EFFECTIVENESS = 4
    L2_CACHE_EFFECTIVENESS = 5
    LL_CACHE_EFFECTIVENESS = 6
    OPERATION_MIX = 7
    LAST = 8


class PerfMetric:
    def __init__(
        self, name: str, description: str, equation: str, unit: Optional[str] = None
    ):
        self.name = name
        self.description = description
        self.equation = equation
        self.unit = unit


def _fill_arm_metrics():
    metrics = [None] * PerfMetricType.LAST

    metrics[PerfMetricType.IPC] = PerfMetric(
        "Instructions Per Cycle (IPC)",
        "Instructions retired per cycle",
        "INST_RETIRED / CPU_CYCLES",
    )

    metrics[PerfMetricType.RETIRING] = PerfMetric(
        "Retiring",
        "Percentage of total slots that are retired operations",
        "100 * (OP_RETIRED / OP_SPEC) * (1 - (STALL_SLOT / (CPU_CYCLES * 8)))",
        "Percentage",
    )

    metrics[PerfMetricType.BACKEND_STALLED_CYCLES] = PerfMetric(
        "Backend Stalled Cycles",
        "Percentage of cycles that were stalled due to resource constraints "
        "in the backend unit of the processor",
        "(STALL_BACKEND / CPU_CYCLES) * 100",
        "Percentage",
    )

    metrics[PerfMetricType.FRONTEND_STALLED_CYCLES] = PerfMetric(
        "Frontend Stalled Cycles",
        "Percentage of cycles that were stalled due to resource constraints "
        "in the frontend unit of the processor",
        "(STALL_FRONTEND / CPU_CYCLES) * 100",
        "Percentage",
    )

    metrics[PerfMetricType.BACKEND_BOUND] = PerfMetric(
        "Backend Bound",
        "Percentage of total slots that were stalled because of "
        "resource constraints in the processor backend",
        "(STALL_SLOT_BACKEND / (CPU_CYCLES * 8) - BR_MIS_PRED * 3 / CPU_CYCLES) * 100",
        "Percentage",
    )

    metrics[PerfMetricType.FRONTEND_BOUND] = PerfMetric(
        "Frontend Bound",
        "Percentage of total slots that were stalled because of "
        "resource constraints in the processor frontend",
        "(STALL_SLOT_FRONTEND / (CPU_CYCLES * 8) - BR_MIS_PRED / CPU_CYCLES) * 100",
        "Percentage",
    )

    metrics[PerfMetricType.SVE_GFLOPS] = PerfMetric(
        "SVE GFLOPS",
        "Giga-floating point operations per second in any precision performed by "
        "the SVE instructions",
        "FP_SCALE_OPS_SPEC / TIME",
        "operations/second",
    )

    metrics[PerfMetricType.NON_SVE_GFLOPS] = PerfMetric(
        "Non-SVE GFLOPS",
        "Giga-floating point operations per second in any precision performed by "
        "an instruction that is not an SVE instruction",
        "FP_FIXED_OPS_SPEC / TIME",
        "operations/second",
    )

    metrics[PerfMetricType.GFLOPS] = PerfMetric(
        "GFLOPS",
        "Giga-floating point operations per second in any precision performed by "
        "any instruction",
        "(FP_SCALE_OPS_SPEC + FP_FIXED_OPS_SPEC) / TIME",
        "operations/second",
    )

    metrics[PerfMetricType.LOAD_OP_PERCENT] = PerfMetric(
        "Load Operations Percentage",
        "Percentage of total operations speculatively executed as load operations",
        "(LD_SPEC / INST_SPEC) * 100",
        "Percentage",
    )

    metrics[PerfMetricType.STORE_OP_PERCENT] = PerfMetric(
        "Store Operations Percentage",
        "Percentage of total operations speculatively executed as store operations",
        "(ST_SPEC / INST_SPEC) * 100",
        "Percentage",
    )

    metrics[PerfMetricType.BRANCH_OP_PERCENT] = PerfMetric(
        "Branch Operations Percentage",
        "Percentage of total operations speculatively executed as branch operations",
        "((BR_IMMED_SPEC + BR_INDIRECT_SPEC) / INST_SPEC) * 100",
        "Percentage",
    )

    metrics[PerfMetricType.INT_OP_PERCENT] = PerfMetric(
        "Integer Operations Percentage",
        "Percentage of total operations speculatively executed as scalar "
        "integer operations",
        "(DP_SPEC / INST_SPEC) * 100",
        "Percentage",
    )

    metrics[PerfMetricType.SCALAR_FLOP_PERCENT] = PerfMetric(
        "Floating Point Operations Percentage",
        "Percentage of total operations speculatively executed as scalar "
        "floating point operations",
        "(VFP_SPEC / INST_SPEC) * 100",
        "Percentage",
    )

    metrics[PerfMetricType.SYNC_INST_PERCENT] = PerfMetric(
        "Synchronization Percentage",
        "Percentage of total instructions that were speculatively executed "
        "because of the synchronization instructions",
        "((ISB_SPEC + DSB_SPEC + DMB_SPEC) / INST_SPEC) * 100",
        "Percentage",
    )

    metrics[PerfMetricType.CRYPTO_OP_PERCENT] = PerfMetric(
        "Crypto Operations Percentage",
        "Percentage of total operations speculatively executed as crypto operations",
        "(CRYPTO_SPEC / INST_SPEC) * 100",
        "Percentage",
    )

    metrics[PerfMetricType.SVE_OP_PERCENT] = PerfMetric(
        "SVE Operations (Load/Store Inclusive) Percentage",
        "Percentage of total operations speculatively executed as "
        "scalable vector operations",
        "(SVE_INST_SPEC / INST_SPEC) * 100",
        "Percentage",
    )

    metrics[PerfMetricType.ASE_OP_PERCENT] = PerfMetric(
        "Advanced SIMD Operations Percentage",
        "Percentage of total operations speculatively executed as advanced "
        "SIMD (NEON) operations",
        "(ASE_SPEC / INST_SPEC) * 100",
        "Percentage",
    )

    metrics[PerfMetricType.SIMD_PERCENT] = PerfMetric(
        "SIMD Percentage",
        "Percentage of total instructions that were speculatively executed "
        "because of the integer or floating point vector/SIMD instructions",
        "((SVE_INST_SPEC + ASE_INST_SPEC) / INST_SPEC) * 100",
        "Percentage",
    )

    metrics[PerfMetricType.FP16_PERCENT] = PerfMetric(
        "FP16 Percentage",
        "Percentage of total instructions that were speculatively executed "
        "because of the half-precision floating point instructions",
        "(FP_HP_SPEC / INST_SPEC) * 100",
        "Percentage",
    )

    metrics[PerfMetricType.FP32_PERCENT] = PerfMetric(
        "FP32 Percentage",
        "Percentage of total instructions that were speculatively executed "
        "because of the single-precision floating point instructions",
        "(FP_SP_SPEC / INST_SPEC) * 100",
        "Percentage",
    )

    metrics[PerfMetricType.FP64_PERCENT] = PerfMetric(
        "FP64 Percentage",
        "Percentage of total instructions that were speculatively executed because "
        "of the double-precision floating point instructions",
        "(FP_DP_SPEC / INST_SPEC) * 100",
        "Percentage",
    )

    metrics[PerfMetricType.FULL_SVE_INST] = PerfMetric(
        "Full SVE Instructions",
        "Fraction of total instructions that were speculatively executed because "
        "of the SVE SIMD instructions with all active predicates",
        "SVE_PRED_FULL_SPEC / INST_SPEC",
        "Fraction",
    )

    metrics[PerfMetricType.PARTIAL_SVE_INST] = PerfMetric(
        "Partial SVE Instructions",
        "Fraction of total instructions that were speculatively executed because "
        "of the SVE SIMD instructions in which at least one element is FALSE",
        "SVE_PRED_PARTIAL_SPEC / INST_SPEC",
        "Fraction",
    )

    metrics[PerfMetricType.EMPTY_SVE_INST] = PerfMetric(
        "Empty SVE Instructions",
        "Fraction of total instructions that were speculatively executed because "
        "of the SVE SIMD instructions with no active predicate",
        "SVE_PRED_EMPTY_SPEC / INST_SPEC",
        "Fraction",
    )

    metrics[PerfMetricType.L1D_CACHE_MISS_RATIO] = PerfMetric(
        "L1D Cache Miss Ratio",
        "Ratio of total level 1 data cache read or write accesses that miss",
        "L1D_CACHE_REFILL / L1D_CACHE",
        "Ratio",
    )

    metrics[PerfMetricType.L1D_CACHE_MPKI] = PerfMetric(
        "L1D Cache MPKI",
        "The number of level 1 data cache accesses missed per thousand instructions "
        "executed",
        "(L1D_CACHE_REFILL / INST_RETIRED) * 1000",
        "misses per 1000 instructions retired",
    )

    metrics[PerfMetricType.L1I_CACHE_MISS_RATIO] = PerfMetric(
        "L1I Cache Miss Ratio",
        "The ratio of level 1 instruction cache accesses missed to the total number "
        "of level 1 instruction cache accesses",
        "L1I_CACHE_REFILL / L1I_CACHE",
        "Ratio",
    )

    metrics[PerfMetricType.L1I_CACHE_MPKI] = PerfMetric(
        "L1I Cache MPKI",
        "The number of level 1 instruction cache accesses missed per thousand "
        "instructions executed",
        "(L1I_CACHE_REFILL / INST_RETIRED) * 1000",
        "misses per 1000 instructions retired",
    )

    metrics[PerfMetricType.L2_CACHE_MISS_RATIO] = PerfMetric(
        "L2 Cache Miss Ratio",
        "The ratio of level 2 cache accesses missed to the total number of level "
        "2 cache accesses",
        "L2D_CACHE_REFILL / L2D_CACHE",
        "Ratio",
    )

    metrics[PerfMetricType.L2_CACHE_MPKI] = PerfMetric(
        "L2 Cache MPKI",
        "The number of level 2 unified cache accesses missed per thousand "
        "instructions executed",
        "(L2D_CACHE_REFILL / INST_RETIRED) * 1000",
        "misses per 1000 instructions retired",
    )

    metrics[PerfMetricType.LL_CACHE_RD_HIT_RATIO] = PerfMetric(
        "LL Cache Read Hit Ratio",
        "The ratio of last level cache read accesses hit in the cache "
        "to the total number of last level cache accesses",
        "(LL_CACHE_RD - LL_CACHE_MISS_RD) / LL_CACHE_RD",
        "Ratio",
    )

    metrics[PerfMetricType.LL_CACHE_RD_MISS_RATIO] = PerfMetric(
        "LL Cache Read Miss Ratio",
        "The ratio of last level cache read accesses missed to the total number "
        "of last level cache accesses",
        "LL_CACHE_MISS_RD / LL_CACHE_RD",
        "Ratio",
    )

    metrics[PerfMetricType.LL_CACHE_RD_MPKI] = PerfMetric(
        "LL Cache Read MPKI",
        "The number of last level cache read accesses missed per thousand "
        "instructions executed",
        "(LL_CACHE_MISS_RD / INST_RETIRED) * 1000",
        "misses per 1000 instruction retired",
    )

    metrics[PerfMetricType.L1D_TLB_MISS_RATIO] = PerfMetric(
        "L1 Data TLB Miss Ratio",
        "The ratio of level 1 data TLB accesses missed to the total number "
        "of level 1 data TLB accesses",
        "L1D_TLB_REFILL / L1D_TLB",
        "Ratio",
    )

    metrics[PerfMetricType.L1D_TLB_MPKI] = PerfMetric(
        "L1 Data TLB MPKI",
        "The number of level 1 data TLB accesses missed per thousand "
        "instructions executed",
        "(L1D_TLB_REFILL / INST_RETIRED) * 1000",
        "misses per 1000 instructions retired",
    )

    metrics[PerfMetricType.L1I_TLB_MISS_RATIO] = PerfMetric(
        "L1 Instruction TLB Miss Ratio",
        "The ratio of level 1 instruction TLB accesses missed to the total number "
        "of level 1 instruction TLB accesses",
        "L1I_TLB_REFILL / L1I_TLB",
        "Ratio",
    )

    metrics[PerfMetricType.L1I_TLB_MPKI] = PerfMetric(
        "L1 Instruction TLB MPKI",
        "The number of level 1 instruction TLB accesses missed per thousand "
        "instructions executed",
        "(L1I_TLB_REFILL / INST_RETIRED) * 1000",
        "misses per 1000 instructions retired",
    )

    metrics[PerfMetricType.BR_MISPRED_RATIO] = PerfMetric(
        "Branch Misprediction Ratio",
        "The ratio of branches mispredicted to the total number of branches "
        "architecturally executed",
        "BR_MIS_PRED_RETIRED / BR_RETIRED",
        "Ratio",
    )

    metrics[PerfMetricType.BR_MISPRED_MPKI] = PerfMetric(
        "Branch MPKI",
        "The number of branch mispredictions per thousand instructions executed",
        "(BR_MIS_PRED_RETIRED / INST_RETIRED) * 1000",
        "misses per 1000 instructions retired",
    )

    metrics[PerfMetricType.BAD_SPECULATION] = PerfMetric(
        "Bad Speculation",
        "Percentage of total slots that executed operations and didn't retire "
        "due to a pipeline flush",
        "100 * ((1-OP_RETIRED/OP_SPEC) * (1-STALL_SLOT/(CPU_CYCLES * 8)) + BR_MIS_PRED * 4 / CPU_CYCLES)",
        "Percentage",
    )

    metrics[PerfMetricType.DTLB_MPKI] = PerfMetric(
        "DTLB MPKI",
        "The number of data TLB Walks per thousand instructions executed",
        "(DTLB_WALK / INST_RETIRED) * 1000",
        "data TLB Walks per 1000 instructions retired",
    )

    metrics[PerfMetricType.DTLB_WALK_RATIO] = PerfMetric(
        "DTLB Walk Ratio",
        "The ratio of data TLB Walks to the total number of data TLB accesses",
        "DTLB_WALK / L1D_TLB",
        "Ratio",
    )

    metrics[PerfMetricType.ITLB_MPKI] = PerfMetric(
        "ITLB MPKI",
        "The number of instruction TLB Walks per thousand instructions executed",
        "(ITLB_WALK / INST_RETIRED) * 1000",
        "instruction TLB Walks per 1000 instructions retired",
    )

    metrics[PerfMetricType.ITLB_WALK_RATIO] = PerfMetric(
        "ITLB Walk Ratio",
        "The ratio of instruction TLB Walks to the total number "
        "of instruction TLB accesses",
        "ITLB_WALK / L1I_TLB",
        "Ratio",
    )

    metrics[PerfMetricType.L2_UNIFIED_TLB_MISS_RATIO] = PerfMetric(
        "L2 Unified TLB Miss Ratio",
        "The ratio of level 2 unified TLB accesses missed to the total number "
        "of level 2 unified TLB accesses",
        "L2D_TLB_REFILL / L2D_TLB",
        "Ratio",
    )

    metrics[PerfMetricType.L2_UNIFIED_TLB_MPKI] = PerfMetric(
        "L2 Unified TLB MPKI",
        "The number of level 2 unified TLB accesses missed per thousand "
        "instructions executed",
        "(L2D_TLB_REFILL / INST_RETIRED) * 1000",
        "level 2 unified TLB accesses per 1000 instructions retired",
    )

    return metrics


_arm_metrics: Optional[list[PerfMetric]] = None


def get_arm_metrics() -> list[PerfMetric]:
    global _arm_metrics
    if _arm_metrics is None:
        _arm_metrics = _fill_arm_metrics()
    return _arm_metrics


class TopDownL1Category:
    def __init__(
        self, name: str, metric_type: PerfMetricType, l2_types: list[TopDownL2Type]
    ) -> None:
        self.name = name
        self.metric_type = metric_type
        self.l2_types = l2_types


def _fill_topdown_l1_categories():
    categories = [None] * TopDownL1Type.LAST

    categories[TopDownL1Type.FRONTEND_BOUND] = TopDownL1Category(
        "Frontend Bound",
        PerfMetricType.FRONTEND_BOUND,
        [
            TopDownL2Type.BRANCH_EFFECTIVENESS,
            TopDownL2Type.ITLB_EFFECTIVENESS,
            TopDownL2Type.L1I_CACHE_EFFECTIVENESS,
            TopDownL2Type.L2_CACHE_EFFECTIVENESS,
            TopDownL2Type.LL_CACHE_EFFECTIVENESS,
        ],
    )
    categories[TopDownL1Type.BACKEND_BOUND] = TopDownL1Category(
        "Backend Bound",
        PerfMetricType.BACKEND_BOUND,
        [
            TopDownL2Type.DTLB_EFFECTIVENESS,
            TopDownL2Type.L1D_CACHE_EFFECTIVENESS,
            TopDownL2Type.L2_CACHE_EFFECTIVENESS,
            TopDownL2Type.LL_CACHE_EFFECTIVENESS,
            TopDownL2Type.OPERATION_MIX,
        ],
    )
    categories[TopDownL1Type.RETIRING] = TopDownL1Category(
        "Retiring", PerfMetricType.RETIRING, [TopDownL2Type.OPERATION_MIX]
    )
    categories[TopDownL1Type.BAD_SPECULATION] = TopDownL1Category(
        "Bad Speculation",
        PerfMetricType.BAD_SPECULATION,
        [TopDownL2Type.BRANCH_EFFECTIVENESS],
    )

    return categories


_topdown_l1_categories: Optional[list[TopDownL1Category]] = None


def get_topdown_l1_categories() -> list[TopDownL1Category]:
    global _topdown_l1_categories
    if _topdown_l1_categories is None:
        _topdown_l1_categories = _fill_topdown_l1_categories()
    return _topdown_l1_categories


class TopDownL2Category:
    def __init__(
        self, name: str, description: str, metric_types: list[PerfMetricType]
    ) -> None:
        self.name = name
        self.description = description
        self.metric_types = metric_types


def _fill_topdown_l2_categories():
    categories = [None] * TopDownL2Type.LAST

    categories[TopDownL2Type.BRANCH_EFFECTIVENESS] = TopDownL2Category(
        "Branch Effectiveness",
        "Metrics to evaluate the effectiveness of branch instruction execution "
        "on this processor",
        [PerfMetricType.BR_MISPRED_MPKI, PerfMetricType.BR_MISPRED_RATIO],
    )
    categories[TopDownL2Type.ITLB_EFFECTIVENESS] = TopDownL2Category(
        "Instruction TLB Effectiveness",
        "Metrics to evaluate the effectiveness of instruction TLB on this processor",
        [
            PerfMetricType.ITLB_MPKI,
            PerfMetricType.L1I_TLB_MPKI,
            PerfMetricType.L2_UNIFIED_TLB_MPKI,
            PerfMetricType.ITLB_WALK_RATIO,
            PerfMetricType.L1I_TLB_MISS_RATIO,
            PerfMetricType.L2_UNIFIED_TLB_MISS_RATIO,
        ],
    )
    categories[TopDownL2Type.DTLB_EFFECTIVENESS] = TopDownL2Category(
        "Data TLB Effectiveness",
        "Metrics to evaluate the effectiveness of data TLB on this processor",
        [
            PerfMetricType.DTLB_MPKI,
            PerfMetricType.L1D_TLB_MPKI,
            PerfMetricType.L2_UNIFIED_TLB_MPKI,
            PerfMetricType.DTLB_WALK_RATIO,
            PerfMetricType.L1D_TLB_MISS_RATIO,
            PerfMetricType.L2_UNIFIED_TLB_MISS_RATIO,
        ],
    )
    categories[TopDownL2Type.L1I_CACHE_EFFECTIVENESS] = TopDownL2Category(
        "L1 Instruction Cache Effectiveness",
        "Metrics to evaluate the effectiveness of L1 Instruction Cache on this processor",
        [PerfMetricType.L1I_CACHE_MPKI, PerfMetricType.L1I_CACHE_MISS_RATIO],
    )
    categories[TopDownL2Type.L1D_CACHE_EFFECTIVENESS] = TopDownL2Category(
        "L1 Data Cache Effectiveness",
        "Metrics to evaluate the effectiveness of L1 Data Cache on this processor",
        [PerfMetricType.L1D_CACHE_MPKI, PerfMetricType.L1D_CACHE_MISS_RATIO],
    )
    categories[TopDownL2Type.L2_CACHE_EFFECTIVENESS] = TopDownL2Category(
        "L2 Unified Cache Effectiveness",
        "Metrics to evaluate the effectiveness of L2 Unified Cache on this processor",
        [PerfMetricType.L2_CACHE_MPKI, PerfMetricType.L2_CACHE_MISS_RATIO],
    )
    categories[TopDownL2Type.LL_CACHE_EFFECTIVENESS] = TopDownL2Category(
        "Last Level Cache Effectiveness",
        "Metrics to evaluate the effectiveness of Last Level Cache on this processor",
        [
            PerfMetricType.LL_CACHE_RD_MPKI,
            PerfMetricType.LL_CACHE_RD_MISS_RATIO,
            PerfMetricType.LL_CACHE_RD_HIT_RATIO,
        ],
    )
    categories[TopDownL2Type.OPERATION_MIX] = TopDownL2Category(
        "Speculative Operation Mix",
        "The distribution of micro-operation types executed for the program",
        [
            PerfMetricType.LOAD_OP_PERCENT,
            PerfMetricType.STORE_OP_PERCENT,
            PerfMetricType.INT_OP_PERCENT,
            PerfMetricType.ASE_OP_PERCENT,
            PerfMetricType.SCALAR_FLOP_PERCENT,
            PerfMetricType.BRANCH_OP_PERCENT,
            PerfMetricType.CRYPTO_OP_PERCENT,
            PerfMetricType.SVE_OP_PERCENT,
        ],
    )

    return categories


_topdown_l2_categories: Optional[list[TopDownL2Category]] = None


def get_topdown_l2_categories() -> list[TopDownL2Category]:
    global _topdown_l2_categories
    if _topdown_l2_categories is None:
        _topdown_l2_categories = _fill_topdown_l2_categories()
    return _topdown_l2_categories


def _fill_misc_metric_types() -> list[PerfMetricType]:
    l1_categories = get_topdown_l1_categories()
    l2_categories = get_topdown_l2_categories()

    topdown_metrics = [x.metric_type for x in l1_categories]
    for l2_category in l2_categories:
        topdown_metrics.extend(l2_category.metric_types)

    all_metrics = [PerfMetricType(x) for x in range(PerfMetricType.LAST)]
    return list(set(all_metrics) - set(topdown_metrics))


_misc_metric_types: Optional[list[PerfMetricType]] = None


def get_misc_metric_types() -> list[PerfMetricType]:
    global _misc_metric_types
    if _misc_metric_types is None:
        _misc_metric_types = _fill_misc_metric_types()
    return _misc_metric_types
