﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ssm/SSM_EXPORTS.h>
#include <aws/ssm/model/OpsItemDataValue.h>
#include <aws/ssm/model/OpsItemStatus.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace SSM {
namespace Model {

/**
 * <p>A count of OpsItems.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/ssm-2014-11-06/OpsItemSummary">AWS
 * API Reference</a></p>
 */
class OpsItemSummary {
 public:
  AWS_SSM_API OpsItemSummary() = default;
  AWS_SSM_API OpsItemSummary(Aws::Utils::Json::JsonView jsonValue);
  AWS_SSM_API OpsItemSummary& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_SSM_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the IAM entity that created the
   * OpsItem.</p>
   */
  inline const Aws::String& GetCreatedBy() const { return m_createdBy; }
  inline bool CreatedByHasBeenSet() const { return m_createdByHasBeenSet; }
  template <typename CreatedByT = Aws::String>
  void SetCreatedBy(CreatedByT&& value) {
    m_createdByHasBeenSet = true;
    m_createdBy = std::forward<CreatedByT>(value);
  }
  template <typename CreatedByT = Aws::String>
  OpsItemSummary& WithCreatedBy(CreatedByT&& value) {
    SetCreatedBy(std::forward<CreatedByT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time the OpsItem was created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreatedTime() const { return m_createdTime; }
  inline bool CreatedTimeHasBeenSet() const { return m_createdTimeHasBeenSet; }
  template <typename CreatedTimeT = Aws::Utils::DateTime>
  void SetCreatedTime(CreatedTimeT&& value) {
    m_createdTimeHasBeenSet = true;
    m_createdTime = std::forward<CreatedTimeT>(value);
  }
  template <typename CreatedTimeT = Aws::Utils::DateTime>
  OpsItemSummary& WithCreatedTime(CreatedTimeT&& value) {
    SetCreatedTime(std::forward<CreatedTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the IAM entity that created the
   * OpsItem.</p>
   */
  inline const Aws::String& GetLastModifiedBy() const { return m_lastModifiedBy; }
  inline bool LastModifiedByHasBeenSet() const { return m_lastModifiedByHasBeenSet; }
  template <typename LastModifiedByT = Aws::String>
  void SetLastModifiedBy(LastModifiedByT&& value) {
    m_lastModifiedByHasBeenSet = true;
    m_lastModifiedBy = std::forward<LastModifiedByT>(value);
  }
  template <typename LastModifiedByT = Aws::String>
  OpsItemSummary& WithLastModifiedBy(LastModifiedByT&& value) {
    SetLastModifiedBy(std::forward<LastModifiedByT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time the OpsItem was last updated.</p>
   */
  inline const Aws::Utils::DateTime& GetLastModifiedTime() const { return m_lastModifiedTime; }
  inline bool LastModifiedTimeHasBeenSet() const { return m_lastModifiedTimeHasBeenSet; }
  template <typename LastModifiedTimeT = Aws::Utils::DateTime>
  void SetLastModifiedTime(LastModifiedTimeT&& value) {
    m_lastModifiedTimeHasBeenSet = true;
    m_lastModifiedTime = std::forward<LastModifiedTimeT>(value);
  }
  template <typename LastModifiedTimeT = Aws::Utils::DateTime>
  OpsItemSummary& WithLastModifiedTime(LastModifiedTimeT&& value) {
    SetLastModifiedTime(std::forward<LastModifiedTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The importance of this OpsItem in relation to other OpsItems in the
   * system.</p>
   */
  inline int GetPriority() const { return m_priority; }
  inline bool PriorityHasBeenSet() const { return m_priorityHasBeenSet; }
  inline void SetPriority(int value) {
    m_priorityHasBeenSet = true;
    m_priority = value;
  }
  inline OpsItemSummary& WithPriority(int value) {
    SetPriority(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The impacted Amazon Web Services resource.</p>
   */
  inline const Aws::String& GetSource() const { return m_source; }
  inline bool SourceHasBeenSet() const { return m_sourceHasBeenSet; }
  template <typename SourceT = Aws::String>
  void SetSource(SourceT&& value) {
    m_sourceHasBeenSet = true;
    m_source = std::forward<SourceT>(value);
  }
  template <typename SourceT = Aws::String>
  OpsItemSummary& WithSource(SourceT&& value) {
    SetSource(std::forward<SourceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The OpsItem status.</p>
   */
  inline OpsItemStatus GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  inline void SetStatus(OpsItemStatus value) {
    m_statusHasBeenSet = true;
    m_status = value;
  }
  inline OpsItemSummary& WithStatus(OpsItemStatus value) {
    SetStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the OpsItem.</p>
   */
  inline const Aws::String& GetOpsItemId() const { return m_opsItemId; }
  inline bool OpsItemIdHasBeenSet() const { return m_opsItemIdHasBeenSet; }
  template <typename OpsItemIdT = Aws::String>
  void SetOpsItemId(OpsItemIdT&& value) {
    m_opsItemIdHasBeenSet = true;
    m_opsItemId = std::forward<OpsItemIdT>(value);
  }
  template <typename OpsItemIdT = Aws::String>
  OpsItemSummary& WithOpsItemId(OpsItemIdT&& value) {
    SetOpsItemId(std::forward<OpsItemIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A short heading that describes the nature of the OpsItem and the impacted
   * resource.</p>
   */
  inline const Aws::String& GetTitle() const { return m_title; }
  inline bool TitleHasBeenSet() const { return m_titleHasBeenSet; }
  template <typename TitleT = Aws::String>
  void SetTitle(TitleT&& value) {
    m_titleHasBeenSet = true;
    m_title = std::forward<TitleT>(value);
  }
  template <typename TitleT = Aws::String>
  OpsItemSummary& WithTitle(TitleT&& value) {
    SetTitle(std::forward<TitleT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Operational data is custom data that provides useful reference details about
   * the OpsItem. </p>
   */
  inline const Aws::Map<Aws::String, OpsItemDataValue>& GetOperationalData() const { return m_operationalData; }
  inline bool OperationalDataHasBeenSet() const { return m_operationalDataHasBeenSet; }
  template <typename OperationalDataT = Aws::Map<Aws::String, OpsItemDataValue>>
  void SetOperationalData(OperationalDataT&& value) {
    m_operationalDataHasBeenSet = true;
    m_operationalData = std::forward<OperationalDataT>(value);
  }
  template <typename OperationalDataT = Aws::Map<Aws::String, OpsItemDataValue>>
  OpsItemSummary& WithOperationalData(OperationalDataT&& value) {
    SetOperationalData(std::forward<OperationalDataT>(value));
    return *this;
  }
  template <typename OperationalDataKeyT = Aws::String, typename OperationalDataValueT = OpsItemDataValue>
  OpsItemSummary& AddOperationalData(OperationalDataKeyT&& key, OperationalDataValueT&& value) {
    m_operationalDataHasBeenSet = true;
    m_operationalData.emplace(std::forward<OperationalDataKeyT>(key), std::forward<OperationalDataValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of OpsItems by category.</p>
   */
  inline const Aws::String& GetCategory() const { return m_category; }
  inline bool CategoryHasBeenSet() const { return m_categoryHasBeenSet; }
  template <typename CategoryT = Aws::String>
  void SetCategory(CategoryT&& value) {
    m_categoryHasBeenSet = true;
    m_category = std::forward<CategoryT>(value);
  }
  template <typename CategoryT = Aws::String>
  OpsItemSummary& WithCategory(CategoryT&& value) {
    SetCategory(std::forward<CategoryT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of OpsItems by severity.</p>
   */
  inline const Aws::String& GetSeverity() const { return m_severity; }
  inline bool SeverityHasBeenSet() const { return m_severityHasBeenSet; }
  template <typename SeverityT = Aws::String>
  void SetSeverity(SeverityT&& value) {
    m_severityHasBeenSet = true;
    m_severity = std::forward<SeverityT>(value);
  }
  template <typename SeverityT = Aws::String>
  OpsItemSummary& WithSeverity(SeverityT&& value) {
    SetSeverity(std::forward<SeverityT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of OpsItem. Systems Manager supports the following types of
   * OpsItems:</p> <ul> <li> <p> <code>/aws/issue</code> </p> <p>This type of OpsItem
   * is used for default OpsItems created by OpsCenter. </p> </li> <li> <p>
   * <code>/aws/changerequest</code> </p> <p>This type of OpsItem is used by Change
   * Manager for reviewing and approving or rejecting change requests. </p> </li>
   * <li> <p> <code>/aws/insight</code> </p> <p>This type of OpsItem is used by
   * OpsCenter for aggregating and reporting on duplicate OpsItems. </p> </li> </ul>
   */
  inline const Aws::String& GetOpsItemType() const { return m_opsItemType; }
  inline bool OpsItemTypeHasBeenSet() const { return m_opsItemTypeHasBeenSet; }
  template <typename OpsItemTypeT = Aws::String>
  void SetOpsItemType(OpsItemTypeT&& value) {
    m_opsItemTypeHasBeenSet = true;
    m_opsItemType = std::forward<OpsItemTypeT>(value);
  }
  template <typename OpsItemTypeT = Aws::String>
  OpsItemSummary& WithOpsItemType(OpsItemTypeT&& value) {
    SetOpsItemType(std::forward<OpsItemTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The time a runbook workflow started. Currently reported only for the OpsItem
   * type <code>/aws/changerequest</code>.</p>
   */
  inline const Aws::Utils::DateTime& GetActualStartTime() const { return m_actualStartTime; }
  inline bool ActualStartTimeHasBeenSet() const { return m_actualStartTimeHasBeenSet; }
  template <typename ActualStartTimeT = Aws::Utils::DateTime>
  void SetActualStartTime(ActualStartTimeT&& value) {
    m_actualStartTimeHasBeenSet = true;
    m_actualStartTime = std::forward<ActualStartTimeT>(value);
  }
  template <typename ActualStartTimeT = Aws::Utils::DateTime>
  OpsItemSummary& WithActualStartTime(ActualStartTimeT&& value) {
    SetActualStartTime(std::forward<ActualStartTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The time a runbook workflow ended. Currently reported only for the OpsItem
   * type <code>/aws/changerequest</code>.</p>
   */
  inline const Aws::Utils::DateTime& GetActualEndTime() const { return m_actualEndTime; }
  inline bool ActualEndTimeHasBeenSet() const { return m_actualEndTimeHasBeenSet; }
  template <typename ActualEndTimeT = Aws::Utils::DateTime>
  void SetActualEndTime(ActualEndTimeT&& value) {
    m_actualEndTimeHasBeenSet = true;
    m_actualEndTime = std::forward<ActualEndTimeT>(value);
  }
  template <typename ActualEndTimeT = Aws::Utils::DateTime>
  OpsItemSummary& WithActualEndTime(ActualEndTimeT&& value) {
    SetActualEndTime(std::forward<ActualEndTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The time specified in a change request for a runbook workflow to start.
   * Currently supported only for the OpsItem type
   * <code>/aws/changerequest</code>.</p>
   */
  inline const Aws::Utils::DateTime& GetPlannedStartTime() const { return m_plannedStartTime; }
  inline bool PlannedStartTimeHasBeenSet() const { return m_plannedStartTimeHasBeenSet; }
  template <typename PlannedStartTimeT = Aws::Utils::DateTime>
  void SetPlannedStartTime(PlannedStartTimeT&& value) {
    m_plannedStartTimeHasBeenSet = true;
    m_plannedStartTime = std::forward<PlannedStartTimeT>(value);
  }
  template <typename PlannedStartTimeT = Aws::Utils::DateTime>
  OpsItemSummary& WithPlannedStartTime(PlannedStartTimeT&& value) {
    SetPlannedStartTime(std::forward<PlannedStartTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The time specified in a change request for a runbook workflow to end.
   * Currently supported only for the OpsItem type
   * <code>/aws/changerequest</code>.</p>
   */
  inline const Aws::Utils::DateTime& GetPlannedEndTime() const { return m_plannedEndTime; }
  inline bool PlannedEndTimeHasBeenSet() const { return m_plannedEndTimeHasBeenSet; }
  template <typename PlannedEndTimeT = Aws::Utils::DateTime>
  void SetPlannedEndTime(PlannedEndTimeT&& value) {
    m_plannedEndTimeHasBeenSet = true;
    m_plannedEndTime = std::forward<PlannedEndTimeT>(value);
  }
  template <typename PlannedEndTimeT = Aws::Utils::DateTime>
  OpsItemSummary& WithPlannedEndTime(PlannedEndTimeT&& value) {
    SetPlannedEndTime(std::forward<PlannedEndTimeT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_createdBy;

  Aws::Utils::DateTime m_createdTime{};

  Aws::String m_lastModifiedBy;

  Aws::Utils::DateTime m_lastModifiedTime{};

  int m_priority{0};

  Aws::String m_source;

  OpsItemStatus m_status{OpsItemStatus::NOT_SET};

  Aws::String m_opsItemId;

  Aws::String m_title;

  Aws::Map<Aws::String, OpsItemDataValue> m_operationalData;

  Aws::String m_category;

  Aws::String m_severity;

  Aws::String m_opsItemType;

  Aws::Utils::DateTime m_actualStartTime{};

  Aws::Utils::DateTime m_actualEndTime{};

  Aws::Utils::DateTime m_plannedStartTime{};

  Aws::Utils::DateTime m_plannedEndTime{};
  bool m_createdByHasBeenSet = false;
  bool m_createdTimeHasBeenSet = false;
  bool m_lastModifiedByHasBeenSet = false;
  bool m_lastModifiedTimeHasBeenSet = false;
  bool m_priorityHasBeenSet = false;
  bool m_sourceHasBeenSet = false;
  bool m_statusHasBeenSet = false;
  bool m_opsItemIdHasBeenSet = false;
  bool m_titleHasBeenSet = false;
  bool m_operationalDataHasBeenSet = false;
  bool m_categoryHasBeenSet = false;
  bool m_severityHasBeenSet = false;
  bool m_opsItemTypeHasBeenSet = false;
  bool m_actualStartTimeHasBeenSet = false;
  bool m_actualEndTimeHasBeenSet = false;
  bool m_plannedStartTimeHasBeenSet = false;
  bool m_plannedEndTimeHasBeenSet = false;
};

}  // namespace Model
}  // namespace SSM
}  // namespace Aws
