﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/glue/Glue_EXPORTS.h>
#include <aws/glue/model/IntegrationError.h>
#include <aws/glue/model/IntegrationStatus.h>
#include <aws/glue/model/Tag.h>

#include <utility>

namespace Aws {
template <typename RESULT_TYPE>
class AmazonWebServiceResult;

namespace Utils {
namespace Json {
class JsonValue;
}  // namespace Json
}  // namespace Utils
namespace Glue {
namespace Model {
class DeleteIntegrationResult {
 public:
  AWS_GLUE_API DeleteIntegrationResult() = default;
  AWS_GLUE_API DeleteIntegrationResult(const Aws::AmazonWebServiceResult<Aws::Utils::Json::JsonValue>& result);
  AWS_GLUE_API DeleteIntegrationResult& operator=(const Aws::AmazonWebServiceResult<Aws::Utils::Json::JsonValue>& result);

  ///@{
  /**
   * <p>The ARN of the source for the integration.</p>
   */
  inline const Aws::String& GetSourceArn() const { return m_sourceArn; }
  template <typename SourceArnT = Aws::String>
  void SetSourceArn(SourceArnT&& value) {
    m_sourceArnHasBeenSet = true;
    m_sourceArn = std::forward<SourceArnT>(value);
  }
  template <typename SourceArnT = Aws::String>
  DeleteIntegrationResult& WithSourceArn(SourceArnT&& value) {
    SetSourceArn(std::forward<SourceArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the target for the integration.</p>
   */
  inline const Aws::String& GetTargetArn() const { return m_targetArn; }
  template <typename TargetArnT = Aws::String>
  void SetTargetArn(TargetArnT&& value) {
    m_targetArnHasBeenSet = true;
    m_targetArn = std::forward<TargetArnT>(value);
  }
  template <typename TargetArnT = Aws::String>
  DeleteIntegrationResult& WithTargetArn(TargetArnT&& value) {
    SetTargetArn(std::forward<TargetArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique name for an integration in Glue.</p>
   */
  inline const Aws::String& GetIntegrationName() const { return m_integrationName; }
  template <typename IntegrationNameT = Aws::String>
  void SetIntegrationName(IntegrationNameT&& value) {
    m_integrationNameHasBeenSet = true;
    m_integrationName = std::forward<IntegrationNameT>(value);
  }
  template <typename IntegrationNameT = Aws::String>
  DeleteIntegrationResult& WithIntegrationName(IntegrationNameT&& value) {
    SetIntegrationName(std::forward<IntegrationNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A description of the integration.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  DeleteIntegrationResult& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) for the integration.</p>
   */
  inline const Aws::String& GetIntegrationArn() const { return m_integrationArn; }
  template <typename IntegrationArnT = Aws::String>
  void SetIntegrationArn(IntegrationArnT&& value) {
    m_integrationArnHasBeenSet = true;
    m_integrationArn = std::forward<IntegrationArnT>(value);
  }
  template <typename IntegrationArnT = Aws::String>
  DeleteIntegrationResult& WithIntegrationArn(IntegrationArnT&& value) {
    SetIntegrationArn(std::forward<IntegrationArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of a KMS key used for encrypting the channel.</p>
   */
  inline const Aws::String& GetKmsKeyId() const { return m_kmsKeyId; }
  template <typename KmsKeyIdT = Aws::String>
  void SetKmsKeyId(KmsKeyIdT&& value) {
    m_kmsKeyIdHasBeenSet = true;
    m_kmsKeyId = std::forward<KmsKeyIdT>(value);
  }
  template <typename KmsKeyIdT = Aws::String>
  DeleteIntegrationResult& WithKmsKeyId(KmsKeyIdT&& value) {
    SetKmsKeyId(std::forward<KmsKeyIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional set of non-secret key–value pairs that contains additional
   * contextual information for encryption.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetAdditionalEncryptionContext() const { return m_additionalEncryptionContext; }
  template <typename AdditionalEncryptionContextT = Aws::Map<Aws::String, Aws::String>>
  void SetAdditionalEncryptionContext(AdditionalEncryptionContextT&& value) {
    m_additionalEncryptionContextHasBeenSet = true;
    m_additionalEncryptionContext = std::forward<AdditionalEncryptionContextT>(value);
  }
  template <typename AdditionalEncryptionContextT = Aws::Map<Aws::String, Aws::String>>
  DeleteIntegrationResult& WithAdditionalEncryptionContext(AdditionalEncryptionContextT&& value) {
    SetAdditionalEncryptionContext(std::forward<AdditionalEncryptionContextT>(value));
    return *this;
  }
  template <typename AdditionalEncryptionContextKeyT = Aws::String, typename AdditionalEncryptionContextValueT = Aws::String>
  DeleteIntegrationResult& AddAdditionalEncryptionContext(AdditionalEncryptionContextKeyT&& key,
                                                          AdditionalEncryptionContextValueT&& value) {
    m_additionalEncryptionContextHasBeenSet = true;
    m_additionalEncryptionContext.emplace(std::forward<AdditionalEncryptionContextKeyT>(key),
                                          std::forward<AdditionalEncryptionContextValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Metadata assigned to the resource consisting of a list of key-value
   * pairs.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  DeleteIntegrationResult& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  DeleteIntegrationResult& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status of the integration being deleted.</p> <p>The possible statuses
   * are:</p> <ul> <li> <p>CREATING: The integration is being created.</p> </li> <li>
   * <p>ACTIVE: The integration creation succeeds.</p> </li> <li> <p>MODIFYING: The
   * integration is being modified.</p> </li> <li> <p>FAILED: The integration
   * creation fails. </p> </li> <li> <p>DELETING: The integration is deleted.</p>
   * </li> <li> <p>SYNCING: The integration is synchronizing.</p> </li> <li>
   * <p>NEEDS_ATTENTION: The integration needs attention, such as
   * synchronization.</p> </li> </ul>
   */
  inline IntegrationStatus GetStatus() const { return m_status; }
  inline void SetStatus(IntegrationStatus value) {
    m_statusHasBeenSet = true;
    m_status = value;
  }
  inline DeleteIntegrationResult& WithStatus(IntegrationStatus value) {
    SetStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The time when the integration was created, in UTC.</p>
   */
  inline const Aws::Utils::DateTime& GetCreateTime() const { return m_createTime; }
  template <typename CreateTimeT = Aws::Utils::DateTime>
  void SetCreateTime(CreateTimeT&& value) {
    m_createTimeHasBeenSet = true;
    m_createTime = std::forward<CreateTimeT>(value);
  }
  template <typename CreateTimeT = Aws::Utils::DateTime>
  DeleteIntegrationResult& WithCreateTime(CreateTimeT&& value) {
    SetCreateTime(std::forward<CreateTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of errors associated with the integration.</p>
   */
  inline const Aws::Vector<IntegrationError>& GetErrors() const { return m_errors; }
  template <typename ErrorsT = Aws::Vector<IntegrationError>>
  void SetErrors(ErrorsT&& value) {
    m_errorsHasBeenSet = true;
    m_errors = std::forward<ErrorsT>(value);
  }
  template <typename ErrorsT = Aws::Vector<IntegrationError>>
  DeleteIntegrationResult& WithErrors(ErrorsT&& value) {
    SetErrors(std::forward<ErrorsT>(value));
    return *this;
  }
  template <typename ErrorsT = IntegrationError>
  DeleteIntegrationResult& AddErrors(ErrorsT&& value) {
    m_errorsHasBeenSet = true;
    m_errors.emplace_back(std::forward<ErrorsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Selects source tables for the integration using Maxwell filter syntax.</p>
   */
  inline const Aws::String& GetDataFilter() const { return m_dataFilter; }
  template <typename DataFilterT = Aws::String>
  void SetDataFilter(DataFilterT&& value) {
    m_dataFilterHasBeenSet = true;
    m_dataFilter = std::forward<DataFilterT>(value);
  }
  template <typename DataFilterT = Aws::String>
  DeleteIntegrationResult& WithDataFilter(DataFilterT&& value) {
    SetDataFilter(std::forward<DataFilterT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::String& GetRequestId() const { return m_requestId; }
  template <typename RequestIdT = Aws::String>
  void SetRequestId(RequestIdT&& value) {
    m_requestIdHasBeenSet = true;
    m_requestId = std::forward<RequestIdT>(value);
  }
  template <typename RequestIdT = Aws::String>
  DeleteIntegrationResult& WithRequestId(RequestIdT&& value) {
    SetRequestId(std::forward<RequestIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_sourceArn;

  Aws::String m_targetArn;

  Aws::String m_integrationName;

  Aws::String m_description;

  Aws::String m_integrationArn;

  Aws::String m_kmsKeyId;

  Aws::Map<Aws::String, Aws::String> m_additionalEncryptionContext;

  Aws::Vector<Tag> m_tags;

  IntegrationStatus m_status{IntegrationStatus::NOT_SET};

  Aws::Utils::DateTime m_createTime{};

  Aws::Vector<IntegrationError> m_errors;

  Aws::String m_dataFilter;

  Aws::String m_requestId;
  bool m_sourceArnHasBeenSet = false;
  bool m_targetArnHasBeenSet = false;
  bool m_integrationNameHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_integrationArnHasBeenSet = false;
  bool m_kmsKeyIdHasBeenSet = false;
  bool m_additionalEncryptionContextHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
  bool m_statusHasBeenSet = false;
  bool m_createTimeHasBeenSet = false;
  bool m_errorsHasBeenSet = false;
  bool m_dataFilterHasBeenSet = false;
  bool m_requestIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace Glue
}  // namespace Aws
