/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.component.google.pubsub;

import java.util.Map;
import java.util.concurrent.ExecutorService;

import com.google.api.gax.retrying.RetrySettings;
import org.apache.camel.Category;
import org.apache.camel.Component;
import org.apache.camel.Consumer;
import org.apache.camel.ExchangePattern;
import org.apache.camel.Processor;
import org.apache.camel.Producer;
import org.apache.camel.component.google.pubsub.serializer.DefaultGooglePubsubSerializer;
import org.apache.camel.component.google.pubsub.serializer.GooglePubsubSerializer;
import org.apache.camel.spi.*;
import org.apache.camel.support.DefaultEndpoint;
import org.apache.camel.util.ObjectHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Send and receive messages to/from Google Cloud Platform PubSub Service.
 * <p/>
 * Built on top of the Google Cloud Pub/Sub libraries.
 */
@UriEndpoint(firstVersion = "2.19.0", scheme = "google-pubsub", title = "Google Pubsub",
             syntax = "google-pubsub:projectId:destinationName", category = { Category.CLOUD, Category.MESSAGING },
             headersClass = GooglePubsubConstants.class)
public class GooglePubsubEndpoint extends DefaultEndpoint implements EndpointServiceLocation, HeaderFilterStrategyAware {

    private Logger log;

    @UriPath(label = "common", description = "The Google Cloud PubSub Project Id")
    @Metadata(required = true)
    private String projectId;
    @UriPath(label = "common",
             description = "The Destination Name. For the consumer this will be the subscription name, while for the producer this will be the topic name.")
    @Metadata(required = true)
    private String destinationName;
    @UriParam(label = "security",
              description = "Use Credentials when interacting with PubSub service (no authentication is required when using emulator).",
              defaultValue = "true")
    private boolean authenticate = true;
    @UriParam(label = "security",
              description = "The Service account key that can be used as credentials for the PubSub publisher/subscriber. It can be loaded by default from "
                            + " classpath, but you can prefix with classpath:, file:, or http: to load the resource from different systems.")
    private String serviceAccountKey;
    @Deprecated
    @UriParam(label = "advanced", description = "To use a custom logger name")
    private String loggerId;
    @UriParam(label = "consumer,advanced", name = "concurrentConsumers",
              description = "The number of parallel streams consuming from the subscription",
              defaultValue = "1")
    private Integer concurrentConsumers = 1;
    @UriParam(label = "consumer,advanced", name = "maxMessagesPerPoll",
              description = "The max number of messages to receive from the server in a single API call", defaultValue = "1")
    private Integer maxMessagesPerPoll = 1;
    @UriParam(label = "consumer,advanced", name = "synchronousPull", description = "Synchronously pull batches of messages",
              defaultValue = "false")
    private boolean synchronousPull;
    @UriParam(label = "consumer", defaultValue = "AUTO", enums = "AUTO,NONE",
              description = "AUTO = exchange gets ack'ed/nack'ed on completion. NONE = downstream process has to ack/nack explicitly")
    private GooglePubsubConstants.AckMode ackMode = GooglePubsubConstants.AckMode.AUTO;
    @UriParam(label = "consumer,advanced", name = "maxAckExtensionPeriod",
              description = "Set the maximum period a message ack deadline will be extended. Value in seconds",
              defaultValue = "3600")
    private int maxAckExtensionPeriod = 3600;
    @UriParam(label = "producer,advanced",
              description = "Should message ordering be enabled")
    private boolean messageOrderingEnabled;
    @UriParam(label = "producer,advanced",
              description = "Pub/Sub endpoint to use. Required when using message ordering, and ensures that messages are received in order even when multiple publishers are used")
    private String pubsubEndpoint;
    @UriParam(label = "producer,advanced",
              description = "A custom GooglePubsubSerializer to use for serializing message payloads in the producer")
    @Metadata(autowired = true)
    private GooglePubsubSerializer serializer;
    @UriParam(label = "producer,advanced",
              description = "A custom RetrySettings to control how the publisher handles retry-able failures")
    private RetrySettings retry;
    @UriParam(label = "advanced",
              description = "Whether to include all Google headers when mapping from Pubsub to Camel Message."
                            + " Setting this to true will include properties such as x-goog etc.")
    private boolean includeAllGoogleProperties;
    @UriParam(label = "advanced",
              description = "To use a custom HeaderFilterStrategy to filter headers to and from Camel message.")
    private HeaderFilterStrategy headerFilterStrategy;

    public GooglePubsubEndpoint(String uri, Component component) {
        super(uri, component);

        if (!(component instanceof GooglePubsubComponent)) {
            throw new IllegalArgumentException(
                    "The component provided is not GooglePubsubComponent : " + component.getClass().getName());
        }
    }

    @Override
    public GooglePubsubComponent getComponent() {
        return (GooglePubsubComponent) super.getComponent();
    }

    public void afterPropertiesSet() {
        if (ObjectHelper.isEmpty(loggerId)) {
            log = LoggerFactory.getLogger(this.getClass().getName());
        } else {
            log = LoggerFactory.getLogger(loggerId);
        }

        // Default pubsub connection.
        // With the publisher endpoints - the main publisher
        // with the consumer endpoints - the ack client

        log.trace("Project ID: {}", this.projectId);
        log.trace("Destination Name: {}", this.destinationName);

        if (headerFilterStrategy == null) {
            headerFilterStrategy = new GooglePubsubHeaderFilterStrategy(includeAllGoogleProperties);
        }
    }

    @Override
    public Producer createProducer() throws Exception {
        afterPropertiesSet();
        if (ObjectHelper.isEmpty(serializer)) {
            serializer = new DefaultGooglePubsubSerializer();
        }
        return new GooglePubsubProducer(this);
    }

    @Override
    public Consumer createConsumer(Processor processor) throws Exception {
        afterPropertiesSet();
        setExchangePattern(ExchangePattern.InOnly);
        GooglePubsubConsumer consumer = new GooglePubsubConsumer(this, processor);
        configureConsumer(consumer);
        return consumer;
    }

    public ExecutorService createExecutor(Object source) {
        return getCamelContext().getExecutorServiceManager().newFixedThreadPool(source,
                "GooglePubsubConsumer[" + getDestinationName() + "]", concurrentConsumers);
    }

    public String getProjectId() {
        return projectId;
    }

    public void setProjectId(String projectId) {
        this.projectId = projectId;
    }

    public String getLoggerId() {
        return loggerId;
    }

    public void setLoggerId(String loggerId) {
        this.loggerId = loggerId;
    }

    public boolean isAuthenticate() {
        return authenticate;
    }

    public void setAuthenticate(boolean authenticate) {
        this.authenticate = authenticate;
    }

    public String getServiceAccountKey() {
        return serviceAccountKey;
    }

    public void setServiceAccountKey(String serviceAccountKey) {
        this.serviceAccountKey = serviceAccountKey;
    }

    public String getDestinationName() {
        return destinationName;
    }

    public void setDestinationName(String destinationName) {
        this.destinationName = destinationName;
    }

    public Integer getConcurrentConsumers() {
        return concurrentConsumers;
    }

    public void setConcurrentConsumers(Integer concurrentConsumers) {
        this.concurrentConsumers = concurrentConsumers;
    }

    public Integer getMaxMessagesPerPoll() {
        return maxMessagesPerPoll;
    }

    public void setMaxMessagesPerPoll(Integer maxMessagesPerPoll) {
        this.maxMessagesPerPoll = maxMessagesPerPoll;
    }

    public boolean isIncludeAllGoogleProperties() {
        return includeAllGoogleProperties;
    }

    public void setIncludeAllGoogleProperties(Boolean includeAllGoogleProperties) {
        this.includeAllGoogleProperties = includeAllGoogleProperties;
    }

    public GooglePubsubConstants.AckMode getAckMode() {
        return ackMode;
    }

    public void setAckMode(GooglePubsubConstants.AckMode ackMode) {
        this.ackMode = ackMode;
    }

    public int getMaxAckExtensionPeriod() {
        return maxAckExtensionPeriod;
    }

    public void setMaxAckExtensionPeriod(int maxAckExtensionPeriod) {
        this.maxAckExtensionPeriod = maxAckExtensionPeriod;
    }

    public GooglePubsubSerializer getSerializer() {
        return serializer;
    }

    public void setSerializer(GooglePubsubSerializer serializer) {
        this.serializer = serializer;
    }

    public boolean isMessageOrderingEnabled() {
        return this.messageOrderingEnabled;
    }

    public void setMessageOrderingEnabled(boolean messageOrderingEnabled) {
        this.messageOrderingEnabled = messageOrderingEnabled;
    }

    public RetrySettings getRetry() {
        return retry;
    }

    public void setRetry(RetrySettings retry) {
        this.retry = retry;
    }

    public boolean isSynchronousPull() {
        return synchronousPull;
    }

    public void setSynchronousPull(Boolean synchronousPull) {
        this.synchronousPull = synchronousPull;
    }

    public String getPubsubEndpoint() {
        return this.pubsubEndpoint;
    }

    public void setPubsubEndpoint(String pubsubEndpoint) {
        this.pubsubEndpoint = pubsubEndpoint;
    }

    @Override
    public String getServiceUrl() {
        if (ObjectHelper.isNotEmpty(getPubsubEndpoint())) {
            return getServiceProtocol() + ":" + getPubsubEndpoint();
        }
        return null;
    }

    @Override
    public String getServiceProtocol() {
        return "pubsub";
    }

    @Override
    public Map<String, String> getServiceMetadata() {
        if (getDestinationName() != null) {
            return Map.of("destinationName", getDestinationName());
        }
        return null;
    }

    @Override
    public HeaderFilterStrategy getHeaderFilterStrategy() {
        return headerFilterStrategy;
    }

    @Override
    public void setHeaderFilterStrategy(HeaderFilterStrategy headerFilterStrategy) {
        this.headerFilterStrategy = headerFilterStrategy;
    }
}
