//
//  MSDocument.m
//  Manuscript
//
//  Created by 二鏡 on 12/01/30.
//  Copyright (c) 2012年 二鏡庵. All rights reserved.
//


#import "MSDocument.h"
#import "MSPager.h"
#import "MSPageFormat.h"
#import "MSTextView.h"
#import "MSLayoutBlockManager.h"
#import "MSDocumentController.h"
#import "MSFindPanelController.h"
#import "MSFontServices.h"
#import "Defaults.h"
#import "MSScriptExecutor.h"

// mapのfile名->辞書で、その中にさらに編集状態が格納されている
// 外部アプリでエンコードを変えた場合は不定
// selectionが不正領域になったら行末とする
static NSString *udkEncodingKey = @"encoding";
static NSString *udkSelectionKey = @"selection";
static NSString *udkFontFamilyKey = @"family";
static NSString *udkFontFaceKey = @"face";
static NSString *udkFontSizeKey = @"fontSize";
static NSString *udkBoxCountKey = @"boxCount";
static NSString *udkGridKey = @"showsGrid";

static NSString *sDocumentRestoreInfo = @"MSDocumentResotreInfo";
static NSArray *sFontFamilyList;
static NSArray *sLocalizedFamilyList;

static NSString *envManuscriptWorkingFile = @"Manuscript_Working_File";

@interface MSDocument ()
@property (readwrite,copy) NSString *_family;
@property (readwrite,copy) NSString *_face;
@end


@interface MSDocument (Font)
@end

@implementation MSDocument
@synthesize encoding, _family, _face;
+ (void)initialize
{
    if(self == [MSDocument class])
    {
        id fontManager = [NSFontManager sharedFontManager];
        id families = [[fontManager availableFontFamilies] mutableCopy];
        [families sortUsingSelector:@selector(caseInsensitiveCompare:)];
        sFontFamilyList = [families copy];
        
        id repList = [NSMutableArray array];
        for(id family in sFontFamilyList)
        {
            id localizedName = [fontManager localizedNameForFamily: family face: nil];
            [repList addObject: localizedName];
        }
        sLocalizedFamilyList = [repList copy];
    }
}

- (id)init
{
    self = [super init];
    if (self) {
        // Add your subclass-specific initialization here.
        // If an error occurs here, return nil.
        content = [[MSLayoutBlockManager alloc] init];
        pager = [[MSPager alloc] init];
        pager.content = content;
        
        id ud = [NSUserDefaults standardUserDefaults];
        CGFloat fontSize = [ud doubleForKey: udFontSize];
        NSInteger boxCount = [ud integerForKey: udBoxCount];
        id fontName = ({
            self._family = [ud stringForKey: udFontFamily];
            self._face = [ud stringForKey: udFontFace];
            [MSFontServices fontNameWithFamily: self._family face: self._face];
        });
        
        MSPageFormat *format = [[MSPageFormat alloc] initWithSpacingMode: eLineSpacingMode_Half
                                                                 padding: NSMakeSize(8.0,16.0) 
                                                                fontName: fontName
                                                                fontSize: fontSize
                                                                boxCount: boxCount];
        format.showsGrid = [ud boolForKey: udShowsGrid];
        pager.format = format;
        self.encoding = NSUTF8StringEncoding;
    }
    return self;
}

- (void)setupFontMenu
{
    [fontFamily removeAllItems];
    [fontFamily addItemsWithTitles: sLocalizedFamilyList];
    NSUInteger index = [sFontFamilyList indexOfObject: self._family];
    [fontFamily selectItemAtIndex: index];

    // faces
    id faces = [MSFontServices facesForFamily:[sFontFamilyList objectAtIndex: index]];
    [self _setFontFaces: faces];
    NSInteger faceIndex = [fontFace indexOfItemWithTitle: self._face];
    if(faceIndex == -1)
    {
        self._face = [faces objectAtIndex: 0];
        faceIndex = 0;
    }
    [fontFace selectItemAtIndex: faceIndex];
}

- (void)_applyFont
{
    pager.format.fontName = [MSFontServices fontNameWithFamily: self._family
                                                          face: self._face];
}

- (void)_setFontFaces:(NSArray*)faces
{
    [fontFace removeAllItems];
    [fontFace addItemsWithTitles: faces];
}

- (IBAction)fontFamilyChanged:(id)sender
{
    NSInteger index = [fontFamily indexOfSelectedItem];
    id aFamily = [sFontFamilyList objectAtIndex: index];
    if([aFamily isEqualToString: self._family])
        return; // 不変更なら何もしない。手抜き実装。
    
    self._family = aFamily;
    id faces = [MSFontServices facesForFamily: aFamily];
    [self _setFontFaces: faces];
    self._face = [faces objectAtIndex: 0];
    [self _applyFont];
}

- (IBAction)fontFaceChanged:(id)sender
{
    self._face = [[fontFace selectedItem] title];
    [self _applyFont];
}

- (void)_loadSelection:(NSRange)selection
{    
    if(NSEqualRanges(selection, NSMakeRange(0,0)))
    {
        [textView moveToEndOfDocument: self];
    }
    else
    {
        NSUInteger length = pager.content.textStorage.length;
        if( NSMaxRange(selection) > length )
            [textView moveToEndOfDocument: self];
        else
            [textView setSelectedRange: selection];
    }
}

- (void)awakeFromNib
{
    pager.textView = textView;
    textView.delegate = self;
    
    id window = [textView window];
    [window useOptimizedDrawing: YES];
    
    NSSize size = [pager.format standardSizeForNewWindow];
    NSRect visibleFrame = [[NSScreen mainScreen] visibleFrame];
    size.width  = MIN(size.width, NSWidth(visibleFrame));
    size.height = MIN(size.height, NSHeight(visibleFrame));
    [window setContentSize: size];
    [window center];
    
    // font メニューを構成
    [self setupFontMenu];

    [self _loadSelection: initialState.selection];
}

- (NSString *)windowNibName
{
    // Override returning the nib file name of the document
    // If you need to use a subclass of NSWindowController or if your document supports multiple NSWindowControllers, you should remove this method and override -makeWindowControllers instead.
    return @"MSDocument";
}

- (NSData *)dataOfType:(NSString *)typeName error:(NSError **)outError
{
    if(UTTypeConformsTo((CFStringRef)typeName, kUTTypePlainText))
    {
        id data = [pager.content dataWithEncoding: self.encoding];
        return data;
    }
    
    if(UTTypeConformsTo((CFStringRef)typeName, kUTTypeRTF))
    {
        id data = [pager.content RTFData];
        return data;
    }
    return nil;
}

- (void)_syncFontMenu
{
    NSUInteger index = [sFontFamilyList indexOfObject: self._family];
    if(index == NSNotFound)
    {
        id ud = [NSUserDefaults standardUserDefaults];
        id defaultFamily = [ud stringForKey: udFontFamily];
        index = [sFontFamilyList indexOfObject: defaultFamily];
    }
    [fontFamily selectItemAtIndex: index];
    [fontFace selectItemWithTitle: self._face];
}

- (void)_restoreFileInfo:(NSDictionary*)info
{
    self.encoding = [[info objectForKey: udkEncodingKey] integerValue];
    
    id rangeData = [info objectForKey: udkSelectionKey];
    id val = [NSUnarchiver unarchiveObjectWithData: rangeData];
    initialState.selection = [val rangeValue];

    // update format
    MSPageFormat *format = pager.format;
    format.boxCount = [[info objectForKey: udkBoxCountKey] integerValue];
    format.fontSize = [[info objectForKey: udkFontSizeKey] floatValue];
    format.showsGrid = [[info objectForKey: udkGridKey] boolValue];
    self._family = [info objectForKey: udkFontFamilyKey];
    self._face = [info objectForKey: udkFontFaceKey];
    format.fontName = [MSFontServices fontNameWithFamily: self._family 
                                                    face: self._face];
}

- (BOOL)readFromData:(NSData *)data ofType:(NSString *)typeName error:(NSError **)outError
{
    if(UTTypeConformsTo((CFStringRef)typeName, kUTTypePlainText))
    {
        
        // 条件が書きにくいため、先にデフォルトの挙動
        NSInteger aEncoding = [[NSDocumentController sharedDocumentController] selectedEncoding];
        if(aEncoding == 0)
            aEncoding = NSUTF8StringEncoding;        
        self.encoding = aEncoding;
        initialState.selection = NSMakeRange(0,0);

        // 可能ならばUDで上書き
        id ud = [NSUserDefaults standardUserDefaults];
        id map = [ud objectForKey: udFileInfoMap];
        if(map)
        {
            id info = [map objectForKey: [self.fileURL path]];
            if(info)
            {
                [self _restoreFileInfo: info];
            }
        }

        id str = [[NSString alloc] initWithData: data
                                       encoding: self.encoding];
        if(str == nil)
        {
            // こけたらS-JISを試みる
            self.encoding = NSShiftJISStringEncoding;
            str = [[NSString alloc] initWithData: data
                                        encoding: self.encoding];
            if(str == nil)
                return NO;
        }
        
        id aStr = [pager.format formatString: str];
        if(aStr == nil)
            return NO;
        
        [pager.content loadString: aStr];
        
        return YES;
    }
    
    if(UTTypeConformsTo((CFStringRef)typeName, kUTTypeRTF))
    {
        id aStr = [[NSAttributedString alloc] initWithRTF: data 
                                       documentAttributes: nil];
        if(aStr == nil)
            return NO;
        
        aStr = [self normalizeAttributedString: aStr];
        [pager.content loadString: aStr];

        return YES;
    }


    return NO;
}

- (void)encodeRestorableStateWithCoder:(NSCoder *)coder
{
    [super encodeRestorableStateWithCoder: coder];
    id info = [self fileInfo];
    [coder encodeObject: info forKey: sDocumentRestoreInfo];
}

- (void)restoreStateWithCoder:(NSCoder *)coder
{
    [super restoreStateWithCoder: coder];
    id info = [coder decodeObjectForKey: sDocumentRestoreInfo];
    // awakeよりも後となる

    [self _restoreFileInfo: info];
    
    [self _loadSelection: initialState.selection];
    // フォント周りの処理。ちょっと整理しないとダメかも
    id faces = [MSFontServices facesForFamily: self._family];
    [self _setFontFaces: faces];
    [self _syncFontMenu];
}

- (NSDictionary*)fileInfo
{
    MSPageFormat *format = pager.format;
    NSRange range = [textView selectedRange];
    id val = [NSValue valueWithRange: range];
    id rangeData = [NSArchiver archivedDataWithRootObject: val];
    id dic = [NSDictionary dictionaryWithObjectsAndKeys: 
              self._family, udkFontFamilyKey,
              self._face, udkFontFaceKey,
              [NSNumber numberWithFloat: format.fontSize], udkFontSizeKey,
              [NSNumber numberWithInteger: format.boxCount], udkBoxCountKey,
              [NSNumber numberWithInteger: self.encoding], udkEncodingKey,
              rangeData, udkSelectionKey,
              [NSNumber numberWithBool: format.showsGrid], udkGridKey,
              nil];
    return dic;
}

+ (BOOL)autosavesInPlace
{
    return YES;
}

- (NSDictionary*)standardAttributes
{
    return [NSDictionary dictionaryWithObjectsAndKeys:
            pager.format.font, NSFontAttributeName,
            [NSColor blackColor], NSForegroundColorAttributeName,
            nil];
}

- (NSAttributedString*)normalizeAttributedString:(NSAttributedString*)aStr
{
    id newStr = [aStr mutableCopy];
    id str = [newStr mutableString];
    id attach = [NSString stringWithFormat: @"%C", NSAttachmentCharacter];
    [str replaceOccurrencesOfString: attach
                         withString: @"" 
                            options: NSLiteralSearch
                              range: NSMakeRange(0, [str length])];
    [newStr setAttributes: [self standardAttributes]
                    range: NSMakeRange(0,[newStr length])];
    return newStr;
}

- (void)_writeStringToPasteboard:(NSString*)string
{
    id pboard = [NSPasteboard generalPasteboard];
    [pboard clearContents];
    id item = [[NSPasteboardItem alloc] init];
    [item setString: string forType: NSPasteboardTypeString];
    [pboard writeObjects: [NSArray arrayWithObject: item]];
}

- (void)_scriptDidComplete:(id)notif
{
    id result = script.executor.result;
    switch(script.writebackMode)
    {
        case eOutputModeDiscard:
            break;
        case eOutputModeReplaceSelection:
            [textView replaceSelectionWithString: result];
            break;
        case eOutputModeReplaceAll:
            [textView replaceAllWithString: result];
            break;
        case eOutputModeInsertAfter:
            [textView insertStringAfterSelection: result];
            break;
        case eOutputModeAppend:
            [textView appendString: result];
            break;
        case eOutputModePasteboard:
            [self _writeStringToPasteboard: result];
            break;
    }
    id center = [NSNotificationCenter defaultCenter];
    [center removeObserver: self
                      name: MSScriptExecutorExecutionDidEndNotification
                    object: script.executor];
    script.executor = nil;
    textView.lock = NO;
}

- (void)_runScriptItem:(MSScriptActionItem*)item
{
    id center = [NSNotificationCenter defaultCenter];
    script.writebackMode = item.outputMode;
    script.executor = [[MSScriptExecutor alloc] init];
    id file = self.fileURL;
    if(file)
    {
        [script.executor setEnvironmentVariable: [file path]
                                         forKey: envManuscriptWorkingFile];
    }
    
    // input設定
    switch(item.inputMode)
    {
        id input;
        case eInputModeNone:
            break;
        case eInputModeSelection:
            input = [textView selectedString];
            if(input != nil)
                script.executor.input = input;
            break;
        case eInputModeAll:
            input = [textView string];
            script.executor.input = input;
            break;
    }
    [center addObserver: self
               selector: @selector(_scriptDidComplete:)
                   name: MSScriptExecutorExecutionDidEndNotification
                 object: script.executor];
    textView.lock = YES;
    
    [script.executor doCommand: item.URL];
}

- (void)windowDidResignKey:(id)notif
{
    [textView stopCaretTimer];
}

- (void)windowDidBecomeKey:(id)sender
{
    [textView startCaretTimer];
}

- (IBAction)performFindPanelAction:(id)sender
{
    id panel = [MSFindPanelController sharedFindPanel];
    [panel showWindow:self];
}

- (IBAction)abortScript:(id)sender
{
    [script.executor abortExecution];
    id center = [NSNotificationCenter defaultCenter];
    [center removeObserver: self
                      name: MSScriptExecutorExecutionDidEndNotification
                    object: script.executor];
    script.executor = nil;
    textView.lock = NO;
}

- (IBAction)runScript:(id)sender
{
    MSScriptActionItem *item = [sender representedObject];
    [self _runScriptItem: item];
}

- (BOOL)validateUserInterfaceItem:(id < NSValidatedUserInterfaceItem >)anItem
{
    if([anItem action] == @selector(abortScript:))
        return script.executor != nil;
    
    if([anItem action] == @selector(runScript:) && [textView hasMarkedText])
        return NO;
    
    if(script.executor)
        return NO;
    
    return [super validateUserInterfaceItem: anItem];
}

- (NSArray*)boxValues
{
    return [NSArray arrayWithObjects: 
            [NSNumber numberWithInt: 15],
            [NSNumber numberWithInt: 20],
            [NSNumber numberWithInt: 25],
            [NSNumber numberWithInt: 30],
            [NSNumber numberWithInt: 35],
            [NSNumber numberWithInt: 40],
            nil];
}

- (NSArray*)fontSizes
{
    return [NSArray arrayWithObjects:
            [NSNumber numberWithFloat: 9],
            [NSNumber numberWithFloat: 12],
            [NSNumber numberWithFloat: 14],
            [NSNumber numberWithFloat: 16],
            [NSNumber numberWithFloat: 18],
            [NSNumber numberWithFloat: 20],
            [NSNumber numberWithFloat: 24],
            [NSNumber numberWithFloat: 28],
            [NSNumber numberWithFloat: 32],
            nil];
}
@end
