﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/Array.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/glacier/GlacierRequest.h>
#include <aws/glacier/Glacier_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Glacier {
namespace Model {

/**
 * <p>Provides options to add an archive to a vault.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/glacier-2012-06-01/UploadArchiveInput">AWS
 * API Reference</a></p>
 */
class UploadArchiveRequest : public StreamingGlacierRequest {
 public:
  AWS_GLACIER_API UploadArchiveRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UploadArchive"; }

  AWS_GLACIER_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the vault.</p>
   */
  inline const Aws::String& GetVaultName() const { return m_vaultName; }
  inline bool VaultNameHasBeenSet() const { return m_vaultNameHasBeenSet; }
  template <typename VaultNameT = Aws::String>
  void SetVaultName(VaultNameT&& value) {
    m_vaultNameHasBeenSet = true;
    m_vaultName = std::forward<VaultNameT>(value);
  }
  template <typename VaultNameT = Aws::String>
  UploadArchiveRequest& WithVaultName(VaultNameT&& value) {
    SetVaultName(std::forward<VaultNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The <code>AccountId</code> value is the AWS account ID of the account that
   * owns the vault. You can either specify an AWS account ID or optionally a single
   * '<code>-</code>' (hyphen), in which case Amazon S3 Glacier uses the AWS account
   * ID associated with the credentials used to sign the request. If you use an
   * account ID, do not include any hyphens ('-') in the ID. </p>
   */
  inline const Aws::String& GetAccountId() const { return m_accountId; }
  inline bool AccountIdHasBeenSet() const { return m_accountIdHasBeenSet; }
  template <typename AccountIdT = Aws::String>
  void SetAccountId(AccountIdT&& value) {
    m_accountIdHasBeenSet = true;
    m_accountId = std::forward<AccountIdT>(value);
  }
  template <typename AccountIdT = Aws::String>
  UploadArchiveRequest& WithAccountId(AccountIdT&& value) {
    SetAccountId(std::forward<AccountIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The optional description of the archive you are uploading.</p>
   */
  inline const Aws::String& GetArchiveDescription() const { return m_archiveDescription; }
  inline bool ArchiveDescriptionHasBeenSet() const { return m_archiveDescriptionHasBeenSet; }
  template <typename ArchiveDescriptionT = Aws::String>
  void SetArchiveDescription(ArchiveDescriptionT&& value) {
    m_archiveDescriptionHasBeenSet = true;
    m_archiveDescription = std::forward<ArchiveDescriptionT>(value);
  }
  template <typename ArchiveDescriptionT = Aws::String>
  UploadArchiveRequest& WithArchiveDescription(ArchiveDescriptionT&& value) {
    SetArchiveDescription(std::forward<ArchiveDescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The SHA256 tree hash of the data being uploaded.</p>
   */
  inline const Aws::String& GetChecksum() const { return m_checksum; }
  inline bool ChecksumHasBeenSet() const { return m_checksumHasBeenSet; }
  template <typename ChecksumT = Aws::String>
  void SetChecksum(ChecksumT&& value) {
    m_checksumHasBeenSet = true;
    m_checksum = std::forward<ChecksumT>(value);
  }
  template <typename ChecksumT = Aws::String>
  UploadArchiveRequest& WithChecksum(ChecksumT&& value) {
    SetChecksum(std::forward<ChecksumT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_vaultName;
  bool m_vaultNameHasBeenSet = false;

  Aws::String m_accountId;
  bool m_accountIdHasBeenSet = false;

  Aws::String m_archiveDescription;
  bool m_archiveDescriptionHasBeenSet = false;

  Aws::String m_checksum;
  bool m_checksumHasBeenSet = false;
};

}  // namespace Model
}  // namespace Glacier
}  // namespace Aws
