! This file is part of mctc-lib.
!
! Licensed under the Apache License, Version 2.0 (the "License");
! you may not use this file except in compliance with the License.
! You may obtain a copy of the License at
!
!     http://www.apache.org/licenses/LICENSE-2.0
!
! Unless required by applicable law or agreed to in writing, software
! distributed under the License is distributed on an "AS IS" BASIS,
! WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
! See the License for the specific language governing permissions and
! limitations under the License.

module test_cutoff
   use mctc_env, only : wp
   use mctc_env_testing, only : new_unittest, unittest_type, error_type, &
   & test_failed
   use mctc_io_structure, only : structure_type
   use testsuite_structure, only : get_structure
   use mctc_cutoff, only : get_lattice_points, wrap_to_central_cell
   implicit none
   private

   public :: collect_cutoff

   real(wp), parameter :: thr = 100*epsilon(1.0_wp)
   real(wp), parameter :: thr2 = sqrt(epsilon(1.0_wp))

contains


   !> Collect all exported unit tests
   subroutine collect_cutoff(testsuite)

      !> Collection of tests
      type(unittest_type), allocatable, intent(out) :: testsuite(:)

      testsuite = [ &
      & new_unittest("lattice-points-cutoff", test_lattice_points_cutoff), &
      & new_unittest("lattice-points-rep", test_lattice_points_rep), &
      & new_unittest("wrap", test_wrap) &
      & ]

   end subroutine collect_cutoff


   subroutine test_lattice_points_cutoff(error)

      !> Error handling
      type(error_type), allocatable, intent(out) :: error

      type(structure_type) :: mol
      real(wp), allocatable :: lattr(:, :)

      real(wp), parameter :: cutoff = 15.0_wp
      real(wp), parameter :: ref(3, 125) = reshape([&
      &  0.00000000000E+0_wp,  0.00000000000E+0_wp,  0.00000000000E+0_wp, &
      &  0.00000000000E+0_wp,  0.00000000000E+0_wp,  9.47387737000E+0_wp, &
      &  0.00000000000E+0_wp,  0.00000000000E+0_wp, -9.47387737000E+0_wp, &
      &  0.00000000000E+0_wp,  0.00000000000E+0_wp,  1.89477547400E+1_wp, &
      &  0.00000000000E+0_wp,  0.00000000000E+0_wp, -1.89477547400E+1_wp, &
      &  0.00000000000E+0_wp,  9.47387737000E+0_wp,  0.00000000000E+0_wp, &
      &  0.00000000000E+0_wp, -9.47387737000E+0_wp,  0.00000000000E+0_wp, &
      &  0.00000000000E+0_wp,  9.47387737000E+0_wp,  9.47387737000E+0_wp, &
      &  0.00000000000E+0_wp,  9.47387737000E+0_wp, -9.47387737000E+0_wp, &
      &  0.00000000000E+0_wp, -9.47387737000E+0_wp,  9.47387737000E+0_wp, &
      &  0.00000000000E+0_wp, -9.47387737000E+0_wp, -9.47387737000E+0_wp, &
      &  0.00000000000E+0_wp,  9.47387737000E+0_wp,  1.89477547400E+1_wp, &
      &  0.00000000000E+0_wp,  9.47387737000E+0_wp, -1.89477547400E+1_wp, &
      &  0.00000000000E+0_wp, -9.47387737000E+0_wp,  1.89477547400E+1_wp, &
      &  0.00000000000E+0_wp, -9.47387737000E+0_wp, -1.89477547400E+1_wp, &
      &  0.00000000000E+0_wp,  1.89477547400E+1_wp,  0.00000000000E+0_wp, &
      &  0.00000000000E+0_wp, -1.89477547400E+1_wp,  0.00000000000E+0_wp, &
      &  0.00000000000E+0_wp,  1.89477547400E+1_wp,  9.47387737000E+0_wp, &
      &  0.00000000000E+0_wp,  1.89477547400E+1_wp, -9.47387737000E+0_wp, &
      &  0.00000000000E+0_wp, -1.89477547400E+1_wp,  9.47387737000E+0_wp, &
      &  0.00000000000E+0_wp, -1.89477547400E+1_wp, -9.47387737000E+0_wp, &
      &  0.00000000000E+0_wp,  1.89477547400E+1_wp,  1.89477547400E+1_wp, &
      &  0.00000000000E+0_wp,  1.89477547400E+1_wp, -1.89477547400E+1_wp, &
      &  0.00000000000E+0_wp, -1.89477547400E+1_wp,  1.89477547400E+1_wp, &
      &  0.00000000000E+0_wp, -1.89477547400E+1_wp, -1.89477547400E+1_wp, &
      &  9.47387737000E+0_wp,  0.00000000000E+0_wp,  0.00000000000E+0_wp, &
      & -9.47387737000E+0_wp,  0.00000000000E+0_wp,  0.00000000000E+0_wp, &
      &  9.47387737000E+0_wp,  0.00000000000E+0_wp,  9.47387737000E+0_wp, &
      &  9.47387737000E+0_wp,  0.00000000000E+0_wp, -9.47387737000E+0_wp, &
      & -9.47387737000E+0_wp,  0.00000000000E+0_wp,  9.47387737000E+0_wp, &
      & -9.47387737000E+0_wp,  0.00000000000E+0_wp, -9.47387737000E+0_wp, &
      &  9.47387737000E+0_wp,  0.00000000000E+0_wp,  1.89477547400E+1_wp, &
      &  9.47387737000E+0_wp,  0.00000000000E+0_wp, -1.89477547400E+1_wp, &
      & -9.47387737000E+0_wp,  0.00000000000E+0_wp,  1.89477547400E+1_wp, &
      & -9.47387737000E+0_wp,  0.00000000000E+0_wp, -1.89477547400E+1_wp, &
      &  9.47387737000E+0_wp,  9.47387737000E+0_wp,  0.00000000000E+0_wp, &
      &  9.47387737000E+0_wp, -9.47387737000E+0_wp,  0.00000000000E+0_wp, &
      & -9.47387737000E+0_wp,  9.47387737000E+0_wp,  0.00000000000E+0_wp, &
      & -9.47387737000E+0_wp, -9.47387737000E+0_wp,  0.00000000000E+0_wp, &
      &  9.47387737000E+0_wp,  9.47387737000E+0_wp,  9.47387737000E+0_wp, &
      &  9.47387737000E+0_wp,  9.47387737000E+0_wp, -9.47387737000E+0_wp, &
      &  9.47387737000E+0_wp, -9.47387737000E+0_wp,  9.47387737000E+0_wp, &
      &  9.47387737000E+0_wp, -9.47387737000E+0_wp, -9.47387737000E+0_wp, &
      & -9.47387737000E+0_wp,  9.47387737000E+0_wp,  9.47387737000E+0_wp, &
      & -9.47387737000E+0_wp,  9.47387737000E+0_wp, -9.47387737000E+0_wp, &
      & -9.47387737000E+0_wp, -9.47387737000E+0_wp,  9.47387737000E+0_wp, &
      & -9.47387737000E+0_wp, -9.47387737000E+0_wp, -9.47387737000E+0_wp, &
      &  9.47387737000E+0_wp,  9.47387737000E+0_wp,  1.89477547400E+1_wp, &
      &  9.47387737000E+0_wp,  9.47387737000E+0_wp, -1.89477547400E+1_wp, &
      &  9.47387737000E+0_wp, -9.47387737000E+0_wp,  1.89477547400E+1_wp, &
      &  9.47387737000E+0_wp, -9.47387737000E+0_wp, -1.89477547400E+1_wp, &
      & -9.47387737000E+0_wp,  9.47387737000E+0_wp,  1.89477547400E+1_wp, &
      & -9.47387737000E+0_wp,  9.47387737000E+0_wp, -1.89477547400E+1_wp, &
      & -9.47387737000E+0_wp, -9.47387737000E+0_wp,  1.89477547400E+1_wp, &
      & -9.47387737000E+0_wp, -9.47387737000E+0_wp, -1.89477547400E+1_wp, &
      &  9.47387737000E+0_wp,  1.89477547400E+1_wp,  0.00000000000E+0_wp, &
      &  9.47387737000E+0_wp, -1.89477547400E+1_wp,  0.00000000000E+0_wp, &
      & -9.47387737000E+0_wp,  1.89477547400E+1_wp,  0.00000000000E+0_wp, &
      & -9.47387737000E+0_wp, -1.89477547400E+1_wp,  0.00000000000E+0_wp, &
      &  9.47387737000E+0_wp,  1.89477547400E+1_wp,  9.47387737000E+0_wp, &
      &  9.47387737000E+0_wp,  1.89477547400E+1_wp, -9.47387737000E+0_wp, &
      &  9.47387737000E+0_wp, -1.89477547400E+1_wp,  9.47387737000E+0_wp, &
      &  9.47387737000E+0_wp, -1.89477547400E+1_wp, -9.47387737000E+0_wp, &
      & -9.47387737000E+0_wp,  1.89477547400E+1_wp,  9.47387737000E+0_wp, &
      & -9.47387737000E+0_wp,  1.89477547400E+1_wp, -9.47387737000E+0_wp, &
      & -9.47387737000E+0_wp, -1.89477547400E+1_wp,  9.47387737000E+0_wp, &
      & -9.47387737000E+0_wp, -1.89477547400E+1_wp, -9.47387737000E+0_wp, &
      &  9.47387737000E+0_wp,  1.89477547400E+1_wp,  1.89477547400E+1_wp, &
      &  9.47387737000E+0_wp,  1.89477547400E+1_wp, -1.89477547400E+1_wp, &
      &  9.47387737000E+0_wp, -1.89477547400E+1_wp,  1.89477547400E+1_wp, &
      &  9.47387737000E+0_wp, -1.89477547400E+1_wp, -1.89477547400E+1_wp, &
      & -9.47387737000E+0_wp,  1.89477547400E+1_wp,  1.89477547400E+1_wp, &
      & -9.47387737000E+0_wp,  1.89477547400E+1_wp, -1.89477547400E+1_wp, &
      & -9.47387737000E+0_wp, -1.89477547400E+1_wp,  1.89477547400E+1_wp, &
      & -9.47387737000E+0_wp, -1.89477547400E+1_wp, -1.89477547400E+1_wp, &
      &  1.89477547400E+1_wp,  0.00000000000E+0_wp,  0.00000000000E+0_wp, &
      & -1.89477547400E+1_wp,  0.00000000000E+0_wp,  0.00000000000E+0_wp, &
      &  1.89477547400E+1_wp,  0.00000000000E+0_wp,  9.47387737000E+0_wp, &
      &  1.89477547400E+1_wp,  0.00000000000E+0_wp, -9.47387737000E+0_wp, &
      & -1.89477547400E+1_wp,  0.00000000000E+0_wp,  9.47387737000E+0_wp, &
      & -1.89477547400E+1_wp,  0.00000000000E+0_wp, -9.47387737000E+0_wp, &
      &  1.89477547400E+1_wp,  0.00000000000E+0_wp,  1.89477547400E+1_wp, &
      &  1.89477547400E+1_wp,  0.00000000000E+0_wp, -1.89477547400E+1_wp, &
      & -1.89477547400E+1_wp,  0.00000000000E+0_wp,  1.89477547400E+1_wp, &
      & -1.89477547400E+1_wp,  0.00000000000E+0_wp, -1.89477547400E+1_wp, &
      &  1.89477547400E+1_wp,  9.47387737000E+0_wp,  0.00000000000E+0_wp, &
      &  1.89477547400E+1_wp, -9.47387737000E+0_wp,  0.00000000000E+0_wp, &
      & -1.89477547400E+1_wp,  9.47387737000E+0_wp,  0.00000000000E+0_wp, &
      & -1.89477547400E+1_wp, -9.47387737000E+0_wp,  0.00000000000E+0_wp, &
      &  1.89477547400E+1_wp,  9.47387737000E+0_wp,  9.47387737000E+0_wp, &
      &  1.89477547400E+1_wp,  9.47387737000E+0_wp, -9.47387737000E+0_wp, &
      &  1.89477547400E+1_wp, -9.47387737000E+0_wp,  9.47387737000E+0_wp, &
      &  1.89477547400E+1_wp, -9.47387737000E+0_wp, -9.47387737000E+0_wp, &
      & -1.89477547400E+1_wp,  9.47387737000E+0_wp,  9.47387737000E+0_wp, &
      & -1.89477547400E+1_wp,  9.47387737000E+0_wp, -9.47387737000E+0_wp, &
      & -1.89477547400E+1_wp, -9.47387737000E+0_wp,  9.47387737000E+0_wp, &
      & -1.89477547400E+1_wp, -9.47387737000E+0_wp, -9.47387737000E+0_wp, &
      &  1.89477547400E+1_wp,  9.47387737000E+0_wp,  1.89477547400E+1_wp, &
      &  1.89477547400E+1_wp,  9.47387737000E+0_wp, -1.89477547400E+1_wp, &
      &  1.89477547400E+1_wp, -9.47387737000E+0_wp,  1.89477547400E+1_wp, &
      &  1.89477547400E+1_wp, -9.47387737000E+0_wp, -1.89477547400E+1_wp, &
      & -1.89477547400E+1_wp,  9.47387737000E+0_wp,  1.89477547400E+1_wp, &
      & -1.89477547400E+1_wp,  9.47387737000E+0_wp, -1.89477547400E+1_wp, &
      & -1.89477547400E+1_wp, -9.47387737000E+0_wp,  1.89477547400E+1_wp, &
      & -1.89477547400E+1_wp, -9.47387737000E+0_wp, -1.89477547400E+1_wp, &
      &  1.89477547400E+1_wp,  1.89477547400E+1_wp,  0.00000000000E+0_wp, &
      &  1.89477547400E+1_wp, -1.89477547400E+1_wp,  0.00000000000E+0_wp, &
      & -1.89477547400E+1_wp,  1.89477547400E+1_wp,  0.00000000000E+0_wp, &
      & -1.89477547400E+1_wp, -1.89477547400E+1_wp,  0.00000000000E+0_wp, &
      &  1.89477547400E+1_wp,  1.89477547400E+1_wp,  9.47387737000E+0_wp, &
      &  1.89477547400E+1_wp,  1.89477547400E+1_wp, -9.47387737000E+0_wp, &
      &  1.89477547400E+1_wp, -1.89477547400E+1_wp,  9.47387737000E+0_wp, &
      &  1.89477547400E+1_wp, -1.89477547400E+1_wp, -9.47387737000E+0_wp, &
      & -1.89477547400E+1_wp,  1.89477547400E+1_wp,  9.47387737000E+0_wp, &
      & -1.89477547400E+1_wp,  1.89477547400E+1_wp, -9.47387737000E+0_wp, &
      & -1.89477547400E+1_wp, -1.89477547400E+1_wp,  9.47387737000E+0_wp, &
      & -1.89477547400E+1_wp, -1.89477547400E+1_wp, -9.47387737000E+0_wp, &
      &  1.89477547400E+1_wp,  1.89477547400E+1_wp,  1.89477547400E+1_wp, &
      &  1.89477547400E+1_wp,  1.89477547400E+1_wp, -1.89477547400E+1_wp, &
      &  1.89477547400E+1_wp, -1.89477547400E+1_wp,  1.89477547400E+1_wp, &
      &  1.89477547400E+1_wp, -1.89477547400E+1_wp, -1.89477547400E+1_wp, &
      & -1.89477547400E+1_wp,  1.89477547400E+1_wp,  1.89477547400E+1_wp, &
      & -1.89477547400E+1_wp,  1.89477547400E+1_wp, -1.89477547400E+1_wp, &
      & -1.89477547400E+1_wp, -1.89477547400E+1_wp,  1.89477547400E+1_wp, &
      & -1.89477547400E+1_wp, -1.89477547400E+1_wp, -1.89477547400E+1_wp],&
      &  shape(ref))

      call get_structure(mol, "x04")

      call get_lattice_points(mol%periodic, mol%lattice, cutoff, lattr)
      
      if (any(abs(lattr - ref) > thr)) then
         call test_failed(error, "Structures do not match")
         print'(3es21.14)', lattr
      end if

   end subroutine test_lattice_points_cutoff

   subroutine test_lattice_points_rep(error)

      !> Error handling
      type(error_type), allocatable, intent(out) :: error

      type(structure_type) :: mol
      real(wp), allocatable :: lattr(:, :)

      integer, parameter :: rep(3) = [2, 1, 1]
      real(wp), parameter :: ref(3, 44) = reshape([&
      &  0.00000000000E+0_wp,  0.00000000000E+0_wp,  9.47387737000E+0_wp, &
      &  0.00000000000E+0_wp,  0.00000000000E+0_wp, -9.47387737000E+0_wp, &
      &  0.00000000000E+0_wp,  9.47387737000E+0_wp,  0.00000000000E+0_wp, &
      &  0.00000000000E+0_wp, -9.47387737000E+0_wp,  0.00000000000E+0_wp, &
      &  0.00000000000E+0_wp,  9.47387737000E+0_wp,  9.47387737000E+0_wp, &
      &  0.00000000000E+0_wp,  9.47387737000E+0_wp, -9.47387737000E+0_wp, &
      &  0.00000000000E+0_wp, -9.47387737000E+0_wp,  9.47387737000E+0_wp, &
      &  0.00000000000E+0_wp, -9.47387737000E+0_wp, -9.47387737000E+0_wp, &
      &  9.47387737000E+0_wp,  0.00000000000E+0_wp,  0.00000000000E+0_wp, &
      & -9.47387737000E+0_wp,  0.00000000000E+0_wp,  0.00000000000E+0_wp, &
      &  9.47387737000E+0_wp,  0.00000000000E+0_wp,  9.47387737000E+0_wp, &
      &  9.47387737000E+0_wp,  0.00000000000E+0_wp, -9.47387737000E+0_wp, &
      & -9.47387737000E+0_wp,  0.00000000000E+0_wp,  9.47387737000E+0_wp, &
      & -9.47387737000E+0_wp,  0.00000000000E+0_wp, -9.47387737000E+0_wp, &
      &  9.47387737000E+0_wp,  9.47387737000E+0_wp,  0.00000000000E+0_wp, &
      &  9.47387737000E+0_wp, -9.47387737000E+0_wp,  0.00000000000E+0_wp, &
      & -9.47387737000E+0_wp,  9.47387737000E+0_wp,  0.00000000000E+0_wp, &
      & -9.47387737000E+0_wp, -9.47387737000E+0_wp,  0.00000000000E+0_wp, &
      &  9.47387737000E+0_wp,  9.47387737000E+0_wp,  9.47387737000E+0_wp, &
      &  9.47387737000E+0_wp,  9.47387737000E+0_wp, -9.47387737000E+0_wp, &
      &  9.47387737000E+0_wp, -9.47387737000E+0_wp,  9.47387737000E+0_wp, &
      &  9.47387737000E+0_wp, -9.47387737000E+0_wp, -9.47387737000E+0_wp, &
      & -9.47387737000E+0_wp,  9.47387737000E+0_wp,  9.47387737000E+0_wp, &
      & -9.47387737000E+0_wp,  9.47387737000E+0_wp, -9.47387737000E+0_wp, &
      & -9.47387737000E+0_wp, -9.47387737000E+0_wp,  9.47387737000E+0_wp, &
      & -9.47387737000E+0_wp, -9.47387737000E+0_wp, -9.47387737000E+0_wp, &
      &  1.89477547400E+1_wp,  0.00000000000E+0_wp,  0.00000000000E+0_wp, &
      & -1.89477547400E+1_wp,  0.00000000000E+0_wp,  0.00000000000E+0_wp, &
      &  1.89477547400E+1_wp,  0.00000000000E+0_wp,  9.47387737000E+0_wp, &
      &  1.89477547400E+1_wp,  0.00000000000E+0_wp, -9.47387737000E+0_wp, &
      & -1.89477547400E+1_wp,  0.00000000000E+0_wp,  9.47387737000E+0_wp, &
      & -1.89477547400E+1_wp,  0.00000000000E+0_wp, -9.47387737000E+0_wp, &
      &  1.89477547400E+1_wp,  9.47387737000E+0_wp,  0.00000000000E+0_wp, &
      &  1.89477547400E+1_wp, -9.47387737000E+0_wp,  0.00000000000E+0_wp, &
      & -1.89477547400E+1_wp,  9.47387737000E+0_wp,  0.00000000000E+0_wp, &
      & -1.89477547400E+1_wp, -9.47387737000E+0_wp,  0.00000000000E+0_wp, &
      &  1.89477547400E+1_wp,  9.47387737000E+0_wp,  9.47387737000E+0_wp, &
      &  1.89477547400E+1_wp,  9.47387737000E+0_wp, -9.47387737000E+0_wp, &
      &  1.89477547400E+1_wp, -9.47387737000E+0_wp,  9.47387737000E+0_wp, &
      &  1.89477547400E+1_wp, -9.47387737000E+0_wp, -9.47387737000E+0_wp, &
      & -1.89477547400E+1_wp,  9.47387737000E+0_wp,  9.47387737000E+0_wp, &
      & -1.89477547400E+1_wp,  9.47387737000E+0_wp, -9.47387737000E+0_wp, &
      & -1.89477547400E+1_wp, -9.47387737000E+0_wp,  9.47387737000E+0_wp, &
      & -1.89477547400E+1_wp, -9.47387737000E+0_wp, -9.47387737000E+0_wp],&
      &  shape(ref))

      call get_structure(mol, "x04")

      call get_lattice_points(mol%lattice, rep, .false., lattr)
            
      if (any(abs(lattr - ref) > thr)) then
         call test_failed(error, "Structures do not match")
         print'(3es21.14)', lattr
      end if

   end subroutine test_lattice_points_rep


   subroutine test_wrap(error)

      !> Error handling
      type(error_type), allocatable, intent(out) :: error

      type(structure_type) :: mol
      real(wp), allocatable :: xyz_shift(:,:)

      real(wp), parameter :: ref(3, 32) = reshape([&
      &  8.66499304004865_wp,  9.33379706146984_wp,  7.91683397818847_wp, &
      & 11.12335640640559_wp,  2.15906103856442_wp, 12.28606540623589_wp, &
      &  2.69933763795316_wp,  8.69604933222823_wp,  5.94619038919326_wp, &
      &  4.40748472136020_wp,  2.73477407301820_wp,  1.34585013180550_wp, &
      &  8.94474331294941_wp,  5.24347823723191_wp,  6.03793064749213_wp, &
      & 10.69920497119952_wp,  4.18256695916720_wp,  4.01290993162606_wp, &
      & 10.03588165896165_wp,  5.08392793928533_wp,  1.39546189302066_wp, &
      &  8.49376544841568_wp,  7.47023679468653_wp,  1.28534952220463_wp, &
      &  9.07166275596235_wp,  9.29463739048189_wp,  3.43672461445391_wp, &
      &  8.87252167134021_wp,  8.08457959995134_wp,  6.01243046499065_wp, &
      &  2.61511156242260_wp,  9.98111860346994_wp,  7.83909475310702_wp, &
      &  1.94055488237084_wp,  8.92607461070971_wp, 10.39160309677660_wp, &
      &  3.94058158183755_wp,  9.62934843688878_wp, 12.34968145677051_wp, &
      &  4.41149849250265_wp, -0.13918998706016_wp, 12.14066417045890_wp, &
      &  4.70428751489632_wp,  1.17418327733924_wp,  9.63142334925047_wp, &
      &  2.97542523741995_wp,  0.23498193061563_wp,  7.50807857572976_wp, &
      & 11.00218693527931_wp, 10.01811156711939_wp,  3.26258886294108_wp, &
      &  6.51725105145864_wp,  6.87025658460207_wp,  1.40398487849503_wp, &
      &  9.16505707508531_wp,  6.55996790419566_wp, 12.22499217557358_wp, &
      &  0.47155553650782_wp,  4.68566204197289_wp,  3.97665598787901_wp, &
      & 10.71305929405516_wp,  2.12295125271661_wp,  4.04673290893442_wp, &
      &  7.00559444915079_wp,  4.60292552897028_wp,  5.68707427274543_wp, &
      &  7.82621004905011_wp, 10.93721749961744_wp,  3.36440794575946_wp, &
      &  9.47562563356497_wp,  4.56669825883223_wp,  7.90994201013468_wp, &
      &  5.72226645413659_wp,  8.73073901566233_wp, 11.81706046742993_wp, &
      &  1.10806523402693_wp,  1.08795999732802_wp,  7.75198050780745_wp, &
      &  4.49303750240828_wp,  3.20681326576017_wp,  9.88850009827761_wp, &
      &  3.68234836535172_wp,  0.82971735354827_wp,  5.66899365523961_wp, &
      &  6.67318941946786_wp,  0.84421918391682_wp,  9.09225122218742_wp, &
      & 12.30020769805011_wp,  9.84555048261104_wp, 11.35093359840778_wp, &
      &  1.70097366892267_wp,  6.88253129604813_wp, 10.28258692216369_wp, &
      &  3.02585151643235_wp, 10.74223721977969_wp,  1.45047192019470_wp],&
      &  shape(ref))

      call get_structure(mol, "x01")

      xyz_shift = mol%xyz - mol%lattice(1, 1)

      call wrap_to_central_cell(xyz_shift, mol%lattice, mol%periodic)

      if (any(abs(mol%xyz - ref) > thr)) then
         call test_failed(error, "Structures do not match")
         print'(3es21.14)', mol%xyz
      end if

   end subroutine test_wrap

end module test_cutoff
