!--------------------------------------------------------------------------------------------------!
!  DFTB+: general package for performing fast atomistic simulations                                !
!  Copyright (C) 2006 - 2023  DFTB+ developers group                                               !
!                                                                                                  !
!  See the LICENSE file for terms of usage and distribution.                                       !
!--------------------------------------------------------------------------------------------------!

#:include 'common.fypp'

!> Program for plotting molecular orbitals as cube files.
program waveplot
  use dftbp_common_accuracy, only : dp
  use dftbp_common_file, only : TFileDescr, openFile, closeFile
  use dftbp_common_globalenv, only : stdOut
  use dftbp_dftb_periodic, only : getCellTranslations
  use dftbp_io_charmanip, only : i2c
  use dftbp_math_simplealgebra, only : invert33
  use dftbp_type_linkedlist, only : TListInt, TListRealR1, len, init, append, asArray
  use dftbp_type_typegeometry, only : TGeometry
  use waveplot_gridcache, only : next
  use waveplot_initwaveplot, only : TProgramVariables, TProgramVariables_init
  use waveplot_molorb, only : getValue
#:if WITH_MPI
  use mpi, only : MPI_THREAD_FUNNELED
  use dftbp_common_mpienv, only : TMpiEnv, TMpiEnv_init
  use dftbp_extlibs_mpifx, only : mpifx_init_thread, mpifx_finalize
#:endif

  implicit none

  !> Container of program variables
  type(TProgramVariables), target :: wp

  !> Pointer to real-valued grid
  real(dp), pointer :: gridValReal(:,:,:)

  !> Pointer to complex-valued grid
  complex(dp), pointer :: gridValCmpl(:,:,:)

  !> Arrays holding the volumetric data
  real(dp), allocatable :: buffer(:,:,:), totChrg(:,:,:), atomicChrg(:,:,:,:), spinUp(:,:,:)

  !> Occupation of orbitals
  real(dp), allocatable :: orbitalOcc(:,:)

  !> Array holding temporary coordinate information
  real(dp), allocatable :: coords(:,:)

  !> Array holding temporary species information
  integer, allocatable :: species(:)

  !> Summation of all grid points
  real(dp) :: sumTotChrg, sumChrg, sumAtomicChrg

  !> Indices of current level, K-point and spin
  integer :: levelIndex(3)

  !> Onedimensional real-valued list of coordinate arrays
  type(TListRealR1) :: coordList

  !> Onedimensional integer-valued list of atomic species indices
  type(TListInt) :: speciesList

  !> Auxiliary variables
  integer :: i1, i2, i3
  integer :: iCell, iLevel, iKPoint, iSpin, iSpinOld, iAtom, iSpecies, iAng, mAng, ind, nBox
  logical :: tFinished, tPlotLevel
  character(len=80) :: comments(2), fileName
  real(dp) :: mDist, dist
  real(dp) :: cellMiddle(3), boxMiddle(3), frac(3), cubeCorner(3), coord(3), shift(3)
  real(dp) :: invBoxVecs(3,3), recVecs2p(3,3)
  real(dp), allocatable :: cellVec(:,:), rCellVec(:,:)

#:if WITH_MPI
  !> MPI environment
  type(TMpiEnv) :: mpiEnv

  ! As this is serial code, trap for run time execution on more than 1 processor with MPI enabled
  call mpifx_init_thread(requiredThreading=MPI_THREAD_FUNNELED)
  call TMpiEnv_init(mpiEnv)
  call mpiEnv%mpiSerialEnv()
#:endif

  ! Allocate resources
  call TProgramVariables_init(wp)
  write(stdout, "(/,A,/)") "Starting main program"

  ! Allocating buffer for general grid, total charge and spin up
  allocate(buffer(wp%opt%nPoints(1), wp%opt%nPoints(2), wp%opt%nPoints(3)))
  if (wp%opt%tCalcTotChrg) then
    allocate(totChrg(wp%opt%nPoints(1), wp%opt%nPoints(2), wp%opt%nPoints(3)))
    totChrg = 0.0_dp
    if (wp%opt%tPlotTotSpin) then
      allocate(spinUp(wp%opt%nPoints(1), wp%opt%nPoints(2), wp%opt%nPoints(3)))
    end if
  end if

  write (comments(1), "(A)") "Cube file generated by WAVEPLOT from data created by DFTB+"

  ! Repeat boxes if necessary
  nBox = product(wp%opt%repeatBox)
  if (nBox > 1) then
    ! If tFillBox is off, coordinates must be repeated here. Otherwise the part for filling with
    ! atoms will do that.
    if (.not. wp%opt%tFillBox) then
      allocate(coords(3, size(wp%input%geo%coords)))
      allocate(species(size(wp%input%geo%species)))
      coords(:,:) = wp%input%geo%coords(:,:)
      species(:) = wp%input%geo%species(:)
      deallocate(wp%input%geo%coords)
      deallocate(wp%input%geo%species)
      allocate(wp%input%geo%coords(3, nBox * wp%input%geo%nAtom))
      allocate(wp%input%geo%species(nBox * wp%input%geo%nAtom))
      ind = 0
      do i1 = 0, wp%opt%repeatBox(1) - 1
        do i2 = 0, wp%opt%repeatBox(2) - 1
          do i3 = 0, wp%opt%repeatBox(3) - 1
            shift(:) = matmul(wp%opt%boxVecs, real([i1, i2, i3], dp))
            do iAtom = 1, wp%input%geo%nAtom
              wp%input%geo%coords(:,ind+iAtom) = coords(:,iAtom) + shift(:)
            end do
            wp%input%geo%species(ind+1:ind+wp%input%geo%nAtom) = species(:)
            ind = ind + wp%input%geo%nAtom
          end do
        end do
      end do
      wp%input%geo%nAtom = nBox * wp%input%geo%nAtom
    end if
    do i1 = 1, 3
      wp%opt%boxVecs(:,i1) = wp%opt%boxVecs(:,i1) * real(wp%opt%repeatBox(i1), dp)
    end do
  end if

  write(stdout, "(A)") "Origin"
  write(stdout, "(2X,3(F0.5,1X))") wp%opt%origin(:)
  write(stdout, "(A)") "Box"
  do i1 = 1, 3
    write(stdout, "(2X,3(F0.5,1X))") wp%opt%boxVecs(:, i1)
  end do
  write(stdout, "(A)") "Spatial resolution [1/Bohr]:"
  write(stdout, "(2X,3(F0.5,1X))") 1.0_dp / sqrt(sum(wp%loc%gridVec**2, dim=1))
  write(stdout, *)

  ! Create density superposition of the atomic orbitals. Occupation is distributed equally on
  ! orbitals with the same angular momentum.
  if (wp%opt%tCalcAtomDens) then
    allocate(atomicChrg(wp%opt%nPoints(1), wp%opt%nPoints(2), wp%opt%nPoints(3), 1))
    allocate(orbitalOcc(wp%input%nOrb, 1))
    ind = 1
    do iAtom = 1, wp%input%geo%nAtom
      iSpecies = wp%input%geo%species(iAtom)
      do iAng = 1, size(wp%basis%basis(iSpecies)%angMoms)
        mAng = 2 * wp%basis%basis(iSpecies)%angMoms(iAng) + 1
        orbitalOcc(ind:ind + mAng - 1,1) = wp%basis%basis(iSpecies)%occupations(iAng)&
            & / real(mAng, dp)
        ind = ind + mAng
      end do
    end do
    call getValue(wp%loc%molorb, wp%opt%gridOrigin, wp%loc%gridVec, orbitalOcc, atomicChrg,&
        & addDensities=.true.)
    sumAtomicChrg = sum(atomicChrg) * wp%loc%gridVol

    if (wp%opt%tVerbose) then
      write(stdout, "('Total charge of atomic densities:',F12.6,/)") sumAtomicChrg
    end if
    if (wp%opt%tPlotAtomDens) then
      write (comments(2), 9989) wp%input%identity
9989  format('Calc-Id:',I11,', atomdens')
      fileName = "wp-atomdens.cube"
      call writeCubeFile(wp%input%geo, wp%aNr%atomicNumbers, wp%loc%gridVec, wp%opt%gridOrigin,&
          & buffer, fileName, comments, wp%opt%repeatBox)
      write(stdout, "(A)") "File '" // trim(fileName) // "' written"
    end if
  end if

  if (wp%opt%tVerbose) then
    write(stdout, "(/,A5,' ',A6,' ',A6,' ',A7,' ',A11,' ',A11)") "Spin", "KPoint", "State",&
        & "Action", "Norm", "W. Occup."
  end if

  ! Fold in coordinates
  if (wp%opt%tFoldCoords) then
    call invert33(invBoxVecs, wp%opt%boxVecs)
    call invert33(recVecs2p, wp%input%geo%latVecs)
    recVecs2p = reshape(recVecs2p, [3, 3], order=[2, 1])
    call wp%boundaryCond%foldCoordsToCell(wp%input%geo%coords(:,:), wp%input%geo%latVecs)
  end if

  ! Fill the box with atoms
  if (wp%opt%tFillBox) then
    ! Shifting plotted region by integer lattice vectors, to have its center as close to the center
    ! of the lattice unit cell as possible.
    cellMiddle(:) = 0.5_dp * sum(wp%input%geo%latVecs, dim=2)
    boxMiddle(:) = wp%opt%origin(:) + 0.5_dp * sum(wp%opt%boxVecs, dim=2)
    ! Workaround for intel 2021 ICE, replacing matmul(boxMiddle - cellMiddle, recVecs2p)
    shift(:) = boxMiddle - cellMiddle
    frac(:) = matmul(shift, recVecs2p)
    wp%opt%origin(:) = wp%opt%origin(:) - matmul(wp%input%geo%latVecs, real(anint(frac), dp))
    wp%opt%gridOrigin(:) = wp%opt%gridOrigin(:)&
        & - matmul(wp%input%geo%latVecs, real(anint(frac), dp))
    ! We need all cells around, which could contain atoms in the sphere, drawn from the center of
    ! the unit cell, containing the entire plotted region
    mDist = 0.0_dp
    do i1 = 0, 1
      do i2 = 0, 1
        do i3 = 0, 1
          cubeCorner(:) = wp%opt%origin(:) &
              &+ matmul(wp%opt%boxVecs, real([i1, i2, i3], dp))
          dist = sqrt(sum((cubeCorner - cellMiddle)**2))
          mDist = max(dist, mDist)
        end do
      end do
    end do
    call getCellTranslations(cellVec, rCellVec, wp%input%geo%latVecs, recVecs2p, mDist)
    ! Check all atoms in the shifted cells, if they fall in the plotted region
    call init(coordList)
    call init(speciesList)
    do iCell = 1, size(rCellVec, dim=2)
      do iAtom = 1, wp%input%geo%nAtom
        coord(:) = wp%input%geo%coords(:,iAtom) + rCellVec(:,iCell)
        frac(:) = matmul(invBoxVecs, coord - wp%opt%origin)
        if (all(frac > -1e-04_dp) .and. all(frac < 1.0_dp + 1e-04_dp)) then
          call append(coordList, coord)
          call append(speciesList, wp%input%geo%species(iAtom))
        end if
      end do
    end do
    deallocate(wp%input%geo%coords)
    deallocate(wp%input%geo%species)
    wp%input%geo%nAtom = len(coordList)
    allocate(wp%input%geo%coords(3, wp%input%geo%nAtom))
    allocate(wp%input%geo%species(wp%input%geo%nAtom))
    call asArray(coordList, wp%input%geo%coords)
    call asArray(speciesList, wp%input%geo%species)
    deallocate(cellVec)
    deallocate(rCellVec)
  end if

  ! Calculate the molecular orbitals and write them to the disk
  iSpinOld = 1
  tFinished = .false.
  lpStates: do while (.not. tFinished)
    ! Get the next grid and its parameters
    if (wp%input%tRealHam) then
      call next(wp%loc%grid, gridValReal, levelIndex, tFinished)
    else
      call next(wp%loc%grid, gridValCmpl, levelIndex, tFinished)
    end if
    iLevel = levelIndex(1)
    iKPoint = levelIndex(2)
    iSpin = levelIndex(3)

    ! Save spin up density before processing first level for spin down
    if (wp%opt%tPlotTotSpin .and. (iSpinOld /= iSpin)) then
      iSpinOld = iSpin
      spinUp(:,:,:) = totChrg(:,:,:)
    end if

    ! Build charge if needed for total charge or was explicitely required
    tPlotLevel = any(wp%opt%plottedSpins == iSpin) &
        &.and. any(wp%opt%plottedKPoints == iKPoint) .and. any(wp%opt%plottedLevels == iLevel)
    if (wp%opt%tCalcTotChrg .or. (tPlotLevel .and. (wp%opt%tPlotChrg .or. wp%opt%tPlotChrgDiff)))&
        & then
      if (wp%input%tRealHam) then
        buffer = gridValReal**2
      else
        buffer = abs(gridValCmpl)**2
      end if
      if (wp%opt%tCalcTotChrg) then
        totChrg(:,:,:) = totChrg(:,:,:) + wp%input%occupations(iLevel, iKPoint, iSpin)&
            & * buffer(:,:,:)
      end if
      sumChrg = sum(buffer) * wp%loc%gridVol
      if (wp%opt%tVerbose) then
        write(stdout, "(I5,I7,I7,A8,F12.6,F12.6)") iSpin, iKPoint, iLevel, "calc", sumChrg,&
            & wp%input%occupations(iLevel, iKPoint, iSpin)
      end if
    end if

    ! Build and dump desired properties of the current level
    if (tPlotLevel) then
      if (wp%opt%tPlotChrg) then
        write (comments(2), 9990) wp%input%identity, iSpin, iKPoint, iLevel
9990    format('Calc-Id:',I11,', Spin:',I2,', K-Point:',I6,', State:',I6, ', abs2')
        fileName = "wp-" // i2c(iSpin) // "-" // i2c(iKPoint) // "-" // i2c(iLevel) // "-abs2.cube"
        call writeCubeFile(wp%input%geo, wp%aNr%atomicNumbers, wp%loc%gridVec, wp%opt%gridOrigin,&
            & buffer, fileName, comments, wp%opt%repeatBox)
        write(stdout, "(A)") "File '" // trim(fileName) // "' written"
      end if

      if (wp%opt%tPlotChrgDiff) then
        buffer = buffer - (sumChrg / sumAtomicChrg) * atomicChrg(:,:,:,1)
        write (comments(2), 9995) wp%input%identity, iSpin, iKPoint, iLevel
9995    format('Calc-Id:',I11,', Spin:',I2,', K-Point:',I6,', State:',I6, ', abs2diff')
        fileName = "wp-" // i2c(iSpin) // "-" // i2c(iKPoint) // "-" // i2c(iLevel) //&
            & "-abs2diff.cube"
        call writeCubeFile(wp%input%geo, wp%aNr%atomicNumbers, wp%loc%gridVec, wp%opt%gridOrigin,&
            & buffer, fileName, comments, wp%opt%repeatBox)
        write(stdout, "(A)") "File '" // trim(fileName) // "' written"
      end if

      if (wp%opt%tPlotReal) then
        if (wp%input%tRealHam) then
          buffer = gridValReal
        else
          buffer = real(gridValCmpl, dp)
        end if
        write (comments(2), 9991) wp%input%identity, iSpin, iKPoint, iLevel
9991    format('Calc-Id:',I11,', Spin:',I2,', K-Point:',I6,', State:',I6, ', real')
        fileName = "wp-" // i2c(iSpin) // "-" // i2c(iKPoint) // "-" // i2c(iLevel) // "-real.cube"
        call writeCubeFile(wp%input%geo, wp%aNr%atomicNumbers, wp%loc%gridVec, wp%opt%gridOrigin,&
            & buffer, fileName, comments, wp%opt%repeatBox)
        write(stdout, "(A)") "File '" // trim(fileName) // "' written"
      end if

      if (wp%opt%tPlotImag) then
        buffer = aimag(gridValCmpl)
        write (comments(2), 9992) wp%input%identity, iSpin, iKPoint, iLevel
9992    format('Calc-Id:',I11,', Spin:',I2,', K-Point:',I6,', State:',I6, ', imag')
        fileName = "wp-" // i2c(iSpin) // "-" // i2c(iKPoint) // "-" // i2c(iLevel) // "-imag.cube"
        call writeCubeFile(wp%input%geo, wp%aNr%atomicNumbers, wp%loc%gridVec, wp%opt%gridOrigin,&
            & buffer, fileName, comments, wp%opt%repeatBox)
        write(stdout, "(A)") "File '" // trim(fileName) // "' written"
      end if
    end if
  end do lpStates

  ! Dump total charge, if required
  if (wp%opt%tCalcTotChrg) then
    sumTotChrg = sum(totChrg) * wp%loc%gridVol
  end if
  if (wp%opt%tPlotTotChrg) then
    write (comments(2), 9993) wp%input%identity
9993 format('Calc-Id:',I11,', abs2')
    fileName = "wp-abs2.cube"
    call writeCubeFile(wp%input%geo, wp%aNr%atomicNumbers, wp%loc%gridVec, wp%opt%gridOrigin,&
        & totChrg, fileName, comments, wp%opt%repeatBox)
    write(stdout, "(A)") "File '" // trim(fileName) // "' written"
    if (wp%opt%tVerbose) then
      write(stdout, "(/,'Total charge:',F12.6,/)") sumTotChrg
    end if
  end if

  ! Dump total charge difference
  if (wp%opt%tPlotTotDiff) then
    buffer = totChrg - (sumTotChrg / sumAtomicChrg) * atomicChrg(:,:,:,1)
    write (comments(2), 9994) wp%input%identity
9994 format('Calc-Id:',I11,', abs2diff')
    fileName = 'wp-abs2diff.cube'
    call writeCubeFile(wp%input%geo, wp%aNr%atomicNumbers, wp%loc%gridVec, wp%opt%gridOrigin,&
        & buffer, fileName, comments, wp%opt%repeatBox)
    write(stdout, "(A)") "File '" // trim(fileName) // "' written"
  end if

  ! Dump spin polarisation
  if (wp%opt%tPlotTotSpin) then
    buffer = 2.0_dp * spinUp - totChrg
    write (comments(2), 9996) wp%input%identity
9996 format('Calc-Id:',I11,', spinpol')
    fileName = 'wp-spinpol.cube'
    call writeCubeFile(wp%input%geo, wp%aNr%atomicNumbers, wp%loc%gridVec, wp%opt%gridOrigin,&
        & buffer, fileName, comments, wp%opt%repeatBox)
    write(stdout, "(A)") "File '" // trim(fileName) // "' written"
  end if

#:if WITH_MPI
  call mpifx_finalize()
#:endif


contains

  !> Writes a 3D function as cube file.
  subroutine writeCubeFile(geo, atomicNumbers, gridVecs, origin, gridVal, fileName, comments,&
      & repeatBox)

    !> Geometry information about the structure
    type(TGeometry), intent(in) :: geo

    !> Atomic numbers corresponding to each species
    integer, intent(in) :: atomicNumbers(:)

    !> Grid vectors
    real(dp), intent(in) :: gridVecs(:,:)

    !> Origin of the grid
    real(dp), intent(in) :: origin(:)

    !> Value of the 3D function in the grid points
    real(dp), intent(in) :: gridVal(:,:,:)

    !> Name of the file to create
    character(len=*), intent(in) :: fileName

    !> First two comment lines of the file
    character(len=*), intent(in), optional :: comments(:)

    !> How often the grid should be repeated along the direction of the grid vectors
    integer, intent(in), optional :: repeatBox(:)

    integer, parameter :: bufferSize = 6
    real(dp) :: buffer(bufferSize)
    character(len=*), parameter :: formBuffer = "(6E16.8)"
    integer :: rep(3)
    integer :: ii, i1, i2, i3, ir1, ir2, ir3
    type(TFileDescr) :: fd

    @:ASSERT(size(atomicNumbers) == size(geo%speciesNames))
    @:ASSERT(all(shape(gridVecs) == [3, 3]))
    @:ASSERT(size(origin) == 3)
    @:ASSERT(all(shape(gridVal) >= [0, 0, 0]))

  #:block DEBUG_CODE
    if (present(comments)) then
      @:ASSERT(size(comments) == 2)
    end if
    if (present(repeatBox)) then
      @:ASSERT(size(repeatBox) == 3)
      @:ASSERT(all(repeatBox > 0))
    end if
  #:endblock DEBUG_CODE

    if (present(repeatBox)) then
      rep(:) = repeatBox(:)
    else
      rep(:) = [1, 1, 1]
    end if

    call openFile(fd, fileName, mode="w")
    if (present(comments)) then
      write (fd%unit, "(A)") trim(comments(1))
      write (fd%unit, "(A)") trim(comments(2))
    else
      write (fd%unit, "(A)") "Made by waveplot"
      write (fd%unit, *)
    end if
    write (fd%unit,"(I5,3F12.6)") geo%nAtom, origin(:)
    write (fd%unit,"(I5,3F12.6)") rep(1) * size(gridVal, dim=1), gridVecs(:,1)
    write (fd%unit,"(I5,3F12.6)") rep(2) * size(gridVal, dim=2), gridVecs(:,2)
    write (fd%unit,"(I5,3F12.6)") rep(3) * size(gridVal, dim=3), gridVecs(:,3)
    do ii = 1, geo%nAtom
      write (fd%unit, "(I5,4F12.6)") atomicNumbers(geo%species(ii)), 0.0_dp, &
          &geo%coords(:, ii)
    end do

    do ir1 = 1, rep(1)
      do i1 = 1, size(gridVal, dim=1)
        do ir2 = 1, rep(2)
          do i2 = 1, size(gridVal, dim=2)
            do ir3 = 1, rep(3)
              do i3 = 1, size(gridVal, dim=3)
                ii = mod(i3 - 1, bufferSize) + 1
                buffer(ii) = gridVal(i1, i2, i3)
                if (ii == bufferSize) then
                  write (fd%unit,formBuffer) real(buffer)
                end if
              end do
              if (ii /= bufferSize) then
                write (fd%unit, "(" // i2c(ii) // "E16.8)") real(buffer(:ii))
              end if
            end do
          end do
        end do
      end do
    end do
    call closeFile(fd)

  end subroutine writeCubeFile

end program waveplot
