!!****f* ABINIT/contstr26
!! NAME
!! contstr26
!!
!! FUNCTION
!! Carries out specialized metric tensor operations needed for contraction
!! of the 2nd strain derivative of the l=0,1,2,3 nonlocal Kleinman-Bylander
!! pseudopotential operation.  Derivatives are wrt a pair of cartesian
!! strain components.
!! Full advantage is taken of the full permutational symmetry of these
!! tensors.
!!
!! COPYRIGHT
!! Copyright (C) 1998-2025 ABINIT group (DRH)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt.
!!
!! INPUTS
!!  istr1=1,...6 specifies cartesian strain component 11,22,33,32,31,21
!!  istr2=seccond strain component
!!  rank=angular momentum
!!  gm(3,3)=metric tensor (array is symmetric but stored as 3x3)
!!  gprimd(3,3)=reciprocal space dimensional primitive translations
!!  aa(2,*)=unique elements of complex right-hand tensor
!!  bb(2,*)=unique elements of complex left-hand tensor
!!
!! OUTPUT
!!  e2nl=contraction for nonlocal 2nd-order strain derivative energy
!!
!! NOTES
!! All tensors are stored in a compressed storage mode defined below;
!! input and output conform to this scheme.
!! When tensor elements occur repeatedly due to symmetry, the
!! WEIGHT IS INCLUDED in the output tensor element to simplify later
!! contractions with other tensors of the same rank and form, i.e. the
!! next contraction is then simply a dot product over the unique elements.
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"


module m_contstr26
contains
!!**



subroutine contstr26(istr1,istr2,rank,gm,gprimd,e2nl,aa,bb)

 use defs_basis
 use m_abicore
 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: istr1,istr2,rank
 real(dp),intent(out) :: e2nl
!arrays
 real(dp),intent(in) :: aa(2,((rank+1)*(rank+2))/2),bb(2,((rank+1)*(rank+2))/2)
 real(dp),intent(in) :: gm(3,3),gprimd(3,3)

!Local variables-------------------------------
!scalars
 integer,parameter :: mrank=3
 integer :: ii,jj,ka,kb,kd,kg
!arrays
 integer,save :: idx(12)=(/1,1,2,2,3,3,3,2,3,1,2,1/)
 real(dp) :: d2gm(3,3),dgm01(3,3),dgm10(3,3),tmp(2)
 real(dp),allocatable :: cm(:,:)

! *************************************************************************
 ABI_MALLOC(cm,(((mrank+1)*(mrank+2))/2,((mrank+1)*(mrank+2))/2))

 ka=idx(2*istr1-1);kb=idx(2*istr1);kg=idx(2*istr2-1);kd=idx(2*istr2)

 do ii = 1,3
   dgm01(:,ii)=-(gprimd(ka,:)*gprimd(kb,ii)+gprimd(kb,:)*gprimd(ka,ii))
   dgm10(:,ii)=-(gprimd(kg,:)*gprimd(kd,ii)+gprimd(kd,:)*gprimd(kg,ii))
 end do

 d2gm(:,:)=0.d0
 do ii = 1,3
   if(ka==kg) d2gm(:,ii)=d2gm(:,ii)&
&   +gprimd(kb,:)*gprimd(kd,ii)+gprimd(kd,:)*gprimd(kb,ii)
   if(ka==kd) d2gm(:,ii)=d2gm(:,ii)&
&   +gprimd(kb,:)*gprimd(kg,ii)+gprimd(kg,:)*gprimd(kb,ii)
   if(kb==kg) d2gm(:,ii)=d2gm(:,ii)&
&   +gprimd(ka,:)*gprimd(kd,ii)+gprimd(kd,:)*gprimd(ka,ii)
   if(kb==kd) d2gm(:,ii)=d2gm(:,ii)&
&   +gprimd(ka,:)*gprimd(kg,ii)+gprimd(kg,:)*gprimd(ka,ii)
 end do
 d2gm(:,:)=0.5d0*d2gm(:,:)

!
!The code below was written by a Mathematica program and formatted by
!a combination of editing scripts.  It is not intended to be read
!by human beings, and certainly not to be modified by one.  Conceivably
!it could be shortened somewhat by identifying common subexpressions.
!
 if(rank==0)then
   cm(1,1)=0.0d0
 elseif(rank==1)then
   cm(1,1)=d2gm(1,1)
   cm(2,1)=d2gm(1,2)
   cm(3,1)=d2gm(1,3)
   cm(1,2)=d2gm(1,2)
   cm(2,2)=d2gm(2,2)
   cm(3,2)=d2gm(2,3)
   cm(1,3)=d2gm(1,3)
   cm(2,3)=d2gm(2,3)
   cm(3,3)=d2gm(3,3)
 elseif(rank==2)then
   cm(1,1)=2*(dgm01(1,1)*dgm10(1,1)+gm(1,1)*d2gm(1,1))
   cm(2,1)=-0.5d0*dgm01(2,2)*dgm10(1,1)+3*dgm01(1,2)*dgm10(1,2)&
&   -0.5d0*dgm01(1,1)*dgm10(2,2)-0.5d0*gm(2,2)*d2gm(1,1)+3*gm(1,2)&
&   *d2gm(1,2)-0.5d0*gm(1,1)*d2gm(2,2)
   cm(3,1)=-0.5d0*dgm01(3,3)*dgm10(1,1)+3*dgm01(1,3)*dgm10(1,3)&
&   -0.5d0*dgm01(1,1)*dgm10(3,3)-0.5d0*gm(3,3)*d2gm(1,1)+3*gm(1,3)&
&   *d2gm(1,3)-0.5d0*gm(1,1)*d2gm(3,3)
   cm(4,1)=-dgm01(2,3)*dgm10(1,1)+3*dgm01(1,3)*dgm10(1,2)+3*dgm01(1,2)&
&   *dgm10(1,3)-dgm01(1,1)*dgm10(2,3)-gm(2,3)*d2gm(1,1)+3*gm(1,3)&
&   *d2gm(1,2)+3*gm(1,2)*d2gm(1,3)-gm(1,1)*d2gm(2,3)
   cm(5,1)=2*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3)+gm(1,3)&
&   *d2gm(1,1)+gm(1,1)*d2gm(1,3))
   cm(6,1)=2*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2)+gm(1,2)&
&   *d2gm(1,1)+gm(1,1)*d2gm(1,2))
   cm(1,2)=-0.5d0*dgm01(2,2)*dgm10(1,1)+3*dgm01(1,2)*dgm10(1,2)&
&   -0.5d0*dgm01(1,1)*dgm10(2,2)-0.5d0*gm(2,2)*d2gm(1,1)+3*gm(1,2)&
&   *d2gm(1,2)-0.5d0*gm(1,1)*d2gm(2,2)
   cm(2,2)=2*(dgm01(2,2)*dgm10(2,2)+gm(2,2)*d2gm(2,2))
   cm(3,2)=-0.5d0*dgm01(3,3)*dgm10(2,2)+3*dgm01(2,3)*dgm10(2,3)&
&   -0.5d0*dgm01(2,2)*dgm10(3,3)-0.5d0*gm(3,3)*d2gm(2,2)+3*gm(2,3)&
&   *d2gm(2,3)-0.5d0*gm(2,2)*d2gm(3,3)
   cm(4,2)=2*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3)+gm(2,3)&
&   *d2gm(2,2)+gm(2,2)*d2gm(2,3))
   cm(5,2)=3*dgm01(2,3)*dgm10(1,2)-dgm01(2,2)*dgm10(1,3)-dgm01(1,3)&
&   *dgm10(2,2)+3*dgm01(1,2)*dgm10(2,3)+3*gm(2,3)*d2gm(1,2)-gm(2,2)&
&   *d2gm(1,3)-gm(1,3)*d2gm(2,2)+3*gm(1,2)*d2gm(2,3)
   cm(6,2)=2*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)+gm(2,2)&
&   *d2gm(1,2)+gm(1,2)*d2gm(2,2))
   cm(1,3)=-0.5d0*dgm01(3,3)*dgm10(1,1)+3*dgm01(1,3)*dgm10(1,3)&
&   -0.5d0*dgm01(1,1)*dgm10(3,3)-0.5d0*gm(3,3)*d2gm(1,1)+3*gm(1,3)&
&   *d2gm(1,3)-0.5d0*gm(1,1)*d2gm(3,3)
   cm(2,3)=-0.5d0*dgm01(3,3)*dgm10(2,2)+3*dgm01(2,3)*dgm10(2,3)&
&   -0.5d0*dgm01(2,2)*dgm10(3,3)-0.5d0*gm(3,3)*d2gm(2,2)+3*gm(2,3)&
&   *d2gm(2,3)-0.5d0*gm(2,2)*d2gm(3,3)
   cm(3,3)=2*(dgm01(3,3)*dgm10(3,3)+gm(3,3)*d2gm(3,3))
   cm(4,3)=2*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)+gm(3,3)&
&   *d2gm(2,3)+gm(2,3)*d2gm(3,3))
   cm(5,3)=2*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)+gm(3,3)&
&   *d2gm(1,3)+gm(1,3)*d2gm(3,3))
   cm(6,3)=-dgm01(3,3)*dgm10(1,2)+3*dgm01(2,3)*dgm10(1,3)+3*dgm01(1,3)&
&   *dgm10(2,3)-dgm01(1,2)*dgm10(3,3)-gm(3,3)*d2gm(1,2)+3*gm(2,3)&
&   *d2gm(1,3)+3*gm(1,3)*d2gm(2,3)-gm(1,2)*d2gm(3,3)
   cm(1,4)=-dgm01(2,3)*dgm10(1,1)+3*dgm01(1,3)*dgm10(1,2)+3*dgm01(1,2)&
&   *dgm10(1,3)-dgm01(1,1)*dgm10(2,3)-gm(2,3)*d2gm(1,1)+3*gm(1,3)&
&   *d2gm(1,2)+3*gm(1,2)*d2gm(1,3)-gm(1,1)*d2gm(2,3)
   cm(2,4)=2*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3)+gm(2,3)&
&   *d2gm(2,2)+gm(2,2)*d2gm(2,3))
   cm(3,4)=2*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)+gm(3,3)&
&   *d2gm(2,3)+gm(2,3)*d2gm(3,3))
   cm(4,4)=3*dgm01(3,3)*dgm10(2,2)+2*dgm01(2,3)*dgm10(2,3)+3*dgm01(2,2)&
&   *dgm10(3,3)+3*gm(3,3)*d2gm(2,2)+2*gm(2,3)*d2gm(2,3)+3*gm(2,2)&
&   *d2gm(3,3)
   cm(5,4)=3*dgm01(3,3)*dgm10(1,2)+dgm01(2,3)*dgm10(1,3)+dgm01(1,3)&
&   *dgm10(2,3)+3*dgm01(1,2)*dgm10(3,3)+3*gm(3,3)*d2gm(1,2)+gm(2,3)&
&   *d2gm(1,3)+gm(1,3)*d2gm(2,3)+3*gm(1,2)*d2gm(3,3)
   cm(6,4)=dgm01(2,3)*dgm10(1,2)+3*dgm01(2,2)*dgm10(1,3)+3*dgm01(1,3)&
&   *dgm10(2,2)+dgm01(1,2)*dgm10(2,3)+gm(2,3)*d2gm(1,2)+3*gm(2,2)&
&   *d2gm(1,3)+3*gm(1,3)*d2gm(2,2)+gm(1,2)*d2gm(2,3)
   cm(1,5)=2*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3)+gm(1,3)&
&   *d2gm(1,1)+gm(1,1)*d2gm(1,3))
   cm(2,5)=3*dgm01(2,3)*dgm10(1,2)-dgm01(2,2)*dgm10(1,3)-dgm01(1,3)&
&   *dgm10(2,2)+3*dgm01(1,2)*dgm10(2,3)+3*gm(2,3)*d2gm(1,2)-gm(2,2)&
&   *d2gm(1,3)-gm(1,3)*d2gm(2,2)+3*gm(1,2)*d2gm(2,3)
   cm(3,5)=2*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)+gm(3,3)&
&   *d2gm(1,3)+gm(1,3)*d2gm(3,3))
   cm(4,5)=3*dgm01(3,3)*dgm10(1,2)+dgm01(2,3)*dgm10(1,3)+dgm01(1,3)&
&   *dgm10(2,3)+3*dgm01(1,2)*dgm10(3,3)+3*gm(3,3)*d2gm(1,2)+gm(2,3)&
&   *d2gm(1,3)+gm(1,3)*d2gm(2,3)+3*gm(1,2)*d2gm(3,3)
   cm(5,5)=3*dgm01(3,3)*dgm10(1,1)+2*dgm01(1,3)*dgm10(1,3)+3*dgm01(1,1)&
&   *dgm10(3,3)+3*gm(3,3)*d2gm(1,1)+2*gm(1,3)*d2gm(1,3)+3*gm(1,1)&
&   *d2gm(3,3)
   cm(6,5)=3*dgm01(2,3)*dgm10(1,1)+dgm01(1,3)*dgm10(1,2)+dgm01(1,2)&
&   *dgm10(1,3)+3*dgm01(1,1)*dgm10(2,3)+3*gm(2,3)*d2gm(1,1)+gm(1,3)&
&   *d2gm(1,2)+gm(1,2)*d2gm(1,3)+3*gm(1,1)*d2gm(2,3)
   cm(1,6)=2*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2)+gm(1,2)&
&   *d2gm(1,1)+gm(1,1)*d2gm(1,2))
   cm(2,6)=2*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)+gm(2,2)&
&   *d2gm(1,2)+gm(1,2)*d2gm(2,2))
   cm(3,6)=-dgm01(3,3)*dgm10(1,2)+3*dgm01(2,3)*dgm10(1,3)+3*dgm01(1,3)&
&   *dgm10(2,3)-dgm01(1,2)*dgm10(3,3)-gm(3,3)*d2gm(1,2)+3*gm(2,3)&
&   *d2gm(1,3)+3*gm(1,3)*d2gm(2,3)-gm(1,2)*d2gm(3,3)
   cm(4,6)=dgm01(2,3)*dgm10(1,2)+3*dgm01(2,2)*dgm10(1,3)+3*dgm01(1,3)&
&   *dgm10(2,2)+dgm01(1,2)*dgm10(2,3)+gm(2,3)*d2gm(1,2)+3*gm(2,2)&
&   *d2gm(1,3)+3*gm(1,3)*d2gm(2,2)+gm(1,2)*d2gm(2,3)
   cm(5,6)=3*dgm01(2,3)*dgm10(1,1)+dgm01(1,3)*dgm10(1,2)+dgm01(1,2)&
&   *dgm10(1,3)+3*dgm01(1,1)*dgm10(2,3)+3*gm(2,3)*d2gm(1,1)+gm(1,3)&
&   *d2gm(1,2)+gm(1,2)*d2gm(1,3)+3*gm(1,1)*d2gm(2,3)
   cm(6,6)=3*dgm01(2,2)*dgm10(1,1)+2*dgm01(1,2)*dgm10(1,2)+3*dgm01(1,1)&
&   *dgm10(2,2)+3*gm(2,2)*d2gm(1,1)+2*gm(1,2)*d2gm(1,2)+3*gm(1,1)&
&   *d2gm(2,2)
 elseif(rank==3)then
   cm(1,1)=gm(1,1)*(6*dgm01(1,1)*dgm10(1,1)+3*gm(1,1)*d2gm(1,1))
   cm(2,1)=4.5d0*gm(1,2)**2*d2gm(1,1)-3*gm(2,2)*(dgm01(1,1)*dgm10(1,1)&
&   +gm(1,1)*d2gm(1,1))+9*gm(1,2)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,2)+gm(1,1)*d2gm(1,2))+gm(1,1)*(-3*dgm01(2,2)*dgm10(1,1)&
&   +9*dgm01(1,2)*dgm10(1,2)-3*dgm01(1,1)*dgm10(2,2)-1.5d0*gm(1,1)&
&   *d2gm(2,2))
   cm(3,1)=4.5d0*gm(1,3)**2*d2gm(1,1)-3*gm(3,3)*(dgm01(1,1)*dgm10(1,1)&
&   +gm(1,1)*d2gm(1,1))+9*gm(1,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,3)+gm(1,1)*d2gm(1,3))+gm(1,1)*(-3*dgm01(3,3)*dgm10(1,1)&
&   +9*dgm01(1,3)*dgm10(1,3)-3*dgm01(1,1)*dgm10(3,3)-1.5d0*gm(1,1)&
&   *d2gm(3,3))
   cm(4,1)=9*gm(1,2)*dgm01(1,3)*dgm10(1,1)-6*gm(1,1)*dgm01(2,3)&
&   *dgm10(1,1)+9*gm(1,1)*dgm01(1,3)*dgm10(1,2)+9*gm(1,2)*dgm01(1,1)&
&   *dgm10(1,3)+9*gm(1,1)*dgm01(1,2)*dgm10(1,3)-6*gm(1,1)*dgm01(1,1)&
&   *dgm10(2,3)-6*gm(2,3)*(dgm01(1,1)*dgm10(1,1)+gm(1,1)*d2gm(1,1))&
&   +9*gm(1,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2)+gm(1,2)&
&   *d2gm(1,1)+gm(1,1)*d2gm(1,2))+9*gm(1,1)*gm(1,2)*d2gm(1,3)-3*gm(1,1)&
&   **2*d2gm(2,3)
   cm(5,1)=6*gm(1,3)*(dgm01(1,1)*dgm10(1,1)+gm(1,1)*d2gm(1,1))&
&   +gm(1,1)*(6*dgm01(1,3)*dgm10(1,1)+6*dgm01(1,1)*dgm10(1,3)+3*gm(1,1)&
&   *d2gm(1,3))
   cm(6,1)=6*gm(1,2)*(dgm01(1,1)*dgm10(1,1)+gm(1,1)*d2gm(1,1))&
&   +gm(1,1)*(6*dgm01(1,2)*dgm10(1,1)+6*dgm01(1,1)*dgm10(1,2)+3*gm(1,1)&
&   *d2gm(1,2))
   cm(7,1)=-1.5d0*gm(1,1)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&   +7.5d0*gm(1,2)**2*d2gm(1,2)-1.5d0*gm(2,2)*(dgm01(1,2)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(1,2)+gm(1,2)*d2gm(1,1)+gm(1,1)*d2gm(1,2))+gm(1,2)&
&   *(-1.5d0*dgm01(2,2)*dgm10(1,1)+15*dgm01(1,2)*dgm10(1,2)-1.5d0*dgm01(1,1)&
&   *dgm10(2,2)-1.5d0*gm(1,1)*d2gm(2,2))
   cm(8,1)=-3*gm(1,3)*dgm01(2,3)*dgm10(1,1)-1.5d0*gm(1,2)*dgm01(3,3)&
&   *dgm10(1,1)+15*gm(1,3)*dgm01(1,3)*dgm10(1,2)-1.5d0*gm(1,1)*dgm01(3,3)&
&   *dgm10(1,2)+15*gm(1,3)*dgm01(1,2)*dgm10(1,3)+15*gm(1,2)*dgm01(1,3)&
&   *dgm10(1,3)-3*gm(1,1)*dgm01(2,3)*dgm10(1,3)-3*gm(1,3)*dgm01(1,1)&
&   *dgm10(2,3)-3*gm(1,1)*dgm01(1,3)*dgm10(2,3)-1.5d0*gm(1,2)*dgm01(1,1)&
&   *dgm10(3,3)-1.5d0*gm(1,1)*dgm01(1,2)*dgm10(3,3)+7.5d0*gm(1,3)&
&   **2*d2gm(1,2)-1.5d0*gm(3,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,2)+gm(1,2)*d2gm(1,1)+gm(1,1)*d2gm(1,2))+15*gm(1,2)*gm(1,3)&
&   *d2gm(1,3)-3*gm(2,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3)&
&   +gm(1,3)*d2gm(1,1)+gm(1,1)*d2gm(1,3))-3*gm(1,1)*gm(1,3)*d2gm(2,3)&
&   -1.5d0*gm(1,1)*gm(1,2)*d2gm(3,3)
   cm(9,1)=-1.5d0*gm(1,3)*dgm01(2,2)*dgm10(1,1)-3*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,1)+15*gm(1,3)*dgm01(1,2)*dgm10(1,2)+15*gm(1,2)*dgm01(1,3)&
&   *dgm10(1,2)-3*gm(1,1)*dgm01(2,3)*dgm10(1,2)+15*gm(1,2)*dgm01(1,2)&
&   *dgm10(1,3)-1.5d0*gm(1,1)*dgm01(2,2)*dgm10(1,3)-1.5d0*gm(1,3)&
&   *dgm01(1,1)*dgm10(2,2)-1.5d0*gm(1,1)*dgm01(1,3)*dgm10(2,2)-3*gm(1,2)&
&   *dgm01(1,1)*dgm10(2,3)-3*gm(1,1)*dgm01(1,2)*dgm10(2,3)+15*gm(1,2)&
&   *gm(1,3)*d2gm(1,2)-3*gm(2,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,2)+gm(1,2)*d2gm(1,1)+gm(1,1)*d2gm(1,2))+7.5d0*gm(1,2)&
&   **2*d2gm(1,3)-1.5d0*gm(2,2)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,3)+gm(1,3)*d2gm(1,1)+gm(1,1)*d2gm(1,3))-1.5d0*gm(1,1)&
&   *gm(1,3)*d2gm(2,2)-3*gm(1,1)*gm(1,2)*d2gm(2,3)
   cm(10,1)=-1.5d0*gm(1,1)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&   +7.5d0*gm(1,3)**2*d2gm(1,3)-1.5d0*gm(3,3)*(dgm01(1,3)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(1,3)+gm(1,3)*d2gm(1,1)+gm(1,1)*d2gm(1,3))+gm(1,3)&
&   *(-1.5d0*dgm01(3,3)*dgm10(1,1)+15*dgm01(1,3)*dgm10(1,3)-1.5d0*dgm01(1,1)&
&   *dgm10(3,3)-1.5d0*gm(1,1)*d2gm(3,3))
   cm(1,2)=4.5d0*gm(1,2)**2*d2gm(1,1)-3*gm(2,2)*(dgm01(1,1)*dgm10(1,1)&
&   +gm(1,1)*d2gm(1,1))+9*gm(1,2)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,2)+gm(1,1)*d2gm(1,2))+gm(1,1)*(-3*dgm01(2,2)*dgm10(1,1)&
&   +9*dgm01(1,2)*dgm10(1,2)-3*dgm01(1,1)*dgm10(2,2)-1.5d0*gm(1,1)&
&   *d2gm(2,2))
   cm(2,2)=12*gm(1,1)*dgm01(2,2)*dgm10(2,2)+6*gm(1,2)*(dgm01(2,2)&
&   *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+6*gm(2,2)**2*d2gm(1,1)+3*gm(1,2)&
&   **2*d2gm(2,2)+gm(2,2)*(12*dgm01(2,2)*dgm10(1,1)+6*dgm01(1,2)&
&   *dgm10(1,2)+12*dgm01(1,1)*dgm10(2,2)+6*gm(1,2)*d2gm(1,2)+12*gm(1,1)&
&   *d2gm(2,2))
   cm(3,2)=-1.5d0*gm(2,2)*dgm01(3,3)*dgm10(1,1)+9*gm(1,3)*dgm01(2,3)&
&   *dgm10(1,2)-6*gm(1,2)*dgm01(3,3)*dgm10(1,2)-6*gm(2,2)*dgm01(1,3)&
&   *dgm10(1,3)-6*gm(1,3)*dgm01(2,2)*dgm10(1,3)+9*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,3)-6*gm(1,3)*dgm01(1,3)*dgm10(2,2)-1.5d0*gm(1,1)*dgm01(3,3)&
&   *dgm10(2,2)+9*gm(1,3)*dgm01(1,2)*dgm10(2,3)+9*gm(1,2)*dgm01(1,3)&
&   *dgm10(2,3)+15*gm(1,1)*dgm01(2,3)*dgm10(2,3)-1.5d0*gm(2,2)*dgm01(1,1)&
&   *dgm10(3,3)-6*gm(1,2)*dgm01(1,2)*dgm10(3,3)-1.5d0*gm(1,1)*dgm01(2,2)&
&   *dgm10(3,3)+7.5d0*gm(2,3)**2*d2gm(1,1)-6*gm(1,3)*gm(2,2)*d2gm(1,3)&
&   -3*gm(1,3)**2*d2gm(2,2)+gm(3,3)*(-1.5d0*dgm01(2,2)*dgm10(1,1)&
&   -6*dgm01(1,2)*dgm10(1,2)-1.5d0*dgm01(1,1)*dgm10(2,2)-1.5d0*gm(2,2)&
&   *d2gm(1,1)-6*gm(1,2)*d2gm(1,2)-1.5d0*gm(1,1)*d2gm(2,2))+9*gm(1,2)&
&   *gm(1,3)*d2gm(2,3)+gm(2,3)*(15*dgm01(2,3)*dgm10(1,1)+9*dgm01(1,3)&
&   *dgm10(1,2)+9*dgm01(1,2)*dgm10(1,3)+15*dgm01(1,1)*dgm10(2,3)&
&   +9*gm(1,3)*d2gm(1,2)+9*gm(1,2)*d2gm(1,3)+15*gm(1,1)*d2gm(2,3))&
&   -3*gm(1,2)**2*d2gm(3,3)-1.5d0*gm(1,1)*gm(2,2)*d2gm(3,3)
   cm(4,2)=3*gm(1,3)*dgm01(2,2)*dgm10(1,2)+6*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,2)+3*gm(1,2)*dgm01(2,2)*dgm10(1,3)+3*gm(1,3)*dgm01(1,2)&
&   *dgm10(2,2)+3*gm(1,2)*dgm01(1,3)*dgm10(2,2)+12*gm(1,1)*dgm01(2,3)&
&   *dgm10(2,2)+6*gm(1,2)*dgm01(1,2)*dgm10(2,3)+12*gm(1,1)*dgm01(2,2)&
&   *dgm10(2,3)+3*gm(1,2)*gm(1,3)*d2gm(2,2)+gm(2,3)*(12*dgm01(2,2)&
&   *dgm10(1,1)+6*dgm01(1,2)*dgm10(1,2)+12*dgm01(1,1)*dgm10(2,2)&
&   +12*gm(2,2)*d2gm(1,1)+6*gm(1,2)*d2gm(1,2)+12*gm(1,1)*d2gm(2,2))&
&   +3*gm(1,2)**2*d2gm(2,3)+gm(2,2)*(12*dgm01(2,3)*dgm10(1,1)+3*dgm01(1,3)&
&   *dgm10(1,2)+3*dgm01(1,2)*dgm10(1,3)+12*dgm01(1,1)*dgm10(2,3)&
&   +3*gm(1,3)*d2gm(1,2)+3*gm(1,2)*d2gm(1,3)+12*gm(1,1)*d2gm(2,3))
   cm(5,2)=-4.5d0*gm(1,3)*dgm01(2,2)*dgm10(1,1)+12*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,1)+3*gm(1,3)*dgm01(1,2)*dgm10(1,2)+3*gm(1,2)*dgm01(1,3)&
&   *dgm10(1,2)+12*gm(1,1)*dgm01(2,3)*dgm10(1,2)+3*gm(1,2)*dgm01(1,2)&
&   *dgm10(1,3)-4.5d0*gm(1,1)*dgm01(2,2)*dgm10(1,3)-4.5d0*gm(1,3)&
&   *dgm01(1,1)*dgm10(2,2)-4.5d0*gm(1,1)*dgm01(1,3)*dgm10(2,2)+12*gm(1,2)&
&   *dgm01(1,1)*dgm10(2,3)+12*gm(1,1)*dgm01(1,2)*dgm10(2,3)+3*gm(1,2)&
&   *gm(1,3)*d2gm(1,2)+12*gm(2,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,2)+gm(1,2)*d2gm(1,1)+gm(1,1)*d2gm(1,2))+1.5d0*gm(1,2)&
&   **2*d2gm(1,3)-4.5d0*gm(2,2)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,3)+gm(1,3)*d2gm(1,1)+gm(1,1)*d2gm(1,3))-4.5d0*gm(1,1)&
&   *gm(1,3)*d2gm(2,2)+12*gm(1,1)*gm(1,2)*d2gm(2,3)
   cm(6,2)=7.5d0*gm(1,1)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&   +4.5d0*gm(1,2)**2*d2gm(1,2)+7.5d0*gm(2,2)*(dgm01(1,2)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(1,2)+gm(1,2)*d2gm(1,1)+gm(1,1)*d2gm(1,2))+gm(1,2)&
&   *(7.5d0*dgm01(2,2)*dgm10(1,1)+9*dgm01(1,2)*dgm10(1,2)+7.5d0*dgm01(1,1)&
&   *dgm10(2,2)+7.5d0*gm(1,1)*d2gm(2,2))
   cm(7,2)=6*gm(1,2)*dgm01(2,2)*dgm10(2,2)+3*gm(2,2)**2*d2gm(1,2)&
&   +6*gm(2,2)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)+gm(1,2)&
&   *d2gm(2,2))
   cm(8,2)=-4.5d0*gm(2,2)*dgm01(3,3)*dgm10(1,2)+12*gm(2,2)*dgm01(2,3)&
&   *dgm10(1,3)+12*gm(1,3)*dgm01(2,3)*dgm10(2,2)-4.5d0*gm(1,2)*dgm01(3,3)&
&   *dgm10(2,2)+12*gm(2,2)*dgm01(1,3)*dgm10(2,3)+12*gm(1,3)*dgm01(2,2)&
&   *dgm10(2,3)+3*gm(1,2)*dgm01(2,3)*dgm10(2,3)-4.5d0*gm(2,2)*dgm01(1,2)&
&   *dgm10(3,3)-4.5d0*gm(1,2)*dgm01(2,2)*dgm10(3,3)+1.5d0*gm(2,3)&
&   **2*d2gm(1,2)-4.5d0*gm(3,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&   *dgm10(2,2)+gm(2,2)*d2gm(1,2)+gm(1,2)*d2gm(2,2))+12*gm(1,3)*gm(2,2)&
&   *d2gm(2,3)+gm(2,3)*(3*dgm01(2,3)*dgm10(1,2)+12*dgm01(2,2)*dgm10(1,3)&
&   +12*dgm01(1,3)*dgm10(2,2)+3*dgm01(1,2)*dgm10(2,3)+12*gm(2,2)&
&   *d2gm(1,3)+12*gm(1,3)*d2gm(2,2)+3*gm(1,2)*d2gm(2,3))-4.5d0*gm(1,2)&
&   *gm(2,2)*d2gm(3,3)
   cm(9,2)=12*gm(1,3)*dgm01(2,2)*dgm10(2,2)+3*gm(1,2)*dgm01(2,3)&
&   *dgm10(2,2)+3*gm(1,2)*dgm01(2,2)*dgm10(2,3)+6*gm(2,2)**2*d2gm(1,3)&
&   +3*gm(2,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)+gm(2,2)&
&   *d2gm(1,2)+gm(1,2)*d2gm(2,2))+gm(2,2)*(3*dgm01(2,3)*dgm10(1,2)&
&   +12*dgm01(2,2)*dgm10(1,3)+12*dgm01(1,3)*dgm10(2,2)+3*dgm01(1,2)&
&   *dgm10(2,3)+12*gm(1,3)*d2gm(2,2)+3*gm(1,2)*d2gm(2,3))
   cm(10,2)=-1.5d0*gm(2,2)*dgm01(3,3)*dgm10(1,3)-1.5d0*gm(1,3)&
&   *dgm01(3,3)*dgm10(2,2)+15*gm(1,3)*dgm01(2,3)*dgm10(2,3)-3*gm(1,2)&
&   *dgm01(3,3)*dgm10(2,3)-1.5d0*gm(2,2)*dgm01(1,3)*dgm10(3,3)-1.5d0*gm(1,3)&
&   *dgm01(2,2)*dgm10(3,3)-3*gm(1,2)*dgm01(2,3)*dgm10(3,3)+7.5d0*gm(2,3)&
&   **2*d2gm(1,3)+gm(3,3)*(-3*dgm01(2,3)*dgm10(1,2)-1.5d0*dgm01(2,2)&
&   *dgm10(1,3)-1.5d0*dgm01(1,3)*dgm10(2,2)-3*dgm01(1,2)*dgm10(2,3)&
&   -3*gm(2,3)*d2gm(1,2)-1.5d0*gm(2,2)*d2gm(1,3)-1.5d0*gm(1,3)*d2gm(2,2)&
&   -3*gm(1,2)*d2gm(2,3))-1.5d0*gm(1,3)*gm(2,2)*d2gm(3,3)+gm(2,3)&
&   *(-3*dgm01(3,3)*dgm10(1,2)+15*dgm01(2,3)*dgm10(1,3)+15*dgm01(1,3)&
&   *dgm10(2,3)-3*dgm01(1,2)*dgm10(3,3)+15*gm(1,3)*d2gm(2,3)-3*gm(1,2)&
&   *d2gm(3,3))
   cm(1,3)=4.5d0*gm(1,3)**2*d2gm(1,1)-3*gm(3,3)*(dgm01(1,1)*dgm10(1,1)&
&   +gm(1,1)*d2gm(1,1))+9*gm(1,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,3)+gm(1,1)*d2gm(1,3))+gm(1,1)*(-3*dgm01(3,3)*dgm10(1,1)&
&   +9*dgm01(1,3)*dgm10(1,3)-3*dgm01(1,1)*dgm10(3,3)-1.5d0*gm(1,1)&
&   *d2gm(3,3))
   cm(2,3)=-1.5d0*gm(2,2)*dgm01(3,3)*dgm10(1,1)+9*gm(1,3)*dgm01(2,3)&
&   *dgm10(1,2)-6*gm(1,2)*dgm01(3,3)*dgm10(1,2)-6*gm(2,2)*dgm01(1,3)&
&   *dgm10(1,3)-6*gm(1,3)*dgm01(2,2)*dgm10(1,3)+9*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,3)-6*gm(1,3)*dgm01(1,3)*dgm10(2,2)-1.5d0*gm(1,1)*dgm01(3,3)&
&   *dgm10(2,2)+9*gm(1,3)*dgm01(1,2)*dgm10(2,3)+9*gm(1,2)*dgm01(1,3)&
&   *dgm10(2,3)+15*gm(1,1)*dgm01(2,3)*dgm10(2,3)-1.5d0*gm(2,2)*dgm01(1,1)&
&   *dgm10(3,3)-6*gm(1,2)*dgm01(1,2)*dgm10(3,3)-1.5d0*gm(1,1)*dgm01(2,2)&
&   *dgm10(3,3)+7.5d0*gm(2,3)**2*d2gm(1,1)-6*gm(1,3)*gm(2,2)*d2gm(1,3)&
&   -3*gm(1,3)**2*d2gm(2,2)+gm(3,3)*(-1.5d0*dgm01(2,2)*dgm10(1,1)&
&   -6*dgm01(1,2)*dgm10(1,2)-1.5d0*dgm01(1,1)*dgm10(2,2)-1.5d0*gm(2,2)&
&   *d2gm(1,1)-6*gm(1,2)*d2gm(1,2)-1.5d0*gm(1,1)*d2gm(2,2))+9*gm(1,2)&
&   *gm(1,3)*d2gm(2,3)+gm(2,3)*(15*dgm01(2,3)*dgm10(1,1)+9*dgm01(1,3)&
&   *dgm10(1,2)+9*dgm01(1,2)*dgm10(1,3)+15*dgm01(1,1)*dgm10(2,3)&
&   +9*gm(1,3)*d2gm(1,2)+9*gm(1,2)*d2gm(1,3)+15*gm(1,1)*d2gm(2,3))&
&   -3*gm(1,2)**2*d2gm(3,3)-1.5d0*gm(1,1)*gm(2,2)*d2gm(3,3)
   cm(3,3)=12*gm(1,1)*dgm01(3,3)*dgm10(3,3)+6*gm(1,3)*(dgm01(3,3)&
&   *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+6*gm(3,3)**2*d2gm(1,1)+3*gm(1,3)&
&   **2*d2gm(3,3)+gm(3,3)*(12*dgm01(3,3)*dgm10(1,1)+6*dgm01(1,3)&
&   *dgm10(1,3)+12*dgm01(1,1)*dgm10(3,3)+6*gm(1,3)*d2gm(1,3)+12*gm(1,1)&
&   *d2gm(3,3))
   cm(4,3)=3*gm(1,3)*dgm01(3,3)*dgm10(1,2)+6*gm(1,3)*dgm01(2,3)&
&   *dgm10(1,3)+3*gm(1,2)*dgm01(3,3)*dgm10(1,3)+6*gm(1,3)*dgm01(1,3)&
&   *dgm10(2,3)+12*gm(1,1)*dgm01(3,3)*dgm10(2,3)+3*gm(1,3)*dgm01(1,2)&
&   *dgm10(3,3)+3*gm(1,2)*dgm01(1,3)*dgm10(3,3)+12*gm(1,1)*dgm01(2,3)&
&   *dgm10(3,3)+3*gm(1,3)**2*d2gm(2,3)+gm(3,3)*(12*dgm01(2,3)*dgm10(1,1)&
&   +3*dgm01(1,3)*dgm10(1,2)+3*dgm01(1,2)*dgm10(1,3)+12*dgm01(1,1)&
&   *dgm10(2,3)+12*gm(2,3)*d2gm(1,1)+3*gm(1,3)*d2gm(1,2)+3*gm(1,2)&
&   *d2gm(1,3)+12*gm(1,1)*d2gm(2,3))+3*gm(1,2)*gm(1,3)*d2gm(3,3)&
&   +gm(2,3)*(12*dgm01(3,3)*dgm10(1,1)+6*dgm01(1,3)*dgm10(1,3)+12*dgm01(1,1)&
&   *dgm10(3,3)+6*gm(1,3)*d2gm(1,3)+12*gm(1,1)*d2gm(3,3))
   cm(5,3)=7.5d0*gm(1,1)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&   +4.5d0*gm(1,3)**2*d2gm(1,3)+7.5d0*gm(3,3)*(dgm01(1,3)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(1,3)+gm(1,3)*d2gm(1,1)+gm(1,1)*d2gm(1,3))+gm(1,3)&
&   *(7.5d0*dgm01(3,3)*dgm10(1,1)+9*dgm01(1,3)*dgm10(1,3)+7.5d0*dgm01(1,1)&
&   *dgm10(3,3)+7.5d0*gm(1,1)*d2gm(3,3))
   cm(6,3)=12*gm(1,3)*dgm01(2,3)*dgm10(1,1)-4.5d0*gm(1,2)*dgm01(3,3)&
&   *dgm10(1,1)+3*gm(1,3)*dgm01(1,3)*dgm10(1,2)-4.5d0*gm(1,1)*dgm01(3,3)&
&   *dgm10(1,2)+3*gm(1,3)*dgm01(1,2)*dgm10(1,3)+3*gm(1,2)*dgm01(1,3)&
&   *dgm10(1,3)+12*gm(1,1)*dgm01(2,3)*dgm10(1,3)+12*gm(1,3)*dgm01(1,1)&
&   *dgm10(2,3)+12*gm(1,1)*dgm01(1,3)*dgm10(2,3)-4.5d0*gm(1,2)*dgm01(1,1)&
&   *dgm10(3,3)-4.5d0*gm(1,1)*dgm01(1,2)*dgm10(3,3)+1.5d0*gm(1,3)&
&   **2*d2gm(1,2)-4.5d0*gm(3,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,2)+gm(1,2)*d2gm(1,1)+gm(1,1)*d2gm(1,2))+3*gm(1,2)*gm(1,3)&
&   *d2gm(1,3)+12*gm(2,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3)&
&   +gm(1,3)*d2gm(1,1)+gm(1,1)*d2gm(1,3))+12*gm(1,1)*gm(1,3)*d2gm(2,3)&
&   -4.5d0*gm(1,1)*gm(1,2)*d2gm(3,3)
   cm(7,3)=-1.5d0*gm(2,2)*dgm01(3,3)*dgm10(1,2)-3*gm(2,2)*dgm01(2,3)&
&   *dgm10(1,3)-3*gm(1,3)*dgm01(2,3)*dgm10(2,2)-1.5d0*gm(1,2)*dgm01(3,3)&
&   *dgm10(2,2)-3*gm(2,2)*dgm01(1,3)*dgm10(2,3)-3*gm(1,3)*dgm01(2,2)&
&   *dgm10(2,3)+15*gm(1,2)*dgm01(2,3)*dgm10(2,3)-1.5d0*gm(2,2)*dgm01(1,2)&
&   *dgm10(3,3)-1.5d0*gm(1,2)*dgm01(2,2)*dgm10(3,3)+7.5d0*gm(2,3)&
&   **2*d2gm(1,2)-1.5d0*gm(3,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&   *dgm10(2,2)+gm(2,2)*d2gm(1,2)+gm(1,2)*d2gm(2,2))-3*gm(1,3)*gm(2,2)&
&   *d2gm(2,3)+gm(2,3)*(15*dgm01(2,3)*dgm10(1,2)-3*dgm01(2,2)*dgm10(1,3)&
&   -3*dgm01(1,3)*dgm10(2,2)+15*dgm01(1,2)*dgm10(2,3)-3*gm(2,2)*d2gm(1,3)&
&   -3*gm(1,3)*d2gm(2,2)+15*gm(1,2)*d2gm(2,3))-1.5d0*gm(1,2)*gm(2,2)&
&   *d2gm(3,3)
   cm(8,3)=3*gm(1,3)*dgm01(3,3)*dgm10(2,3)+3*gm(1,3)*dgm01(2,3)&
&   *dgm10(3,3)+12*gm(1,2)*dgm01(3,3)*dgm10(3,3)+6*gm(3,3)**2*d2gm(1,2)&
&   +gm(3,3)*(12*dgm01(3,3)*dgm10(1,2)+3*dgm01(2,3)*dgm10(1,3)+3*dgm01(1,3)&
&   *dgm10(2,3)+12*dgm01(1,2)*dgm10(3,3)+3*gm(2,3)*d2gm(1,3)+3*gm(1,3)&
&   *d2gm(2,3)+12*gm(1,2)*d2gm(3,3))+3*gm(2,3)*(dgm01(3,3)*dgm10(1,3)&
&   +dgm01(1,3)*dgm10(3,3)+gm(1,3)*d2gm(3,3))
   cm(9,3)=-4.5d0*gm(2,2)*dgm01(3,3)*dgm10(1,3)-4.5d0*gm(1,3)*dgm01(3,3)&
&   *dgm10(2,2)+3*gm(1,3)*dgm01(2,3)*dgm10(2,3)+12*gm(1,2)*dgm01(3,3)&
&   *dgm10(2,3)-4.5d0*gm(2,2)*dgm01(1,3)*dgm10(3,3)-4.5d0*gm(1,3)&
&   *dgm01(2,2)*dgm10(3,3)+12*gm(1,2)*dgm01(2,3)*dgm10(3,3)+1.5d0*gm(2,3)&
&   **2*d2gm(1,3)+gm(3,3)*(12*dgm01(2,3)*dgm10(1,2)-4.5d0*dgm01(2,2)&
&   *dgm10(1,3)-4.5d0*dgm01(1,3)*dgm10(2,2)+12*dgm01(1,2)*dgm10(2,3)&
&   +12*gm(2,3)*d2gm(1,2)-4.5d0*gm(2,2)*d2gm(1,3)-4.5d0*gm(1,3)*d2gm(2,2)&
&   +12*gm(1,2)*d2gm(2,3))-4.5d0*gm(1,3)*gm(2,2)*d2gm(3,3)+gm(2,3)&
&   *(12*dgm01(3,3)*dgm10(1,2)+3*dgm01(2,3)*dgm10(1,3)+3*dgm01(1,3)&
&   *dgm10(2,3)+12*dgm01(1,2)*dgm10(3,3)+3*gm(1,3)*d2gm(2,3)+12*gm(1,2)&
&   *d2gm(3,3))
   cm(10,3)=6*gm(1,3)*dgm01(3,3)*dgm10(3,3)+3*gm(3,3)**2*d2gm(1,3)&
&   +6*gm(3,3)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)+gm(1,3)&
&   *d2gm(3,3))
   cm(1,4)=9*gm(1,2)*dgm01(1,3)*dgm10(1,1)-6*gm(1,1)*dgm01(2,3)&
&   *dgm10(1,1)+9*gm(1,1)*dgm01(1,3)*dgm10(1,2)+9*gm(1,2)*dgm01(1,1)&
&   *dgm10(1,3)+9*gm(1,1)*dgm01(1,2)*dgm10(1,3)-6*gm(1,1)*dgm01(1,1)&
&   *dgm10(2,3)-6*gm(2,3)*(dgm01(1,1)*dgm10(1,1)+gm(1,1)*d2gm(1,1))&
&   +9*gm(1,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2)+gm(1,2)&
&   *d2gm(1,1)+gm(1,1)*d2gm(1,2))+9*gm(1,1)*gm(1,2)*d2gm(1,3)-3*gm(1,1)&
&   **2*d2gm(2,3)
   cm(2,4)=3*gm(1,3)*dgm01(2,2)*dgm10(1,2)+6*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,2)+3*gm(1,2)*dgm01(2,2)*dgm10(1,3)+3*gm(1,3)*dgm01(1,2)&
&   *dgm10(2,2)+3*gm(1,2)*dgm01(1,3)*dgm10(2,2)+12*gm(1,1)*dgm01(2,3)&
&   *dgm10(2,2)+6*gm(1,2)*dgm01(1,2)*dgm10(2,3)+12*gm(1,1)*dgm01(2,2)&
&   *dgm10(2,3)+3*gm(1,2)*gm(1,3)*d2gm(2,2)+gm(2,3)*(12*dgm01(2,2)&
&   *dgm10(1,1)+6*dgm01(1,2)*dgm10(1,2)+12*dgm01(1,1)*dgm10(2,2)&
&   +12*gm(2,2)*d2gm(1,1)+6*gm(1,2)*d2gm(1,2)+12*gm(1,1)*d2gm(2,2))&
&   +3*gm(1,2)**2*d2gm(2,3)+gm(2,2)*(12*dgm01(2,3)*dgm10(1,1)+3*dgm01(1,3)&
&   *dgm10(1,2)+3*dgm01(1,2)*dgm10(1,3)+12*dgm01(1,1)*dgm10(2,3)&
&   +3*gm(1,3)*d2gm(1,2)+3*gm(1,2)*d2gm(1,3)+12*gm(1,1)*d2gm(2,3))
   cm(3,4)=3*gm(1,3)*dgm01(3,3)*dgm10(1,2)+6*gm(1,3)*dgm01(2,3)&
&   *dgm10(1,3)+3*gm(1,2)*dgm01(3,3)*dgm10(1,3)+6*gm(1,3)*dgm01(1,3)&
&   *dgm10(2,3)+12*gm(1,1)*dgm01(3,3)*dgm10(2,3)+3*gm(1,3)*dgm01(1,2)&
&   *dgm10(3,3)+3*gm(1,2)*dgm01(1,3)*dgm10(3,3)+12*gm(1,1)*dgm01(2,3)&
&   *dgm10(3,3)+3*gm(1,3)**2*d2gm(2,3)+gm(3,3)*(12*dgm01(2,3)*dgm10(1,1)&
&   +3*dgm01(1,3)*dgm10(1,2)+3*dgm01(1,2)*dgm10(1,3)+12*dgm01(1,1)&
&   *dgm10(2,3)+12*gm(2,3)*d2gm(1,1)+3*gm(1,3)*d2gm(1,2)+3*gm(1,2)&
&   *d2gm(1,3)+12*gm(1,1)*d2gm(2,3))+3*gm(1,2)*gm(1,3)*d2gm(3,3)&
&   +gm(2,3)*(12*dgm01(3,3)*dgm10(1,1)+6*dgm01(1,3)*dgm10(1,3)+12*dgm01(1,1)&
&   *dgm10(3,3)+6*gm(1,3)*d2gm(1,3)+12*gm(1,1)*d2gm(3,3))
   cm(4,4)=15*gm(2,2)*dgm01(3,3)*dgm10(1,1)-6*gm(1,3)*dgm01(2,3)&
&   *dgm10(1,2)+18*gm(1,2)*dgm01(3,3)*dgm10(1,2)+18*gm(2,2)*dgm01(1,3)&
&   *dgm10(1,3)+18*gm(1,3)*dgm01(2,2)*dgm10(1,3)-6*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,3)+18*gm(1,3)*dgm01(1,3)*dgm10(2,2)+15*gm(1,1)*dgm01(3,3)&
&   *dgm10(2,2)-6*gm(1,3)*dgm01(1,2)*dgm10(2,3)-6*gm(1,2)*dgm01(1,3)&
&   *dgm10(2,3)+18*gm(1,1)*dgm01(2,3)*dgm10(2,3)+15*gm(2,2)*dgm01(1,1)&
&   *dgm10(3,3)+18*gm(1,2)*dgm01(1,2)*dgm10(3,3)+15*gm(1,1)*dgm01(2,2)&
&   *dgm10(3,3)+9*gm(2,3)**2*d2gm(1,1)+18*gm(1,3)*gm(2,2)*d2gm(1,3)&
&   +9*gm(1,3)**2*d2gm(2,2)+gm(3,3)*(15*dgm01(2,2)*dgm10(1,1)+18*dgm01(1,2)&
&   *dgm10(1,2)+15*dgm01(1,1)*dgm10(2,2)+15*gm(2,2)*d2gm(1,1)+18*gm(1,2)&
&   *d2gm(1,2)+15*gm(1,1)*d2gm(2,2))-6*gm(1,2)*gm(1,3)*d2gm(2,3)&
&   +gm(2,3)*(18*dgm01(2,3)*dgm10(1,1)-6*dgm01(1,3)*dgm10(1,2)-6*dgm01(1,2)&
&   *dgm10(1,3)+18*dgm01(1,1)*dgm10(2,3)-6*gm(1,3)*d2gm(1,2)-6*gm(1,2)&
&   *d2gm(1,3)+18*gm(1,1)*d2gm(2,3))+9*gm(1,2)**2*d2gm(3,3)+15*gm(1,1)&
&   *gm(2,2)*d2gm(3,3)
   cm(5,4)=3*gm(1,3)*dgm01(2,3)*dgm10(1,1)+12*gm(1,2)*dgm01(3,3)&
&   *dgm10(1,1)+6*gm(1,3)*dgm01(1,3)*dgm10(1,2)+12*gm(1,1)*dgm01(3,3)&
&   *dgm10(1,2)+6*gm(1,3)*dgm01(1,2)*dgm10(1,3)+6*gm(1,2)*dgm01(1,3)&
&   *dgm10(1,3)+3*gm(1,1)*dgm01(2,3)*dgm10(1,3)+3*gm(1,3)*dgm01(1,1)&
&   *dgm10(2,3)+3*gm(1,1)*dgm01(1,3)*dgm10(2,3)+12*gm(1,2)*dgm01(1,1)&
&   *dgm10(3,3)+12*gm(1,1)*dgm01(1,2)*dgm10(3,3)+3*gm(1,3)**2*d2gm(1,2)&
&   +12*gm(3,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2)+gm(1,2)&
&   *d2gm(1,1)+gm(1,1)*d2gm(1,2))+6*gm(1,2)*gm(1,3)*d2gm(1,3)+3*gm(2,3)&
&   *(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3)+gm(1,3)*d2gm(1,1)&
&   +gm(1,1)*d2gm(1,3))+3*gm(1,1)*gm(1,3)*d2gm(2,3)+12*gm(1,1)*gm(1,2)&
&   *d2gm(3,3)
   cm(6,4)=12*gm(1,3)*dgm01(2,2)*dgm10(1,1)+3*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,1)+6*gm(1,3)*dgm01(1,2)*dgm10(1,2)+6*gm(1,2)*dgm01(1,3)&
&   *dgm10(1,2)+3*gm(1,1)*dgm01(2,3)*dgm10(1,2)+6*gm(1,2)*dgm01(1,2)&
&   *dgm10(1,3)+12*gm(1,1)*dgm01(2,2)*dgm10(1,3)+12*gm(1,3)*dgm01(1,1)&
&   *dgm10(2,2)+12*gm(1,1)*dgm01(1,3)*dgm10(2,2)+3*gm(1,2)*dgm01(1,1)&
&   *dgm10(2,3)+3*gm(1,1)*dgm01(1,2)*dgm10(2,3)+6*gm(1,2)*gm(1,3)&
&   *d2gm(1,2)+3*gm(2,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2)&
&   +gm(1,2)*d2gm(1,1)+gm(1,1)*d2gm(1,2))+3*gm(1,2)**2*d2gm(1,3)&
&   +12*gm(2,2)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3)+gm(1,3)&
&   *d2gm(1,1)+gm(1,1)*d2gm(1,3))+12*gm(1,1)*gm(1,3)*d2gm(2,2)+3*gm(1,1)&
&   *gm(1,2)*d2gm(2,3)
   cm(7,4)=-6*gm(1,3)*dgm01(2,2)*dgm10(2,2)+9*gm(1,2)*dgm01(2,3)&
&   *dgm10(2,2)+9*gm(1,2)*dgm01(2,2)*dgm10(2,3)-3*gm(2,2)**2*d2gm(1,3)&
&   +9*gm(2,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)+gm(2,2)&
&   *d2gm(1,2)+gm(1,2)*d2gm(2,2))+gm(2,2)*(9*dgm01(2,3)*dgm10(1,2)&
&   -6*dgm01(2,2)*dgm10(1,3)-6*dgm01(1,3)*dgm10(2,2)+9*dgm01(1,2)&
&   *dgm10(2,3)-6*gm(1,3)*d2gm(2,2)+9*gm(1,2)*d2gm(2,3))
   cm(8,4)=12*gm(2,2)*dgm01(3,3)*dgm10(1,3)+12*gm(1,3)*dgm01(3,3)&
&   *dgm10(2,2)+6*gm(1,3)*dgm01(2,3)*dgm10(2,3)+3*gm(1,2)*dgm01(3,3)&
&   *dgm10(2,3)+12*gm(2,2)*dgm01(1,3)*dgm10(3,3)+12*gm(1,3)*dgm01(2,2)&
&   *dgm10(3,3)+3*gm(1,2)*dgm01(2,3)*dgm10(3,3)+3*gm(2,3)**2*d2gm(1,3)&
&   +gm(3,3)*(3*dgm01(2,3)*dgm10(1,2)+12*dgm01(2,2)*dgm10(1,3)+12*dgm01(1,3)&
&   *dgm10(2,2)+3*dgm01(1,2)*dgm10(2,3)+3*gm(2,3)*d2gm(1,2)+12*gm(2,2)&
&   *d2gm(1,3)+12*gm(1,3)*d2gm(2,2)+3*gm(1,2)*d2gm(2,3))+12*gm(1,3)&
&   *gm(2,2)*d2gm(3,3)+gm(2,3)*(3*dgm01(3,3)*dgm10(1,2)+6*dgm01(2,3)&
&   *dgm10(1,3)+6*dgm01(1,3)*dgm10(2,3)+3*dgm01(1,2)*dgm10(3,3)+6*gm(1,3)&
&   *d2gm(2,3)+3*gm(1,2)*d2gm(3,3))
   cm(9,4)=12*gm(2,2)*dgm01(3,3)*dgm10(1,2)+3*gm(2,2)*dgm01(2,3)&
&   *dgm10(1,3)+3*gm(1,3)*dgm01(2,3)*dgm10(2,2)+12*gm(1,2)*dgm01(3,3)&
&   *dgm10(2,2)+3*gm(2,2)*dgm01(1,3)*dgm10(2,3)+3*gm(1,3)*dgm01(2,2)&
&   *dgm10(2,3)+6*gm(1,2)*dgm01(2,3)*dgm10(2,3)+12*gm(2,2)*dgm01(1,2)&
&   *dgm10(3,3)+12*gm(1,2)*dgm01(2,2)*dgm10(3,3)+3*gm(2,3)**2*d2gm(1,2)&
&   +12*gm(3,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)+gm(2,2)&
&   *d2gm(1,2)+gm(1,2)*d2gm(2,2))+3*gm(1,3)*gm(2,2)*d2gm(2,3)+gm(2,3)&
&   *(6*dgm01(2,3)*dgm10(1,2)+3*dgm01(2,2)*dgm10(1,3)+3*dgm01(1,3)&
&   *dgm10(2,2)+6*dgm01(1,2)*dgm10(2,3)+3*gm(2,2)*d2gm(1,3)+3*gm(1,3)&
&   *d2gm(2,2)+6*gm(1,2)*d2gm(2,3))+12*gm(1,2)*gm(2,2)*d2gm(3,3)
   cm(10,4)=9*gm(1,3)*dgm01(3,3)*dgm10(2,3)+9*gm(1,3)*dgm01(2,3)&
&   *dgm10(3,3)-6*gm(1,2)*dgm01(3,3)*dgm10(3,3)-3*gm(3,3)**2*d2gm(1,2)&
&   +gm(3,3)*(-6*dgm01(3,3)*dgm10(1,2)+9*dgm01(2,3)*dgm10(1,3)+9*dgm01(1,3)&
&   *dgm10(2,3)-6*dgm01(1,2)*dgm10(3,3)+9*gm(2,3)*d2gm(1,3)+9*gm(1,3)&
&   *d2gm(2,3)-6*gm(1,2)*d2gm(3,3))+9*gm(2,3)*(dgm01(3,3)*dgm10(1,3)&
&   +dgm01(1,3)*dgm10(3,3)+gm(1,3)*d2gm(3,3))
   cm(1,5)=6*gm(1,3)*(dgm01(1,1)*dgm10(1,1)+gm(1,1)*d2gm(1,1))&
&   +gm(1,1)*(6*dgm01(1,3)*dgm10(1,1)+6*dgm01(1,1)*dgm10(1,3)+3*gm(1,1)&
&   *d2gm(1,3))
   cm(2,5)=-4.5d0*gm(1,3)*dgm01(2,2)*dgm10(1,1)+12*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,1)+3*gm(1,3)*dgm01(1,2)*dgm10(1,2)+3*gm(1,2)*dgm01(1,3)&
&   *dgm10(1,2)+12*gm(1,1)*dgm01(2,3)*dgm10(1,2)+3*gm(1,2)*dgm01(1,2)&
&   *dgm10(1,3)-4.5d0*gm(1,1)*dgm01(2,2)*dgm10(1,3)-4.5d0*gm(1,3)&
&   *dgm01(1,1)*dgm10(2,2)-4.5d0*gm(1,1)*dgm01(1,3)*dgm10(2,2)+12*gm(1,2)&
&   *dgm01(1,1)*dgm10(2,3)+12*gm(1,1)*dgm01(1,2)*dgm10(2,3)+3*gm(1,2)&
&   *gm(1,3)*d2gm(1,2)+12*gm(2,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,2)+gm(1,2)*d2gm(1,1)+gm(1,1)*d2gm(1,2))+1.5d0*gm(1,2)&
&   **2*d2gm(1,3)-4.5d0*gm(2,2)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,3)+gm(1,3)*d2gm(1,1)+gm(1,1)*d2gm(1,3))-4.5d0*gm(1,1)&
&   *gm(1,3)*d2gm(2,2)+12*gm(1,1)*gm(1,2)*d2gm(2,3)
   cm(3,5)=7.5d0*gm(1,1)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&   +4.5d0*gm(1,3)**2*d2gm(1,3)+7.5d0*gm(3,3)*(dgm01(1,3)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(1,3)+gm(1,3)*d2gm(1,1)+gm(1,1)*d2gm(1,3))+gm(1,3)&
&   *(7.5d0*dgm01(3,3)*dgm10(1,1)+9*dgm01(1,3)*dgm10(1,3)+7.5d0*dgm01(1,1)&
&   *dgm10(3,3)+7.5d0*gm(1,1)*d2gm(3,3))
   cm(4,5)=3*gm(1,3)*dgm01(2,3)*dgm10(1,1)+12*gm(1,2)*dgm01(3,3)&
&   *dgm10(1,1)+6*gm(1,3)*dgm01(1,3)*dgm10(1,2)+12*gm(1,1)*dgm01(3,3)&
&   *dgm10(1,2)+6*gm(1,3)*dgm01(1,2)*dgm10(1,3)+6*gm(1,2)*dgm01(1,3)&
&   *dgm10(1,3)+3*gm(1,1)*dgm01(2,3)*dgm10(1,3)+3*gm(1,3)*dgm01(1,1)&
&   *dgm10(2,3)+3*gm(1,1)*dgm01(1,3)*dgm10(2,3)+12*gm(1,2)*dgm01(1,1)&
&   *dgm10(3,3)+12*gm(1,1)*dgm01(1,2)*dgm10(3,3)+3*gm(1,3)**2*d2gm(1,2)&
&   +12*gm(3,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2)+gm(1,2)&
&   *d2gm(1,1)+gm(1,1)*d2gm(1,2))+6*gm(1,2)*gm(1,3)*d2gm(1,3)+3*gm(2,3)&
&   *(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3)+gm(1,3)*d2gm(1,1)&
&   +gm(1,1)*d2gm(1,3))+3*gm(1,1)*gm(1,3)*d2gm(2,3)+12*gm(1,1)*gm(1,2)&
&   *d2gm(3,3)
   cm(5,5)=3*gm(1,3)**2*d2gm(1,1)+12*gm(3,3)*(dgm01(1,1)*dgm10(1,1)&
&   +gm(1,1)*d2gm(1,1))+6*gm(1,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,3)+gm(1,1)*d2gm(1,3))+gm(1,1)*(12*dgm01(3,3)*dgm10(1,1)&
&   +6*dgm01(1,3)*dgm10(1,3)+12*dgm01(1,1)*dgm10(3,3)+6*gm(1,1)*d2gm(3,3))
   cm(6,5)=3*gm(1,2)*dgm01(1,3)*dgm10(1,1)+12*gm(1,1)*dgm01(2,3)&
&   *dgm10(1,1)+3*gm(1,1)*dgm01(1,3)*dgm10(1,2)+3*gm(1,2)*dgm01(1,1)&
&   *dgm10(1,3)+3*gm(1,1)*dgm01(1,2)*dgm10(1,3)+12*gm(1,1)*dgm01(1,1)&
&   *dgm10(2,3)+12*gm(2,3)*(dgm01(1,1)*dgm10(1,1)+gm(1,1)*d2gm(1,1))&
&   +3*gm(1,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2)+gm(1,2)&
&   *d2gm(1,1)+gm(1,1)*d2gm(1,2))+3*gm(1,1)*gm(1,2)*d2gm(1,3)+6*gm(1,1)&
&   **2*d2gm(2,3)
   cm(7,5)=(-36*gm(1,3)*dgm01(2,2)*dgm10(1,2)+180*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,2)-36*gm(1,2)*dgm01(2,2)*dgm10(1,3)-36*gm(1,3)*dgm01(1,2)&
&   *dgm10(2,2)-36*gm(1,2)*dgm01(1,3)*dgm10(2,2)-18*gm(1,1)*dgm01(2,3)&
&   *dgm10(2,2)+180*gm(1,2)*dgm01(1,2)*dgm10(2,3)-18*gm(1,1)*dgm01(2,2)&
&   *dgm10(2,3)-36*gm(1,2)*gm(1,3)*d2gm(2,2)+gm(2,3)*(-18*dgm01(2,2)&
&   *dgm10(1,1)+180*dgm01(1,2)*dgm10(1,2)-18*dgm01(1,1)*dgm10(2,2)&
&   -18*gm(2,2)*d2gm(1,1)+180*gm(1,2)*d2gm(1,2)-18*gm(1,1)*d2gm(2,2))&
&   +90*gm(1,2)**2*d2gm(2,3)+gm(2,2)*(-18*dgm01(2,3)*dgm10(1,1)-36*dgm01(1,3)&
&   *dgm10(1,2)-36*dgm01(1,2)*dgm10(1,3)-18*dgm01(1,1)*dgm10(2,3)&
&   -36*gm(1,3)*d2gm(1,2)-36*gm(1,2)*d2gm(1,3)-18*gm(1,1)*d2gm(2,3)))&
&   /12.d0
   cm(8,5)=12*gm(1,3)*dgm01(3,3)*dgm10(1,2)+3*gm(1,3)*dgm01(2,3)&
&   *dgm10(1,3)+12*gm(1,2)*dgm01(3,3)*dgm10(1,3)+3*gm(1,3)*dgm01(1,3)&
&   *dgm10(2,3)-4.5d0*gm(1,1)*dgm01(3,3)*dgm10(2,3)+12*gm(1,3)*dgm01(1,2)&
&   *dgm10(3,3)+12*gm(1,2)*dgm01(1,3)*dgm10(3,3)-4.5d0*gm(1,1)*dgm01(2,3)&
&   *dgm10(3,3)+1.5d0*gm(1,3)**2*d2gm(2,3)+gm(3,3)*(-4.5d0*dgm01(2,3)&
&   *dgm10(1,1)+12*dgm01(1,3)*dgm10(1,2)+12*dgm01(1,2)*dgm10(1,3)&
&   -4.5d0*dgm01(1,1)*dgm10(2,3)-4.5d0*gm(2,3)*d2gm(1,1)+12*gm(1,3)&
&   *d2gm(1,2)+12*gm(1,2)*d2gm(1,3)-4.5d0*gm(1,1)*d2gm(2,3))+12*gm(1,2)&
&   *gm(1,3)*d2gm(3,3)+gm(2,3)*(-4.5d0*dgm01(3,3)*dgm10(1,1)+3*dgm01(1,3)&
&   *dgm10(1,3)-4.5d0*dgm01(1,1)*dgm10(3,3)+3*gm(1,3)*d2gm(1,3)-4.5d0*gm(1,1)&
&   *d2gm(3,3))
   cm(9,5)=-1.5d0*gm(2,2)*dgm01(3,3)*dgm10(1,1)+9*gm(1,3)*dgm01(2,3)&
&   *dgm10(1,2)+15*gm(1,2)*dgm01(3,3)*dgm10(1,2)-6*gm(2,2)*dgm01(1,3)&
&   *dgm10(1,3)-6*gm(1,3)*dgm01(2,2)*dgm10(1,3)+9*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,3)-6*gm(1,3)*dgm01(1,3)*dgm10(2,2)-1.5d0*gm(1,1)*dgm01(3,3)&
&   *dgm10(2,2)+9*gm(1,3)*dgm01(1,2)*dgm10(2,3)+9*gm(1,2)*dgm01(1,3)&
&   *dgm10(2,3)-6*gm(1,1)*dgm01(2,3)*dgm10(2,3)-1.5d0*gm(2,2)*dgm01(1,1)&
&   *dgm10(3,3)+15*gm(1,2)*dgm01(1,2)*dgm10(3,3)-1.5d0*gm(1,1)*dgm01(2,2)&
&   *dgm10(3,3)-3*gm(2,3)**2*d2gm(1,1)-6*gm(1,3)*gm(2,2)*d2gm(1,3)&
&   -3*gm(1,3)**2*d2gm(2,2)+gm(3,3)*(-1.5d0*dgm01(2,2)*dgm10(1,1)&
&   +15*dgm01(1,2)*dgm10(1,2)-1.5d0*dgm01(1,1)*dgm10(2,2)-1.5d0*gm(2,2)&
&   *d2gm(1,1)+15*gm(1,2)*d2gm(1,2)-1.5d0*gm(1,1)*d2gm(2,2))+9*gm(1,2)&
&   *gm(1,3)*d2gm(2,3)+gm(2,3)*(-6*dgm01(2,3)*dgm10(1,1)+9*dgm01(1,3)&
&   *dgm10(1,2)+9*dgm01(1,2)*dgm10(1,3)-6*dgm01(1,1)*dgm10(2,3)+9*gm(1,3)&
&   *d2gm(1,2)+9*gm(1,2)*d2gm(1,3)-6*gm(1,1)*d2gm(2,3))+7.5d0*gm(1,2)&
&   **2*d2gm(3,3)-1.5d0*gm(1,1)*gm(2,2)*d2gm(3,3)
   cm(10,5)=-3*gm(1,1)*dgm01(3,3)*dgm10(3,3)+9*gm(1,3)*(dgm01(3,3)&
&   *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))-1.5d0*gm(3,3)**2*d2gm(1,1)&
&   +4.5d0*gm(1,3)**2*d2gm(3,3)+gm(3,3)*(-3*dgm01(3,3)*dgm10(1,1)&
&   +9*dgm01(1,3)*dgm10(1,3)-3*dgm01(1,1)*dgm10(3,3)+9*gm(1,3)*d2gm(1,3)&
&   -3*gm(1,1)*d2gm(3,3))
   cm(1,6)=6*gm(1,2)*(dgm01(1,1)*dgm10(1,1)+gm(1,1)*d2gm(1,1))&
&   +gm(1,1)*(6*dgm01(1,2)*dgm10(1,1)+6*dgm01(1,1)*dgm10(1,2)+3*gm(1,1)&
&   *d2gm(1,2))
   cm(2,6)=7.5d0*gm(1,1)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&   +4.5d0*gm(1,2)**2*d2gm(1,2)+7.5d0*gm(2,2)*(dgm01(1,2)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(1,2)+gm(1,2)*d2gm(1,1)+gm(1,1)*d2gm(1,2))+gm(1,2)&
&   *(7.5d0*dgm01(2,2)*dgm10(1,1)+9*dgm01(1,2)*dgm10(1,2)+7.5d0*dgm01(1,1)&
&   *dgm10(2,2)+7.5d0*gm(1,1)*d2gm(2,2))
   cm(3,6)=12*gm(1,3)*dgm01(2,3)*dgm10(1,1)-4.5d0*gm(1,2)*dgm01(3,3)&
&   *dgm10(1,1)+3*gm(1,3)*dgm01(1,3)*dgm10(1,2)-4.5d0*gm(1,1)*dgm01(3,3)&
&   *dgm10(1,2)+3*gm(1,3)*dgm01(1,2)*dgm10(1,3)+3*gm(1,2)*dgm01(1,3)&
&   *dgm10(1,3)+12*gm(1,1)*dgm01(2,3)*dgm10(1,3)+12*gm(1,3)*dgm01(1,1)&
&   *dgm10(2,3)+12*gm(1,1)*dgm01(1,3)*dgm10(2,3)-4.5d0*gm(1,2)*dgm01(1,1)&
&   *dgm10(3,3)-4.5d0*gm(1,1)*dgm01(1,2)*dgm10(3,3)+1.5d0*gm(1,3)&
&   **2*d2gm(1,2)-4.5d0*gm(3,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,2)+gm(1,2)*d2gm(1,1)+gm(1,1)*d2gm(1,2))+3*gm(1,2)*gm(1,3)&
&   *d2gm(1,3)+12*gm(2,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3)&
&   +gm(1,3)*d2gm(1,1)+gm(1,1)*d2gm(1,3))+12*gm(1,1)*gm(1,3)*d2gm(2,3)&
&   -4.5d0*gm(1,1)*gm(1,2)*d2gm(3,3)
   cm(4,6)=12*gm(1,3)*dgm01(2,2)*dgm10(1,1)+3*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,1)+6*gm(1,3)*dgm01(1,2)*dgm10(1,2)+6*gm(1,2)*dgm01(1,3)&
&   *dgm10(1,2)+3*gm(1,1)*dgm01(2,3)*dgm10(1,2)+6*gm(1,2)*dgm01(1,2)&
&   *dgm10(1,3)+12*gm(1,1)*dgm01(2,2)*dgm10(1,3)+12*gm(1,3)*dgm01(1,1)&
&   *dgm10(2,2)+12*gm(1,1)*dgm01(1,3)*dgm10(2,2)+3*gm(1,2)*dgm01(1,1)&
&   *dgm10(2,3)+3*gm(1,1)*dgm01(1,2)*dgm10(2,3)+6*gm(1,2)*gm(1,3)&
&   *d2gm(1,2)+3*gm(2,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2)&
&   +gm(1,2)*d2gm(1,1)+gm(1,1)*d2gm(1,2))+3*gm(1,2)**2*d2gm(1,3)&
&   +12*gm(2,2)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3)+gm(1,3)&
&   *d2gm(1,1)+gm(1,1)*d2gm(1,3))+12*gm(1,1)*gm(1,3)*d2gm(2,2)+3*gm(1,1)&
&   *gm(1,2)*d2gm(2,3)
   cm(5,6)=3*gm(1,2)*dgm01(1,3)*dgm10(1,1)+12*gm(1,1)*dgm01(2,3)&
&   *dgm10(1,1)+3*gm(1,1)*dgm01(1,3)*dgm10(1,2)+3*gm(1,2)*dgm01(1,1)&
&   *dgm10(1,3)+3*gm(1,1)*dgm01(1,2)*dgm10(1,3)+12*gm(1,1)*dgm01(1,1)&
&   *dgm10(2,3)+12*gm(2,3)*(dgm01(1,1)*dgm10(1,1)+gm(1,1)*d2gm(1,1))&
&   +3*gm(1,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2)+gm(1,2)&
&   *d2gm(1,1)+gm(1,1)*d2gm(1,2))+3*gm(1,1)*gm(1,2)*d2gm(1,3)+6*gm(1,1)&
&   **2*d2gm(2,3)
   cm(6,6)=3*gm(1,2)**2*d2gm(1,1)+12*gm(2,2)*(dgm01(1,1)*dgm10(1,1)&
&   +gm(1,1)*d2gm(1,1))+6*gm(1,2)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,2)+gm(1,1)*d2gm(1,2))+gm(1,1)*(12*dgm01(2,2)*dgm10(1,1)&
&   +6*dgm01(1,2)*dgm10(1,2)+12*dgm01(1,1)*dgm10(2,2)+6*gm(1,1)*d2gm(2,2))
   cm(7,6)=-3*gm(1,1)*dgm01(2,2)*dgm10(2,2)+9*gm(1,2)*(dgm01(2,2)&
&   *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))-1.5d0*gm(2,2)**2*d2gm(1,1)&
&   +4.5d0*gm(1,2)**2*d2gm(2,2)+gm(2,2)*(-3*dgm01(2,2)*dgm10(1,1)&
&   +9*dgm01(1,2)*dgm10(1,2)-3*dgm01(1,1)*dgm10(2,2)+9*gm(1,2)*d2gm(1,2)&
&   -3*gm(1,1)*d2gm(2,2))
   cm(8,6)=-1.5d0*gm(2,2)*dgm01(3,3)*dgm10(1,1)+9*gm(1,3)*dgm01(2,3)&
&   *dgm10(1,2)-6*gm(1,2)*dgm01(3,3)*dgm10(1,2)+15*gm(2,2)*dgm01(1,3)&
&   *dgm10(1,3)+15*gm(1,3)*dgm01(2,2)*dgm10(1,3)+9*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,3)+15*gm(1,3)*dgm01(1,3)*dgm10(2,2)-1.5d0*gm(1,1)*dgm01(3,3)&
&   *dgm10(2,2)+9*gm(1,3)*dgm01(1,2)*dgm10(2,3)+9*gm(1,2)*dgm01(1,3)&
&   *dgm10(2,3)-6*gm(1,1)*dgm01(2,3)*dgm10(2,3)-1.5d0*gm(2,2)*dgm01(1,1)&
&   *dgm10(3,3)-6*gm(1,2)*dgm01(1,2)*dgm10(3,3)-1.5d0*gm(1,1)*dgm01(2,2)&
&   *dgm10(3,3)-3*gm(2,3)**2*d2gm(1,1)+15*gm(1,3)*gm(2,2)*d2gm(1,3)&
&   +7.5d0*gm(1,3)**2*d2gm(2,2)+gm(3,3)*(-1.5d0*dgm01(2,2)*dgm10(1,1)&
&   -6*dgm01(1,2)*dgm10(1,2)-1.5d0*dgm01(1,1)*dgm10(2,2)-1.5d0*gm(2,2)&
&   *d2gm(1,1)-6*gm(1,2)*d2gm(1,2)-1.5d0*gm(1,1)*d2gm(2,2))+9*gm(1,2)&
&   *gm(1,3)*d2gm(2,3)+gm(2,3)*(-6*dgm01(2,3)*dgm10(1,1)+9*dgm01(1,3)&
&   *dgm10(1,2)+9*dgm01(1,2)*dgm10(1,3)-6*dgm01(1,1)*dgm10(2,3)+9*gm(1,3)&
&   *d2gm(1,2)+9*gm(1,2)*d2gm(1,3)-6*gm(1,1)*d2gm(2,3))-3*gm(1,2)&
&   **2*d2gm(3,3)-1.5d0*gm(1,1)*gm(2,2)*d2gm(3,3)
   cm(9,6)=12*gm(1,3)*dgm01(2,2)*dgm10(1,2)+3*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,2)+12*gm(1,2)*dgm01(2,2)*dgm10(1,3)+12*gm(1,3)*dgm01(1,2)&
&   *dgm10(2,2)+12*gm(1,2)*dgm01(1,3)*dgm10(2,2)-4.5d0*gm(1,1)*dgm01(2,3)&
&   *dgm10(2,2)+3*gm(1,2)*dgm01(1,2)*dgm10(2,3)-4.5d0*gm(1,1)*dgm01(2,2)&
&   *dgm10(2,3)+12*gm(1,2)*gm(1,3)*d2gm(2,2)+gm(2,3)*(-4.5d0*dgm01(2,2)&
&   *dgm10(1,1)+3*dgm01(1,2)*dgm10(1,2)-4.5d0*dgm01(1,1)*dgm10(2,2)&
&   -4.5d0*gm(2,2)*d2gm(1,1)+3*gm(1,2)*d2gm(1,2)-4.5d0*gm(1,1)*d2gm(2,2))&
&   +1.5d0*gm(1,2)**2*d2gm(2,3)+gm(2,2)*(-4.5d0*dgm01(2,3)*dgm10(1,1)&
&   +12*dgm01(1,3)*dgm10(1,2)+12*dgm01(1,2)*dgm10(1,3)-4.5d0*dgm01(1,1)&
&   *dgm10(2,3)+12*gm(1,3)*d2gm(1,2)+12*gm(1,2)*d2gm(1,3)-4.5d0*gm(1,1)&
&   *d2gm(2,3))
   cm(10,6)=(-36*gm(1,3)*dgm01(3,3)*dgm10(1,2)+180*gm(1,3)*dgm01(2,3)&
&   *dgm10(1,3)-36*gm(1,2)*dgm01(3,3)*dgm10(1,3)+180*gm(1,3)*dgm01(1,3)&
&   *dgm10(2,3)-18*gm(1,1)*dgm01(3,3)*dgm10(2,3)-36*gm(1,3)*dgm01(1,2)&
&   *dgm10(3,3)-36*gm(1,2)*dgm01(1,3)*dgm10(3,3)-18*gm(1,1)*dgm01(2,3)&
&   *dgm10(3,3)+90*gm(1,3)**2*d2gm(2,3)+gm(3,3)*(-18*dgm01(2,3)*dgm10(1,1)&
&   -36*dgm01(1,3)*dgm10(1,2)-36*dgm01(1,2)*dgm10(1,3)-18*dgm01(1,1)&
&   *dgm10(2,3)-18*gm(2,3)*d2gm(1,1)-36*gm(1,3)*d2gm(1,2)-36*gm(1,2)&
&   *d2gm(1,3)-18*gm(1,1)*d2gm(2,3))-36*gm(1,2)*gm(1,3)*d2gm(3,3)&
&   +gm(2,3)*(-18*dgm01(3,3)*dgm10(1,1)+180*dgm01(1,3)*dgm10(1,3)&
&   -18*dgm01(1,1)*dgm10(3,3)+180*gm(1,3)*d2gm(1,3)-18*gm(1,1)*d2gm(3,3)))&
&   /12.d0
   cm(1,7)=-1.5d0*gm(1,1)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&   +7.5d0*gm(1,2)**2*d2gm(1,2)-1.5d0*gm(2,2)*(dgm01(1,2)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(1,2)+gm(1,2)*d2gm(1,1)+gm(1,1)*d2gm(1,2))+gm(1,2)&
&   *(-1.5d0*dgm01(2,2)*dgm10(1,1)+15*dgm01(1,2)*dgm10(1,2)-1.5d0*dgm01(1,1)&
&   *dgm10(2,2)-1.5d0*gm(1,1)*d2gm(2,2))
   cm(2,7)=6*gm(1,2)*dgm01(2,2)*dgm10(2,2)+3*gm(2,2)**2*d2gm(1,2)&
&   +6*gm(2,2)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)+gm(1,2)&
&   *d2gm(2,2))
   cm(3,7)=-1.5d0*gm(2,2)*dgm01(3,3)*dgm10(1,2)-3*gm(2,2)*dgm01(2,3)&
&   *dgm10(1,3)-3*gm(1,3)*dgm01(2,3)*dgm10(2,2)-1.5d0*gm(1,2)*dgm01(3,3)&
&   *dgm10(2,2)-3*gm(2,2)*dgm01(1,3)*dgm10(2,3)-3*gm(1,3)*dgm01(2,2)&
&   *dgm10(2,3)+15*gm(1,2)*dgm01(2,3)*dgm10(2,3)-1.5d0*gm(2,2)*dgm01(1,2)&
&   *dgm10(3,3)-1.5d0*gm(1,2)*dgm01(2,2)*dgm10(3,3)+7.5d0*gm(2,3)&
&   **2*d2gm(1,2)-1.5d0*gm(3,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&   *dgm10(2,2)+gm(2,2)*d2gm(1,2)+gm(1,2)*d2gm(2,2))-3*gm(1,3)*gm(2,2)&
&   *d2gm(2,3)+gm(2,3)*(15*dgm01(2,3)*dgm10(1,2)-3*dgm01(2,2)*dgm10(1,3)&
&   -3*dgm01(1,3)*dgm10(2,2)+15*dgm01(1,2)*dgm10(2,3)-3*gm(2,2)*d2gm(1,3)&
&   -3*gm(1,3)*d2gm(2,2)+15*gm(1,2)*d2gm(2,3))-1.5d0*gm(1,2)*gm(2,2)&
&   *d2gm(3,3)
   cm(4,7)=-6*gm(1,3)*dgm01(2,2)*dgm10(2,2)+9*gm(1,2)*dgm01(2,3)&
&   *dgm10(2,2)+9*gm(1,2)*dgm01(2,2)*dgm10(2,3)-3*gm(2,2)**2*d2gm(1,3)&
&   +9*gm(2,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)+gm(2,2)&
&   *d2gm(1,2)+gm(1,2)*d2gm(2,2))+gm(2,2)*(9*dgm01(2,3)*dgm10(1,2)&
&   -6*dgm01(2,2)*dgm10(1,3)-6*dgm01(1,3)*dgm10(2,2)+9*dgm01(1,2)&
&   *dgm10(2,3)-6*gm(1,3)*d2gm(2,2)+9*gm(1,2)*d2gm(2,3))
   cm(5,7)=-3*gm(1,3)*dgm01(2,2)*dgm10(1,2)+15*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,2)-3*gm(1,2)*dgm01(2,2)*dgm10(1,3)-3*gm(1,3)*dgm01(1,2)&
&   *dgm10(2,2)-3*gm(1,2)*dgm01(1,3)*dgm10(2,2)-1.5d0*gm(1,1)*dgm01(2,3)&
&   *dgm10(2,2)+15*gm(1,2)*dgm01(1,2)*dgm10(2,3)-1.5d0*gm(1,1)*dgm01(2,2)&
&   *dgm10(2,3)-3*gm(1,2)*gm(1,3)*d2gm(2,2)+gm(2,3)*(-1.5d0*dgm01(2,2)&
&   *dgm10(1,1)+15*dgm01(1,2)*dgm10(1,2)-1.5d0*dgm01(1,1)*dgm10(2,2)&
&   -1.5d0*gm(2,2)*d2gm(1,1)+15*gm(1,2)*d2gm(1,2)-1.5d0*gm(1,1)*d2gm(2,2))&
&   +7.5d0*gm(1,2)**2*d2gm(2,3)+gm(2,2)*(-1.5d0*dgm01(2,3)*dgm10(1,1)&
&   -3*dgm01(1,3)*dgm10(1,2)-3*dgm01(1,2)*dgm10(1,3)-1.5d0*dgm01(1,1)&
&   *dgm10(2,3)-3*gm(1,3)*d2gm(1,2)-3*gm(1,2)*d2gm(1,3)-1.5d0*gm(1,1)&
&   *d2gm(2,3))
   cm(6,7)=-3*gm(1,1)*dgm01(2,2)*dgm10(2,2)+9*gm(1,2)*(dgm01(2,2)&
&   *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))-1.5d0*gm(2,2)**2*d2gm(1,1)&
&   +4.5d0*gm(1,2)**2*d2gm(2,2)+gm(2,2)*(-3*dgm01(2,2)*dgm10(1,1)&
&   +9*dgm01(1,2)*dgm10(1,2)-3*dgm01(1,1)*dgm10(2,2)+9*gm(1,2)*d2gm(1,2)&
&   -3*gm(1,1)*d2gm(2,2))
   cm(7,7)=gm(2,2)*(6*dgm01(2,2)*dgm10(2,2)+3*gm(2,2)*d2gm(2,2))
   cm(8,7)=4.5d0*gm(2,3)**2*d2gm(2,2)-3*gm(3,3)*(dgm01(2,2)*dgm10(2,2)&
&   +gm(2,2)*d2gm(2,2))+9*gm(2,3)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&   *dgm10(2,3)+gm(2,2)*d2gm(2,3))+gm(2,2)*(-3*dgm01(3,3)*dgm10(2,2)&
&   +9*dgm01(2,3)*dgm10(2,3)-3*dgm01(2,2)*dgm10(3,3)-1.5d0*gm(2,2)&
&   *d2gm(3,3))
   cm(9,7)=6*gm(2,3)*(dgm01(2,2)*dgm10(2,2)+gm(2,2)*d2gm(2,2))&
&   +gm(2,2)*(6*dgm01(2,3)*dgm10(2,2)+6*dgm01(2,2)*dgm10(2,3)+3*gm(2,2)&
&   *d2gm(2,3))
   cm(10,7)=-1.5d0*gm(2,2)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))&
&   +7.5d0*gm(2,3)**2*d2gm(2,3)-1.5d0*gm(3,3)*(dgm01(2,3)*dgm10(2,2)&
&   +dgm01(2,2)*dgm10(2,3)+gm(2,3)*d2gm(2,2)+gm(2,2)*d2gm(2,3))+gm(2,3)&
&   *(-1.5d0*dgm01(3,3)*dgm10(2,2)+15*dgm01(2,3)*dgm10(2,3)-1.5d0*dgm01(2,2)&
&   *dgm10(3,3)-1.5d0*gm(2,2)*d2gm(3,3))
   cm(1,8)=-3*gm(1,3)*dgm01(2,3)*dgm10(1,1)-1.5d0*gm(1,2)*dgm01(3,3)&
&   *dgm10(1,1)+15*gm(1,3)*dgm01(1,3)*dgm10(1,2)-1.5d0*gm(1,1)*dgm01(3,3)&
&   *dgm10(1,2)+15*gm(1,3)*dgm01(1,2)*dgm10(1,3)+15*gm(1,2)*dgm01(1,3)&
&   *dgm10(1,3)-3*gm(1,1)*dgm01(2,3)*dgm10(1,3)-3*gm(1,3)*dgm01(1,1)&
&   *dgm10(2,3)-3*gm(1,1)*dgm01(1,3)*dgm10(2,3)-1.5d0*gm(1,2)*dgm01(1,1)&
&   *dgm10(3,3)-1.5d0*gm(1,1)*dgm01(1,2)*dgm10(3,3)+7.5d0*gm(1,3)&
&   **2*d2gm(1,2)-1.5d0*gm(3,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,2)+gm(1,2)*d2gm(1,1)+gm(1,1)*d2gm(1,2))+15*gm(1,2)*gm(1,3)&
&   *d2gm(1,3)-3*gm(2,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3)&
&   +gm(1,3)*d2gm(1,1)+gm(1,1)*d2gm(1,3))-3*gm(1,1)*gm(1,3)*d2gm(2,3)&
&   -1.5d0*gm(1,1)*gm(1,2)*d2gm(3,3)
   cm(2,8)=-4.5d0*gm(2,2)*dgm01(3,3)*dgm10(1,2)+12*gm(2,2)*dgm01(2,3)&
&   *dgm10(1,3)+12*gm(1,3)*dgm01(2,3)*dgm10(2,2)-4.5d0*gm(1,2)*dgm01(3,3)&
&   *dgm10(2,2)+12*gm(2,2)*dgm01(1,3)*dgm10(2,3)+12*gm(1,3)*dgm01(2,2)&
&   *dgm10(2,3)+3*gm(1,2)*dgm01(2,3)*dgm10(2,3)-4.5d0*gm(2,2)*dgm01(1,2)&
&   *dgm10(3,3)-4.5d0*gm(1,2)*dgm01(2,2)*dgm10(3,3)+1.5d0*gm(2,3)&
&   **2*d2gm(1,2)-4.5d0*gm(3,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&   *dgm10(2,2)+gm(2,2)*d2gm(1,2)+gm(1,2)*d2gm(2,2))+12*gm(1,3)*gm(2,2)&
&   *d2gm(2,3)+gm(2,3)*(3*dgm01(2,3)*dgm10(1,2)+12*dgm01(2,2)*dgm10(1,3)&
&   +12*dgm01(1,3)*dgm10(2,2)+3*dgm01(1,2)*dgm10(2,3)+12*gm(2,2)&
&   *d2gm(1,3)+12*gm(1,3)*d2gm(2,2)+3*gm(1,2)*d2gm(2,3))-4.5d0*gm(1,2)&
&   *gm(2,2)*d2gm(3,3)
   cm(3,8)=3*gm(1,3)*dgm01(3,3)*dgm10(2,3)+3*gm(1,3)*dgm01(2,3)&
&   *dgm10(3,3)+12*gm(1,2)*dgm01(3,3)*dgm10(3,3)+6*gm(3,3)**2*d2gm(1,2)&
&   +gm(3,3)*(12*dgm01(3,3)*dgm10(1,2)+3*dgm01(2,3)*dgm10(1,3)+3*dgm01(1,3)&
&   *dgm10(2,3)+12*dgm01(1,2)*dgm10(3,3)+3*gm(2,3)*d2gm(1,3)+3*gm(1,3)&
&   *d2gm(2,3)+12*gm(1,2)*d2gm(3,3))+3*gm(2,3)*(dgm01(3,3)*dgm10(1,3)&
&   +dgm01(1,3)*dgm10(3,3)+gm(1,3)*d2gm(3,3))
   cm(4,8)=12*gm(2,2)*dgm01(3,3)*dgm10(1,3)+12*gm(1,3)*dgm01(3,3)&
&   *dgm10(2,2)+6*gm(1,3)*dgm01(2,3)*dgm10(2,3)+3*gm(1,2)*dgm01(3,3)&
&   *dgm10(2,3)+12*gm(2,2)*dgm01(1,3)*dgm10(3,3)+12*gm(1,3)*dgm01(2,2)&
&   *dgm10(3,3)+3*gm(1,2)*dgm01(2,3)*dgm10(3,3)+3*gm(2,3)**2*d2gm(1,3)&
&   +gm(3,3)*(3*dgm01(2,3)*dgm10(1,2)+12*dgm01(2,2)*dgm10(1,3)+12*dgm01(1,3)&
&   *dgm10(2,2)+3*dgm01(1,2)*dgm10(2,3)+3*gm(2,3)*d2gm(1,2)+12*gm(2,2)&
&   *d2gm(1,3)+12*gm(1,3)*d2gm(2,2)+3*gm(1,2)*d2gm(2,3))+12*gm(1,3)&
&   *gm(2,2)*d2gm(3,3)+gm(2,3)*(3*dgm01(3,3)*dgm10(1,2)+6*dgm01(2,3)&
&   *dgm10(1,3)+6*dgm01(1,3)*dgm10(2,3)+3*dgm01(1,2)*dgm10(3,3)+6*gm(1,3)&
&   *d2gm(2,3)+3*gm(1,2)*d2gm(3,3))
   cm(5,8)=12*gm(1,3)*dgm01(3,3)*dgm10(1,2)+3*gm(1,3)*dgm01(2,3)&
&   *dgm10(1,3)+12*gm(1,2)*dgm01(3,3)*dgm10(1,3)+3*gm(1,3)*dgm01(1,3)&
&   *dgm10(2,3)-4.5d0*gm(1,1)*dgm01(3,3)*dgm10(2,3)+12*gm(1,3)*dgm01(1,2)&
&   *dgm10(3,3)+12*gm(1,2)*dgm01(1,3)*dgm10(3,3)-4.5d0*gm(1,1)*dgm01(2,3)&
&   *dgm10(3,3)+1.5d0*gm(1,3)**2*d2gm(2,3)+gm(3,3)*(-4.5d0*dgm01(2,3)&
&   *dgm10(1,1)+12*dgm01(1,3)*dgm10(1,2)+12*dgm01(1,2)*dgm10(1,3)&
&   -4.5d0*dgm01(1,1)*dgm10(2,3)-4.5d0*gm(2,3)*d2gm(1,1)+12*gm(1,3)&
&   *d2gm(1,2)+12*gm(1,2)*d2gm(1,3)-4.5d0*gm(1,1)*d2gm(2,3))+12*gm(1,2)&
&   *gm(1,3)*d2gm(3,3)+gm(2,3)*(-4.5d0*dgm01(3,3)*dgm10(1,1)+3*dgm01(1,3)&
&   *dgm10(1,3)-4.5d0*dgm01(1,1)*dgm10(3,3)+3*gm(1,3)*d2gm(1,3)-4.5d0*gm(1,1)&
&   *d2gm(3,3))
   cm(6,8)=-1.5d0*gm(2,2)*dgm01(3,3)*dgm10(1,1)+9*gm(1,3)*dgm01(2,3)&
&   *dgm10(1,2)-6*gm(1,2)*dgm01(3,3)*dgm10(1,2)+15*gm(2,2)*dgm01(1,3)&
&   *dgm10(1,3)+15*gm(1,3)*dgm01(2,2)*dgm10(1,3)+9*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,3)+15*gm(1,3)*dgm01(1,3)*dgm10(2,2)-1.5d0*gm(1,1)*dgm01(3,3)&
&   *dgm10(2,2)+9*gm(1,3)*dgm01(1,2)*dgm10(2,3)+9*gm(1,2)*dgm01(1,3)&
&   *dgm10(2,3)-6*gm(1,1)*dgm01(2,3)*dgm10(2,3)-1.5d0*gm(2,2)*dgm01(1,1)&
&   *dgm10(3,3)-6*gm(1,2)*dgm01(1,2)*dgm10(3,3)-1.5d0*gm(1,1)*dgm01(2,2)&
&   *dgm10(3,3)-3*gm(2,3)**2*d2gm(1,1)+15*gm(1,3)*gm(2,2)*d2gm(1,3)&
&   +7.5d0*gm(1,3)**2*d2gm(2,2)+gm(3,3)*(-1.5d0*dgm01(2,2)*dgm10(1,1)&
&   -6*dgm01(1,2)*dgm10(1,2)-1.5d0*dgm01(1,1)*dgm10(2,2)-1.5d0*gm(2,2)&
&   *d2gm(1,1)-6*gm(1,2)*d2gm(1,2)-1.5d0*gm(1,1)*d2gm(2,2))+9*gm(1,2)&
&   *gm(1,3)*d2gm(2,3)+gm(2,3)*(-6*dgm01(2,3)*dgm10(1,1)+9*dgm01(1,3)&
&   *dgm10(1,2)+9*dgm01(1,2)*dgm10(1,3)-6*dgm01(1,1)*dgm10(2,3)+9*gm(1,3)&
&   *d2gm(1,2)+9*gm(1,2)*d2gm(1,3)-6*gm(1,1)*d2gm(2,3))-3*gm(1,2)&
&   **2*d2gm(3,3)-1.5d0*gm(1,1)*gm(2,2)*d2gm(3,3)
   cm(7,8)=4.5d0*gm(2,3)**2*d2gm(2,2)-3*gm(3,3)*(dgm01(2,2)*dgm10(2,2)&
&   +gm(2,2)*d2gm(2,2))+9*gm(2,3)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&   *dgm10(2,3)+gm(2,2)*d2gm(2,3))+gm(2,2)*(-3*dgm01(3,3)*dgm10(2,2)&
&   +9*dgm01(2,3)*dgm10(2,3)-3*dgm01(2,2)*dgm10(3,3)-1.5d0*gm(2,2)&
&   *d2gm(3,3))
   cm(8,8)=12*gm(2,2)*dgm01(3,3)*dgm10(3,3)+6*gm(2,3)*(dgm01(3,3)&
&   *dgm10(2,3)+dgm01(2,3)*dgm10(3,3))+6*gm(3,3)**2*d2gm(2,2)+3*gm(2,3)&
&   **2*d2gm(3,3)+gm(3,3)*(12*dgm01(3,3)*dgm10(2,2)+6*dgm01(2,3)&
&   *dgm10(2,3)+12*dgm01(2,2)*dgm10(3,3)+6*gm(2,3)*d2gm(2,3)+12*gm(2,2)&
&   *d2gm(3,3))
   cm(9,8)=7.5d0*gm(2,2)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))&
&   +4.5d0*gm(2,3)**2*d2gm(2,3)+7.5d0*gm(3,3)*(dgm01(2,3)*dgm10(2,2)&
&   +dgm01(2,2)*dgm10(2,3)+gm(2,3)*d2gm(2,2)+gm(2,2)*d2gm(2,3))+gm(2,3)&
&   *(7.5d0*dgm01(3,3)*dgm10(2,2)+9*dgm01(2,3)*dgm10(2,3)+7.5d0*dgm01(2,2)&
&   *dgm10(3,3)+7.5d0*gm(2,2)*d2gm(3,3))
   cm(10,8)=6*gm(2,3)*dgm01(3,3)*dgm10(3,3)+3*gm(3,3)**2*d2gm(2,3)&
&   +6*gm(3,3)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)+gm(2,3)&
&   *d2gm(3,3))
   cm(1,9)=(-18*gm(1,3)*dgm01(2,2)*dgm10(1,1)-36*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,1)+180*gm(1,3)*dgm01(1,2)*dgm10(1,2)+180*gm(1,2)*dgm01(1,3)&
&   *dgm10(1,2)-36*gm(1,1)*dgm01(2,3)*dgm10(1,2)+180*gm(1,2)*dgm01(1,2)&
&   *dgm10(1,3)-18*gm(1,1)*dgm01(2,2)*dgm10(1,3)-18*gm(1,3)*dgm01(1,1)&
&   *dgm10(2,2)-18*gm(1,1)*dgm01(1,3)*dgm10(2,2)-36*gm(1,2)*dgm01(1,1)&
&   *dgm10(2,3)-36*gm(1,1)*dgm01(1,2)*dgm10(2,3)+180*gm(1,2)*gm(1,3)&
&   *d2gm(1,2)-36*gm(2,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2)&
&   +gm(1,2)*d2gm(1,1)+gm(1,1)*d2gm(1,2))+90*gm(1,2)**2*d2gm(1,3)&
&   -18*gm(2,2)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3)+gm(1,3)&
&   *d2gm(1,1)+gm(1,1)*d2gm(1,3))-18*gm(1,1)*gm(1,3)*d2gm(2,2)-36*gm(1,1)&
&   *gm(1,2)*d2gm(2,3))/12.d0
   cm(2,9)=12*gm(1,3)*dgm01(2,2)*dgm10(2,2)+3*gm(1,2)*dgm01(2,3)&
&   *dgm10(2,2)+3*gm(1,2)*dgm01(2,2)*dgm10(2,3)+6*gm(2,2)**2*d2gm(1,3)&
&   +3*gm(2,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)+gm(2,2)&
&   *d2gm(1,2)+gm(1,2)*d2gm(2,2))+gm(2,2)*(3*dgm01(2,3)*dgm10(1,2)&
&   +12*dgm01(2,2)*dgm10(1,3)+12*dgm01(1,3)*dgm10(2,2)+3*dgm01(1,2)&
&   *dgm10(2,3)+12*gm(1,3)*d2gm(2,2)+3*gm(1,2)*d2gm(2,3))
   cm(3,9)=-4.5d0*gm(2,2)*dgm01(3,3)*dgm10(1,3)-4.5d0*gm(1,3)*dgm01(3,3)&
&   *dgm10(2,2)+3*gm(1,3)*dgm01(2,3)*dgm10(2,3)+12*gm(1,2)*dgm01(3,3)&
&   *dgm10(2,3)-4.5d0*gm(2,2)*dgm01(1,3)*dgm10(3,3)-4.5d0*gm(1,3)&
&   *dgm01(2,2)*dgm10(3,3)+12*gm(1,2)*dgm01(2,3)*dgm10(3,3)+1.5d0*gm(2,3)&
&   **2*d2gm(1,3)+gm(3,3)*(12*dgm01(2,3)*dgm10(1,2)-4.5d0*dgm01(2,2)&
&   *dgm10(1,3)-4.5d0*dgm01(1,3)*dgm10(2,2)+12*dgm01(1,2)*dgm10(2,3)&
&   +12*gm(2,3)*d2gm(1,2)-4.5d0*gm(2,2)*d2gm(1,3)-4.5d0*gm(1,3)*d2gm(2,2)&
&   +12*gm(1,2)*d2gm(2,3))-4.5d0*gm(1,3)*gm(2,2)*d2gm(3,3)+gm(2,3)&
&   *(12*dgm01(3,3)*dgm10(1,2)+3*dgm01(2,3)*dgm10(1,3)+3*dgm01(1,3)&
&   *dgm10(2,3)+12*dgm01(1,2)*dgm10(3,3)+3*gm(1,3)*d2gm(2,3)+12*gm(1,2)&
&   *d2gm(3,3))
   cm(4,9)=12*gm(2,2)*dgm01(3,3)*dgm10(1,2)+3*gm(2,2)*dgm01(2,3)&
&   *dgm10(1,3)+3*gm(1,3)*dgm01(2,3)*dgm10(2,2)+12*gm(1,2)*dgm01(3,3)&
&   *dgm10(2,2)+3*gm(2,2)*dgm01(1,3)*dgm10(2,3)+3*gm(1,3)*dgm01(2,2)&
&   *dgm10(2,3)+6*gm(1,2)*dgm01(2,3)*dgm10(2,3)+12*gm(2,2)*dgm01(1,2)&
&   *dgm10(3,3)+12*gm(1,2)*dgm01(2,2)*dgm10(3,3)+3*gm(2,3)**2*d2gm(1,2)&
&   +12*gm(3,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)+gm(2,2)&
&   *d2gm(1,2)+gm(1,2)*d2gm(2,2))+3*gm(1,3)*gm(2,2)*d2gm(2,3)+gm(2,3)&
&   *(6*dgm01(2,3)*dgm10(1,2)+3*dgm01(2,2)*dgm10(1,3)+3*dgm01(1,3)&
&   *dgm10(2,2)+6*dgm01(1,2)*dgm10(2,3)+3*gm(2,2)*d2gm(1,3)+3*gm(1,3)&
&   *d2gm(2,2)+6*gm(1,2)*d2gm(2,3))+12*gm(1,2)*gm(2,2)*d2gm(3,3)
   cm(5,9)=-1.5d0*gm(2,2)*dgm01(3,3)*dgm10(1,1)+9*gm(1,3)*dgm01(2,3)&
&   *dgm10(1,2)+15*gm(1,2)*dgm01(3,3)*dgm10(1,2)-6*gm(2,2)*dgm01(1,3)&
&   *dgm10(1,3)-6*gm(1,3)*dgm01(2,2)*dgm10(1,3)+9*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,3)-6*gm(1,3)*dgm01(1,3)*dgm10(2,2)-1.5d0*gm(1,1)*dgm01(3,3)&
&   *dgm10(2,2)+9*gm(1,3)*dgm01(1,2)*dgm10(2,3)+9*gm(1,2)*dgm01(1,3)&
&   *dgm10(2,3)-6*gm(1,1)*dgm01(2,3)*dgm10(2,3)-1.5d0*gm(2,2)*dgm01(1,1)&
&   *dgm10(3,3)+15*gm(1,2)*dgm01(1,2)*dgm10(3,3)-1.5d0*gm(1,1)*dgm01(2,2)&
&   *dgm10(3,3)-3*gm(2,3)**2*d2gm(1,1)-6*gm(1,3)*gm(2,2)*d2gm(1,3)&
&   -3*gm(1,3)**2*d2gm(2,2)+gm(3,3)*(-1.5d0*dgm01(2,2)*dgm10(1,1)&
&   +15*dgm01(1,2)*dgm10(1,2)-1.5d0*dgm01(1,1)*dgm10(2,2)-1.5d0*gm(2,2)&
&   *d2gm(1,1)+15*gm(1,2)*d2gm(1,2)-1.5d0*gm(1,1)*d2gm(2,2))+9*gm(1,2)&
&   *gm(1,3)*d2gm(2,3)+gm(2,3)*(-6*dgm01(2,3)*dgm10(1,1)+9*dgm01(1,3)&
&   *dgm10(1,2)+9*dgm01(1,2)*dgm10(1,3)-6*dgm01(1,1)*dgm10(2,3)+9*gm(1,3)&
&   *d2gm(1,2)+9*gm(1,2)*d2gm(1,3)-6*gm(1,1)*d2gm(2,3))+7.5d0*gm(1,2)&
&   **2*d2gm(3,3)-1.5d0*gm(1,1)*gm(2,2)*d2gm(3,3)
   cm(6,9)=12*gm(1,3)*dgm01(2,2)*dgm10(1,2)+3*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,2)+12*gm(1,2)*dgm01(2,2)*dgm10(1,3)+12*gm(1,3)*dgm01(1,2)&
&   *dgm10(2,2)+12*gm(1,2)*dgm01(1,3)*dgm10(2,2)-4.5d0*gm(1,1)*dgm01(2,3)&
&   *dgm10(2,2)+3*gm(1,2)*dgm01(1,2)*dgm10(2,3)-4.5d0*gm(1,1)*dgm01(2,2)&
&   *dgm10(2,3)+12*gm(1,2)*gm(1,3)*d2gm(2,2)+gm(2,3)*(-4.5d0*dgm01(2,2)&
&   *dgm10(1,1)+3*dgm01(1,2)*dgm10(1,2)-4.5d0*dgm01(1,1)*dgm10(2,2)&
&   -4.5d0*gm(2,2)*d2gm(1,1)+3*gm(1,2)*d2gm(1,2)-4.5d0*gm(1,1)*d2gm(2,2))&
&   +1.5d0*gm(1,2)**2*d2gm(2,3)+gm(2,2)*(-4.5d0*dgm01(2,3)*dgm10(1,1)&
&   +12*dgm01(1,3)*dgm10(1,2)+12*dgm01(1,2)*dgm10(1,3)-4.5d0*dgm01(1,1)&
&   *dgm10(2,3)+12*gm(1,3)*d2gm(1,2)+12*gm(1,2)*d2gm(1,3)-4.5d0*gm(1,1)&
&   *d2gm(2,3))
   cm(7,9)=6*gm(2,3)*(dgm01(2,2)*dgm10(2,2)+gm(2,2)*d2gm(2,2))&
&   +gm(2,2)*(6*dgm01(2,3)*dgm10(2,2)+6*dgm01(2,2)*dgm10(2,3)+3*gm(2,2)&
&   *d2gm(2,3))
   cm(8,9)=7.5d0*gm(2,2)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))&
&   +4.5d0*gm(2,3)**2*d2gm(2,3)+7.5d0*gm(3,3)*(dgm01(2,3)*dgm10(2,2)&
&   +dgm01(2,2)*dgm10(2,3)+gm(2,3)*d2gm(2,2)+gm(2,2)*d2gm(2,3))+gm(2,3)&
&   *(7.5d0*dgm01(3,3)*dgm10(2,2)+9*dgm01(2,3)*dgm10(2,3)+7.5d0*dgm01(2,2)&
&   *dgm10(3,3)+7.5d0*gm(2,2)*d2gm(3,3))
   cm(9,9)=3*gm(2,3)**2*d2gm(2,2)+12*gm(3,3)*(dgm01(2,2)*dgm10(2,2)&
&   +gm(2,2)*d2gm(2,2))+6*gm(2,3)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&   *dgm10(2,3)+gm(2,2)*d2gm(2,3))+gm(2,2)*(12*dgm01(3,3)*dgm10(2,2)&
&   +6*dgm01(2,3)*dgm10(2,3)+12*dgm01(2,2)*dgm10(3,3)+6*gm(2,2)*d2gm(3,3))
   cm(10,9)=-3*gm(2,2)*dgm01(3,3)*dgm10(3,3)+9*gm(2,3)*(dgm01(3,3)&
&   *dgm10(2,3)+dgm01(2,3)*dgm10(3,3))-1.5d0*gm(3,3)**2*d2gm(2,2)&
&   +4.5d0*gm(2,3)**2*d2gm(3,3)+gm(3,3)*(-3*dgm01(3,3)*dgm10(2,2)&
&   +9*dgm01(2,3)*dgm10(2,3)-3*dgm01(2,2)*dgm10(3,3)+9*gm(2,3)*d2gm(2,3)&
&   -3*gm(2,2)*d2gm(3,3))
   cm(1,10)=-1.5d0*gm(1,1)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&   +7.5d0*gm(1,3)**2*d2gm(1,3)-1.5d0*gm(3,3)*(dgm01(1,3)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(1,3)+gm(1,3)*d2gm(1,1)+gm(1,1)*d2gm(1,3))+gm(1,3)&
&   *(-1.5d0*dgm01(3,3)*dgm10(1,1)+15*dgm01(1,3)*dgm10(1,3)-1.5d0*dgm01(1,1)&
&   *dgm10(3,3)-1.5d0*gm(1,1)*d2gm(3,3))
   cm(2,10)=-1.5d0*gm(2,2)*dgm01(3,3)*dgm10(1,3)-1.5d0*gm(1,3)&
&   *dgm01(3,3)*dgm10(2,2)+15*gm(1,3)*dgm01(2,3)*dgm10(2,3)-3*gm(1,2)&
&   *dgm01(3,3)*dgm10(2,3)-1.5d0*gm(2,2)*dgm01(1,3)*dgm10(3,3)-1.5d0*gm(1,3)&
&   *dgm01(2,2)*dgm10(3,3)-3*gm(1,2)*dgm01(2,3)*dgm10(3,3)+7.5d0*gm(2,3)&
&   **2*d2gm(1,3)+gm(3,3)*(-3*dgm01(2,3)*dgm10(1,2)-1.5d0*dgm01(2,2)&
&   *dgm10(1,3)-1.5d0*dgm01(1,3)*dgm10(2,2)-3*dgm01(1,2)*dgm10(2,3)&
&   -3*gm(2,3)*d2gm(1,2)-1.5d0*gm(2,2)*d2gm(1,3)-1.5d0*gm(1,3)*d2gm(2,2)&
&   -3*gm(1,2)*d2gm(2,3))-1.5d0*gm(1,3)*gm(2,2)*d2gm(3,3)+gm(2,3)&
&   *(-3*dgm01(3,3)*dgm10(1,2)+15*dgm01(2,3)*dgm10(1,3)+15*dgm01(1,3)&
&   *dgm10(2,3)-3*dgm01(1,2)*dgm10(3,3)+15*gm(1,3)*d2gm(2,3)-3*gm(1,2)&
&   *d2gm(3,3))
   cm(3,10)=6*gm(1,3)*dgm01(3,3)*dgm10(3,3)+3*gm(3,3)**2*d2gm(1,3)&
&   +6*gm(3,3)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)+gm(1,3)&
&   *d2gm(3,3))
   cm(4,10)=9*gm(1,3)*dgm01(3,3)*dgm10(2,3)+9*gm(1,3)*dgm01(2,3)&
&   *dgm10(3,3)-6*gm(1,2)*dgm01(3,3)*dgm10(3,3)-3*gm(3,3)**2*d2gm(1,2)&
&   +gm(3,3)*(-6*dgm01(3,3)*dgm10(1,2)+9*dgm01(2,3)*dgm10(1,3)+9*dgm01(1,3)&
&   *dgm10(2,3)-6*dgm01(1,2)*dgm10(3,3)+9*gm(2,3)*d2gm(1,3)+9*gm(1,3)&
&   *d2gm(2,3)-6*gm(1,2)*d2gm(3,3))+9*gm(2,3)*(dgm01(3,3)*dgm10(1,3)&
&   +dgm01(1,3)*dgm10(3,3)+gm(1,3)*d2gm(3,3))
   cm(5,10)=-3*gm(1,1)*dgm01(3,3)*dgm10(3,3)+9*gm(1,3)*(dgm01(3,3)&
&   *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))-1.5d0*gm(3,3)**2*d2gm(1,1)&
&   +4.5d0*gm(1,3)**2*d2gm(3,3)+gm(3,3)*(-3*dgm01(3,3)*dgm10(1,1)&
&   +9*dgm01(1,3)*dgm10(1,3)-3*dgm01(1,1)*dgm10(3,3)+9*gm(1,3)*d2gm(1,3)&
&   -3*gm(1,1)*d2gm(3,3))
   cm(6,10)=-3*gm(1,3)*dgm01(3,3)*dgm10(1,2)+15*gm(1,3)*dgm01(2,3)&
&   *dgm10(1,3)-3*gm(1,2)*dgm01(3,3)*dgm10(1,3)+15*gm(1,3)*dgm01(1,3)&
&   *dgm10(2,3)-1.5d0*gm(1,1)*dgm01(3,3)*dgm10(2,3)-3*gm(1,3)*dgm01(1,2)&
&   *dgm10(3,3)-3*gm(1,2)*dgm01(1,3)*dgm10(3,3)-1.5d0*gm(1,1)*dgm01(2,3)&
&   *dgm10(3,3)+7.5d0*gm(1,3)**2*d2gm(2,3)+gm(3,3)*(-1.5d0*dgm01(2,3)&
&   *dgm10(1,1)-3*dgm01(1,3)*dgm10(1,2)-3*dgm01(1,2)*dgm10(1,3)-1.5d0*dgm01(1,1)&
&   *dgm10(2,3)-1.5d0*gm(2,3)*d2gm(1,1)-3*gm(1,3)*d2gm(1,2)-3*gm(1,2)&
&   *d2gm(1,3)-1.5d0*gm(1,1)*d2gm(2,3))-3*gm(1,2)*gm(1,3)*d2gm(3,3)&
&   +gm(2,3)*(-1.5d0*dgm01(3,3)*dgm10(1,1)+15*dgm01(1,3)*dgm10(1,3)&
&   -1.5d0*dgm01(1,1)*dgm10(3,3)+15*gm(1,3)*d2gm(1,3)-1.5d0*gm(1,1)&
&   *d2gm(3,3))
   cm(7,10)=-1.5d0*gm(2,2)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))&
&   +7.5d0*gm(2,3)**2*d2gm(2,3)-1.5d0*gm(3,3)*(dgm01(2,3)*dgm10(2,2)&
&   +dgm01(2,2)*dgm10(2,3)+gm(2,3)*d2gm(2,2)+gm(2,2)*d2gm(2,3))+gm(2,3)&
&   *(-1.5d0*dgm01(3,3)*dgm10(2,2)+15*dgm01(2,3)*dgm10(2,3)-1.5d0*dgm01(2,2)&
&   *dgm10(3,3)-1.5d0*gm(2,2)*d2gm(3,3))
   cm(8,10)=6*gm(2,3)*dgm01(3,3)*dgm10(3,3)+3*gm(3,3)**2*d2gm(2,3)&
&   +6*gm(3,3)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)+gm(2,3)&
&   *d2gm(3,3))
   cm(9,10)=-3*gm(2,2)*dgm01(3,3)*dgm10(3,3)+9*gm(2,3)*(dgm01(3,3)&
&   *dgm10(2,3)+dgm01(2,3)*dgm10(3,3))-1.5d0*gm(3,3)**2*d2gm(2,2)&
&   +4.5d0*gm(2,3)**2*d2gm(3,3)+gm(3,3)*(-3*dgm01(3,3)*dgm10(2,2)&
&   +9*dgm01(2,3)*dgm10(2,3)-3*dgm01(2,2)*dgm10(3,3)+9*gm(2,3)*d2gm(2,3)&
&   -3*gm(2,2)*d2gm(3,3))
   cm(10,10)=gm(3,3)*(6*dgm01(3,3)*dgm10(3,3)+3*gm(3,3)*d2gm(3,3))


 end if
!
!contraction to output scalar
!
 e2nl=0.d0
 do jj=1,((rank+1)*(rank+2))/2
   tmp(:)=0.d0
   do ii=1,((rank+1)*(rank+2))/2
     tmp(:)=tmp(:)+aa(:,ii)*cm(ii,jj)
   end do
   e2nl=e2nl+tmp(1)*bb(1,jj)+tmp(2)*bb(2,jj)
 end do

 ABI_FREE(cm)

end subroutine contstr26
!!***

end module m_contstr26
!!***
