--
--  Copyright (C) 2014-2022, AdaCore
--  SPDX-License-Identifier: Apache-2.0
--

with Ada.Unchecked_Conversion;

with System;

with Langkit_Support.Internal.Analysis;
use Langkit_Support.Internal.Analysis;
with Langkit_Support.Internal.Conversions;

with Libadalang.Generic_Impl;      use Libadalang.Generic_Impl;
with Libadalang.Generic_Introspection;
use Libadalang.Generic_Introspection;
with Libadalang.Implementation;
with Libadalang.Public_Converters; use Libadalang.Public_Converters;

package body Libadalang.Generic_API is

   Desc_Address : constant System.Address := Desc'Address
     with Export, External_Name => "Libadalang__language_id";

   package Lk_Convs renames Langkit_Support.Internal.Conversions;

   subtype Generic_Internal_Context is
     Langkit_Support.Internal.Analysis.Internal_Context;
   subtype Specific_Internal_Context is
     Libadalang.Implementation.Internal_Context;

   subtype Generic_Internal_Unit is
     Langkit_Support.Internal.Analysis.Internal_Unit;
   subtype Specific_Internal_Unit is
     Libadalang.Implementation.Internal_Unit;

   function "+" is new Ada.Unchecked_Conversion
     (Generic_Internal_Context, Specific_Internal_Context);
   function "+" is new Ada.Unchecked_Conversion
     (Specific_Internal_Context, Generic_Internal_Context);

   function "+" is new Ada.Unchecked_Conversion
     (Generic_Internal_Unit, Specific_Internal_Unit);
   function "+" is new Ada.Unchecked_Conversion
     (Specific_Internal_Unit, Generic_Internal_Unit);

   ------------------------
   -- To_Generic_Context --
   ------------------------

   function To_Generic_Context (Context : Analysis_Context) return Lk_Context
   is
      Ctx : constant Specific_Internal_Context := Unwrap_Context.all (Context);
   begin
      return Lk_Convs.Wrap_Context (Self_Id, +Ctx);
   end To_Generic_Context;

   --------------------------
   -- From_Generic_Context --
   --------------------------

   function From_Generic_Context (Context : Lk_Context) return Analysis_Context
   is
   begin
      if Context = No_Lk_Context then
         return No_Analysis_Context;
      elsif Context.Language /= Self_Id then
         raise Precondition_Failure with "context belongs to another language";
      else
         declare
            Ctx : constant Generic_Internal_Context :=
              Lk_Convs.Unwrap_Context (Context);
         begin
            return Wrap_Context.all (+Ctx);
         end;
      end if;
   end From_Generic_Context;

   ---------------------
   -- To_Generic_Unit --
   ---------------------

   function To_Generic_Unit (Unit : Analysis_Unit) return Lk_Unit is
      U : constant Specific_Internal_Unit := Unwrap_Unit.all (Unit);
   begin
      return Lk_Convs.Wrap_Unit (Self_Id, +U);
   end To_Generic_Unit;

   -----------------------
   -- From_Generic_Unit --
   -----------------------

   function From_Generic_Unit (Unit : Lk_Unit) return Analysis_Unit is
   begin
      if Unit = No_Lk_Unit then
         return No_Analysis_Unit;
      elsif Unit.Language /= Self_Id then
         raise Precondition_Failure with "unit belongs to another language";
      else
         declare
            U : constant Generic_Internal_Unit := Lk_Convs.Unwrap_Unit (Unit);
         begin
            return Wrap_Unit.all (+U);
         end;
      end if;
   end From_Generic_Unit;

   -----------------------------
   -- To_Generic_Grammar_Rule --
   -----------------------------

   function To_Generic_Grammar_Rule
     (Rule : Grammar_Rule) return Langkit_Support.Generic_API.Grammar_Rule_Ref
   is
   begin
      --  'Pos is 0-based whereas Grammar_Rule_Index is 1-based

      return From_Index (Self_Id, Grammar_Rule'Pos (Rule) + 1);
   end To_Generic_Grammar_Rule;

   -------------------------------
   -- From_Generic_Grammar_Rule --
   -------------------------------

   function From_Generic_Grammar_Rule
     (Rule : Langkit_Support.Generic_API.Grammar_Rule_Ref) return Grammar_Rule
   is
   begin
      if Rule = Langkit_Support.Generic_API.No_Grammar_Rule_Ref then
         raise Precondition_Failure
           with "null grammar rule";
      elsif Language (Rule) /= Self_Id then
         raise Precondition_Failure
           with "grammar rule belongs to another language";
      end if;

      --  'Pos is 0-based whereas Grammar_Rule_Index is 1-based

      return Grammar_Rule'Val (To_Index (Rule) - 1);
   end From_Generic_Grammar_Rule;

   ---------------------
   -- To_Generic_Node --
   ---------------------

   function To_Generic_Node
     (Node : Ada_Node'Class) return Lk_Node
   is
      E : constant Implementation.Internal_Entity :=
        Unwrap_Entity.all (Node);
   begin
      return Lk_Convs.Wrap_Node (Self_Id, +E);
   end To_Generic_Node;

   -----------------------
   -- From_Generic_Node --
   -----------------------

   function From_Generic_Node (Node : Lk_Node) return Ada_Node
   is
   begin
      if Node.Is_Null then
         return No_Ada_Node;
      elsif Node.Language /= Self_Id then
         raise Precondition_Failure with "node belongs to another language";
      else
         declare
            N : constant Langkit_Support.Internal.Analysis.Internal_Entity :=
              Lk_Convs.Unwrap_Node (Node);
            E : constant Implementation.Internal_Entity := +N;
         begin
            return Wrap_Node.all (E.Node, E.Info);
         end;
      end if;
   end From_Generic_Node;

   --------------------------
   -- To_Generic_Node_Type --
   --------------------------

   function To_Generic_Node_Type
     (Kind : Ada_Node_Kind_Type)
      return Langkit_Support.Generic_API.Introspection.Type_Ref
   is
      use Langkit_Support.Generic_API.Introspection;
   begin
      return From_Index (Self_Id, Node_Kinds (Kind));
   end To_Generic_Node_Type;

   ----------------------------
   -- From_Generic_Node_Type --
   ----------------------------

   function From_Generic_Node_Type
     (Kind : Langkit_Support.Generic_API.Introspection.Type_Ref)
      return Ada_Node_Kind_Type
   is
      use Langkit_Support.Generic_API.Introspection;
   begin
      if Kind /= Langkit_Support.Generic_API.Introspection.No_Type_Ref then
         if Kind.Language /= Self_Id then
            raise Precondition_Failure with "type belongs to another language";
         end if;

         case To_Index (Kind) is
                  when Type_Index_For_Abort_Absent =>
                     return Ada_Abort_Absent;
                  when Type_Index_For_Abort_Present =>
                     return Ada_Abort_Present;
                  when Type_Index_For_Abstract_Absent =>
                     return Ada_Abstract_Absent;
                  when Type_Index_For_Abstract_Present =>
                     return Ada_Abstract_Present;
                  when Type_Index_For_Ada_Node_List =>
                     return Ada_Ada_Node_List;
                  when Type_Index_For_Abstract_State_Decl_List =>
                     return Ada_Abstract_State_Decl_List;
                  when Type_Index_For_Alternatives_List =>
                     return Ada_Alternatives_List;
                  when Type_Index_For_Constraint_List =>
                     return Ada_Constraint_List;
                  when Type_Index_For_Decl_List =>
                     return Ada_Decl_List;
                  when Type_Index_For_Stmt_List =>
                     return Ada_Stmt_List;
                  when Type_Index_For_Aspect_Assoc_List =>
                     return Ada_Aspect_Assoc_List;
                  when Type_Index_For_Base_Assoc_List =>
                     return Ada_Base_Assoc_List;
                  when Type_Index_For_Assoc_List =>
                     return Ada_Assoc_List;
                  when Type_Index_For_Case_Expr_Alternative_List =>
                     return Ada_Case_Expr_Alternative_List;
                  when Type_Index_For_Case_Stmt_Alternative_List =>
                     return Ada_Case_Stmt_Alternative_List;
                  when Type_Index_For_Compilation_Unit_List =>
                     return Ada_Compilation_Unit_List;
                  when Type_Index_For_Concat_Operand_List =>
                     return Ada_Concat_Operand_List;
                  when Type_Index_For_Contract_Case_Assoc_List =>
                     return Ada_Contract_Case_Assoc_List;
                  when Type_Index_For_Defining_Name_List =>
                     return Ada_Defining_Name_List;
                  when Type_Index_For_Discriminant_Spec_List =>
                     return Ada_Discriminant_Spec_List;
                  when Type_Index_For_Elsif_Expr_Part_List =>
                     return Ada_Elsif_Expr_Part_List;
                  when Type_Index_For_Elsif_Stmt_Part_List =>
                     return Ada_Elsif_Stmt_Part_List;
                  when Type_Index_For_Enum_Literal_Decl_List =>
                     return Ada_Enum_Literal_Decl_List;
                  when Type_Index_For_Expr_Alternatives_List =>
                     return Ada_Expr_Alternatives_List;
                  when Type_Index_For_Format_String_Chunk_List =>
                     return Ada_Format_String_Chunk_List;
                  when Type_Index_For_Discriminant_Choice_List =>
                     return Ada_Discriminant_Choice_List;
                  when Type_Index_For_Name_List =>
                     return Ada_Name_List;
                  when Type_Index_For_Parent_List =>
                     return Ada_Parent_List;
                  when Type_Index_For_Param_Spec_List =>
                     return Ada_Param_Spec_List;
                  when Type_Index_For_Pragma_Node_List =>
                     return Ada_Pragma_Node_List;
                  when Type_Index_For_Select_When_Part_List =>
                     return Ada_Select_When_Part_List;
                  when Type_Index_For_Unconstrained_Array_Index_List =>
                     return Ada_Unconstrained_Array_Index_List;
                  when Type_Index_For_Variant_List =>
                     return Ada_Variant_List;
                  when Type_Index_For_Aliased_Absent =>
                     return Ada_Aliased_Absent;
                  when Type_Index_For_Aliased_Present =>
                     return Ada_Aliased_Present;
                  when Type_Index_For_All_Absent =>
                     return Ada_All_Absent;
                  when Type_Index_For_All_Present =>
                     return Ada_All_Present;
                  when Type_Index_For_Constrained_Array_Indices =>
                     return Ada_Constrained_Array_Indices;
                  when Type_Index_For_Unconstrained_Array_Indices =>
                     return Ada_Unconstrained_Array_Indices;
                  when Type_Index_For_Aspect_Assoc =>
                     return Ada_Aspect_Assoc;
                  when Type_Index_For_At_Clause =>
                     return Ada_At_Clause;
                  when Type_Index_For_Attribute_Def_Clause =>
                     return Ada_Attribute_Def_Clause;
                  when Type_Index_For_Enum_Rep_Clause =>
                     return Ada_Enum_Rep_Clause;
                  when Type_Index_For_Record_Rep_Clause =>
                     return Ada_Record_Rep_Clause;
                  when Type_Index_For_Aspect_Spec =>
                     return Ada_Aspect_Spec;
                  when Type_Index_For_Contract_Case_Assoc =>
                     return Ada_Contract_Case_Assoc;
                  when Type_Index_For_Pragma_Argument_Assoc =>
                     return Ada_Pragma_Argument_Assoc;
                  when Type_Index_For_Entry_Spec =>
                     return Ada_Entry_Spec;
                  when Type_Index_For_Enum_Subp_Spec =>
                     return Ada_Enum_Subp_Spec;
                  when Type_Index_For_Subp_Spec =>
                     return Ada_Subp_Spec;
                  when Type_Index_For_Synthetic_Binary_Spec =>
                     return Ada_Synthetic_Binary_Spec;
                  when Type_Index_For_Synthetic_Unary_Spec =>
                     return Ada_Synthetic_Unary_Spec;
                  when Type_Index_For_Component_List =>
                     return Ada_Component_List;
                  when Type_Index_For_Known_Discriminant_Part =>
                     return Ada_Known_Discriminant_Part;
                  when Type_Index_For_Unknown_Discriminant_Part =>
                     return Ada_Unknown_Discriminant_Part;
                  when Type_Index_For_Entry_Completion_Formal_Params =>
                     return Ada_Entry_Completion_Formal_Params;
                  when Type_Index_For_Generic_Formal_Part =>
                     return Ada_Generic_Formal_Part;
                  when Type_Index_For_Null_Record_Def =>
                     return Ada_Null_Record_Def;
                  when Type_Index_For_Record_Def =>
                     return Ada_Record_Def;
                  when Type_Index_For_Aggregate_Assoc =>
                     return Ada_Aggregate_Assoc;
                  when Type_Index_For_Multi_Dim_Array_Assoc =>
                     return Ada_Multi_Dim_Array_Assoc;
                  when Type_Index_For_Composite_Constraint_Assoc =>
                     return Ada_Composite_Constraint_Assoc;
                  when Type_Index_For_Iterated_Assoc =>
                     return Ada_Iterated_Assoc;
                  when Type_Index_For_Param_Assoc =>
                     return Ada_Param_Assoc;
                  when Type_Index_For_Abstract_State_Decl =>
                     return Ada_Abstract_State_Decl;
                  when Type_Index_For_Anonymous_Expr_Decl =>
                     return Ada_Anonymous_Expr_Decl;
                  when Type_Index_For_Component_Decl =>
                     return Ada_Component_Decl;
                  when Type_Index_For_Discriminant_Spec =>
                     return Ada_Discriminant_Spec;
                  when Type_Index_For_Generic_Formal_Obj_Decl =>
                     return Ada_Generic_Formal_Obj_Decl;
                  when Type_Index_For_Generic_Formal_Package =>
                     return Ada_Generic_Formal_Package;
                  when Type_Index_For_Generic_Formal_Subp_Decl =>
                     return Ada_Generic_Formal_Subp_Decl;
                  when Type_Index_For_Generic_Formal_Type_Decl =>
                     return Ada_Generic_Formal_Type_Decl;
                  when Type_Index_For_Param_Spec =>
                     return Ada_Param_Spec;
                  when Type_Index_For_Synthetic_Formal_Param_Decl =>
                     return Ada_Synthetic_Formal_Param_Decl;
                  when Type_Index_For_Generic_Package_Internal =>
                     return Ada_Generic_Package_Internal;
                  when Type_Index_For_Package_Decl =>
                     return Ada_Package_Decl;
                  when Type_Index_For_Discrete_Base_Subtype_Decl =>
                     return Ada_Discrete_Base_Subtype_Decl;
                  when Type_Index_For_Subtype_Decl =>
                     return Ada_Subtype_Decl;
                  when Type_Index_For_Classwide_Type_Decl =>
                     return Ada_Classwide_Type_Decl;
                  when Type_Index_For_Incomplete_Type_Decl =>
                     return Ada_Incomplete_Type_Decl;
                  when Type_Index_For_Incomplete_Formal_Type_Decl =>
                     return Ada_Incomplete_Formal_Type_Decl;
                  when Type_Index_For_Incomplete_Tagged_Type_Decl =>
                     return Ada_Incomplete_Tagged_Type_Decl;
                  when Type_Index_For_Protected_Type_Decl =>
                     return Ada_Protected_Type_Decl;
                  when Type_Index_For_Task_Type_Decl =>
                     return Ada_Task_Type_Decl;
                  when Type_Index_For_Single_Task_Type_Decl =>
                     return Ada_Single_Task_Type_Decl;
                  when Type_Index_For_Anonymous_Type_Decl =>
                     return Ada_Anonymous_Type_Decl;
                  when Type_Index_For_Synth_Anonymous_Type_Decl =>
                     return Ada_Synth_Anonymous_Type_Decl;
                  when Type_Index_For_Concrete_Type_Decl =>
                     return Ada_Concrete_Type_Decl;
                  when Type_Index_For_Formal_Type_Decl =>
                     return Ada_Formal_Type_Decl;
                  when Type_Index_For_Abstract_Subp_Decl =>
                     return Ada_Abstract_Subp_Decl;
                  when Type_Index_For_Abstract_Formal_Subp_Decl =>
                     return Ada_Abstract_Formal_Subp_Decl;
                  when Type_Index_For_Concrete_Formal_Subp_Decl =>
                     return Ada_Concrete_Formal_Subp_Decl;
                  when Type_Index_For_Subp_Decl =>
                     return Ada_Subp_Decl;
                  when Type_Index_For_Entry_Decl =>
                     return Ada_Entry_Decl;
                  when Type_Index_For_Enum_Literal_Decl =>
                     return Ada_Enum_Literal_Decl;
                  when Type_Index_For_Synthetic_Char_Enum_Lit =>
                     return Ada_Synthetic_Char_Enum_Lit;
                  when Type_Index_For_Generic_Subp_Internal =>
                     return Ada_Generic_Subp_Internal;
                  when Type_Index_For_Synthetic_Subp_Decl =>
                     return Ada_Synthetic_Subp_Decl;
                  when Type_Index_For_Accept_Stmt_Body =>
                     return Ada_Accept_Stmt_Body;
                  when Type_Index_For_Expr_Function =>
                     return Ada_Expr_Function;
                  when Type_Index_For_Null_Subp_Decl =>
                     return Ada_Null_Subp_Decl;
                  when Type_Index_For_Subp_Body =>
                     return Ada_Subp_Body;
                  when Type_Index_For_Subp_Renaming_Decl =>
                     return Ada_Subp_Renaming_Decl;
                  when Type_Index_For_Package_Body_Stub =>
                     return Ada_Package_Body_Stub;
                  when Type_Index_For_Protected_Body_Stub =>
                     return Ada_Protected_Body_Stub;
                  when Type_Index_For_Subp_Body_Stub =>
                     return Ada_Subp_Body_Stub;
                  when Type_Index_For_Task_Body_Stub =>
                     return Ada_Task_Body_Stub;
                  when Type_Index_For_Entry_Body =>
                     return Ada_Entry_Body;
                  when Type_Index_For_Package_Body =>
                     return Ada_Package_Body;
                  when Type_Index_For_Protected_Body =>
                     return Ada_Protected_Body;
                  when Type_Index_For_Task_Body =>
                     return Ada_Task_Body;
                  when Type_Index_For_Entry_Index_Spec =>
                     return Ada_Entry_Index_Spec;
                  when Type_Index_For_Error_Decl =>
                     return Ada_Error_Decl;
                  when Type_Index_For_Exception_Decl =>
                     return Ada_Exception_Decl;
                  when Type_Index_For_Exception_Handler =>
                     return Ada_Exception_Handler;
                  when Type_Index_For_For_Loop_Var_Decl =>
                     return Ada_For_Loop_Var_Decl;
                  when Type_Index_For_Generic_Package_Decl =>
                     return Ada_Generic_Package_Decl;
                  when Type_Index_For_Generic_Subp_Decl =>
                     return Ada_Generic_Subp_Decl;
                  when Type_Index_For_Generic_Package_Instantiation =>
                     return Ada_Generic_Package_Instantiation;
                  when Type_Index_For_Generic_Subp_Instantiation =>
                     return Ada_Generic_Subp_Instantiation;
                  when Type_Index_For_Generic_Package_Renaming_Decl =>
                     return Ada_Generic_Package_Renaming_Decl;
                  when Type_Index_For_Generic_Subp_Renaming_Decl =>
                     return Ada_Generic_Subp_Renaming_Decl;
                  when Type_Index_For_Label_Decl =>
                     return Ada_Label_Decl;
                  when Type_Index_For_Named_Stmt_Decl =>
                     return Ada_Named_Stmt_Decl;
                  when Type_Index_For_Number_Decl =>
                     return Ada_Number_Decl;
                  when Type_Index_For_Object_Decl =>
                     return Ada_Object_Decl;
                  when Type_Index_For_Extended_Return_Stmt_Object_Decl =>
                     return Ada_Extended_Return_Stmt_Object_Decl;
                  when Type_Index_For_No_Type_Object_Renaming_Decl =>
                     return Ada_No_Type_Object_Renaming_Decl;
                  when Type_Index_For_Package_Renaming_Decl =>
                     return Ada_Package_Renaming_Decl;
                  when Type_Index_For_Single_Protected_Decl =>
                     return Ada_Single_Protected_Decl;
                  when Type_Index_For_Single_Task_Decl =>
                     return Ada_Single_Task_Decl;
                  when Type_Index_For_Synthetic_Object_Decl =>
                     return Ada_Synthetic_Object_Decl;
                  when Type_Index_For_Case_Stmt_Alternative =>
                     return Ada_Case_Stmt_Alternative;
                  when Type_Index_For_Compilation_Unit =>
                     return Ada_Compilation_Unit;
                  when Type_Index_For_Component_Clause =>
                     return Ada_Component_Clause;
                  when Type_Index_For_Component_Def =>
                     return Ada_Component_Def;
                  when Type_Index_For_Constant_Absent =>
                     return Ada_Constant_Absent;
                  when Type_Index_For_Constant_Present =>
                     return Ada_Constant_Present;
                  when Type_Index_For_Composite_Constraint =>
                     return Ada_Composite_Constraint;
                  when Type_Index_For_Delta_Constraint =>
                     return Ada_Delta_Constraint;
                  when Type_Index_For_Digits_Constraint =>
                     return Ada_Digits_Constraint;
                  when Type_Index_For_Range_Constraint =>
                     return Ada_Range_Constraint;
                  when Type_Index_For_Declarative_Part =>
                     return Ada_Declarative_Part;
                  when Type_Index_For_Private_Part =>
                     return Ada_Private_Part;
                  when Type_Index_For_Public_Part =>
                     return Ada_Public_Part;
                  when Type_Index_For_Else_Part =>
                     return Ada_Else_Part;
                  when Type_Index_For_Elsif_Expr_Part =>
                     return Ada_Elsif_Expr_Part;
                  when Type_Index_For_Elsif_Stmt_Part =>
                     return Ada_Elsif_Stmt_Part;
                  when Type_Index_For_Abstract_State_Decl_Expr =>
                     return Ada_Abstract_State_Decl_Expr;
                  when Type_Index_For_Allocator =>
                     return Ada_Allocator;
                  when Type_Index_For_Aggregate =>
                     return Ada_Aggregate;
                  when Type_Index_For_Bracket_Aggregate =>
                     return Ada_Bracket_Aggregate;
                  when Type_Index_For_Delta_Aggregate =>
                     return Ada_Delta_Aggregate;
                  when Type_Index_For_Bracket_Delta_Aggregate =>
                     return Ada_Bracket_Delta_Aggregate;
                  when Type_Index_For_Null_Record_Aggregate =>
                     return Ada_Null_Record_Aggregate;
                  when Type_Index_For_Bin_Op =>
                     return Ada_Bin_Op;
                  when Type_Index_For_Relation_Op =>
                     return Ada_Relation_Op;
                  when Type_Index_For_Box_Expr =>
                     return Ada_Box_Expr;
                  when Type_Index_For_Case_Expr_Alternative =>
                     return Ada_Case_Expr_Alternative;
                  when Type_Index_For_Concat_Op =>
                     return Ada_Concat_Op;
                  when Type_Index_For_Concat_Operand =>
                     return Ada_Concat_Operand;
                  when Type_Index_For_Case_Expr =>
                     return Ada_Case_Expr;
                  when Type_Index_For_If_Expr =>
                     return Ada_If_Expr;
                  when Type_Index_For_Contract_Cases =>
                     return Ada_Contract_Cases;
                  when Type_Index_For_Decl_Expr =>
                     return Ada_Decl_Expr;
                  when Type_Index_For_Format_String_Literal =>
                     return Ada_Format_String_Literal;
                  when Type_Index_For_Membership_Expr =>
                     return Ada_Membership_Expr;
                  when Type_Index_For_Array_Subcomponent_Choice_Name =>
                     return Ada_Array_Subcomponent_Choice_Name;
                  when Type_Index_For_Attribute_Ref =>
                     return Ada_Attribute_Ref;
                  when Type_Index_For_Call_Expr =>
                     return Ada_Call_Expr;
                  when Type_Index_For_Defining_Name =>
                     return Ada_Defining_Name;
                  when Type_Index_For_Synthetic_Defining_Name =>
                     return Ada_Synthetic_Defining_Name;
                  when Type_Index_For_Discrete_Subtype_Name =>
                     return Ada_Discrete_Subtype_Name;
                  when Type_Index_For_Dotted_Name =>
                     return Ada_Dotted_Name;
                  when Type_Index_For_End_Name =>
                     return Ada_End_Name;
                  when Type_Index_For_Explicit_Deref =>
                     return Ada_Explicit_Deref;
                  when Type_Index_For_Qual_Expr =>
                     return Ada_Qual_Expr;
                  when Type_Index_For_Reduce_Attribute_Ref =>
                     return Ada_Reduce_Attribute_Ref;
                  when Type_Index_For_Char_Literal =>
                     return Ada_Char_Literal;
                  when Type_Index_For_Identifier =>
                     return Ada_Identifier;
                  when Type_Index_For_Op_Abs =>
                     return Ada_Op_Abs;
                  when Type_Index_For_Op_And =>
                     return Ada_Op_And;
                  when Type_Index_For_Op_And_Then =>
                     return Ada_Op_And_Then;
                  when Type_Index_For_Op_Concat =>
                     return Ada_Op_Concat;
                  when Type_Index_For_Op_Div =>
                     return Ada_Op_Div;
                  when Type_Index_For_Op_Double_Dot =>
                     return Ada_Op_Double_Dot;
                  when Type_Index_For_Op_Eq =>
                     return Ada_Op_Eq;
                  when Type_Index_For_Op_Gt =>
                     return Ada_Op_Gt;
                  when Type_Index_For_Op_Gte =>
                     return Ada_Op_Gte;
                  when Type_Index_For_Op_In =>
                     return Ada_Op_In;
                  when Type_Index_For_Op_Lt =>
                     return Ada_Op_Lt;
                  when Type_Index_For_Op_Lte =>
                     return Ada_Op_Lte;
                  when Type_Index_For_Op_Minus =>
                     return Ada_Op_Minus;
                  when Type_Index_For_Op_Mod =>
                     return Ada_Op_Mod;
                  when Type_Index_For_Op_Mult =>
                     return Ada_Op_Mult;
                  when Type_Index_For_Op_Neq =>
                     return Ada_Op_Neq;
                  when Type_Index_For_Op_Not =>
                     return Ada_Op_Not;
                  when Type_Index_For_Op_Not_In =>
                     return Ada_Op_Not_In;
                  when Type_Index_For_Op_Or =>
                     return Ada_Op_Or;
                  when Type_Index_For_Op_Or_Else =>
                     return Ada_Op_Or_Else;
                  when Type_Index_For_Op_Plus =>
                     return Ada_Op_Plus;
                  when Type_Index_For_Op_Pow =>
                     return Ada_Op_Pow;
                  when Type_Index_For_Op_Rem =>
                     return Ada_Op_Rem;
                  when Type_Index_For_Op_Xor =>
                     return Ada_Op_Xor;
                  when Type_Index_For_String_Literal =>
                     return Ada_String_Literal;
                  when Type_Index_For_Null_Literal =>
                     return Ada_Null_Literal;
                  when Type_Index_For_Int_Literal =>
                     return Ada_Int_Literal;
                  when Type_Index_For_Real_Literal =>
                     return Ada_Real_Literal;
                  when Type_Index_For_Synthetic_Identifier =>
                     return Ada_Synthetic_Identifier;
                  when Type_Index_For_Target_Name =>
                     return Ada_Target_Name;
                  when Type_Index_For_Update_Attribute_Ref =>
                     return Ada_Update_Attribute_Ref;
                  when Type_Index_For_Paren_Expr =>
                     return Ada_Paren_Expr;
                  when Type_Index_For_Quantified_Expr =>
                     return Ada_Quantified_Expr;
                  when Type_Index_For_Raise_Expr =>
                     return Ada_Raise_Expr;
                  when Type_Index_For_Un_Op =>
                     return Ada_Un_Op;
                  when Type_Index_For_For_Loop_Iter_Filter =>
                     return Ada_For_Loop_Iter_Filter;
                  when Type_Index_For_Format_String_Chunk =>
                     return Ada_Format_String_Chunk;
                  when Type_Index_For_Format_String_Tok_End =>
                     return Ada_Format_String_Tok_End;
                  when Type_Index_For_Format_String_Tok_Mid =>
                     return Ada_Format_String_Tok_Mid;
                  when Type_Index_For_Format_String_Tok_Start =>
                     return Ada_Format_String_Tok_Start;
                  when Type_Index_For_Format_String_Tok_String =>
                     return Ada_Format_String_Tok_String;
                  when Type_Index_For_Handled_Stmts =>
                     return Ada_Handled_Stmts;
                  when Type_Index_For_Interface_Kind_Limited =>
                     return Ada_Interface_Kind_Limited;
                  when Type_Index_For_Interface_Kind_Protected =>
                     return Ada_Interface_Kind_Protected;
                  when Type_Index_For_Interface_Kind_Synchronized =>
                     return Ada_Interface_Kind_Synchronized;
                  when Type_Index_For_Interface_Kind_Task =>
                     return Ada_Interface_Kind_Task;
                  when Type_Index_For_Iter_Type_In =>
                     return Ada_Iter_Type_In;
                  when Type_Index_For_Iter_Type_Of =>
                     return Ada_Iter_Type_Of;
                  when Type_Index_For_Library_Item =>
                     return Ada_Library_Item;
                  when Type_Index_For_Limited_Absent =>
                     return Ada_Limited_Absent;
                  when Type_Index_For_Limited_Present =>
                     return Ada_Limited_Present;
                  when Type_Index_For_For_Loop_Spec =>
                     return Ada_For_Loop_Spec;
                  when Type_Index_For_While_Loop_Spec =>
                     return Ada_While_Loop_Spec;
                  when Type_Index_For_Mode_Default =>
                     return Ada_Mode_Default;
                  when Type_Index_For_Mode_In =>
                     return Ada_Mode_In;
                  when Type_Index_For_Mode_In_Out =>
                     return Ada_Mode_In_Out;
                  when Type_Index_For_Mode_Out =>
                     return Ada_Mode_Out;
                  when Type_Index_For_Multi_Abstract_State_Decl =>
                     return Ada_Multi_Abstract_State_Decl;
                  when Type_Index_For_Not_Null_Absent =>
                     return Ada_Not_Null_Absent;
                  when Type_Index_For_Not_Null_Present =>
                     return Ada_Not_Null_Present;
                  when Type_Index_For_Null_Component_Decl =>
                     return Ada_Null_Component_Decl;
                  when Type_Index_For_Others_Designator =>
                     return Ada_Others_Designator;
                  when Type_Index_For_Overriding_Not_Overriding =>
                     return Ada_Overriding_Not_Overriding;
                  when Type_Index_For_Overriding_Overriding =>
                     return Ada_Overriding_Overriding;
                  when Type_Index_For_Overriding_Unspecified =>
                     return Ada_Overriding_Unspecified;
                  when Type_Index_For_Params =>
                     return Ada_Params;
                  when Type_Index_For_Paren_Abstract_State_Decl =>
                     return Ada_Paren_Abstract_State_Decl;
                  when Type_Index_For_Pp_Else_Directive =>
                     return Ada_Pp_Else_Directive;
                  when Type_Index_For_Pp_Elsif_Directive =>
                     return Ada_Pp_Elsif_Directive;
                  when Type_Index_For_Pp_End_If_Directive =>
                     return Ada_Pp_End_If_Directive;
                  when Type_Index_For_Pp_If_Directive =>
                     return Ada_Pp_If_Directive;
                  when Type_Index_For_Pp_Then_Kw =>
                     return Ada_Pp_Then_Kw;
                  when Type_Index_For_Pragma =>
                     return Ada_Pragma_Node;
                  when Type_Index_For_Private_Absent =>
                     return Ada_Private_Absent;
                  when Type_Index_For_Private_Present =>
                     return Ada_Private_Present;
                  when Type_Index_For_Protected_Def =>
                     return Ada_Protected_Def;
                  when Type_Index_For_Protected_Absent =>
                     return Ada_Protected_Absent;
                  when Type_Index_For_Protected_Present =>
                     return Ada_Protected_Present;
                  when Type_Index_For_Quantifier_All =>
                     return Ada_Quantifier_All;
                  when Type_Index_For_Quantifier_Some =>
                     return Ada_Quantifier_Some;
                  when Type_Index_For_Range_Spec =>
                     return Ada_Range_Spec;
                  when Type_Index_For_Renaming_Clause =>
                     return Ada_Renaming_Clause;
                  when Type_Index_For_Synthetic_Renaming_Clause =>
                     return Ada_Synthetic_Renaming_Clause;
                  when Type_Index_For_Reverse_Absent =>
                     return Ada_Reverse_Absent;
                  when Type_Index_For_Reverse_Present =>
                     return Ada_Reverse_Present;
                  when Type_Index_For_Select_When_Part =>
                     return Ada_Select_When_Part;
                  when Type_Index_For_Accept_Stmt =>
                     return Ada_Accept_Stmt;
                  when Type_Index_For_Accept_Stmt_With_Stmts =>
                     return Ada_Accept_Stmt_With_Stmts;
                  when Type_Index_For_For_Loop_Stmt =>
                     return Ada_For_Loop_Stmt;
                  when Type_Index_For_Loop_Stmt =>
                     return Ada_Loop_Stmt;
                  when Type_Index_For_While_Loop_Stmt =>
                     return Ada_While_Loop_Stmt;
                  when Type_Index_For_Begin_Block =>
                     return Ada_Begin_Block;
                  when Type_Index_For_Decl_Block =>
                     return Ada_Decl_Block;
                  when Type_Index_For_Case_Stmt =>
                     return Ada_Case_Stmt;
                  when Type_Index_For_Extended_Return_Stmt =>
                     return Ada_Extended_Return_Stmt;
                  when Type_Index_For_If_Stmt =>
                     return Ada_If_Stmt;
                  when Type_Index_For_Named_Stmt =>
                     return Ada_Named_Stmt;
                  when Type_Index_For_Select_Stmt =>
                     return Ada_Select_Stmt;
                  when Type_Index_For_Error_Stmt =>
                     return Ada_Error_Stmt;
                  when Type_Index_For_Abort_Stmt =>
                     return Ada_Abort_Stmt;
                  when Type_Index_For_Assign_Stmt =>
                     return Ada_Assign_Stmt;
                  when Type_Index_For_Call_Stmt =>
                     return Ada_Call_Stmt;
                  when Type_Index_For_Delay_Stmt =>
                     return Ada_Delay_Stmt;
                  when Type_Index_For_Exit_Stmt =>
                     return Ada_Exit_Stmt;
                  when Type_Index_For_Goto_Stmt =>
                     return Ada_Goto_Stmt;
                  when Type_Index_For_Label =>
                     return Ada_Label;
                  when Type_Index_For_Null_Stmt =>
                     return Ada_Null_Stmt;
                  when Type_Index_For_Raise_Stmt =>
                     return Ada_Raise_Stmt;
                  when Type_Index_For_Requeue_Stmt =>
                     return Ada_Requeue_Stmt;
                  when Type_Index_For_Return_Stmt =>
                     return Ada_Return_Stmt;
                  when Type_Index_For_Simple_Decl_Stmt =>
                     return Ada_Simple_Decl_Stmt;
                  when Type_Index_For_Terminate_Alternative =>
                     return Ada_Terminate_Alternative;
                  when Type_Index_For_Subp_Kind_Function =>
                     return Ada_Subp_Kind_Function;
                  when Type_Index_For_Subp_Kind_Procedure =>
                     return Ada_Subp_Kind_Procedure;
                  when Type_Index_For_Subunit =>
                     return Ada_Subunit;
                  when Type_Index_For_Synchronized_Absent =>
                     return Ada_Synchronized_Absent;
                  when Type_Index_For_Synchronized_Present =>
                     return Ada_Synchronized_Present;
                  when Type_Index_For_Tagged_Absent =>
                     return Ada_Tagged_Absent;
                  when Type_Index_For_Tagged_Present =>
                     return Ada_Tagged_Present;
                  when Type_Index_For_Task_Def =>
                     return Ada_Task_Def;
                  when Type_Index_For_Then_Abort_Part =>
                     return Ada_Then_Abort_Part;
                  when Type_Index_For_Type_Attributes_Repository =>
                     return Ada_Type_Attributes_Repository;
                  when Type_Index_For_Access_To_Subp_Def =>
                     return Ada_Access_To_Subp_Def;
                  when Type_Index_For_Anonymous_Type_Access_Def =>
                     return Ada_Anonymous_Type_Access_Def;
                  when Type_Index_For_Type_Access_Def =>
                     return Ada_Type_Access_Def;
                  when Type_Index_For_Array_Type_Def =>
                     return Ada_Array_Type_Def;
                  when Type_Index_For_Derived_Type_Def =>
                     return Ada_Derived_Type_Def;
                  when Type_Index_For_Enum_Type_Def =>
                     return Ada_Enum_Type_Def;
                  when Type_Index_For_Formal_Discrete_Type_Def =>
                     return Ada_Formal_Discrete_Type_Def;
                  when Type_Index_For_Interface_Type_Def =>
                     return Ada_Interface_Type_Def;
                  when Type_Index_For_Mod_Int_Type_Def =>
                     return Ada_Mod_Int_Type_Def;
                  when Type_Index_For_Private_Type_Def =>
                     return Ada_Private_Type_Def;
                  when Type_Index_For_Decimal_Fixed_Point_Def =>
                     return Ada_Decimal_Fixed_Point_Def;
                  when Type_Index_For_Floating_Point_Def =>
                     return Ada_Floating_Point_Def;
                  when Type_Index_For_Ordinary_Fixed_Point_Def =>
                     return Ada_Ordinary_Fixed_Point_Def;
                  when Type_Index_For_Record_Type_Def =>
                     return Ada_Record_Type_Def;
                  when Type_Index_For_Signed_Int_Type_Def =>
                     return Ada_Signed_Int_Type_Def;
                  when Type_Index_For_Anonymous_Type =>
                     return Ada_Anonymous_Type;
                  when Type_Index_For_Enum_Lit_Synth_Type_Expr =>
                     return Ada_Enum_Lit_Synth_Type_Expr;
                  when Type_Index_For_Subtype_Indication =>
                     return Ada_Subtype_Indication;
                  when Type_Index_For_Constrained_Subtype_Indication =>
                     return Ada_Constrained_Subtype_Indication;
                  when Type_Index_For_Discrete_Subtype_Indication =>
                     return Ada_Discrete_Subtype_Indication;
                  when Type_Index_For_Synthetic_Type_Expr =>
                     return Ada_Synthetic_Type_Expr;
                  when Type_Index_For_Unconstrained_Array_Index =>
                     return Ada_Unconstrained_Array_Index;
                  when Type_Index_For_Until_Absent =>
                     return Ada_Until_Absent;
                  when Type_Index_For_Until_Present =>
                     return Ada_Until_Present;
                  when Type_Index_For_Use_Package_Clause =>
                     return Ada_Use_Package_Clause;
                  when Type_Index_For_Use_Type_Clause =>
                     return Ada_Use_Type_Clause;
                  when Type_Index_For_Value_Sequence =>
                     return Ada_Value_Sequence;
                  when Type_Index_For_Variant =>
                     return Ada_Variant;
                  when Type_Index_For_Variant_Part =>
                     return Ada_Variant_Part;
                  when Type_Index_For_With_Clause =>
                     return Ada_With_Clause;
                  when Type_Index_For_With_Private_Absent =>
                     return Ada_With_Private_Absent;
                  when Type_Index_For_With_Private_Present =>
                     return Ada_With_Private_Present;

            when others =>
               null;
         end case;
      end if;

      raise Precondition_Failure with "concrete node type expected";
   end From_Generic_Node_Type;

end Libadalang.Generic_API;
