/*!
 * \file   include/TFEL/System/getFunction.h
 * \brief  this file declares some C wrappers around the dlsym
 * system call.
 * \author Helfer Thomas
 * \date   13 déc 2008
 */

#ifndef LIB_TFEL_SYSTEM_GETFUNCTION_H
#define LIB_TFEL_SYSTEM_GETFUNCTION_H

#if (defined _WIN32 || defined _WIN64) && (!defined __CYGWIN__)
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <windows.h>
#ifdef small
#undef small
#endif /* small */
typedef HINSTANCE__* LibraryHandlerPtr;
#else
typedef void* LibraryHandlerPtr;
#endif /* (defined _WIN32 || defined _WIN64) && (!defined __CYGWIN__) */

#include "TFEL/System/ExternalFunctionsPrototypes.hxx"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*!
 * \brief this function returns true if the given behaviour requires
 * an offset for the elastic properties.
 * \param l: link to library opened through dlopen
 * \param f: name of the behaviour to be checked
 * \return int, 0 if false, 1 if true, -1 if an error occured.
 */
int tfel_getUMATRequiresStiffnessTensor(LibraryHandlerPtr, const char* const);
/*!
 * \brief this function returns true if the given behaviour requires
 * an offset of the thermal expansions coefficients.
 * \param l: link to library opened through dlopen
 * \param f: name of the behaviour to be checked
 * \return int, 0 if false, 1 if true, -1 if an error occured.
 */
int tfel_getUMATRequiresThermalExpansionCoefficientTensor(LibraryHandlerPtr,
                                                          const char* const);

/*!
 * \brief this function returns true if the given behaviour saves the tangent
 * operator
 * \param l: link to library opened through dlopen
 * \param f: name of the behaviour to be checked
 * \return int, 0 if false, 1 if true, -1 if an error occured.
 */
int tfel_checkIfAsterBehaviourSavesTangentOperator(LibraryHandlerPtr,
                                                   const char* const);

int tfel_getBool(LibraryHandlerPtr, const char* const);

int tfel_getUnsignedShort(LibraryHandlerPtr, const char* const);

double tfel_getDouble(LibraryHandlerPtr, const char* const);

long double tfel_getLongDouble(LibraryHandlerPtr, const char* const);

int tfel_getInteger(LibraryHandlerPtr, const char* const);

char** tfel_getArrayOfStrings(LibraryHandlerPtr, const char* const);

int* tfel_getArrayOfInts(LibraryHandlerPtr, const char* const);

int(TFEL_ADDCALL_PTR tfel_getSetOutOfBoundsPolicyFunction(
    LibraryHandlerPtr lib, const char* const name))(const int);

int(TFEL_ADDCALL_PTR tfel_getSetParameterFunction(
    LibraryHandlerPtr lib, const char* const name))(const char* const,
                                                    const double);

int(TFEL_ADDCALL_PTR tfel_getSetIntegerParameterFunction(
    LibraryHandlerPtr lib, const char* const name))(const char* const,
                                                    const int);

int(TFEL_ADDCALL_PTR tfel_getSetUnsignedShortParameterFunction(
    LibraryHandlerPtr lib, const char* const name))(const char* const,
                                                    const unsigned short);
/*!
 * \brief this function returns a function of type `GenericMaterialPropertyPtr`
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
double(TFEL_ADDCALL_PTR tfel_getGenericMaterialProperty(
    LibraryHandlerPtr, const char* const))(mfront_gmp_OutputStatus* const,
                                           const mfront_gmp_real* const,
                                           const mfront_gmp_size_type,
                                           const mfront_gmp_OutOfBoundsPolicy);
/*!
 * \brief return a function generated by the generic behaviour interface
 * \param l: library handler
 * \param f: function name
 * \return the searched function pointer if the call succeed, the NULL pointer
 * if not.
 */
int(TFEL_ADDCALL_PTR tfel_getGenericBehaviourFunction(LibraryHandlerPtr,
                                                      const char* const))(
    struct mfront_gb_BehaviourData* const);
/*!
 * \brief return a function generated by the generic behaviour interface
 * associated with an initialize functions.
 * \param l: library handler
 * \param f: function name
 * \return the searched function pointer if the call succeed, the NULL pointer
 * if not.
 */
int(TFEL_ADDCALL_PTR tfel_getGenericBehaviourInitializeFunction(
    LibraryHandlerPtr,
    const char* const))(struct mfront_gb_BehaviourData* const,
                        const mfront_gb_real* const);
/*!
 * \brief return a function generated by the generic behaviour interface
 * associated with a post-processing.
 * \param l: library handler
 * \param f: function name
 * \return the searched function pointer if the call succeed, the NULL pointer
 * if not.
 */
int(TFEL_ADDCALL_PTR tfel_getGenericBehaviourPostProcessingFunction(
    LibraryHandlerPtr,
    const char* const))(mfront_gb_real* const,
                        struct mfront_gb_BehaviourData* const);
/*!
 * \brief return a function generated by the generic behaviour interface to
 * rotate the gradients from the global frame to the material frame.
 * \param l: library handler
 * \param f: function name
 * \return the searched function pointer if the call succeed, the NULL pointer
 * if not.
 */
void(TFEL_ADDCALL_PTR tfel_getGenericBehaviourRotateGradientsFunction(
    LibraryHandlerPtr, const char* const))(mfront_gb_real* const,
                                           const mfront_gb_real* const,
                                           const mfront_gb_real* const);
/*!
 * \brief return a function generated by the generic behaviour interface to
 * rotate an array of gradients from the global frame to the material frame.
 * \param l: library handler
 * \param f: function name
 * \return the searched function pointer if the call succeed, the NULL pointer
 * if not.
 */
void(TFEL_ADDCALL_PTR tfel_getGenericBehaviourRotateArrayOfGradientsFunction(
    LibraryHandlerPtr, const char* const))(mfront_gb_real* const,
                                           const mfront_gb_real* const,
                                           const mfront_gb_real* const,
                                           const mfront_gb_size_type);
/*!
 * \brief return a function generated by the generic behaviour interface to
 * rotate the thermodynamic forces from the material frame to the global frame.
 * \param l: library handler
 * \param f: function name
 * \return the searched function pointer if the call succeed, the NULL pointer
 * if not.
 */
void(TFEL_ADDCALL_PTR tfel_getGenericBehaviourRotateThermodynamicForcesFunction(
    LibraryHandlerPtr, const char* const))(mfront_gb_real* const,
                                           const mfront_gb_real* const,
                                           const mfront_gb_real* const);
/*!
 * \brief return a function generated by the generic behaviour interface to
 * rotate an array of thermodynamic forces from the material frame to the global
 * frame.
 * \param l: library handler
 * \param f: function name
 * \return the searched function pointer if the call succeed, the NULL pointer
 * if not.
 */
void(TFEL_ADDCALL_PTR
         tfel_getGenericBehaviourRotateArrayOfThermodynamicForcesFunction(
             LibraryHandlerPtr, const char* const))(mfront_gb_real* const,
                                                    const mfront_gb_real* const,
                                                    const mfront_gb_real* const,
                                                    const mfront_gb_size_type);
/*!
 * \brief return a function generated by the generic behaviour interface to
 * rotate the tangent operator blocks from the material frame to the global
 * frame.
 * \param l: library handler
 * \param f: function name
 * \return the searched function pointer if the call succeed, the NULL pointer
 * if not.
 */
void(TFEL_ADDCALL_PTR
         tfel_getGenericBehaviourRotateTangentOperatorBlocksFunction(
             LibraryHandlerPtr,
             const char* const))(mfront_gb_real* const,
                                 const mfront_gb_real* const,
                                 const mfront_gb_real* const);
/*!
 * \brief return a function generated by the generic behaviour interface to
 * rotate an array of tangent operator blocks from the material frame to the
 * global frame.
 * \param l: library handler
 * \param f: function name
 * \return the searched function pointer if the call succeed, the NULL pointer
 * if not.
 */
void(TFEL_ADDCALL_PTR
         tfel_getGenericBehaviourRotateArrayOfTangentOperatorBlocksFunction(
             LibraryHandlerPtr, const char* const))(mfront_gb_real* const,
                                                    const mfront_gb_real* const,
                                                    const mfront_gb_real* const,
                                                    const mfront_gb_size_type);
/*!
 * \brief this function gets the number of arguments of the
 * specified material property
 *
 * This function search a symbol called name+'_nargs' and interpret
 * it as and unsigned short. The value of that unsigned short
 * compared to the number given in argument.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return int, the number of variables (-1 if on error).
 */
int tfel_getMaterialPropertyNumberOfVariables(LibraryHandlerPtr,
                                              const char* const);
/*!
 * \brief this function gets the name of the arguments of the specified
 * material property
 * \param l: library opened through dlopen
 * \param n: name of the function
 * \return a pointer to the array of the arguments'names
 */
char** tfel_getMaterialPropertyVariables(LibraryHandlerPtr, const char* const);
/*!
 * \brief this function returns a function of type `CyranoMaterialPropertyPtr`
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
double(TFEL_ADDCALL_PTR tfel_getCyranoMaterialProperty(
    LibraryHandlerPtr, const char* const))(CyranoOutputStatus* const,
                                           const CyranoRealType* const,
                                           const CyranoIntegerType,
                                           const CyranoOutOfBoundsPolicy);
/*!
 * \brief this function returns a function of type `CyranoBehaviourPtr`
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
void(TFEL_ADDCALL_PTR tfel_getCyranoFunction(
    LibraryHandlerPtr, const char* const))(const CyranoIntegerType* const,
                                           const CyranoRealType* const,
                                           const CyranoRealType* const,
                                           CyranoRealType* const,
                                           const CyranoRealType* const,
                                           const CyranoRealType* const,
                                           const CyranoRealType* const,
                                           const CyranoRealType* const,
                                           const CyranoRealType* const,
                                           const CyranoIntegerType* const,
                                           const CyranoRealType* const,
                                           const CyranoRealType* const,
                                           CyranoRealType* const,
                                           const CyranoIntegerType* const,
                                           CyranoRealType* const,
                                           const CyranoIntegerType* const,
                                           CyranoIntegerType* const);
/*!
 * \return a pointer to an Abaqus/Standard (umat) external behaviour
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param[in] l: link to library opened through dlopen
 * \param[in] n: name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
void(TFEL_ADDCALL_PTR tfel_getAbaqusExternalBehaviourFunction(
    LibraryHandlerPtr, const char* const))(AbaqusRealType* const,
                                           AbaqusRealType* const,
                                           AbaqusRealType* const,
                                           AbaqusRealType* const,
                                           AbaqusRealType* const,
                                           AbaqusRealType* const,
                                           AbaqusRealType* const,
                                           AbaqusRealType* const,
                                           AbaqusRealType* const,
                                           AbaqusRealType* const,
                                           const AbaqusRealType* const,
                                           const AbaqusRealType* const,
                                           const AbaqusRealType* const,
                                           const AbaqusRealType* const,
                                           const AbaqusRealType* const,
                                           const AbaqusRealType* const,
                                           const AbaqusRealType* const,
                                           const AbaqusRealType* const,
                                           const char* const,
                                           const AbaqusIntegerType* const,
                                           const AbaqusIntegerType* const,
                                           const AbaqusIntegerType* const,
                                           const AbaqusIntegerType* const,
                                           const AbaqusRealType* const,
                                           const AbaqusIntegerType* const,
                                           const AbaqusRealType* const,
                                           const AbaqusRealType* const,
                                           AbaqusRealType* const,
                                           const AbaqusRealType* const,
                                           const AbaqusRealType* const,
                                           const AbaqusRealType* const,
                                           const AbaqusIntegerType* const,
                                           const AbaqusIntegerType* const,
                                           const AbaqusIntegerType* const,
                                           const AbaqusIntegerType* const,
                                           const AbaqusIntegerType* const,
                                           AbaqusIntegerType* const,
                                           const int);
/*!
 * \return a pointer to a Abaqus/Explicit (vumat) external behaviour
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param[in] l: link to library opened through dlopen
 * \param[in] n: name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
void(TFEL_ADDCALL_PTR tfel_getAbaqusExplicitExternalBehaviourFunction(
    LibraryHandlerPtr, const char* const))(const AbaqusIntegerType* const,
                                           const AbaqusIntegerType* const,
                                           const AbaqusIntegerType* const,
                                           const AbaqusIntegerType* const,
                                           const AbaqusIntegerType* const,
                                           const AbaqusIntegerType* const,
                                           const AbaqusIntegerType* const,
                                           const AbaqusRealType* const,
                                           const AbaqusRealType* const,
                                           const AbaqusRealType* const,
                                           const char* const,
                                           const AbaqusRealType* const,
                                           const AbaqusRealType* const,
                                           const AbaqusRealType* const,
                                           const AbaqusRealType* const,
                                           const AbaqusRealType* const,
                                           const AbaqusRealType* const,
                                           const AbaqusRealType* const,
                                           const AbaqusRealType* const,
                                           const AbaqusRealType* const,
                                           const AbaqusRealType* const,
                                           const AbaqusRealType* const,
                                           const AbaqusRealType* const,
                                           const AbaqusRealType* const,
                                           const AbaqusRealType* const,
                                           const AbaqusRealType* const,
                                           const AbaqusRealType* const,
                                           const AbaqusRealType* const,
                                           const AbaqusRealType* const,
                                           AbaqusRealType* const,
                                           AbaqusRealType* const,
                                           AbaqusRealType* const,
                                           AbaqusRealType* const,
                                           const int);
/*!
 * \return a pointer to an Ansys (usermat) behaviour
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param[in] l: link to library opened through dlopen
 * \param[in] n: name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
void(TFEL_ADDCALL_PTR tfel_getAnsysExternalBehaviourFunction(
    LibraryHandlerPtr, const char* const))(const AnsysIntegerType* const,
                                           const AnsysIntegerType* const,
                                           const AnsysIntegerType* const,
                                           const AnsysIntegerType* const,
                                           const AnsysIntegerType* const,
                                           const AnsysIntegerType* const,
                                           const AnsysIntegerType* const,
                                           AnsysIntegerType* const,
                                           const AnsysIntegerType* const,
                                           const AnsysIntegerType* const,
                                           const AnsysIntegerType* const,
                                           const AnsysIntegerType* const,
                                           const AnsysIntegerType* const,
                                           const AnsysRealType* const,
                                           const AnsysRealType* const,
                                           const AnsysRealType* const,
                                           const AnsysRealType* const,
                                           AnsysRealType* const,
                                           AnsysRealType* const,
                                           AnsysRealType* const,
                                           AnsysRealType* const,
                                           AnsysRealType* const,
                                           AnsysRealType* const,
                                           const AnsysRealType* const,
                                           const AnsysRealType* const,
                                           AnsysRealType* const,
                                           const AnsysRealType* const,
                                           const AnsysRealType* const,
                                           const AnsysRealType* const,
                                           const AnsysRealType* const,
                                           const AnsysRealType* const,
                                           const AnsysRealType* const,
                                           AnsysRealType* const,
                                           const AnsysRealType* const,
                                           const AnsysRealType* const,
                                           const AnsysRealType* const,
                                           const AnsysRealType* const,
                                           const AnsysRealType* const,
                                           const AnsysRealType* const,
                                           const AnsysRealType* const,
                                           const AnsysRealType* const);
/*!
 * \return a pointer to a Castem external behaviour
 * \param[in] l : library
 * \param[in] n : external behaviour name
 */
void(TFEL_ADDCALL_PTR tfel_getCastemExternalBehaviourFunction(
    LibraryHandlerPtr, const char* const))(CastemRealType* const,
                                           CastemRealType* const,
                                           CastemRealType* const,
                                           CastemRealType* const,
                                           CastemRealType* const,
                                           CastemRealType* const,
                                           CastemRealType* const,
                                           CastemRealType* const,
                                           CastemRealType* const,
                                           CastemRealType* const,
                                           const CastemRealType* const,
                                           const CastemRealType* const,
                                           const CastemRealType* const,
                                           const CastemRealType* const,
                                           const CastemRealType* const,
                                           const CastemRealType* const,
                                           const CastemRealType* const,
                                           const CastemRealType* const,
                                           const char* const,
                                           const CastemIntegerType* const,
                                           const CastemIntegerType* const,
                                           const CastemIntegerType* const,
                                           const CastemIntegerType* const,
                                           const CastemRealType* const,
                                           const CastemIntegerType* const,
                                           const CastemRealType* const,
                                           const CastemRealType* const,
                                           CastemRealType* const,
                                           const CastemRealType* const,
                                           const CastemRealType* const,
                                           const CastemRealType* const,
                                           const CastemIntegerType* const,
                                           const CastemIntegerType* const,
                                           const CastemIntegerType* const,
                                           const CastemIntegerType* const,
                                           const CastemIntegerType* const,
                                           CastemIntegerType* const,
                                           const int);

/*!
 * \brief this function returns a function of type
 *  void (*)(AsterRealType *const,
 *	       AsterRealType *const,
 *	       AsterRealType *const,
 *	       const AsterRealType *const,
 *	       const AsterRealType *const,
 *	       const AsterRealType *const,
 *	       const AsterRealType *const,
 *	       const AsterRealType *const,
 *	       const AsterRealType *const,
 *	       const AsterRealType *const,
 *	       const AsterIntegerType  *const,
 *	       const AsterIntegerType  *const,
 *	       const AsterRealType *const,
 *	       const AsterIntegerType  *const,
 *	       const AsterRealType *const,
 *	       AsterRealType *const,
 *	       const AsterIntegerType  *const)
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
void(TFEL_ADDCALL_PTR tfel_getAsterFunction(
    LibraryHandlerPtr, const char* const))(AsterRealType* const,
                                           AsterRealType* const,
                                           AsterRealType* const,
                                           const AsterRealType* const,
                                           const AsterRealType* const,
                                           const AsterRealType* const,
                                           const AsterRealType* const,
                                           const AsterRealType* const,
                                           const AsterRealType* const,
                                           const AsterRealType* const,
                                           const AsterIntegerType* const,
                                           const AsterIntegerType* const,
                                           const AsterRealType* const,
                                           const AsterIntegerType* const,
                                           const AsterRealType* const,
                                           AsterRealType* const,
                                           const AsterIntegerType* const);
/*!
 * \brief this function returns a function of type
 *  const char* (*)(void)
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param[in] lib: handler to the library opened through dlopen
 * \param[in] f: name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
const char*(TFEL_ADDCALL_PTR tfel_getAsterIntegrationErrorMessage(
    LibraryHandlerPtr, const char* const))();
/*!
 * \return a pointer to an CalculiX external behaviour
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param[in] l: link to library opened through dlopen
 * \param[in] n: name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
void(TFEL_ADDCALL_PTR tfel_getCalculiXExternalBehaviourFunction(
    LibraryHandlerPtr, const char* const))(const char* const,
                                           const CalculiXIntegerType* const,
                                           const CalculiXIntegerType* const,
                                           const CalculiXIntegerType* const,
                                           const CalculiXRealType* const,
                                           const CalculiXRealType* const,
                                           const CalculiXRealType* const,
                                           const CalculiXRealType* const,
                                           const CalculiXRealType* const,
                                           const CalculiXRealType* const,
                                           const CalculiXRealType* const,
                                           const CalculiXRealType* const,
                                           const CalculiXIntegerType* const,
                                           const CalculiXRealType* const,
                                           const CalculiXRealType* const,
                                           const CalculiXRealType* const,
                                           const CalculiXRealType* const,
                                           const CalculiXIntegerType* const,
                                           const CalculiXIntegerType* const,
                                           const CalculiXIntegerType* const,
                                           const CalculiXIntegerType* const,
                                           const CalculiXRealType* const,
                                           CalculiXRealType* const,
                                           CalculiXRealType* const,
                                           CalculiXRealType* const,
                                           const CalculiXIntegerType* const,
                                           const CalculiXRealType* const,
                                           const CalculiXRealType* const,
                                           CalculiXRealType* const,
                                           const CalculiXIntegerType* const,
                                           const int);
/*!
 * \return a pointer toan Diana FEA external behaviour
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param[in] l: link to library opened through dlopen
 * \param[in] n: name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
void(TFEL_ADDCALL_PTR tfel_getDianaFEAExternalBehaviourFunction(
    LibraryHandlerPtr, const char* const))(DianaFEARealType* const,
                                           DianaFEARealType* const,
                                           DianaFEARealType* const,
                                           const DianaFEAIntegerType* const,
                                           const DianaFEAIntegerType* const,
                                           const DianaFEAIntegerType* const,
                                           const DianaFEARealType* const,
                                           const DianaFEARealType* const,
                                           const DianaFEARealType* const,
                                           const DianaFEARealType* const,
                                           const DianaFEARealType* const,
                                           const DianaFEARealType* const);
/*!
 * \brief this function returns a function of type
 * double (*)(const double * const)
 *
 * The name 'castem' comes from an extension we wrote for that
 * finite element code that uses function with the following
 * prototype double (*)(const double* const);
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
double(TFEL_ADDCALL_PTR tfel_getCastemFunction(
    LibraryHandlerPtr, const char* const))(const double* const);

/*!
 * \brief this function returns a function of type
 * double (*)(void)
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
double(TFEL_ADDCALL_PTR tfel_getCFunction0(LibraryHandlerPtr,
                                           const char* const))(void);

/*!
 * \brief this function returns a function of type
 * double (*)(double)
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
double(TFEL_ADDCALL_PTR tfel_getCFunction1(LibraryHandlerPtr,
                                           const char* const))(double);

/*!
 * \brief this function returns a function of type
 * double (*)(double,double)
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
double(TFEL_ADDCALL_PTR tfel_getCFunction2(LibraryHandlerPtr,
                                           const char* const))(double, double);

/*!
 * \brief this function returns a function of type
 * double (*)(double,double,double)
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
double(TFEL_ADDCALL_PTR tfel_getCFunction3(LibraryHandlerPtr,
                                           const char* const))(double,
                                                               double,
                                                               double);

/*!
 * \brief this function returns a function of type
 * double (*)(double,double,double,double)
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
double(TFEL_ADDCALL_PTR tfel_getCFunction4(
    LibraryHandlerPtr, const char* const))(double, double, double, double);

/*!
 * \brief this function returns a function of type
 * double (*)(double,double,double,double,double)
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
double(TFEL_ADDCALL_PTR tfel_getCFunction5(LibraryHandlerPtr,
                                           const char* const))(
    double, double, double, double, double);

/*!
 * \brief this function returns a function of type
 * double (*)(double,double,double,double,double,double)
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
double(TFEL_ADDCALL_PTR tfel_getCFunction6(LibraryHandlerPtr,
                                           const char* const))(
    double, double, double, double, double, double);

/*!
 * \brief this function returns a function of type
 * double (*)(double,double,double,double,double,double,double)
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
double(TFEL_ADDCALL_PTR tfel_getCFunction7(LibraryHandlerPtr,
                                           const char* const))(
    double, double, double, double, double, double, double);

/*!
 * \brief this function returns a function of type
 * double (*)(double,double,double,double,double,double,double,double)
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
double(TFEL_ADDCALL_PTR tfel_getCFunction8(LibraryHandlerPtr,
                                           const char* const))(
    double, double, double, double, double, double, double, double);

/*!
 * \brief this function returns a function of type
 * double (*)(double,double,double,double,double,double,double,double,double)
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
double(TFEL_ADDCALL_PTR tfel_getCFunction9(LibraryHandlerPtr,
                                           const char* const))(
    double, double, double, double, double, double, double, double, double);

/*!
 * \brief this function returns a function of type
 * double
 * (*)(double,double,double,double,double,double,double,double,double,double)
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
double(TFEL_ADDCALL_PTR tfel_getCFunction10(LibraryHandlerPtr,
                                            const char* const))(double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double);

/*!
 * \brief this function returns a function of type
 * double (*)(double,double,double,double,
 *          double,double,double,double,
 *          double,double,double)
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
double(TFEL_ADDCALL_PTR tfel_getCFunction11(LibraryHandlerPtr,
                                            const char* const))(double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double);

/*!
 * \brief this function returns a function of type
 * double (*)(double,double,double,double,
 *          double,double,double,double,
 *          double,double,double,double)
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
double(TFEL_ADDCALL_PTR tfel_getCFunction12(LibraryHandlerPtr,
                                            const char* const))(double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double);

/*!
 * \brief this function returns a function of type
 * double (*)(double,double,double,double,
 *          double,double,double,double,
 *          double,double,double,double,
 *          double)
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
double(TFEL_ADDCALL_PTR tfel_getCFunction13(LibraryHandlerPtr,
                                            const char* const))(double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double);

/*!
 * \brief this function returns a function of type
 * double (*)(double,double,double,double,
 *          double,double,double,double,
 *          double,double,double,double,
 *          double,double)
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
double(TFEL_ADDCALL_PTR tfel_getCFunction14(LibraryHandlerPtr,
                                            const char* const))(double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double);

/*!
 * \brief this function returns a function of type
 * double (*)(double,double,double,double,
 *          double,double,double,double,
 *          double,double,double,double,
 *          double,double,double)
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
double(TFEL_ADDCALL_PTR tfel_getCFunction15(LibraryHandlerPtr,
                                            const char* const))(double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double,
                                                                double);

/*!
 * \brief this function returns a function of type
 * double (*)(void)
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
double(TFEL_ADDCALL_PTR tfel_getFortranFunction0(LibraryHandlerPtr,
                                                 const char* const))(void);

/*!
 * \brief this function returns a function of type
 * double (*)(const double* const)
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
double(TFEL_ADDCALL_PTR tfel_getFortranFunction1(
    LibraryHandlerPtr, const char* const))(const double* const);

/*!
 * \brief this function returns a function of type
 * double (*)(const double* const,const double* const)
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
double(TFEL_ADDCALL_PTR tfel_getFortranFunction2(
    LibraryHandlerPtr, const char* const))(const double* const,
                                           const double* const);

/*!
 * \brief this function returns a function of type
 * double (*)(const double* const,const double* const,const double* const)
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
double(TFEL_ADDCALL_PTR tfel_getFortranFunction3(LibraryHandlerPtr,
                                                 const char* const))(
    const double* const, const double* const, const double* const);

/*!
 * \brief this function returns a function of type
 * double (*)(const double* const,const double* const,const double* const,const
 * double* const)
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
double(TFEL_ADDCALL_PTR tfel_getFortranFunction4(
    LibraryHandlerPtr, const char* const))(const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const);

/*!
 * \brief this function returns a function of type
 * double (*)(const double* const,const double* const,const double* const,const
 * double* const,const double* const)
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
double(TFEL_ADDCALL_PTR tfel_getFortranFunction5(
    LibraryHandlerPtr, const char* const))(const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const);

/*!
 * \brief this function returns a function of type
 * double (*)(const double* const,const double* const,const double* const,const
 * double* const,const double* const,const double* const)
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
double(TFEL_ADDCALL_PTR tfel_getFortranFunction6(
    LibraryHandlerPtr, const char* const))(const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const);

/*!
 * \brief this function returns a function of type
 * double (*)(const double* const,const double* const,const double* const,const
 * double* const,const double* const,const double* const,const double* const)
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
double(TFEL_ADDCALL_PTR tfel_getFortranFunction7(
    LibraryHandlerPtr, const char* const))(const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const);

/*!
 * \brief this function returns a function of type
 * double (*)(const double* const,const double* const,const double* const,const
 * double* const,const double* const,const double* const,const double*
 * const,const double* const)
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
double(TFEL_ADDCALL_PTR tfel_getFortranFunction8(
    LibraryHandlerPtr, const char* const))(const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const);

/*!
 * \brief this function returns a function of type
 * double (*)(const double* const,const double* const,const double* const,const
 * double* const,const double* const,const double* const,const double*
 * const,const double* const,const double* const)
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
double(TFEL_ADDCALL_PTR tfel_getFortranFunction9(
    LibraryHandlerPtr, const char* const))(const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const);

/*!
 * \brief this function returns a function of type
 * double (*)(const double* const,const double* const,const double* const,const
 * double* const,const double* const,const double* const,const double*
 * const,const double* const,const double* const,const double* const)
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
double(TFEL_ADDCALL_PTR tfel_getFortranFunction10(
    LibraryHandlerPtr, const char* const))(const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const);

/*!
 * \brief this function returns a function of type
 * double (*)(const double* const,const double* const,const double* const,const
 * double* const,
 *          const double* const,const double* const,const double* const,const
 * double* const,
 *          const double* const,const double* const,const double* const)
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
double(TFEL_ADDCALL_PTR tfel_getFortranFunction11(
    LibraryHandlerPtr, const char* const))(const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const);

/*!
 * \brief this function returns a function of type
 * double (*)(const double* const,const double* const,const double* const,const
 * double* const,
 *          const double* const,const double* const,const double* const,const
 * double* const,
 *          const double* const,const double* const,const double* const,const
 * double* const)
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
double(TFEL_ADDCALL_PTR tfel_getFortranFunction12(
    LibraryHandlerPtr, const char* const))(const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const);

/*!
 * \brief this function returns a function of type
 * double (*)(const double* const,const double* const,const double* const,const
 * double* const,
 *          const double* const,const double* const,const double* const,const
 * double* const,
 *          const double* const,const double* const,const double* const,const
 * double* const,
 *          const double* const)
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
double(TFEL_ADDCALL_PTR tfel_getFortranFunction13(
    LibraryHandlerPtr, const char* const))(const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const);

/*!
 * \brief this function returns a function of type
 * double (*)(const double* const,const double* const,const double* const,const
 * double* const,
 *          const double* const,const double* const,const double* const,const
 * double* const,
 *          const double* const,const double* const,const double* const,const
 * double* const,
 *          const double* const,const double* const)
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
double(TFEL_ADDCALL_PTR tfel_getFortranFunction14(
    LibraryHandlerPtr, const char* const))(const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const);

/*!
 * \brief this function returns a function of type
 * double (*)(const double* const,const double* const,const double* const,const
 * double* const,
 *          const double* const,const double* const,const double* const,const
 * double* const,
 *          const double* const,const double* const,const double* const,const
 * double* const,
 *          const double* const,const double* const,const double* const)
 *
 * This function is a simple wrapper around dlsym. We can check if a
 * symbol with the given name exists but cannot check if it points to
 * a function with that prototype.
 *
 * \param LibraryHandlerPtr, link to library opened through dlopen
 * \param const char * const, name of the function to be checked
 * \return a function pointer if the call succeed, the NULL pointer if not.
 */
double(TFEL_ADDCALL_PTR tfel_getFortranFunction15(
    LibraryHandlerPtr, const char* const))(const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const,
                                           const double* const);
#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* LIB_TFEL_SYSTEM_GETFUNCTION_H */
