"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.clearSplitCookies = clearSplitCookies;
exports.getExtraAuthStorageValue = getExtraAuthStorageValue;
exports.setExtraAuthStorage = setExtraAuthStorage;
exports.splitValueIntoCookies = splitValueIntoCookies;
exports.unsplitCookiesIntoValue = unsplitCookiesIntoValue;
var _router = require("../../../../src/core/server/http/router");
var _compression = require("../utils/compression");
var _common = require("../../common");
/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */

function getExtraAuthStorageValue(request, options) {
  let compressedContent = '';
  let content = '';
  if (options.additionalCookies > 0) {
    compressedContent = unsplitCookiesIntoValue(request, options.cookiePrefix, options.additionalCookies);
  }
  try {
    content = (0, _compression.inflateValue)(Buffer.from(compressedContent, 'base64')).toString();
  } catch (error) {
    throw error;
  }
  return content;
}

/**
 * Compress and split up the given value into multiple cookies
 * @param request
 * @param cookie
 * @param options
 */
function setExtraAuthStorage(request, content, options) {
  const compressedAuthorizationHeaderValue = (0, _compression.deflateValue)(content);
  const compressedContent = compressedAuthorizationHeaderValue.toString('base64');
  splitValueIntoCookies(request, options.cookiePrefix, compressedContent, options.additionalCookies, options.logger);
}
function splitValueIntoCookies(request,
// @todo Should be OpenSearchDashboardsRequest, I believe?
cookiePrefix, value, additionalCookies, logger) {
  /**
   * Assume that Iron adds around 50%.
   * Remember that an empty cookie is around 30 bytes
   */

  const maxLengthPerCookie = Math.floor(_common.MAX_LENGTH_OF_COOKIE_BYTES / _common.ESTIMATED_IRON_COOKIE_OVERHEAD);
  const cookiesNeeded = value.length / maxLengthPerCookie; // Assume 1 bit per character since this value is encoded
  // If the amount of additional cookies aren't enough for our logic, we try to write the value anyway
  // TODO We could also consider throwing an error, since a failed cookie leads to weird redirects.
  // But throwing would probably also lead to a weird redirect, since we'd get the token from the IdP again and again
  let splitValueAt = maxLengthPerCookie;
  if (cookiesNeeded > additionalCookies) {
    splitValueAt = Math.ceil(value.length / additionalCookies);
    if (logger) {
      logger.warn('The payload may be too large for the cookies. To be safe, we would need ' + Math.ceil(cookiesNeeded) + ' cookies in total, but we only have ' + additionalCookies + '. This can be changed with opensearch_security.openid.extra_storage.additional_cookies.');
    }
  }
  const rawRequest = (0, _router.ensureRawRequest)(request);
  const values = [];
  for (let i = 1; i <= additionalCookies; i++) {
    values.push(value.substring((i - 1) * splitValueAt, i * splitValueAt));
  }
  values.forEach(async (cookieSplitValue, index) => {
    const cookieName = cookiePrefix + (index + 1);
    if (cookieSplitValue === '') {
      // Make sure we clean up cookies that are not needed for the given value
      rawRequest.cookieAuth.h.unstate(cookieName);
    } else {
      rawRequest.cookieAuth.h.state(cookieName, cookieSplitValue);
    }
  });
}
function unsplitCookiesIntoValue(request, cookiePrefix, additionalCookies) {
  const rawRequest = (0, _router.ensureRawRequest)(request);
  let fullCookieValue = '';

  // We don't want to mix and match between _states and .state.
  // If we find the first additional cookie in _states, we
  // use _states for all subsequent additional cookies
  const requestHasNewerCookieState = rawRequest._states && rawRequest._states[cookiePrefix + 1];
  for (let i = 1; i <= additionalCookies; i++) {
    const cookieName = cookiePrefix + i;
    if (requestHasNewerCookieState && rawRequest._states[cookieName] && rawRequest._states[cookieName].value) {
      fullCookieValue = fullCookieValue + rawRequest._states[cookieName].value;
    } else if (!requestHasNewerCookieState && rawRequest.state[cookieName]) {
      fullCookieValue = fullCookieValue + rawRequest.state[cookieName];
    }
  }
  return fullCookieValue;
}
function clearSplitCookies(request, options) {
  const rawRequest = (0, _router.ensureRawRequest)(request);
  for (let i = 1; i <= options.additionalCookies; i++) {
    const cookieName = options.cookiePrefix + i;
    if (rawRequest.state[cookieName]) {
      rawRequest.cookieAuth.h.unstate(cookieName);
    }
  }
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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