/*
 * Decompiled with CFR 0.152.
 */
package org.duckdb;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.nio.DoubleBuffer;
import java.nio.FloatBuffer;
import java.nio.IntBuffer;
import java.nio.LongBuffer;
import java.nio.ShortBuffer;
import java.nio.charset.StandardCharsets;
import java.sql.SQLException;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.OffsetDateTime;
import java.time.OffsetTime;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.duckdb.DuckDBBindings;
import org.duckdb.DuckDBConnection;
import org.duckdb.DuckDBHugeInt;

public class DuckDBAppender
implements AutoCloseable {
    private static final Set<Integer> supportedTypes = new LinkedHashSet<Integer>();
    private static final DuckDBBindings.CAPIType[] int8Types;
    private static final DuckDBBindings.CAPIType[] int16Types;
    private static final DuckDBBindings.CAPIType[] int32Types;
    private static final DuckDBBindings.CAPIType[] int64Types;
    private static final DuckDBBindings.CAPIType[] int128Types;
    private static final DuckDBBindings.CAPIType[] timestampLocalTypes;
    private static final DuckDBBindings.CAPIType[] timestampMicrosTypes;
    private static final int STRING_MAX_INLINE_BYTES = 12;
    private static final LocalDateTime EPOCH_DATE_TIME;
    private final DuckDBConnection conn;
    private final String catalog;
    private final String schema;
    private final String table;
    private final long maxRows;
    private ByteBuffer appenderRef;
    private final Lock appenderRefLock = new ReentrantLock();
    private final ByteBuffer chunkRef;
    private final Column[] columns;
    private long rowIdx = 0L;
    private int colIdx = 0;
    private int structFieldIdx = 0;
    private int unionFieldIdx = 0;
    private boolean appendingRow = false;
    private boolean appendingStruct = false;
    private boolean writeInlinedStrings = true;

    DuckDBAppender(DuckDBConnection conn, String catalog, String schema, String table) throws SQLException {
        this.conn = conn;
        this.catalog = catalog;
        this.schema = schema;
        this.table = table;
        this.maxRows = DuckDBBindings.duckdb_vector_size();
        ByteBuffer appenderRef = null;
        ByteBuffer[] colTypes = null;
        ByteBuffer chunkRef = null;
        Column[] vectors = null;
        try {
            appenderRef = DuckDBAppender.createAppender(conn, catalog, schema, table);
            colTypes = DuckDBAppender.readTableTypes(appenderRef);
            chunkRef = DuckDBAppender.createChunk(colTypes);
            vectors = DuckDBAppender.createVectors(chunkRef, colTypes);
        }
        catch (Exception e) {
            if (null != chunkRef) {
                DuckDBBindings.duckdb_destroy_data_chunk(chunkRef);
            }
            if (null != colTypes) {
                for (ByteBuffer ct : colTypes) {
                    if (null == ct) continue;
                    DuckDBBindings.duckdb_destroy_logical_type(ct);
                }
            }
            if (null != appenderRef) {
                DuckDBBindings.duckdb_appender_destroy(appenderRef);
            }
            throw new SQLException(this.createErrMsg(e.getMessage()), e);
        }
        this.appenderRef = appenderRef;
        this.chunkRef = chunkRef;
        this.columns = vectors;
    }

    public DuckDBAppender beginRow() throws SQLException {
        this.checkOpen();
        this.checkAppendingRow(false);
        this.checkAppendingStruct(false);
        if (0 != this.colIdx) {
            throw new SQLException(this.createErrMsg("'endRow' must be called before adding next row"));
        }
        this.appendingRow = true;
        return this;
    }

    public DuckDBAppender endRow() throws SQLException {
        this.checkOpen();
        this.checkAppendingRow(true);
        this.checkAppendingStruct(false);
        if (this.columns.length != this.colIdx) {
            throw new SQLException(this.createErrMsg("'endRow' can be called only after adding all columns, expected: " + this.columns.length + ", actual: " + this.colIdx));
        }
        ++this.rowIdx;
        this.appendingRow = false;
        if (this.rowIdx >= this.maxRows) {
            try {
                this.flush();
            }
            catch (SQLException e) {
                this.appendingRow = true;
                --this.rowIdx;
                throw e;
            }
        }
        this.colIdx = 0;
        return this;
    }

    public DuckDBAppender beginStruct() throws SQLException {
        this.checkOpen();
        this.checkCurrentColumnType(DuckDBBindings.CAPIType.DUCKDB_TYPE_STRUCT);
        this.checkAppendingStruct(false);
        this.appendingStruct = true;
        return this;
    }

    public DuckDBAppender endStruct() throws SQLException {
        this.checkOpen();
        this.checkAppendingStruct(true);
        Column structCol = this.currentTopLevelColumn();
        if (structCol.children.size() != this.structFieldIdx) {
            throw new SQLException(this.createErrMsg("'endStruct' can be called only after adding all struct fields, expected: " + structCol.children.size() + ", actual: " + this.structFieldIdx));
        }
        this.structFieldIdx = 0;
        this.appendingStruct = false;
        this.incrementColOrStructFieldIdx();
        return this;
    }

    public DuckDBAppender beginUnion(String tag) throws SQLException {
        Column childCol;
        int i;
        this.checkOpen();
        this.checkCurrentColumnType(DuckDBBindings.CAPIType.DUCKDB_TYPE_UNION);
        this.checkAppendingUnion(false);
        Column structCol = this.currentTopLevelColumn();
        int fieldWithTag = 0;
        for (i = 1; i < structCol.children.size(); ++i) {
            childCol = (Column)structCol.children.get(i);
            if (!childCol.structFieldName.equals(tag)) continue;
            fieldWithTag = i;
        }
        if (0 == fieldWithTag) {
            throw new SQLException(this.createErrMsg("specified union field not found, value: '" + tag + "'"));
        }
        this.appendingStruct = true;
        this.append((byte)(fieldWithTag - 1));
        for (i = 1; i < structCol.children.size(); ++i) {
            if (i == fieldWithTag) continue;
            childCol = (Column)structCol.children.get(i);
            childCol.setNull(this.rowIdx);
        }
        this.unionFieldIdx = fieldWithTag;
        return this;
    }

    public DuckDBAppender endUnion() throws SQLException {
        this.checkOpen();
        this.checkAppendingUnion(true);
        this.structFieldIdx = 0;
        this.unionFieldIdx = 0;
        this.appendingStruct = false;
        this.incrementColOrStructFieldIdx();
        return this;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long flush() throws SQLException {
        this.checkOpen();
        this.checkAppendingRow(false);
        this.checkAppendingStruct(false);
        if (0L == this.rowIdx) {
            return this.rowIdx;
        }
        this.appenderRefLock.lock();
        try {
            this.checkOpen();
            DuckDBBindings.duckdb_data_chunk_set_size(this.chunkRef, this.rowIdx);
            int appendState = DuckDBBindings.duckdb_append_data_chunk(this.appenderRef, this.chunkRef);
            if (0 != appendState) {
                byte[] errorUTF8 = DuckDBBindings.duckdb_appender_error(this.appenderRef);
                String error = DuckDBAppender.strFromUTF8(errorUTF8);
                throw new SQLException(this.createErrMsg(error));
            }
            int flushState = DuckDBBindings.duckdb_appender_flush(this.appenderRef);
            if (0 != flushState) {
                byte[] errorUTF8 = DuckDBBindings.duckdb_appender_error(this.appenderRef);
                String error = DuckDBAppender.strFromUTF8(errorUTF8);
                throw new SQLException(this.createErrMsg(error));
            }
            DuckDBBindings.duckdb_data_chunk_reset(this.chunkRef);
            try {
                for (Column col : this.columns) {
                    col.reset();
                }
            }
            catch (SQLException e) {
                throw new SQLException(this.createErrMsg(e.getMessage()), e);
            }
            long ret = this.rowIdx;
            this.rowIdx = 0L;
            long l = ret;
            return l;
        }
        finally {
            this.appenderRefLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() throws SQLException {
        if (this.isClosed()) {
            return;
        }
        this.appenderRefLock.lock();
        try {
            if (this.isClosed()) {
                return;
            }
            if (this.rowIdx > 0L) {
                try {
                    this.flush();
                }
                catch (SQLException sQLException) {
                    // empty catch block
                }
            }
            for (Column col : this.columns) {
                col.destroy();
            }
            DuckDBBindings.duckdb_destroy_data_chunk(this.chunkRef);
            DuckDBBindings.duckdb_appender_close(this.appenderRef);
            DuckDBBindings.duckdb_appender_destroy(this.appenderRef);
            if (!this.conn.closing) {
                this.conn.connRefLock.lock();
                try {
                    this.conn.appenders.remove(this);
                }
                finally {
                    this.conn.connRefLock.unlock();
                }
            }
            this.appenderRef = null;
        }
        finally {
            this.appenderRefLock.unlock();
        }
    }

    public boolean isClosed() throws SQLException {
        return this.appenderRef == null;
    }

    public DuckDBAppender append(boolean value) throws SQLException {
        Column col = this.currentColumnWithRowPos(DuckDBBindings.CAPIType.DUCKDB_TYPE_BOOLEAN);
        byte val = (byte)(value ? 1 : 0);
        col.data.put(val);
        this.incrementColOrStructFieldIdx();
        return this;
    }

    public DuckDBAppender append(char value) throws SQLException {
        String str = String.valueOf(value);
        return this.append(str);
    }

    public DuckDBAppender append(byte value) throws SQLException {
        Column col = this.currentColumnWithRowPos(int8Types);
        col.data.put(value);
        this.incrementColOrStructFieldIdx();
        return this;
    }

    public DuckDBAppender append(short value) throws SQLException {
        Column col = this.currentColumnWithRowPos(int16Types);
        col.data.putShort(value);
        this.incrementColOrStructFieldIdx();
        return this;
    }

    public DuckDBAppender append(int value) throws SQLException {
        Column col = this.currentColumnWithRowPos(int32Types);
        col.data.putInt(value);
        this.incrementColOrStructFieldIdx();
        return this;
    }

    public DuckDBAppender append(long value) throws SQLException {
        Column col = this.currentColumnWithRowPos(int64Types);
        col.data.putLong(value);
        this.incrementColOrStructFieldIdx();
        return this;
    }

    public DuckDBAppender append(float value) throws SQLException {
        Column col = this.currentColumnWithRowPos(DuckDBBindings.CAPIType.DUCKDB_TYPE_FLOAT);
        col.data.putFloat(value);
        this.incrementColOrStructFieldIdx();
        return this;
    }

    public DuckDBAppender append(double value) throws SQLException {
        Column col = this.currentColumnWithRowPos(DuckDBBindings.CAPIType.DUCKDB_TYPE_DOUBLE);
        col.data.putDouble(value);
        this.incrementColOrStructFieldIdx();
        return this;
    }

    public DuckDBAppender append(Boolean value) throws SQLException {
        this.checkCurrentColumnType(DuckDBBindings.CAPIType.DUCKDB_TYPE_BOOLEAN);
        if (value == null) {
            return this.appendNull();
        }
        return this.append((boolean)value);
    }

    public DuckDBAppender append(Character value) throws SQLException {
        this.checkCurrentColumnType(DuckDBBindings.CAPIType.DUCKDB_TYPE_VARCHAR);
        if (value == null) {
            return this.appendNull();
        }
        return this.append(value.charValue());
    }

    public DuckDBAppender append(Byte value) throws SQLException {
        this.checkCurrentColumnType(int8Types);
        if (value == null) {
            return this.appendNull();
        }
        return this.append((byte)value);
    }

    public DuckDBAppender append(Short value) throws SQLException {
        this.checkCurrentColumnType(int16Types);
        if (value == null) {
            return this.appendNull();
        }
        return this.append((short)value);
    }

    public DuckDBAppender append(Integer value) throws SQLException {
        this.checkCurrentColumnType(int32Types);
        if (value == null) {
            return this.appendNull();
        }
        return this.append((int)value);
    }

    public DuckDBAppender append(Long value) throws SQLException {
        this.checkCurrentColumnType(int64Types);
        if (value == null) {
            return this.appendNull();
        }
        return this.append((long)value);
    }

    public DuckDBAppender append(Float value) throws SQLException {
        this.checkCurrentColumnType(DuckDBBindings.CAPIType.DUCKDB_TYPE_FLOAT);
        if (value == null) {
            return this.appendNull();
        }
        return this.append(value.floatValue());
    }

    public DuckDBAppender append(Double value) throws SQLException {
        this.checkCurrentColumnType(DuckDBBindings.CAPIType.DUCKDB_TYPE_DOUBLE);
        if (value == null) {
            return this.appendNull();
        }
        return this.append((double)value);
    }

    public DuckDBAppender appendHugeInt(long lower, long upper) throws SQLException {
        Column col = this.currentColumnWithRowPos(int128Types);
        col.data.putLong(lower);
        col.data.putLong(upper);
        this.incrementColOrStructFieldIdx();
        return this;
    }

    public DuckDBAppender append(BigInteger value) throws SQLException {
        this.checkCurrentColumnType(int128Types);
        if (value == null) {
            return this.appendNull();
        }
        if (value.compareTo(DuckDBHugeInt.HUGE_INT_MIN) < 0 || value.compareTo(DuckDBHugeInt.HUGE_INT_MAX) > 0) {
            throw new SQLException("Specified BigInteger value is out of range for HUGEINT field");
        }
        long lower = value.longValue();
        long upper = value.shiftRight(64).longValue();
        return this.appendHugeInt(lower, upper);
    }

    public DuckDBAppender appendDecimal(short value) throws SQLException {
        Column col = this.currentDecimalColumnWithRowPos(DuckDBBindings.CAPIType.DUCKDB_TYPE_SMALLINT);
        col.data.putShort(value);
        this.incrementColOrStructFieldIdx();
        return this;
    }

    public DuckDBAppender appendDecimal(int value) throws SQLException {
        Column col = this.currentDecimalColumnWithRowPos(DuckDBBindings.CAPIType.DUCKDB_TYPE_INTEGER);
        col.data.putInt(value);
        this.incrementColOrStructFieldIdx();
        return this;
    }

    public DuckDBAppender appendDecimal(long value) throws SQLException {
        Column col = this.currentDecimalColumnWithRowPos(DuckDBBindings.CAPIType.DUCKDB_TYPE_BIGINT);
        col.data.putLong(value);
        this.incrementColOrStructFieldIdx();
        return this;
    }

    public DuckDBAppender appendDecimal(long lower, long upper) throws SQLException {
        Column col = this.currentDecimalColumnWithRowPos(DuckDBBindings.CAPIType.DUCKDB_TYPE_HUGEINT);
        col.data.putLong(lower);
        col.data.putLong(upper);
        this.incrementColOrStructFieldIdx();
        return this;
    }

    public DuckDBAppender append(BigDecimal value) throws SQLException {
        Column col = this.currentColumnWithRowPos(DuckDBBindings.CAPIType.DUCKDB_TYPE_DECIMAL);
        if (value == null) {
            return this.appendNull();
        }
        if (value.precision() > col.decimalPrecision) {
            throw new SQLException(this.createErrMsg("invalid decimal precision, max expected: " + col.decimalPrecision + ", actual: " + value.precision()));
        }
        if (col.decimalScale != value.scale()) {
            throw new SQLException(this.createErrMsg("invalid decimal scale, expected: " + col.decimalScale + ", actual: " + value.scale()));
        }
        switch (col.decimalInternalType) {
            case DUCKDB_TYPE_SMALLINT: {
                this.checkDecimalPrecision(value, DuckDBBindings.CAPIType.DUCKDB_TYPE_SMALLINT, 4);
                short shortValue = value.unscaledValue().shortValueExact();
                return this.appendDecimal(shortValue);
            }
            case DUCKDB_TYPE_INTEGER: {
                this.checkDecimalPrecision(value, DuckDBBindings.CAPIType.DUCKDB_TYPE_INTEGER, 9);
                int intValue = value.unscaledValue().intValueExact();
                return this.appendDecimal(intValue);
            }
            case DUCKDB_TYPE_BIGINT: {
                this.checkDecimalPrecision(value, DuckDBBindings.CAPIType.DUCKDB_TYPE_BIGINT, 18);
                long longValue = value.unscaledValue().longValueExact();
                return this.appendDecimal(longValue);
            }
            case DUCKDB_TYPE_HUGEINT: {
                this.checkDecimalPrecision(value, DuckDBBindings.CAPIType.DUCKDB_TYPE_HUGEINT, 38);
                BigInteger unscaledValue = value.unscaledValue();
                long lower = unscaledValue.longValue();
                long upper = unscaledValue.shiftRight(64).longValue();
                return this.appendDecimal(lower, upper);
            }
        }
        throw new SQLException(this.createErrMsg("invalid decimal internal type: '" + (Object)((Object)col.decimalInternalType) + "'"));
    }

    public DuckDBAppender append(boolean[] values) throws SQLException {
        return this.append(values, null);
    }

    public DuckDBAppender append(boolean[] values, boolean[] nullMask) throws SQLException {
        Column col = this.currentArrayInnerColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_BOOLEAN);
        if (values == null) {
            return this.appendNull();
        }
        byte[] bytes = new byte[values.length];
        for (int i = 0; i < values.length; ++i) {
            bytes[i] = (byte)(values[i] ? 1 : 0);
        }
        this.checkArrayLength(col, values.length);
        this.setArrayNullMask(col, nullMask);
        int pos = (int)(this.rowIdx * col.arraySize);
        col.data.position(pos);
        col.data.put(bytes);
        this.incrementColOrStructFieldIdx();
        return this;
    }

    public DuckDBAppender append(boolean[][] values) throws SQLException {
        return this.append(values, (boolean[][])null);
    }

    public DuckDBAppender append(boolean[][] values, boolean[][] nullMask) throws SQLException {
        Column arrayCol = this.currentArrayInnerColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY);
        if (values == null) {
            return this.appendNull();
        }
        this.checkArrayLength(arrayCol, values.length);
        Column col = this.currentNestedArrayInnerColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_BOOLEAN);
        byte[] buf = new byte[(int)col.arraySize];
        for (int i = 0; i < values.length; ++i) {
            boolean[] childValues = values[i];
            if (childValues == null) {
                arrayCol.setNull(this.rowIdx, i);
                continue;
            }
            this.checkArrayLength(col, childValues.length);
            if (nullMask != null) {
                this.setArrayNullMask(col, nullMask[i], i);
            }
            for (int j = 0; j < childValues.length; ++j) {
                buf[j] = (byte)(childValues[j] ? 1 : 0);
            }
            int pos = (int)((this.rowIdx * arrayCol.arraySize + (long)i) * col.arraySize);
            col.data.position(pos);
            col.data.put(buf);
        }
        this.incrementColOrStructFieldIdx();
        return this;
    }

    public DuckDBAppender appendByteArray(byte[] values) throws SQLException {
        return this.appendByteArray(values, null);
    }

    public DuckDBAppender appendByteArray(byte[] values, boolean[] nullMask) throws SQLException {
        Column col = this.currentArrayInnerColumn(int8Types);
        if (values == null) {
            return this.appendNull();
        }
        this.checkArrayLength(col, values.length);
        this.setArrayNullMask(col, nullMask);
        int pos = (int)(this.rowIdx * col.arraySize);
        col.data.position(pos);
        col.data.put(values);
        this.incrementColOrStructFieldIdx();
        return this;
    }

    public DuckDBAppender appendByteArray(byte[][] values) throws SQLException {
        return this.appendByteArray(values, (boolean[][])null);
    }

    public DuckDBAppender appendByteArray(byte[][] values, boolean[][] nullMask) throws SQLException {
        Column arrayCol = this.currentArrayInnerColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY);
        if (values == null) {
            return this.appendNull();
        }
        this.checkArrayLength(arrayCol, values.length);
        Column col = this.currentNestedArrayInnerColumn(int8Types);
        for (int i = 0; i < values.length; ++i) {
            byte[] childValues = values[i];
            if (childValues == null) {
                arrayCol.setNull(this.rowIdx, i);
                continue;
            }
            this.checkArrayLength(col, childValues.length);
            if (nullMask != null) {
                this.setArrayNullMask(col, nullMask[i], i);
            }
            int pos = (int)((this.rowIdx * arrayCol.arraySize + (long)i) * col.arraySize);
            col.data.position(pos);
            col.data.put(childValues);
        }
        this.incrementColOrStructFieldIdx();
        return this;
    }

    public DuckDBAppender append(byte[] values) throws SQLException {
        this.checkCurrentColumnType(DuckDBBindings.CAPIType.DUCKDB_TYPE_BLOB);
        if (values == null) {
            return this.appendNull();
        }
        return this.appendStringOrBlobInternal(DuckDBBindings.CAPIType.DUCKDB_TYPE_BLOB, values);
    }

    public DuckDBAppender append(char[] characters) throws SQLException {
        this.checkCurrentColumnType(DuckDBBindings.CAPIType.DUCKDB_TYPE_VARCHAR);
        if (characters == null) {
            return this.appendNull();
        }
        String str = String.valueOf(characters);
        return this.append(str);
    }

    public DuckDBAppender append(short[] values) throws SQLException {
        return this.append(values, (boolean[])null);
    }

    public DuckDBAppender append(short[] values, boolean[] nullMask) throws SQLException {
        Column col = this.currentArrayInnerColumn(int16Types);
        if (values == null) {
            return this.appendNull();
        }
        this.checkArrayLength(col, values.length);
        this.setArrayNullMask(col, nullMask);
        ShortBuffer shortData = col.data.asShortBuffer();
        int pos = (int)(this.rowIdx * col.arraySize);
        shortData.position(pos);
        shortData.put(values);
        this.incrementColOrStructFieldIdx();
        return this;
    }

    public DuckDBAppender append(short[][] values) throws SQLException {
        return this.append(values, (boolean[][])null);
    }

    public DuckDBAppender append(short[][] values, boolean[][] nullMask) throws SQLException {
        Column arrayCol = this.currentArrayInnerColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY);
        if (values == null) {
            return this.appendNull();
        }
        this.checkArrayLength(arrayCol, values.length);
        Column col = this.currentNestedArrayInnerColumn(int16Types);
        for (int i = 0; i < values.length; ++i) {
            short[] childValues = values[i];
            if (childValues == null) {
                arrayCol.setNull(this.rowIdx, i);
                continue;
            }
            this.checkArrayLength(col, childValues.length);
            if (nullMask != null) {
                this.setArrayNullMask(col, nullMask[i], i);
            }
            ShortBuffer shortBuffer = col.data.asShortBuffer();
            int pos = (int)((this.rowIdx * arrayCol.arraySize + (long)i) * col.arraySize);
            shortBuffer.position(pos);
            shortBuffer.put(childValues);
        }
        this.incrementColOrStructFieldIdx();
        return this;
    }

    public DuckDBAppender append(int[] values) throws SQLException {
        return this.append(values, (boolean[])null);
    }

    public DuckDBAppender append(int[] values, boolean[] nullMask) throws SQLException {
        Column col = this.currentArrayInnerColumn(int32Types);
        if (values == null) {
            return this.appendNull();
        }
        this.checkArrayLength(col, values.length);
        this.setArrayNullMask(col, nullMask);
        IntBuffer intData = col.data.asIntBuffer();
        int pos = (int)(this.rowIdx * col.arraySize);
        intData.position(pos);
        intData.put(values);
        this.incrementColOrStructFieldIdx();
        return this;
    }

    public DuckDBAppender append(int[][] values) throws SQLException {
        return this.append(values, (boolean[][])null);
    }

    public DuckDBAppender append(int[][] values, boolean[][] nullMask) throws SQLException {
        Column arrayCol = this.currentArrayInnerColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY);
        if (values == null) {
            return this.appendNull();
        }
        this.checkArrayLength(arrayCol, values.length);
        Column col = this.currentNestedArrayInnerColumn(int32Types);
        for (int i = 0; i < values.length; ++i) {
            int[] childValues = values[i];
            if (childValues == null) {
                arrayCol.setNull(this.rowIdx, i);
                continue;
            }
            this.checkArrayLength(col, childValues.length);
            if (nullMask != null) {
                this.setArrayNullMask(col, nullMask[i], i);
            }
            IntBuffer intData = col.data.asIntBuffer();
            int pos = (int)((this.rowIdx * arrayCol.arraySize + (long)i) * col.arraySize);
            intData.position(pos);
            intData.put(childValues);
        }
        this.incrementColOrStructFieldIdx();
        return this;
    }

    public DuckDBAppender append(long[] values) throws SQLException {
        return this.append(values, (boolean[])null);
    }

    public DuckDBAppender append(long[] values, boolean[] nullMask) throws SQLException {
        Column col = this.currentArrayInnerColumn(int64Types);
        if (values == null) {
            return this.appendNull();
        }
        this.checkArrayLength(col, values.length);
        this.setArrayNullMask(col, nullMask);
        LongBuffer longData = col.data.asLongBuffer();
        int pos = (int)(this.rowIdx * col.arraySize);
        longData.position(pos);
        longData.put(values);
        this.incrementColOrStructFieldIdx();
        return this;
    }

    public DuckDBAppender append(long[][] values) throws SQLException {
        return this.append(values, (boolean[][])null);
    }

    public DuckDBAppender append(long[][] values, boolean[][] nullMask) throws SQLException {
        Column arrayCol = this.currentArrayInnerColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY);
        if (values == null) {
            return this.appendNull();
        }
        this.checkArrayLength(arrayCol, values.length);
        Column col = this.currentNestedArrayInnerColumn(int64Types);
        for (int i = 0; i < values.length; ++i) {
            long[] childValues = values[i];
            if (childValues == null) {
                arrayCol.setNull(this.rowIdx, i);
                continue;
            }
            this.checkArrayLength(col, childValues.length);
            if (nullMask != null) {
                this.setArrayNullMask(col, nullMask[i], i);
            }
            LongBuffer longData = col.data.asLongBuffer();
            int pos = (int)((this.rowIdx * arrayCol.arraySize + (long)i) * col.arraySize);
            longData.position(pos);
            longData.put(childValues);
        }
        this.incrementColOrStructFieldIdx();
        return this;
    }

    public DuckDBAppender append(float[] values) throws SQLException {
        return this.append(values, (boolean[])null);
    }

    public DuckDBAppender append(float[] values, boolean[] nullMask) throws SQLException {
        Column col = this.currentArrayInnerColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_FLOAT);
        if (values == null) {
            return this.appendNull();
        }
        this.checkArrayLength(col, values.length);
        this.setArrayNullMask(col, nullMask);
        FloatBuffer floatData = col.data.asFloatBuffer();
        int pos = (int)(this.rowIdx * col.arraySize);
        floatData.position(pos);
        floatData.put(values);
        this.incrementColOrStructFieldIdx();
        return this;
    }

    public DuckDBAppender append(float[][] values) throws SQLException {
        return this.append(values, (boolean[][])null);
    }

    public DuckDBAppender append(float[][] values, boolean[][] nullMask) throws SQLException {
        Column arrayCol = this.currentArrayInnerColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY);
        if (values == null) {
            return this.appendNull();
        }
        this.checkArrayLength(arrayCol, values.length);
        Column col = this.currentNestedArrayInnerColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_FLOAT);
        for (int i = 0; i < values.length; ++i) {
            float[] childValues = values[i];
            if (childValues == null) {
                arrayCol.setNull(this.rowIdx, i);
                continue;
            }
            this.checkArrayLength(col, childValues.length);
            if (nullMask != null) {
                this.setArrayNullMask(col, nullMask[i], i);
            }
            FloatBuffer floatData = col.data.asFloatBuffer();
            int pos = (int)((this.rowIdx * arrayCol.arraySize + (long)i) * col.arraySize);
            floatData.position(pos);
            floatData.put(childValues);
        }
        this.incrementColOrStructFieldIdx();
        return this;
    }

    public DuckDBAppender append(double[] values) throws SQLException {
        return this.append(values, null);
    }

    public DuckDBAppender append(double[] values, boolean[] nullMask) throws SQLException {
        Column col = this.currentArrayInnerColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_DOUBLE);
        if (values == null) {
            return this.appendNull();
        }
        this.checkArrayLength(col, values.length);
        this.setArrayNullMask(col, nullMask);
        DoubleBuffer doubleData = col.data.asDoubleBuffer();
        int pos = (int)(this.rowIdx * col.arraySize);
        doubleData.position(pos);
        doubleData.put(values);
        this.incrementColOrStructFieldIdx();
        return this;
    }

    public DuckDBAppender append(double[][] values) throws SQLException {
        return this.append(values, (boolean[][])null);
    }

    public DuckDBAppender append(double[][] values, boolean[][] nullMask) throws SQLException {
        Column arrayCol = this.currentArrayInnerColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY);
        if (values == null) {
            return this.appendNull();
        }
        this.checkArrayLength(arrayCol, values.length);
        Column col = this.currentNestedArrayInnerColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_DOUBLE);
        for (int i = 0; i < values.length; ++i) {
            double[] childValues = values[i];
            if (childValues == null) {
                arrayCol.setNull(this.rowIdx, i);
                continue;
            }
            this.checkArrayLength(col, childValues.length);
            if (nullMask != null) {
                this.setArrayNullMask(col, nullMask[i], i);
            }
            DoubleBuffer doubleBuffer = col.data.asDoubleBuffer();
            int pos = (int)((this.rowIdx * arrayCol.arraySize + (long)i) * col.arraySize);
            doubleBuffer.position(pos);
            doubleBuffer.put(childValues);
        }
        this.incrementColOrStructFieldIdx();
        return this;
    }

    public DuckDBAppender append(String value) throws SQLException {
        this.checkCurrentColumnType(DuckDBBindings.CAPIType.DUCKDB_TYPE_VARCHAR);
        if (value == null) {
            return this.appendNull();
        }
        byte[] bytes = value.getBytes(StandardCharsets.UTF_8);
        return this.appendStringOrBlobInternal(DuckDBBindings.CAPIType.DUCKDB_TYPE_VARCHAR, bytes);
    }

    public DuckDBAppender appendUUID(long mostSigBits, long leastSigBits) throws SQLException {
        Column col = this.currentColumnWithRowPos(DuckDBBindings.CAPIType.DUCKDB_TYPE_UUID);
        col.data.putLong(leastSigBits);
        col.data.putLong(mostSigBits ^= Long.MIN_VALUE);
        this.incrementColOrStructFieldIdx();
        return this;
    }

    public DuckDBAppender append(UUID value) throws SQLException {
        this.checkCurrentColumnType(DuckDBBindings.CAPIType.DUCKDB_TYPE_UUID);
        if (value == null) {
            return this.appendNull();
        }
        long mostSigBits = value.getMostSignificantBits();
        long leastSigBits = value.getLeastSignificantBits();
        return this.appendUUID(mostSigBits, leastSigBits);
    }

    public DuckDBAppender appendEpochDays(int days) throws SQLException {
        Column col = this.currentColumnWithRowPos(DuckDBBindings.CAPIType.DUCKDB_TYPE_DATE);
        col.data.putInt(days);
        this.incrementColOrStructFieldIdx();
        return this;
    }

    public DuckDBAppender append(LocalDate value) throws SQLException {
        if (value == null) {
            return this.appendNull();
        }
        long days = value.toEpochDay();
        if (days < Integer.MIN_VALUE || days > Integer.MAX_VALUE) {
            throw new SQLException(this.createErrMsg("unsupported number of days: " + days + ", must fit into 'int32_t'"));
        }
        return this.appendEpochDays((int)days);
    }

    public DuckDBAppender appendDayMicros(long micros) throws SQLException {
        Column col = this.currentColumnWithRowPos(DuckDBBindings.CAPIType.DUCKDB_TYPE_TIME);
        col.data.putLong(micros);
        this.incrementColOrStructFieldIdx();
        return this;
    }

    public DuckDBAppender append(LocalTime value) throws SQLException {
        this.checkCurrentColumnType(DuckDBBindings.CAPIType.DUCKDB_TYPE_TIME);
        if (value == null) {
            return this.appendNull();
        }
        long micros = value.toNanoOfDay() / 1000L;
        return this.appendDayMicros(micros);
    }

    public DuckDBAppender appendDayMicros(long micros, int offset) throws SQLException {
        Column col = this.currentColumnWithRowPos(DuckDBBindings.CAPIType.DUCKDB_TYPE_TIME_TZ);
        long packed = (micros & 0xFFFFFFFFFFL) << 24 | (long)(offset & 0xFFFFFF);
        col.data.putLong(packed);
        this.incrementColOrStructFieldIdx();
        return this;
    }

    public DuckDBAppender append(OffsetTime value) throws SQLException {
        this.checkCurrentColumnType(DuckDBBindings.CAPIType.DUCKDB_TYPE_TIME_TZ);
        if (value == null) {
            return this.appendNull();
        }
        int offset = value.getOffset().getTotalSeconds();
        long micros = value.toLocalTime().toNanoOfDay() / 1000L;
        return this.appendDayMicros(micros, offset);
    }

    public DuckDBAppender appendEpochSeconds(long seconds) throws SQLException {
        Column col = this.currentColumnWithRowPos(DuckDBBindings.CAPIType.DUCKDB_TYPE_TIMESTAMP_S);
        col.data.putLong(seconds);
        this.incrementColOrStructFieldIdx();
        return this;
    }

    public DuckDBAppender appendEpochMillis(long millis) throws SQLException {
        Column col = this.currentColumnWithRowPos(DuckDBBindings.CAPIType.DUCKDB_TYPE_TIMESTAMP_MS);
        col.data.putLong(millis);
        this.incrementColOrStructFieldIdx();
        return this;
    }

    public DuckDBAppender appendEpochMicros(long micros) throws SQLException {
        Column col = this.currentColumnWithRowPos(timestampMicrosTypes);
        col.data.putLong(micros);
        this.incrementColOrStructFieldIdx();
        return this;
    }

    public DuckDBAppender appendEpochNanos(long nanos) throws SQLException {
        Column col = this.currentColumnWithRowPos(DuckDBBindings.CAPIType.DUCKDB_TYPE_TIMESTAMP_NS);
        col.data.putLong(nanos);
        this.incrementColOrStructFieldIdx();
        return this;
    }

    public DuckDBAppender append(LocalDateTime value) throws SQLException {
        Column col = this.currentColumn();
        this.checkCurrentColumnType(timestampLocalTypes);
        if (value == null) {
            return this.appendNull();
        }
        switch (col.colType) {
            case DUCKDB_TYPE_TIMESTAMP_S: {
                long seconds = EPOCH_DATE_TIME.until(value, ChronoUnit.SECONDS);
                return this.appendEpochSeconds(seconds);
            }
            case DUCKDB_TYPE_TIMESTAMP_MS: {
                long millis = EPOCH_DATE_TIME.until(value, ChronoUnit.MILLIS);
                return this.appendEpochMillis(millis);
            }
            case DUCKDB_TYPE_TIMESTAMP: {
                long micros = EPOCH_DATE_TIME.until(value, ChronoUnit.MICROS);
                return this.appendEpochMicros(micros);
            }
            case DUCKDB_TYPE_TIMESTAMP_NS: {
                long nanos = EPOCH_DATE_TIME.until(value, ChronoUnit.NANOS);
                return this.appendEpochNanos(nanos);
            }
        }
        throw new SQLException(this.createErrMsg("invalid column type: " + (Object)((Object)col.colType)));
    }

    public DuckDBAppender append(Date value) throws SQLException {
        Column col = this.currentColumn();
        this.checkCurrentColumnType(timestampLocalTypes);
        if (value == null) {
            return this.appendNull();
        }
        switch (col.colType) {
            case DUCKDB_TYPE_TIMESTAMP_S: {
                long seconds = value.getTime() / 1000L;
                return this.appendEpochSeconds(seconds);
            }
            case DUCKDB_TYPE_TIMESTAMP_MS: {
                long millis = value.getTime();
                return this.appendEpochMillis(millis);
            }
            case DUCKDB_TYPE_TIMESTAMP: {
                long micros = Math.multiplyExact(value.getTime(), 1000L);
                return this.appendEpochMicros(micros);
            }
            case DUCKDB_TYPE_TIMESTAMP_NS: {
                long nanos = Math.multiplyExact(value.getTime(), 1000000L);
                return this.appendEpochNanos(nanos);
            }
        }
        throw new SQLException(this.createErrMsg("invalid column type: " + (Object)((Object)col.colType)));
    }

    public DuckDBAppender append(OffsetDateTime value) throws SQLException {
        this.checkCurrentColumnType(DuckDBBindings.CAPIType.DUCKDB_TYPE_TIMESTAMP_TZ);
        if (value == null) {
            return this.appendNull();
        }
        ZonedDateTime zdt = value.atZoneSameInstant(ZoneOffset.UTC);
        LocalDateTime ldt = zdt.toLocalDateTime();
        long micros = EPOCH_DATE_TIME.until(ldt, ChronoUnit.MICROS);
        return this.appendEpochMicros(micros);
    }

    public DuckDBAppender appendNull() throws SQLException {
        Column col = this.currentColumn();
        col.setNull(this.rowIdx);
        this.incrementColOrStructFieldIdx();
        return this;
    }

    public DuckDBAppender appendDefault() throws SQLException {
        this.currentColumn();
        this.appenderRefLock.lock();
        try {
            this.checkOpen();
            DuckDBBindings.duckdb_append_default_to_chunk(this.appenderRef, this.chunkRef, this.colIdx, this.rowIdx);
        }
        finally {
            this.appenderRefLock.unlock();
        }
        this.incrementColOrStructFieldIdx();
        return this;
    }

    public boolean getWriteInlinedStrings() {
        return this.writeInlinedStrings;
    }

    public void setWriteInlinedStrings(boolean writeInlinedStrings) {
        this.writeInlinedStrings = writeInlinedStrings;
    }

    private String createErrMsg(String error) {
        return "Appender error, catalog: '" + this.catalog + "', schema: '" + this.schema + "', table: '" + this.table + "', message: " + (null != error ? error : "N/A");
    }

    private void checkOpen() throws SQLException {
        if (this.isClosed()) {
            throw new SQLException(this.createErrMsg("appender was closed"));
        }
    }

    private void checkAppendingRow(boolean expected) throws SQLException {
        if (this.appendingRow != expected) {
            throw new SQLException(this.createErrMsg("'beginRow' and 'endRow' calls must be paired"));
        }
    }

    private void checkAppendingStruct(boolean expected) throws SQLException {
        if (this.appendingStruct != expected) {
            throw new SQLException(this.createErrMsg("'beginStruct' and 'endStruct' calls must be paired and cannot be interleaved with 'beginRow' and 'endRow'"));
        }
    }

    private void checkAppendingUnion(boolean expected) throws SQLException {
        if (this.appendingStruct != expected) {
            throw new SQLException(this.createErrMsg("'beginUnion' and 'endUnion' calls must be paired and cannot be interleaved with 'beginRow' and 'endRow'"));
        }
        if (this.appendingStruct && this.unionFieldIdx == 0) {
            throw new SQLException(this.createErrMsg("invalid zero union field index"));
        }
        if (!this.appendingStruct && this.unionFieldIdx != 0) {
            throw new SQLException(this.createErrMsg("invalid non-zero union field index"));
        }
    }

    private void incrementColOrStructFieldIdx() throws SQLException {
        if (this.appendingStruct) {
            ++this.structFieldIdx;
            return;
        }
        if (this.appendingRow) {
            ++this.colIdx;
            return;
        }
        throw new SQLException(this.createErrMsg("'beginRow' must be called before calling `append`"));
    }

    private Column currentTopLevelColumn() throws SQLException {
        this.checkOpen();
        if (this.colIdx >= this.columns.length) {
            throw new SQLException(this.createErrMsg("invalid columns count, expected: " + this.columns.length + ", actual: " + (this.colIdx + 1)));
        }
        return this.columns[this.colIdx];
    }

    private Column currentColumn() throws SQLException {
        Column col = this.currentTopLevelColumn();
        if (!this.appendingStruct || col.colType != DuckDBBindings.CAPIType.DUCKDB_TYPE_STRUCT && col.colType != DuckDBBindings.CAPIType.DUCKDB_TYPE_UNION) {
            return col;
        }
        if (this.unionFieldIdx > 0) {
            if (this.unionFieldIdx > col.children.size()) {
                throw new SQLException(this.createErrMsg("invalid union fields count, expected: " + this.columns.length + ", actual: " + (this.structFieldIdx + 1)));
            }
            return (Column)col.children.get(this.unionFieldIdx);
        }
        if (this.structFieldIdx >= col.children.size()) {
            throw new SQLException(this.createErrMsg("invalid struct fields count, expected: " + this.columns.length + ", actual: " + (this.structFieldIdx + 1)));
        }
        return (Column)col.children.get(this.structFieldIdx);
    }

    private Column currentArrayInnerColumn(DuckDBBindings.CAPIType ctype) throws SQLException {
        return this.currentArrayInnerColumn(ctype.typeArray);
    }

    private Column currentArrayInnerColumn(DuckDBBindings.CAPIType[] ctypes) throws SQLException {
        Column parentCol = this.currentColumn();
        if (parentCol.colType != DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY) {
            throw new SQLException(this.createErrMsg("invalid array column type: '" + (Object)((Object)parentCol.colType) + "'"));
        }
        Column col = (Column)parentCol.children.get(0);
        for (DuckDBBindings.CAPIType ct : ctypes) {
            if (col.colType != ct) continue;
            return col;
        }
        throw new SQLException(this.createErrMsg("invalid array inner column type, expected one of: '" + Arrays.toString((Object[])ctypes) + "', actual: '" + (Object)((Object)col.colType) + "'"));
    }

    private Column currentNestedArrayInnerColumn(DuckDBBindings.CAPIType ctype) throws SQLException {
        return this.currentNestedArrayInnerColumn(ctype.typeArray);
    }

    private Column currentNestedArrayInnerColumn(DuckDBBindings.CAPIType[] ctypes) throws SQLException {
        Column parentCol = this.currentColumn();
        if (parentCol.colType != DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY) {
            throw new SQLException(this.createErrMsg("invalid array column type: '" + (Object)((Object)parentCol.colType) + "'"));
        }
        Column arrayCol = (Column)parentCol.children.get(0);
        if (arrayCol.colType != DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY) {
            throw new SQLException(this.createErrMsg("invalid nested array column type: '" + (Object)((Object)arrayCol.colType) + "'"));
        }
        Column col = (Column)arrayCol.children.get(0);
        for (DuckDBBindings.CAPIType ct : ctypes) {
            if (col.colType != ct) continue;
            return col;
        }
        throw new SQLException(this.createErrMsg("invalid  nested array inner column type, expected one of: '" + Arrays.toString((Object[])ctypes) + "', actual: '" + (Object)((Object)col.colType) + "'"));
    }

    private void checkCurrentColumnType(DuckDBBindings.CAPIType ctype) throws SQLException {
        this.checkCurrentColumnType(ctype.typeArray);
    }

    private void checkCurrentColumnType(DuckDBBindings.CAPIType[] ctypes) throws SQLException {
        Column col = this.currentColumn();
        this.checkColumnType(col, ctypes);
    }

    private void checkColumnType(Column col, DuckDBBindings.CAPIType ctype) throws SQLException {
        this.checkColumnType(col, ctype.typeArray);
    }

    private void checkColumnType(Column col, DuckDBBindings.CAPIType[] ctypes) throws SQLException {
        for (DuckDBBindings.CAPIType ct : ctypes) {
            if (col.colType != ct) continue;
            return;
        }
        throw new SQLException(this.createErrMsg("invalid column type, expected one of: '" + Arrays.toString((Object[])ctypes) + "', actual: '" + (Object)((Object)col.colType) + "'"));
    }

    private void checkArrayLength(Column col, int length) throws SQLException {
        if (col.arraySize != (long)length) {
            throw new SQLException(this.createErrMsg("invalid array size, expected: " + col.arraySize + ", actual: " + length));
        }
    }

    private void setArrayNullMask(Column col, boolean[] nullMask) throws SQLException {
        this.setArrayNullMask(col, nullMask, 0);
    }

    private void setArrayNullMask(Column col, boolean[] nullMask, int parentArrayIdx) throws SQLException {
        if (null == nullMask) {
            return;
        }
        for (int i = 0; i < nullMask.length; ++i) {
            if (!nullMask[i]) continue;
            col.setNull(this.rowIdx, (int)((long)i + col.arraySize * (long)parentArrayIdx));
        }
    }

    private Column currentDecimalColumnWithRowPos(DuckDBBindings.CAPIType decimalInternalType) throws SQLException {
        Column col = this.currentColumnWithRowPos(DuckDBBindings.CAPIType.DUCKDB_TYPE_DECIMAL);
        if (col.decimalInternalType != decimalInternalType) {
            throw new SQLException(this.createErrMsg("invalid decimal internal type, expected: '" + (Object)((Object)col.decimalInternalType) + "', actual: '" + (Object)((Object)decimalInternalType) + "'"));
        }
        this.setRowPos(col, ((Column)col).decimalInternalType.widthBytes);
        return col;
    }

    private Column currentColumnWithRowPos(DuckDBBindings.CAPIType ctype) throws SQLException {
        return this.currentColumnWithRowPos(ctype.typeArray);
    }

    private void setRowPos(Column col, long widthBytes) throws SQLException {
        long pos = this.rowIdx * widthBytes;
        if (pos >= (long)col.data.capacity()) {
            throw new SQLException(this.createErrMsg("invalid calculated position: " + pos + ", type: '" + (Object)((Object)col.colType) + "'"));
        }
        col.data.position((int)pos);
    }

    private Column currentColumnWithRowPos(DuckDBBindings.CAPIType[] ctypes) throws SQLException {
        Column col = this.currentColumn();
        boolean typeMatches = false;
        for (DuckDBBindings.CAPIType ct : ctypes) {
            if (((Column)col).colType.typeId == ct.typeId) {
                typeMatches = true;
            }
            if (((Column)col).colType.widthBytes == ct.widthBytes) continue;
            throw new SQLException(this.createErrMsg("invalid columns type width, expected: '" + (Object)((Object)ct) + "', actual: '" + (Object)((Object)col.colType) + "'"));
        }
        if (!typeMatches) {
            Object[] typeStrs = new String[ctypes.length];
            for (int i = 0; i < ctypes.length; ++i) {
                typeStrs[i] = String.valueOf((Object)ctypes[i]);
            }
            throw new SQLException(this.createErrMsg("invalid columns type, expected one of: '" + Arrays.toString(typeStrs) + "', actual: '" + (Object)((Object)col.colType) + "'"));
        }
        if (((Column)col).colType.widthBytes > 0L) {
            this.setRowPos(col, ((Column)col).colType.widthBytes);
        }
        return col;
    }

    private void checkDecimalPrecision(BigDecimal value, DuckDBBindings.CAPIType decimalInternalType, int maxPrecision) throws SQLException {
        if (value.precision() > maxPrecision) {
            throw new SQLException(this.createErrMsg("invalid decimal precision, value: " + value.precision() + ", max value: " + maxPrecision + ", decimal internal type: " + (Object)((Object)decimalInternalType)));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private DuckDBAppender appendStringOrBlobInternal(DuckDBBindings.CAPIType ctype, byte[] bytes) throws SQLException {
        if (this.writeInlinedStrings && bytes.length < 12) {
            Column col = this.currentColumnWithRowPos(ctype);
            col.data.putInt(bytes.length);
            if (bytes.length > 0) {
                col.data.put(bytes);
            }
        } else {
            Column col = this.currentColumn();
            this.checkColumnType(col, ctype);
            this.appenderRefLock.lock();
            try {
                this.checkOpen();
                DuckDBBindings.duckdb_vector_assign_string_element_len(col.vectorRef, this.rowIdx, bytes);
            }
            finally {
                this.appenderRefLock.unlock();
            }
        }
        this.incrementColOrStructFieldIdx();
        return this;
    }

    private static byte[] utf8(String str) {
        if (null == str) {
            return null;
        }
        return str.getBytes(StandardCharsets.UTF_8);
    }

    private static String strFromUTF8(byte[] utf8) {
        if (null == utf8) {
            return "";
        }
        return new String(utf8, StandardCharsets.UTF_8);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static ByteBuffer createAppender(DuckDBConnection conn, String catalog, String schema, String table) throws SQLException {
        conn.checkOpen();
        ReentrantLock connRefLock = conn.connRefLock;
        connRefLock.lock();
        try {
            ByteBuffer[] out = new ByteBuffer[1];
            int state = DuckDBBindings.duckdb_appender_create_ext(conn.connRef, DuckDBAppender.utf8(catalog), DuckDBAppender.utf8(schema), DuckDBAppender.utf8(table), out);
            if (0 != state) {
                throw new SQLException("duckdb_appender_create_ext error");
            }
            ByteBuffer byteBuffer = out[0];
            return byteBuffer;
        }
        finally {
            connRefLock.unlock();
        }
    }

    private static ByteBuffer[] readTableTypes(ByteBuffer appenderRef) throws SQLException {
        long colCountLong = DuckDBBindings.duckdb_appender_column_count(appenderRef);
        if (colCountLong > Integer.MAX_VALUE || colCountLong < 0L) {
            throw new SQLException("invalid columns count: " + colCountLong);
        }
        int colCount = (int)colCountLong;
        ByteBuffer[] res = new ByteBuffer[colCount];
        for (int i = 0; i < colCount; ++i) {
            ByteBuffer colType = DuckDBBindings.duckdb_appender_column_type(appenderRef, i);
            if (null == colType) {
                throw new SQLException("cannot get logical type for column: " + i);
            }
            int typeId = DuckDBBindings.duckdb_get_type_id(colType);
            if (!supportedTypes.contains(typeId)) {
                for (ByteBuffer lt : res) {
                    if (null == lt) continue;
                    DuckDBBindings.duckdb_destroy_logical_type(lt);
                }
                throw new SQLException("unsupported C API type: " + typeId);
            }
            res[i] = colType;
        }
        return res;
    }

    private static ByteBuffer createChunk(ByteBuffer[] colTypes) throws SQLException {
        ByteBuffer chunkRef = DuckDBBindings.duckdb_create_data_chunk(colTypes);
        if (null == chunkRef) {
            throw new SQLException("cannot create data chunk");
        }
        return chunkRef;
    }

    private static void initVecChildren(Column parent) throws SQLException {
        switch (parent.colType) {
            case DUCKDB_TYPE_LIST: 
            case DUCKDB_TYPE_MAP: {
                ByteBuffer vec = DuckDBBindings.duckdb_list_vector_get_child(parent.vectorRef);
                Column col = new Column(parent, null, vec);
                parent.children.add(col);
                break;
            }
            case DUCKDB_TYPE_STRUCT: 
            case DUCKDB_TYPE_UNION: {
                long count = DuckDBBindings.duckdb_struct_type_child_count(parent.colTypeRef);
                int i = 0;
                while ((long)i < count) {
                    ByteBuffer vec = DuckDBBindings.duckdb_struct_vector_get_child(parent.vectorRef, i);
                    Column col = new Column(parent, null, vec, i);
                    parent.children.add(col);
                    ++i;
                }
                break;
            }
            case DUCKDB_TYPE_ARRAY: {
                ByteBuffer vec = DuckDBBindings.duckdb_array_vector_get_child(parent.vectorRef);
                Column col = new Column(parent, null, vec);
                parent.children.add(col);
                break;
            }
        }
    }

    private static Column[] createVectors(ByteBuffer chunkRef, ByteBuffer[] colTypes) throws SQLException {
        Column[] vectors = new Column[colTypes.length];
        try {
            for (int i = 0; i < colTypes.length; ++i) {
                ByteBuffer vector = DuckDBBindings.duckdb_data_chunk_get_vector(chunkRef, i);
                vectors[i] = new Column(null, colTypes[i], vector);
                colTypes[i] = null;
            }
        }
        catch (Exception e) {
            for (Column col : vectors) {
                if (null == col) continue;
                col.destroy();
            }
            throw e;
        }
        return vectors;
    }

    static {
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_BOOLEAN.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_TINYINT.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_UTINYINT.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_SMALLINT.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_USMALLINT.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_INTEGER.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_UINTEGER.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_BIGINT.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_UBIGINT.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_HUGEINT.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_UHUGEINT.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_FLOAT.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_DOUBLE.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_DECIMAL.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_VARCHAR.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_BLOB.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_DATE.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_TIME.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_TIME_TZ.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_TIMESTAMP_S.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_TIMESTAMP_MS.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_TIMESTAMP.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_TIMESTAMP_TZ.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_TIMESTAMP_NS.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_UUID.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_STRUCT.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_UNION.typeId);
        int8Types = new DuckDBBindings.CAPIType[]{DuckDBBindings.CAPIType.DUCKDB_TYPE_TINYINT, DuckDBBindings.CAPIType.DUCKDB_TYPE_UTINYINT};
        int16Types = new DuckDBBindings.CAPIType[]{DuckDBBindings.CAPIType.DUCKDB_TYPE_SMALLINT, DuckDBBindings.CAPIType.DUCKDB_TYPE_USMALLINT};
        int32Types = new DuckDBBindings.CAPIType[]{DuckDBBindings.CAPIType.DUCKDB_TYPE_INTEGER, DuckDBBindings.CAPIType.DUCKDB_TYPE_UINTEGER};
        int64Types = new DuckDBBindings.CAPIType[]{DuckDBBindings.CAPIType.DUCKDB_TYPE_BIGINT, DuckDBBindings.CAPIType.DUCKDB_TYPE_UBIGINT};
        int128Types = new DuckDBBindings.CAPIType[]{DuckDBBindings.CAPIType.DUCKDB_TYPE_HUGEINT, DuckDBBindings.CAPIType.DUCKDB_TYPE_UHUGEINT};
        timestampLocalTypes = new DuckDBBindings.CAPIType[]{DuckDBBindings.CAPIType.DUCKDB_TYPE_TIMESTAMP_S, DuckDBBindings.CAPIType.DUCKDB_TYPE_TIMESTAMP_MS, DuckDBBindings.CAPIType.DUCKDB_TYPE_TIMESTAMP, DuckDBBindings.CAPIType.DUCKDB_TYPE_TIMESTAMP_NS};
        timestampMicrosTypes = new DuckDBBindings.CAPIType[]{DuckDBBindings.CAPIType.DUCKDB_TYPE_TIMESTAMP, DuckDBBindings.CAPIType.DUCKDB_TYPE_TIMESTAMP_TZ};
        EPOCH_DATE_TIME = LocalDateTime.ofEpochSecond(0L, 0, ZoneOffset.UTC);
    }

    private static class Column {
        private final Column parent;
        private ByteBuffer colTypeRef;
        private final DuckDBBindings.CAPIType colType;
        private final DuckDBBindings.CAPIType decimalInternalType;
        private final int decimalPrecision;
        private final int decimalScale;
        private final long arraySize;
        private final String structFieldName;
        private final ByteBuffer vectorRef;
        private ByteBuffer data;
        private ByteBuffer validity;
        private final List<Column> children = new ArrayList<Column>();

        private Column(Column parent, ByteBuffer colTypeRef, ByteBuffer vector) throws SQLException {
            this(parent, colTypeRef, vector, -1);
        }

        private Column(Column parent, ByteBuffer colTypeRef, ByteBuffer vector, int structFieldIdx) throws SQLException {
            this.parent = parent;
            if (null == vector) {
                throw new SQLException("cannot initialize data chunk vector");
            }
            if (null == colTypeRef) {
                this.colTypeRef = DuckDBBindings.duckdb_vector_get_column_type(vector);
                if (null == this.colTypeRef) {
                    throw new SQLException("cannot initialize data chunk vector type");
                }
            } else {
                this.colTypeRef = colTypeRef;
            }
            int colTypeId = DuckDBBindings.duckdb_get_type_id(this.colTypeRef);
            this.colType = DuckDBBindings.CAPIType.capiTypeFromTypeId(colTypeId);
            if (this.colType == DuckDBBindings.CAPIType.DUCKDB_TYPE_DECIMAL) {
                int decimalInternalTypeId = DuckDBBindings.duckdb_decimal_internal_type(this.colTypeRef);
                this.decimalInternalType = DuckDBBindings.CAPIType.capiTypeFromTypeId(decimalInternalTypeId);
                this.decimalPrecision = DuckDBBindings.duckdb_decimal_width(this.colTypeRef);
                this.decimalScale = DuckDBBindings.duckdb_decimal_scale(this.colTypeRef);
            } else {
                this.decimalInternalType = DuckDBBindings.CAPIType.DUCKDB_TYPE_INVALID;
                this.decimalPrecision = -1;
                this.decimalScale = -1;
            }
            this.arraySize = null == parent || parent.colType != DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY ? 1L : DuckDBBindings.duckdb_array_type_array_size(parent.colTypeRef);
            if (structFieldIdx >= 0) {
                byte[] nameUTF8 = DuckDBBindings.duckdb_struct_type_child_name(parent.colTypeRef, structFieldIdx);
                this.structFieldName = DuckDBAppender.strFromUTF8(nameUTF8);
            } else {
                this.structFieldName = null;
            }
            this.vectorRef = vector;
            if (this.colType.widthBytes > 0L || this.colType == DuckDBBindings.CAPIType.DUCKDB_TYPE_DECIMAL) {
                this.data = DuckDBBindings.duckdb_vector_get_data(this.vectorRef, this.widthBytes() * this.arraySize * this.parentArraySize());
                if (null == this.data) {
                    throw new SQLException("cannot initialize data chunk vector data");
                }
            } else {
                this.data = null;
            }
            DuckDBBindings.duckdb_vector_ensure_validity_writable(this.vectorRef);
            this.validity = DuckDBBindings.duckdb_vector_get_validity(this.vectorRef, this.arraySize * this.parentArraySize());
            if (null == this.validity) {
                throw new SQLException("cannot initialize data chunk vector validity");
            }
            DuckDBAppender.initVecChildren(this);
        }

        void reset() throws SQLException {
            if (null != this.data) {
                this.data = DuckDBBindings.duckdb_vector_get_data(this.vectorRef, this.widthBytes() * this.arraySize * this.parentArraySize());
                if (null == this.data) {
                    throw new SQLException("cannot reset data chunk vector data");
                }
            }
            DuckDBBindings.duckdb_vector_ensure_validity_writable(this.vectorRef);
            this.validity = DuckDBBindings.duckdb_vector_get_validity(this.vectorRef, this.arraySize * this.parentArraySize());
            if (null == this.validity) {
                throw new SQLException("cannot reset data chunk vector validity");
            }
            for (Column col : this.children) {
                col.reset();
            }
        }

        void destroy() {
            for (Column cvec : this.children) {
                cvec.destroy();
            }
            this.children.clear();
            if (null != this.colTypeRef) {
                DuckDBBindings.duckdb_destroy_logical_type(this.colTypeRef);
                this.colTypeRef = null;
            }
        }

        void setNull(long rowIdx) throws SQLException {
            if (1L != this.arraySize) {
                throw new SQLException("Invalid API usage for array, size: " + this.arraySize);
            }
            this.setNull(rowIdx, 0);
            for (Column col : this.children) {
                int i = 0;
                while ((long)i < col.arraySize) {
                    col.setNull(rowIdx, i);
                    ++i;
                }
            }
        }

        void setNull(long rowIdx, int arrayIdx) {
            LongBuffer entries = this.validity.asLongBuffer();
            long vectorPos = rowIdx * this.arraySize * this.parentArraySize() + (long)arrayIdx;
            long validityPos = vectorPos / 64L;
            entries.position((int)validityPos);
            long mask = entries.get();
            long idxInEntry = vectorPos % 64L;
            entries.position((int)validityPos);
            entries.put(mask &= 1L << (int)idxInEntry ^ 0xFFFFFFFFFFFFFFFFL);
        }

        long widthBytes() {
            if (this.colType == DuckDBBindings.CAPIType.DUCKDB_TYPE_DECIMAL) {
                return this.decimalInternalType.widthBytes;
            }
            return this.colType.widthBytes;
        }

        long parentArraySize() {
            if (null == this.parent) {
                return 1L;
            }
            return this.parent.arraySize;
        }
    }
}

