#!/usr/bin/env bash

set -e
set -x

readonly tmpdir="$(mktemp -d)"
trap "rm -rf '$tmpdir'" EXIT

readonly outdir="${BASH_SOURCE%/*}"
readonly defaultBranch='master'

export GIT_AUTHOR_NAME='testauthor'
export GIT_AUTHOR_EMAIL='testauthor@cmake.org'
export GIT_COMMITTER_NAME='testauthor'
export GIT_COMMITTER_EMAIL='testauthor@cmake.org'
export GIT_CONFIG_NOSYSTEM=1
export GIT_CONFIG_GLOBAL="$tmpdir/.gitconfig"
git config --global protocol.file.allow always

(
cd "$tmpdir"

git --bare init -b "$defaultBranch" gitrepo.git
rm -f gitrepo.git/hooks/*.sample

mkdir gitrepo
cd gitrepo
git init -b "$defaultBranch"
echo 'cmake_minimum_required(VERSION 3.31)
project(Example NONE)
add_custom_target(example ALL
  COMMAND ${CMAKE_COMMAND} -E copy ${CMAKE_CURRENT_SOURCE_DIR}/example.cmake.in example.cmake
)' >CMakeLists.txt
echo 'message(STATUS example)' >example.cmake.in
git add CMakeLists.txt example.cmake.in
git commit -m "Initial import into repo."
git push ../gitrepo.git "$defaultBranch"
cd ../gitrepo.git
git gc --prune
cd ..
ln -s gitrepo.git GIT

git --bare init -b "$defaultBranch" gitrepo-sub.git
rm -f gitrepo-sub.git/hooks/*.sample

mkdir gitrepo-sub
cd gitrepo-sub
git init -b "$defaultBranch"
echo 'cmake_minimum_required(VERSION 3.31)
project(ExampleWithSub NONE)
file(STRINGS "${CMAKE_SOURCE_DIR}/.git/config" git_config_strings
     REGEX "^\\[submodule")
foreach(submodule m1 m2)
  option(WITH_${submodule} "Enable ${submodule}" OFF)
  set(${submodule}_INITED FALSE)
  set(${submodule}_UPDATED FALSE)
  foreach(s ${git_config_strings})
    if("${s}" MATCHES "${submodule}")
      set(${submodule}_INITED TRUE)
    endif()
  endforeach()
  if(EXISTS "${CMAKE_SOURCE_DIR}/m/${submodule}/CMakeLists.txt")
    set(${submodule}_UPDATED TRUE)
  endif()
  if(WITH_${submodule})
    if(NOT ${submodule}_INITED)
      message(FATAL_ERROR "${submodule} not inited")
    elseif(NOT ${submodule}_UPDATED)
      message(FATAL_ERROR "${submodule} not updated")
    endif()
  else()
    if(${submodule}_INITED)
      message(FATAL_ERROR "${submodule} inited")
    elseif(${submodule}_UPDATED)
      message(FATAL_ERROR "${submodule} updated")
    endif()
  endif()
endforeach()' >CMakeLists.txt
git add CMakeLists.txt
git submodule add -- ../GIT m/m1
git submodule add -- ../GIT m/m2
git submodule add -- ../GIT m/m3
git commit -m "Initial import into repo."
git push ../gitrepo-sub.git "$defaultBranch"
cd ../gitrepo-sub.git
git gc --prune
cd ..
ln -s gitrepo-sub.git GIT-with-submodules

git --bare init -b "$defaultBranch" gitrepo-sub-rec.git
rm -f gitrepo-sub-rec.git/hooks/*.sample

mkdir gitrepo-sub-rec
cd gitrepo-sub-rec
git init -b "$defaultBranch"
echo 'cmake_minimum_required(VERSION 3.19)
project(ExampleWithRecSub NONE)
set(top_submodule_dir "${CMAKE_SOURCE_DIR}/submodule")
if(NOT EXISTS "${top_submodule_dir}/CMakeLists.txt")
  message(FATAL_ERROR "Top submodule not updated")
endif()
option(WITH_RECURSIVE "Submodules are updated recursively" ON)
foreach(submodule m1 m2 m3)
  set(${submodule}_UPDATED FALSE)
  if(EXISTS "${top_submodule_dir}/m/${submodule}/CMakeLists.txt")
    set(${submodule}_UPDATED TRUE)
  endif()
  if(WITH_RECURSIVE)
    if(NOT ${submodule}_UPDATED)
      message(FATAL_ERROR "${submodule} not updated")
    endif()
  else()
    if(${submodule}_UPDATED)
      message(FATAL_ERROR "${submodule} updated")
    endif()
  endif()
endforeach()' >CMakeLists.txt
git add CMakeLists.txt
git submodule add -- ../GIT-with-submodules submodule
git commit -m "Initial import into repo."
git push ../gitrepo-sub-rec.git "$defaultBranch"
cd ../gitrepo-sub-rec.git
git gc --prune
cd ..
)

tar cvzf "$outdir/gitrepo.tgz" -C "$tmpdir" gitrepo.git
tar cvzf "$outdir/gitrepo-sub.tgz" -C "$tmpdir" gitrepo-sub.git
tar cvzf "$outdir/gitrepo-sub-rec.tgz" -C "$tmpdir" gitrepo-sub-rec.git

git_tag=$(cd "$tmpdir/gitrepo.git" ; git rev-parse HEAD)
sed -i "/generated by gitrepo.bash/ s/ [0-9a-f]\\+ / $git_tag /" "$outdir/CMakeLists.txt"
