import type { ReactNode } from 'react';
import Ink, { type RenderMetrics } from './ink.js';
export type RenderOptions = {
    /**
    Output stream where app will be rendered.

    @default process.stdout
    */
    stdout?: NodeJS.WriteStream;
    /**
    Input stream where app will listen for input.

    @default process.stdin
    */
    stdin?: NodeJS.ReadStream;
    /**
    Error stream.
    @default process.stderr
    */
    stderr?: NodeJS.WriteStream;
    /**
    If true, each update will be rendered as separate output, without replacing the previous one.

    @default false
    */
    debug?: boolean;
    /**
    Configure whether Ink should listen for Ctrl+C keyboard input and exit the app. This is needed in case `process.stdin` is in raw mode, because then Ctrl+C is ignored by default and the process is expected to handle it manually.

    @default true
    */
    exitOnCtrlC?: boolean;
    /**
    Patch console methods to ensure console output doesn't mix with Ink's output.

    @default true
    */
    patchConsole?: boolean;
    /**
    Runs the given callback after each render and re-render.
    */
    onRender?: (metrics: RenderMetrics) => void;
    /**
    Enable screen reader support. See https://github.com/vadimdemedes/ink/blob/master/readme.md#screen-reader-support

    @default process.env['INK_SCREEN_READER'] === 'true'
    */
    isScreenReaderEnabled?: boolean;
    /**
    Maximum frames per second for render updates.
    This controls how frequently the UI can update to prevent excessive re-rendering.
    Higher values allow more frequent updates but may impact performance.

    @default 30
    */
    maxFps?: number;
    /**
     * Render the app in an alternate screen buffer.
     * This is useful for fullscreen applications.
     * @default false
     */
    alternateBuffer?: boolean;
};
export type Instance = {
    /**
    Replace the previous root node with a new one or update props of the current root node.
    */
    rerender: Ink['render'];
    /**
    Manually unmount the whole Ink app.
    */
    unmount: Ink['unmount'];
    /**
    Returns a promise that resolves when the app is unmounted.
    */
    waitUntilExit: Ink['waitUntilExit'];
    cleanup: () => void;
    /**
    Clear output.
    */
    clear: () => void;
    /**
     * Manually recalculate layout.
     */
    recalculateLayout: Ink['recalculateLayout'];
};
/**
Mount a component and render the output.
*/
declare const render: (node: ReactNode, options?: NodeJS.WriteStream | RenderOptions) => Instance;
export default render;
