﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/eks/EKSRequest.h>
#include <aws/eks/EKS_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace EKS {
namespace Model {

/**
 */
class DeleteAddonRequest : public EKSRequest {
 public:
  AWS_EKS_API DeleteAddonRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DeleteAddon"; }

  AWS_EKS_API Aws::String SerializePayload() const override;

  AWS_EKS_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The name of your cluster.</p>
   */
  inline const Aws::String& GetClusterName() const { return m_clusterName; }
  inline bool ClusterNameHasBeenSet() const { return m_clusterNameHasBeenSet; }
  template <typename ClusterNameT = Aws::String>
  void SetClusterName(ClusterNameT&& value) {
    m_clusterNameHasBeenSet = true;
    m_clusterName = std::forward<ClusterNameT>(value);
  }
  template <typename ClusterNameT = Aws::String>
  DeleteAddonRequest& WithClusterName(ClusterNameT&& value) {
    SetClusterName(std::forward<ClusterNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the add-on. The name must match one of the names returned by <a
   * href="https://docs.aws.amazon.com/eks/latest/APIReference/API_ListAddons.html">
   * <code>ListAddons</code> </a>.</p>
   */
  inline const Aws::String& GetAddonName() const { return m_addonName; }
  inline bool AddonNameHasBeenSet() const { return m_addonNameHasBeenSet; }
  template <typename AddonNameT = Aws::String>
  void SetAddonName(AddonNameT&& value) {
    m_addonNameHasBeenSet = true;
    m_addonName = std::forward<AddonNameT>(value);
  }
  template <typename AddonNameT = Aws::String>
  DeleteAddonRequest& WithAddonName(AddonNameT&& value) {
    SetAddonName(std::forward<AddonNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifying this option preserves the add-on software on your cluster but
   * Amazon EKS stops managing any settings for the add-on. If an IAM account is
   * associated with the add-on, it isn't removed.</p>
   */
  inline bool GetPreserve() const { return m_preserve; }
  inline bool PreserveHasBeenSet() const { return m_preserveHasBeenSet; }
  inline void SetPreserve(bool value) {
    m_preserveHasBeenSet = true;
    m_preserve = value;
  }
  inline DeleteAddonRequest& WithPreserve(bool value) {
    SetPreserve(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_clusterName;

  Aws::String m_addonName;

  bool m_preserve{false};
  bool m_clusterNameHasBeenSet = false;
  bool m_addonNameHasBeenSet = false;
  bool m_preserveHasBeenSet = false;
};

}  // namespace Model
}  // namespace EKS
}  // namespace Aws
