"use strict";
/*********************************************************************
 * Copyright (c) 2020 Ericsson and others
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
exports.ForkedFile = void 0;
const assert = require("assert");
const child_process_1 = require("child_process");
const fs_1 = require("fs");
const file_1 = require("./file");
/**
 * Open and read a file from a subprocess (mode `r+` only).
 *
 * This is useful when opening a ptmx/pts pair at the same time.
 * When both files are opened by the same process, closing does not correctly release
 * the read callbacks, leaving node hanging at exit.
 *
 * Instead, we open one of the two files in a subprocess in order to kill it once done,
 * which will properly release read callbacks for some reason.
 */
class ForkedFile {
    constructor(path) {
        this.path = path;
        // To write to the file, we'll write to stdin.
        // To read from the file, we'll read from stdout.
        this._fork = (0, child_process_1.spawn)(process.execPath, [...process.execArgv, __filename, path], {
            stdio: ['pipe', 'pipe', 'inherit'],
        });
    }
    get reader() {
        if (!this._fork.stdout) {
            throw new Error('Forked process missing stdout');
        }
        return this._fork.stdout;
    }
    get writer() {
        if (!this._fork.stdin) {
            throw new Error('Forked process missing stdin');
        }
        return this._fork.stdin;
    }
    destroy() {
        if (this._fork.exitCode === null && this._fork.signalCode === null) {
            this._fork.kill();
        }
    }
}
exports.ForkedFile = ForkedFile;
const [, script, path] = process.argv;
// Check if we are forked:
if (script === __filename) {
    assert(typeof path === 'string', 'argv[2] must be a string');
    const file = new file_1.File((0, fs_1.openSync)(path, 'r+'));
    process.stdin.pipe(file.writer);
    file.reader.pipe(process.stdout);
}
//# sourceMappingURL=forked-file.js.map