"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getFunctionName = exports.getBaseClassDependencyCount = exports.getDependencies = void 0;
const inject_1 = require("../annotation/inject");
const ERROR_MSGS = require("../constants/error_msgs");
const literal_types_1 = require("../constants/literal_types");
const METADATA_KEY = require("../constants/metadata_keys");
const serialization_1 = require("../utils/serialization");
Object.defineProperty(exports, "getFunctionName", { enumerable: true, get: function () { return serialization_1.getFunctionName; } });
const target_1 = require("./target");
function getDependencies(metadataReader, func) {
    const constructorName = serialization_1.getFunctionName(func);
    const targets = getTargets(metadataReader, constructorName, func, false);
    return targets;
}
exports.getDependencies = getDependencies;
function getTargets(metadataReader, constructorName, func, isBaseClass) {
    const metadata = metadataReader.getConstructorMetadata(func);
    const serviceIdentifiers = metadata.compilerGeneratedMetadata;
    if (serviceIdentifiers === undefined) {
        const msg = `${ERROR_MSGS.MISSING_INJECTABLE_ANNOTATION} ${constructorName}.`;
        throw new Error(msg);
    }
    const constructorArgsMetadata = metadata.userGeneratedMetadata;
    const keys = Object.keys(constructorArgsMetadata);
    const hasUserDeclaredUnknownInjections = (func.length === 0 && keys.length > 0);
    const hasOptionalParameters = keys.length > func.length;
    const iterations = (hasUserDeclaredUnknownInjections || hasOptionalParameters) ? keys.length : func.length;
    const constructorTargets = getConstructorArgsAsTargets(isBaseClass, constructorName, serviceIdentifiers, constructorArgsMetadata, iterations);
    const propertyTargets = getClassPropsAsTargets(metadataReader, func);
    const targets = [
        ...constructorTargets,
        ...propertyTargets
    ];
    return targets;
}
function getConstructorArgsAsTarget(index, isBaseClass, constructorName, serviceIdentifiers, constructorArgsMetadata) {
    const targetMetadata = constructorArgsMetadata[index.toString()] || [];
    const metadata = formatTargetMetadata(targetMetadata);
    const isManaged = metadata.unmanaged !== true;
    let serviceIdentifier = serviceIdentifiers[index];
    const injectIdentifier = (metadata.inject || metadata.multiInject);
    serviceIdentifier = (injectIdentifier) ? (injectIdentifier) : serviceIdentifier;
    if (serviceIdentifier instanceof inject_1.LazyServiceIdentifer) {
        serviceIdentifier = serviceIdentifier.unwrap();
    }
    if (isManaged) {
        const isObject = serviceIdentifier === Object;
        const isFunction = serviceIdentifier === Function;
        const isUndefined = serviceIdentifier === undefined;
        const isUnknownType = (isObject || isFunction || isUndefined);
        if (!isBaseClass && isUnknownType) {
            const msg = `${ERROR_MSGS.MISSING_INJECT_ANNOTATION} argument ${index} in class ${constructorName}.`;
            throw new Error(msg);
        }
        const target = new target_1.Target(literal_types_1.TargetTypeEnum.ConstructorArgument, metadata.targetName, serviceIdentifier);
        target.metadata = targetMetadata;
        return target;
    }
    return null;
}
function getConstructorArgsAsTargets(isBaseClass, constructorName, serviceIdentifiers, constructorArgsMetadata, iterations) {
    const targets = [];
    for (let i = 0; i < iterations; i++) {
        const index = i;
        const target = getConstructorArgsAsTarget(index, isBaseClass, constructorName, serviceIdentifiers, constructorArgsMetadata);
        if (target !== null) {
            targets.push(target);
        }
    }
    return targets;
}
function getClassPropsAsTargets(metadataReader, constructorFunc) {
    const classPropsMetadata = metadataReader.getPropertiesMetadata(constructorFunc);
    let targets = [];
    const keys = Object.keys(classPropsMetadata);
    for (const key of keys) {
        const targetMetadata = classPropsMetadata[key];
        const metadata = formatTargetMetadata(classPropsMetadata[key]);
        const targetName = metadata.targetName || key;
        const serviceIdentifier = (metadata.inject || metadata.multiInject);
        const target = new target_1.Target(literal_types_1.TargetTypeEnum.ClassProperty, targetName, serviceIdentifier);
        target.metadata = targetMetadata;
        targets.push(target);
    }
    const baseConstructor = Object.getPrototypeOf(constructorFunc.prototype).constructor;
    if (baseConstructor !== Object) {
        const baseTargets = getClassPropsAsTargets(metadataReader, baseConstructor);
        targets = [
            ...targets,
            ...baseTargets
        ];
    }
    return targets;
}
function getBaseClassDependencyCount(metadataReader, func) {
    const baseConstructor = Object.getPrototypeOf(func.prototype).constructor;
    if (baseConstructor !== Object) {
        const baseConstructorName = serialization_1.getFunctionName(baseConstructor);
        const targets = getTargets(metadataReader, baseConstructorName, baseConstructor, true);
        const metadata = targets.map((t) => t.metadata.filter((m) => m.key === METADATA_KEY.UNMANAGED_TAG));
        const unmanagedCount = [].concat.apply([], metadata).length;
        const dependencyCount = targets.length - unmanagedCount;
        if (dependencyCount > 0) {
            return dependencyCount;
        }
        else {
            return getBaseClassDependencyCount(metadataReader, baseConstructor);
        }
    }
    else {
        return 0;
    }
}
exports.getBaseClassDependencyCount = getBaseClassDependencyCount;
function formatTargetMetadata(targetMetadata) {
    const targetMetadataMap = {};
    targetMetadata.forEach((m) => {
        targetMetadataMap[m.key.toString()] = m.value;
    });
    return {
        inject: targetMetadataMap[METADATA_KEY.INJECT_TAG],
        multiInject: targetMetadataMap[METADATA_KEY.MULTI_INJECT_TAG],
        targetName: targetMetadataMap[METADATA_KEY.NAME_TAG],
        unmanaged: targetMetadataMap[METADATA_KEY.UNMANAGED_TAG]
    };
}
//# sourceMappingURL=reflection_utils.js.map