"use strict";
/********************************************************************************
 * Copyright (C) 2018 TypeFox and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var WebSocketConnectionProvider_1;
Object.defineProperty(exports, "__esModule", { value: true });
exports.WebSocketConnectionProvider = exports.DEFAULT_HTTP_FALLBACK_OPTIONS = exports.HttpFallbackOptions = void 0;
const inversify_1 = require("inversify");
const common_1 = require("../../common");
const web_socket_channel_1 = require("../../common/messaging/web-socket-channel");
const endpoint_1 = require("../endpoint");
const reconnecting_websocket_1 = require("reconnecting-websocket");
const abstract_connection_provider_1 = require("../../common/messaging/abstract-connection-provider");
const uuid_1 = require("uuid");
inversify_1.decorate(inversify_1.injectable(), common_1.JsonRpcProxyFactory);
inversify_1.decorate(inversify_1.unmanaged(), common_1.JsonRpcProxyFactory, 0);
exports.HttpFallbackOptions = Symbol('HttpFallbackOptions');
exports.DEFAULT_HTTP_FALLBACK_OPTIONS = {
    allowed: true,
    maxAttempts: 2,
    errorTimeout: 5000,
    pollingTimeout: 5000,
    requestTimeout: 0
};
let WebSocketConnectionProvider = WebSocketConnectionProvider_1 = class WebSocketConnectionProvider extends abstract_connection_provider_1.AbstractConnectionProvider {
    constructor() {
        super();
        this.onSocketDidOpenEmitter = new common_1.Emitter();
        this.onSocketDidOpen = this.onSocketDidOpenEmitter.event;
        this.onSocketDidCloseEmitter = new common_1.Emitter();
        this.onSocketDidClose = this.onSocketDidCloseEmitter.event;
        this.onHttpFallbackDidActivateEmitter = new common_1.Emitter();
        this.onHttpFallbackDidActivate = this.onHttpFallbackDidActivateEmitter.event;
        this.useHttpFallback = false;
        this.websocketErrorCounter = 0;
        this.httpFallbackId = uuid_1.v4();
        this.httpFallbackDisconnected = true;
        const url = this.createWebSocketUrl(web_socket_channel_1.WebSocketChannel.wsPath);
        const socket = this.createWebSocket(url);
        socket.onerror = event => this.handleSocketError(event);
        socket.onopen = () => {
            this.fireSocketDidOpen();
        };
        socket.onclose = ({ code, reason }) => {
            for (const channel of [...this.channels.values()]) {
                channel.close(code, reason);
            }
            this.fireSocketDidClose();
        };
        socket.onmessage = ({ data }) => {
            this.websocketErrorCounter = 0;
            this.handleIncomingRawMessage(data);
        };
        this.socket = socket;
        window.addEventListener('offline', () => this.tryReconnect());
        window.addEventListener('online', () => this.tryReconnect());
    }
    static createProxy(container, path, arg) {
        return container.get(WebSocketConnectionProvider_1).createProxy(path, arg);
    }
    handleSocketError(event) {
        var _a, _b;
        this.websocketErrorCounter += 1;
        if (((_a = this.httpFallbackOptions) === null || _a === void 0 ? void 0 : _a.allowed) && this.websocketErrorCounter >= ((_b = this.httpFallbackOptions) === null || _b === void 0 ? void 0 : _b.maxAttempts)) {
            this.useHttpFallback = true;
            this.socket.close();
            const httpUrl = this.createHttpWebSocketUrl(web_socket_channel_1.WebSocketChannel.wsPath);
            this.onHttpFallbackDidActivateEmitter.fire(undefined);
            this.doLongPolling(httpUrl);
            this.messageService().warn('Could not establish a websocket connection. The application will be using the HTTP fallback mode. This may affect performance and the behavior of some features.');
        }
        console.error(event);
    }
    async doLongPolling(url) {
        var _a, _b, _c;
        let timeoutDuration = ((_a = this.httpFallbackOptions) === null || _a === void 0 ? void 0 : _a.requestTimeout) || 0;
        const controller = new AbortController();
        const pollingId = window.setTimeout(() => controller.abort(), (_b = this.httpFallbackOptions) === null || _b === void 0 ? void 0 : _b.pollingTimeout);
        try {
            const response = await fetch(url, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                signal: controller.signal,
                keepalive: true,
                body: JSON.stringify({ id: this.httpFallbackId, polling: true })
            });
            if (response.status === 200) {
                window.clearTimeout(pollingId);
                if (this.httpFallbackDisconnected) {
                    this.fireSocketDidOpen();
                }
                const json = await response.json();
                if (Array.isArray(json)) {
                    for (const item of json) {
                        this.handleIncomingRawMessage(item);
                    }
                }
                else {
                    throw new Error('Received invalid long polling response.');
                }
            }
            else {
                timeoutDuration = ((_c = this.httpFallbackOptions) === null || _c === void 0 ? void 0 : _c.errorTimeout) || 0;
                this.httpFallbackDisconnected = true;
                this.fireSocketDidClose();
                throw new Error('Response has error code: ' + response.status);
            }
        }
        catch (e) {
            console.error('Error occurred during long polling', e);
        }
        setTimeout(() => this.doLongPolling(url), timeoutDuration);
    }
    openChannel(path, handler, options) {
        if (this.useHttpFallback || this.socket.readyState === WebSocket.OPEN) {
            super.openChannel(path, handler, options);
        }
        else {
            const openChannel = () => {
                this.socket.removeEventListener('open', openChannel);
                this.openChannel(path, handler, options);
            };
            this.socket.addEventListener('open', openChannel);
            this.onHttpFallbackDidActivate(openChannel);
        }
    }
    createChannel(id) {
        const httpUrl = this.createHttpWebSocketUrl(web_socket_channel_1.WebSocketChannel.wsPath);
        return new web_socket_channel_1.WebSocketChannel(id, content => {
            if (this.useHttpFallback) {
                fetch(httpUrl, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({ id: this.httpFallbackId, content })
                });
            }
            else if (this.socket.readyState < WebSocket.CLOSING) {
                this.socket.send(content);
            }
        });
    }
    /**
     * Creates a websocket URL to the current location
     */
    createWebSocketUrl(path) {
        const endpoint = new endpoint_1.Endpoint({ path });
        return endpoint.getWebSocketUrl().toString();
    }
    createHttpWebSocketUrl(path) {
        const endpoint = new endpoint_1.Endpoint({ path });
        return endpoint.getRestUrl().toString();
    }
    /**
     * Creates a web socket for the given url
     */
    createWebSocket(url) {
        return new reconnecting_websocket_1.default(url, undefined, {
            maxReconnectionDelay: 10000,
            minReconnectionDelay: 1000,
            reconnectionDelayGrowFactor: 1.3,
            connectionTimeout: 10000,
            maxRetries: Infinity,
            debug: false
        });
    }
    fireSocketDidOpen() {
        this.onSocketDidOpenEmitter.fire(undefined);
    }
    fireSocketDidClose() {
        this.onSocketDidCloseEmitter.fire(undefined);
    }
    tryReconnect() {
        if (!this.useHttpFallback && this.socket.readyState !== WebSocket.CONNECTING) {
            this.socket.reconnect();
        }
    }
};
__decorate([
    inversify_1.inject(common_1.MessageServiceFactory),
    __metadata("design:type", Function)
], WebSocketConnectionProvider.prototype, "messageService", void 0);
__decorate([
    inversify_1.inject(exports.HttpFallbackOptions),
    inversify_1.optional(),
    __metadata("design:type", Object)
], WebSocketConnectionProvider.prototype, "httpFallbackOptions", void 0);
WebSocketConnectionProvider = WebSocketConnectionProvider_1 = __decorate([
    inversify_1.injectable(),
    __metadata("design:paramtypes", [])
], WebSocketConnectionProvider);
exports.WebSocketConnectionProvider = WebSocketConnectionProvider;
//# sourceMappingURL=ws-connection-provider.js.map