/*******************************************************************************
 * Copyright (c) 2024 École Polytechnique de Montréal
 *
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse Public License 2.0 which
 * accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *******************************************************************************/
package org.eclipse.tracecompass.incubator.rocm.core.exatracer.trace;

import java.util.Collection;
import java.util.List;

import org.eclipse.jdt.annotation.NonNull;
import org.eclipse.jdt.annotation.Nullable;
import org.eclipse.tracecompass.incubator.gpu.core.trace.IGpuTraceEventLayout;
import org.eclipse.tracecompass.tmf.core.event.ITmfEvent;

import com.google.common.collect.ImmutableList;

/**
 * Event layout for events in traces generated by exatracer
 *
 * @author Arnaud Fiorini
 */
public class RocmExatracerTraceEventLayout implements IGpuTraceEventLayout {

    /** Suffix for API events */
    public static final String BEGIN_SUFFIX = "_entry"; //$NON-NLS-1$\
    /** HIP string */
    public static final String HIP = "hip"; //$NON-NLS-1$
    /** HSA string */
    public static final String HSA = "hsa"; //$NON-NLS-1$

    private static @Nullable RocmExatracerTraceEventLayout INSTANCE;
    private static List<IApiEventLayout> fApiLayouts = ImmutableList.of(new HsaApiEventLayout(), new HipApiEventLayout());

    /**
     * The instance of this event layout
     * <p>
     * This object is completely immutable, so no need to create additional
     * instances via the constructor.
     *
     * @return the instance
     */
    public static synchronized RocmExatracerTraceEventLayout getInstance() {
        RocmExatracerTraceEventLayout instance = INSTANCE;
        if (instance == null) {
            instance = new RocmExatracerTraceEventLayout();
            INSTANCE = instance;
        }
        return instance;
    }

    @Override
    public @NonNull Collection<IApiEventLayout> getApiLayouts() {
        return fApiLayouts;
    }

    @Override
    public @NonNull IApiEventLayout getCorrespondingApiLayout(ITmfEvent event) {
        if (event.getName().startsWith(HSA)) {
            return fApiLayouts.get(0);
        }
        return fApiLayouts.get(1);
    }

    @Override
    public boolean isMemcpyBegin(ITmfEvent event) {
        return false;
    }

    @Override
    public boolean isLaunchBegin(ITmfEvent event) {
        return false;
    }

    @Override
    public boolean isApiEvent(ITmfEvent event) {
        return event.getName().startsWith(HSA) || event.getName().startsWith(HIP);
    }

    @Override
    public @NonNull String fieldThreadId() {
        return "lttng_thread_id"; //$NON-NLS-1$
    }

    @Override
    public @NonNull String fieldDuration() {
        return ""; //$NON-NLS-1$
    }

    /**
     * Event layout for HSA api events
     */
    public static class HsaApiEventLayout extends RocmApiEventLayout {
        @Override
        public String getApiName() {
            return "HSA"; //$NON-NLS-1$
        }
    }

    /**
     * Event layout for HIP api events
     */
    public static class HipApiEventLayout extends RocmApiEventLayout {
        @Override
        public String getApiName() {
            return "HIP"; //$NON-NLS-1$
        }
    }
}
